/* conversion.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <glib.h>       
#include "conversion.h"

/* Converts an MMD2 module to a song */
struct song *MMD2_convert(struct MMD2 *mmd)
{
  struct song *song;
  int i, j, k, l;
  unsigned char commandconversion[256];

  if (mmd == NULL) {
    fprintf(stderr, "MMD2_convert() called with null MMD2\n");
    return NULL;
  }
  
  /* Create a table for converting MMD commands; values need special handling */
  for (i = 0; i < 256; i++)
    commandconversion[i] = i;
  commandconversion[0] = COMMAND_PREVIOUS_COMMAND_VALUE;
  commandconversion[1] = COMMAND_PITCH_WHEEL;
  commandconversion[2] = COMMAND_PITCH_WHEEL;
  commandconversion[3] = COMMAND_PITCH_WHEEL;
  commandconversion[4] = COMMAND_MIDI_CONTROLLERS + 1;
  commandconversion[5] = COMMAND_NOT_DEFINED;
  commandconversion[6] = COMMAND_NOT_DEFINED;
  commandconversion[7] = COMMAND_NOT_DEFINED;
  commandconversion[8] = COMMAND_HOLD;
  commandconversion[9] = COMMAND_TPL;
  commandconversion[10] = COMMAND_VELOCITY;
  commandconversion[11] = COMMAND_PLAYSEQ_POSITION;
  commandconversion[12] = COMMAND_VELOCITY;
  commandconversion[13] = COMMAND_CHANNEL_PRESSURE;
  commandconversion[14] = COMMAND_MIDI_CONTROLLERS + 10;
  commandconversion[15] = COMMAND_TEMPO;
  commandconversion[16] = COMMAND_SEND_MESSAGE;
  commandconversion[17] = COMMAND_NOT_DEFINED;
  commandconversion[18] = COMMAND_NOT_DEFINED;
  commandconversion[19] = COMMAND_NOT_DEFINED;
  commandconversion[20] = COMMAND_NOT_DEFINED;
  commandconversion[21] = COMMAND_NOT_DEFINED;
  commandconversion[22] = COMMAND_MIDI_CONTROLLERS + 7;
  commandconversion[23] = COMMAND_NOT_DEFINED;
  commandconversion[24] = COMMAND_NOT_DEFINED;
  commandconversion[25] = COMMAND_NOT_DEFINED;
  commandconversion[26] = COMMAND_NOT_DEFINED;
  commandconversion[27] = COMMAND_NOT_DEFINED;
  commandconversion[28] = COMMAND_NOT_DEFINED;
  commandconversion[29] = COMMAND_END_BLOCK;
  commandconversion[30] = COMMAND_NOT_DEFINED;
  commandconversion[31] = COMMAND_DELAY;

  song = (struct song *)calloc(1, sizeof(struct song));
  if (mmd->expdata != NULL && mmd->expdata->songname != NULL) {
    gchar *name=g_convert(mmd->expdata->songname, strlen(mmd->expdata->songname), "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
    song->name = strdup(name);
    g_free(name);
  } else
    song->name = "Untitled";
  if (mmd->song) {
    /* Tempo */
    song->tempo = mmd->song->deftempo;
    song->ticksperline = mmd->song->tempo2;
    /* Sections */
    song->numsections = mmd->song->songlen;
    song->sections = (unsigned int *)calloc(song->numsections, sizeof(unsigned int));
    for (i = 0; i < song->numsections; i++)
      song->sections[i] = mmd->song->sectiontable[i];
    /* Playing sequences */
    song->numplayseqs = mmd->song->numpseqs;
    song->playseqs = (struct playseq **)calloc(song->numplayseqs, sizeof(struct playseq *));
    for (i = 0; i < song->numplayseqs; i++) {
      struct playseq *playseq = (struct playseq *)calloc(1, sizeof(struct playseq));
      struct PlaySeq *PlaySeq = mmd->song->playseqtable[i];
      song->playseqs[i] = playseq;
      if (PlaySeq->name != NULL) {
	gchar *name=g_convert(PlaySeq->name, strlen(PlaySeq->name), "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
	playseq->name = strdup(name);
	g_free(name);
      }
      playseq->length = PlaySeq->length;
      playseq->blocknumbers = (unsigned int *)calloc(playseq->length, sizeof(unsigned int));
      for (j = 0; j < song->playseqs[i]->length; j++)
	playseq->blocknumbers[j] = PlaySeq->seq[j];
    }
    /* Blocks */
    song->numblocks = mmd->song->numblocks;
    song->blocks = (struct block **)calloc(song->numblocks, sizeof(struct block *));
    for (i = 0; i < song->numblocks; i++) {
      struct block *block;
      struct MMD1Block *MMD1Block = mmd->blockarr[i];
      unsigned char *notes = (unsigned char *)MMD1Block + sizeof(struct MMD1Block);
      if (MMD1Block->info != NULL && MMD1Block->info->pagetable != NULL)
	block = block_alloc(MMD1Block->numtracks, MMD1Block->lines + 1, MMD1Block->info->pagetable->num_pages + 1);
      else
	block = block_alloc(MMD1Block->numtracks, MMD1Block->lines + 1, 1);

      song->blocks[i] = block;

      if (MMD1Block->info != NULL && MMD1Block->info->blockname != NULL) {
	gchar *name=g_convert(MMD1Block->info->blockname, strlen(MMD1Block->info->blockname), "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
	block->name = strdup(name);
	g_free(name);
      }

      for (j = 0; j < block->length; j++)
	for (k = 0; k < block->tracks; k++) {
	  block->notes[(j * block->tracks + k) * 2] = notes[(j * block->tracks + k) * 4];
	  block->notes[(j * block->tracks + k) * 2 + 1] = notes[(j * block->tracks + k) * 4 + 1];
	  block->commands[(j * block->tracks + k) * 2] = commandconversion[notes[(j * block->tracks + k) * 4 + 2]];
	  block->commands[(j * block->tracks + k) * 2 + 1] = notes[(j * block->tracks + k) * 4 + 3];

	  /* Command values may need special handling */
	  MMD2_commands_convert(&(block->commands[(j * block->tracks + k) * 2]), mmd->expdata->mmdcmd3x);
	}

      if (MMD1Block->info != NULL && MMD1Block->info->pagetable!=NULL) {
	for (j = 0; j < MMD1Block->info->pagetable->num_pages; j++) {
	  for (k = 0; k < (MMD1Block->lines + 1); k++)
	    for (l = 0; l < MMD1Block->numtracks; l++) {
              unsigned char *commands=(unsigned char *)MMD1Block->info->pagetable->page[j];

	      block->commands[(j + 1) * 2 * block->tracks * block->length + (k * block->tracks + l) * 2] = commandconversion[commands[(k * MMD1Block->numtracks + l) * 2]];
	      block->commands[(j + 1) * 2 * block->tracks * block->length + (k * block->tracks + l) * 2 + 1] = commands[(k * MMD1Block->numtracks + l) * 2 + 1];

	      /* Command values may need special handling */
	      MMD2_commands_convert(&(block->commands[(j + 1) * 2 * block->tracks * block->length + (k * block->tracks + l) * 2]), mmd->expdata->mmdcmd3x);
	    }
	}
      }
    }
    /* Instruments */
    song->numinstruments = mmd->song->numsamples;
    song->instruments = (struct instrument **)calloc(song->numinstruments, sizeof(struct instrument *));
    for (i = 0; i < song->numinstruments; i++) {
      song->instruments[i] = instrument_alloc();
      if (mmd->expdata) {
	gchar *name=g_convert(mmd->expdata->iinfo[i].name, strlen(mmd->expdata->iinfo[i].name), "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
	song->instruments[i]->name = strdup(name);
	song->instruments[i]->hold = mmd->expdata->exp_smp[i].hold;
	g_free(name);
      }
      song->instruments[i]->midichannel = mmd->song->sample[i].midich - 1;
      song->instruments[i]->midipreset = mmd->song->sample[i].midipreset;
      song->instruments[i]->transpose = mmd->song->sample[i].strans;
      if (mmd->song->sample[i].svol == 64)
	song->instruments[i]->defaultvelocity = 127;
      else
	song->instruments[i]->defaultvelocity = mmd->song->sample[i].svol * 2;
    }

    /* Track volumes */
    song->maxtracks = mmd->song->numtracks;
    song->trackvolumes = (unsigned int *)calloc(song->maxtracks, sizeof(unsigned int));
    for (i = 0; i < song->maxtracks; i++) {
      if (mmd->song->trackvols[i] < 64)
	song->trackvolumes[i] = mmd->song->trackvols[i] * 2;
      else
	song->trackvolumes[i] = 127;
    }
    if (mmd->song->mastervol < 64)
      song->mastervolume = mmd->song->mastervol * 2;
    else
      song->mastervolume = 127;

    /* SysEX dumps */
    if (mmd->expdata && mmd->expdata->dumps) {
      /* Is this really the only way to do this? I know, this is horrible but
       * this is what the lame MMD spec says */
      struct MMDDump **dumps = (struct MMDDump **)((char *)mmd->expdata->dumps + sizeof(struct MMDDumpData));

      song->nummessages = mmd->expdata->dumps->numdumps;
      song->messages = (struct message **)calloc(song->nummessages, sizeof(struct message *));
      for (i = 0; i < song->nummessages; i++) {
	song->messages[i] = (struct message *)calloc(1, sizeof(struct message));
	song->messages[i]->length = dumps[i]->length;
	song->messages[i]->data = (unsigned char *)calloc(song->messages[i]->length, sizeof(unsigned char));
	memcpy(song->messages[i]->data, dumps[i]->data, song->messages[i]->length);

	if (dumps[i]->ext_len >= 20) {
	  gchar *name=g_convert(dumps[i]->name, strlen(dumps[i]->name), "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
	  song->messages[i]->name = strdup(name);
	  g_free(name);
	}
      }
    }
  } else {
    /* Nothing can be done in this case... should never happen though */
    free(song);
    song = song_alloc();
  }

  return song;
}

/* Converts MMD command values at given location */
void MMD2_commands_convert(unsigned char *command, struct MMDMIDICmd3x *mmdcmd3x)
{
  switch (command[0]) {
  case COMMAND_PREVIOUS_COMMAND_VALUE:
    if (command[1] != 0) {
      command[0] = 0x80;
      command[1] &= 0x7f;
    }
    break;
  case COMMAND_VELOCITY:
    /* Fix volume range to 0..127 */
    if (command[1] == 64)
      command[1] = 127;
    else
      command[1] *= 2;
    break;
  case COMMAND_DELAY:
    /* Command 1F in MMD is both note delay and retrigger */
    if (command[1] > 15)
      command[1] = command[1] >> 4;
    else
      command[0] = COMMAND_RETRIGGER;
    break;
  case COMMAND_TEMPO:
    /* Values >0xf0 have special meanings in OctaMED */
    if (command[1] > 0xf0) {
      switch (command[1]) {
      case 0xf1:
	/* Makes a single note play twice */
	command[0] = COMMAND_RETRIGGER;
	command[1] = 3;
	break;
      case 0xf2:
	/* Delays the start of a note by half a line */
	command[0] = COMMAND_DELAY;
	command[1] = 3;
	break;
      case 0xf3:
	/* Works like 0FF1 except the note is played three times */
	command[0] = COMMAND_RETRIGGER;
	command[1] = 2;
	break;
      case 0xf4:
	/* Delays the note one-third of a line */
	command[0] = COMMAND_DELAY;
	command[1] = 2;
	break;
      case 0xf5:
	/* Delays the note two thirds of a line */
	command[0] = COMMAND_DELAY;
	command[1] = 4;
	break;
      case 0xfe:
	/* Stops the song playing */
	command[0] = COMMAND_TEMPO;
	command[1] = 0;
	break;
      case 0xff:
	/* Stops the note on the current track */
	command[0] = COMMAND_VELOCITY;
	command[1] = 0;
	break;
      default:
	command[0] = 0;
	command[1] = 0;
	break;
      }
    }
    break;
  case 0x31:
  case 0x32:
  case 0x33:
  case 0x34:
  case 0x35:
  case 0x36:
  case 0x37:
  case 0x38:
  case 0x39:
  case 0x3a:
  case 0x3b:
  case 0x3c:
  case 0x3d:
  case 0x3e:
  case 0x3f:
    if (mmdcmd3x->ctrlr_types[command[0] - 0x31] == MCS_TYPE_STD_MSB)
      command[0] = mmdcmd3x->ctrlr_numbers[command[0] - 0x31] + 0x80;
    break;
  default:
    break;
  }
}

/* Converts a song to an MMD2 module */
struct MMD2 *song_convert(struct song *song)
{
  struct MMD2 *mmd;
  int i, j, k, l;
  unsigned char commandconversion[256];
  
  if (song == NULL) {
    fprintf(stderr, "song_convert() called with null song\n");
    return NULL;
  }

  /* Create a table for converting song commands; values need special handling */
  for (i = 0; i < 256; i++)
    commandconversion[i] = i; 
  commandconversion[COMMAND_PITCH_WHEEL] = 0x03;
  commandconversion[COMMAND_END_BLOCK] = 0x1d;
  commandconversion[COMMAND_PLAYSEQ_POSITION] = 0x0b;
  commandconversion[COMMAND_SEND_MESSAGE] = 0x10;
  commandconversion[COMMAND_HOLD] = 0x08;
  commandconversion[COMMAND_RETRIGGER] = 0x1f;
  commandconversion[COMMAND_DELAY] = 0x20;
  commandconversion[COMMAND_VELOCITY] = 0x0c;
  commandconversion[COMMAND_CHANNEL_PRESSURE] = 0x0d;
  commandconversion[COMMAND_TPL] = 0x09;
  commandconversion[COMMAND_TEMPO] = 0x0f;

  mmd = (struct MMD2 *)calloc(1, sizeof(struct MMD2));
  mmd->id = ID_MMD2;
  mmd->song = (struct MMD2song *)(calloc(1, sizeof(struct MMD2song)));
  mmd->blockarr=(struct MMD1Block **)calloc(song->numblocks, sizeof(struct MMD1Block *));
  mmd->expdata = (struct MMD0exp *)calloc(1, sizeof(struct MMD0exp));

  /* Song name */
  if(song->name!=NULL) {
    gchar *name=g_convert(song->name, strlen(song->name), "ISO-8859-1", "UTF-8", NULL, NULL, NULL);
    mmd->expdata->songname=strdup(name);
    mmd->expdata->songnamelen=strlen(mmd->expdata->songname)+1;
    g_free(name);
  }
  /* Tempo */
  mmd->song->deftempo = song->tempo;
  mmd->song->flags2 = (4-1)|FLAG2_BPM;
  mmd->song->tempo2 = song->ticksperline;
  /* Sections */
  mmd->song->songlen = song->numsections;
  mmd->song->sectiontable = (unsigned short *)calloc(mmd->song->songlen, sizeof(unsigned char));
  for (i = 0; i < song->numsections; i++)
    mmd->song->sectiontable[i] = song->sections[i];
  /* Playing sequences */
  mmd->song->numpseqs = song->numplayseqs;
  mmd->song->playseqtable = (struct PlaySeq **)calloc(mmd->song->numpseqs, sizeof(struct PlaySeq *));
  for (i = 0; i < song->numplayseqs; i++) {
    struct playseq *playseq = song->playseqs[i];
    struct PlaySeq *PlaySeq = (struct PlaySeq *)calloc(1, sizeof(struct PlaySeq)+playseq->length*sizeof(unsigned short));
    mmd->song->playseqtable[i] = PlaySeq;
    if (playseq->name!=NULL) {
      gchar *name=g_convert(playseq->name, strlen(playseq->name), "ISO-8859-1", "UTF-8", NULL, NULL, NULL);
      strncpy(PlaySeq->name, name, 32);
      PlaySeq->name[31]=0;
      g_free(name);
    }
    PlaySeq->length = playseq->length;
    for (j = 0; j < mmd->song->playseqtable[i]->length; j++)
      PlaySeq->seq[j] = playseq->blocknumbers[j];
  }
  /* Blocks */
  mmd->song->numblocks = song->numblocks;
  mmd->blockarr = (struct MMD1Block **)calloc(mmd->song->numblocks, sizeof(struct MMD1Block *));
  for (i = 0; i < mmd->song->numblocks; i++) {
    struct block *block=song->blocks[i];
    struct MMD1Block *MMD1Block=(struct MMD1Block *)calloc(1, sizeof(struct MMD1Block)+4*block->tracks*block->length);
    unsigned char *notes=(unsigned char *)MMD1Block + sizeof(struct MMD1Block);
    MMD1Block->info=(struct BlockInfo *)calloc(1, sizeof(struct BlockInfo));
    if(block->commandpages>1) {
      MMD1Block->info->pagetable=(struct BlockCmdPageTable *)calloc(1, sizeof(struct BlockCmdPageTable)+(block->commandpages - 1)*sizeof(unsigned short *));
      MMD1Block->info->pagetable->num_pages=block->commandpages - 1;
    }
    
    mmd->blockarr[i] = MMD1Block;

    if (block->name != NULL) {
      gchar *name=g_convert(block->name, strlen(block->name), "ISO-8859-1", "UTF-8", NULL, NULL, NULL);
      MMD1Block->info->blockname = strdup(name);
      MMD1Block->info->blocknamelen=strlen(MMD1Block->info->blockname);
      g_free(name);
    }
    MMD1Block->numtracks = block->tracks;
    MMD1Block->lines = block->length - 1;

    for (j = 0; j < (MMD1Block->lines + 1); j++)
      for (k = 0; k < MMD1Block->numtracks; k++) {
        notes[(j * block->tracks + k) * 4] = block->notes[(j * block->tracks + k) * 2];
        notes[(j * block->tracks + k) * 4 + 1] = block->notes[(j * block->tracks + k) * 2 + 1];
        notes[(j * block->tracks + k) * 4 + 2] = commandconversion[block->commands[(j * block->tracks + k) * 2]];
        notes[(j * block->tracks + k) * 4 + 3] = block->commands[(j * block->tracks + k) * 2 + 1];
        /* Command values may need special handling */
        song_commands_convert(&(notes[(j * block->tracks + k) * 4 + 2]));
      }

    if(MMD1Block->info->pagetable!=NULL) {
      for (j = 0; j < MMD1Block->info->pagetable->num_pages; j++) {
        MMD1Block->info->pagetable->page[j] = (unsigned short *)calloc(MMD1Block->numtracks * (MMD1Block->lines + 1) * 2, sizeof(unsigned short));
        for (k = 0; k < (MMD1Block->lines + 1); k++)
          for (l = 0; l < MMD1Block->numtracks; l++) {
            unsigned char *commands=(unsigned char *)MMD1Block->info->pagetable->page[j];
            commands[(k * MMD1Block->numtracks + l) * 2] = commandconversion[block->commands[(j + 1) * 2 * block->tracks * block->length + (k * block->tracks + l) * 2]];
            commands[(k * MMD1Block->numtracks + l) * 2 + 1] = block->commands[(j + 1) * 2 * block->tracks * block->length + (k * block->tracks + l) * 2 + 1];
            /* Command values may need special handling */
            song_commands_convert(&(commands[(k * MMD1Block->numtracks + l) * 2]));
          }
      }
    }
  }
  /* Instruments */
  mmd->song->numsamples = song->numinstruments;
  mmd->expdata->s_ext_entries=mmd->song->numsamples;
  mmd->expdata->s_ext_entrsz=sizeof(struct InstrExt);
  mmd->expdata->i_ext_entries=mmd->song->numsamples;
  mmd->expdata->i_ext_entrsz=sizeof(struct MMDInstrInfo);
  mmd->expdata->exp_smp=(struct InstrExt *)calloc(mmd->expdata->s_ext_entries, sizeof(struct InstrExt));
  mmd->expdata->iinfo=(struct MMDInstrInfo *)calloc(mmd->expdata->i_ext_entries, sizeof(struct MMDInstrInfo));
  for (i = 0; i < mmd->song->numsamples; i++) {
    gchar *name=g_convert(song->instruments[i]->name, strlen(song->instruments[i]->name), "ISO-8859-1", "UTF-8", NULL, NULL, NULL);
    strncpy(mmd->expdata->iinfo[i].name, name, 40);
    mmd->expdata->iinfo[i].name[39]=0;
    mmd->expdata->exp_smp[i].hold = song->instruments[i]->hold;
    mmd->song->sample[i].midich = song->instruments[i]->midichannel + 1;
    mmd->song->sample[i].midipreset = song->instruments[i]->midipreset;
    mmd->song->sample[i].strans = song->instruments[i]->transpose;
    mmd->song->sample[i].svol = (song->instruments[i]->defaultvelocity + 1) / 2;
    g_free(name);
  }

  /* Track volumes */
  mmd->song->numtracks = song->maxtracks;
  mmd->song->trackvols = (unsigned char *)calloc(mmd->song->numtracks, sizeof(unsigned char));
  mmd->song->trackpans = (unsigned char *)calloc(mmd->song->numtracks, sizeof(unsigned char));
  for (i = 0; i < mmd->song->numtracks; i++)
    mmd->song->trackvols[i] = (song->trackvolumes[i] + 1) / 2;
  mmd->song->mastervol = (song->mastervolume + 1) / 2;

  /* SysEX dumps */
  mmd->expdata->dumps=(struct MMDDumpData *)calloc(1, sizeof(struct MMDDumpData)+song->nummessages*sizeof(struct MMDDump *));
  mmd->expdata->dumps->numdumps = song->nummessages;
  struct MMDDump **dumps = (struct MMDDump **)((char *)mmd->expdata->dumps + sizeof(struct MMDDumpData));
      
  for (i = 0; i < mmd->expdata->dumps->numdumps; i++) {
    dumps[i] = (struct MMDDump *)calloc(1, sizeof(struct MMDDump));
    dumps[i]->length = song->messages[i]->length;
    dumps[i]->data = (unsigned char *)calloc(dumps[i]->length, sizeof(unsigned char));
    memcpy(dumps[i]->data, song->messages[i]->data, dumps[i]->length);
      
    if(song->messages[i]->name!=NULL) {
      gchar *name=g_convert(song->messages[i]->name, strlen(song->messages[i]->name), "ISO-8859-1", "UTF-8", NULL, NULL, NULL);
      strncpy(dumps[i]->name, name, 20);
      dumps[i]->name[19]=0;
      dumps[i]->ext_len=strlen(name);
      g_free(name);
    }
  }

  return mmd;
}

/* Converts song command values at given location */
void song_commands_convert(unsigned char *command)
{
  switch (command[0]) {
  case 0x0c:
    /* Fix volume range to 0..64 */
    command[1]=(command[1]+1)/2;
    break;
  case 0x20:
    /* Delay */
    command[0] = 0x1f;
    command[1]<<=4;
    break;  
  default:
    break;
  }
}
