/* keyboard.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include <glade/glade.h>

#include "gui.h"
#include "callbacks.h"
#include "editor.h"
#include "song.h"
#include "player.h"
#include "trackerwidget.h"

static guint chord_previous_keyval=0;

gboolean key_press_event(GtkWidget *widget, GdkEventKey *event,
			 gpointer user_data) {
  return keyboard_event(widget, event, TRUE, (struct gui *)user_data);
}

gboolean key_release_event(GtkWidget *widget, GdkEventKey *event,
			   gpointer user_data) {
  return keyboard_event(widget, event, FALSE, (struct gui *)user_data);
}

gboolean keyboard_event(GtkWidget *widget, GdkEventKey *event,
			gboolean pressed, struct gui *gui) {
  struct editor *editor=gui->editor;
  struct song *song=editor->song;

  gboolean handled=FALSE;

  if(!(GTK_IS_ENTRY(GTK_WINDOW(widget)->focus_widget))) {
    Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");
    int shift=event->state & GDK_SHIFT_MASK;
    int ctrl=event->state & GDK_CONTROL_MASK;
    int alt=event->state & GDK_MOD1_MASK;

    if(pressed) {
      if(ctrl) {
	switch (event->keyval) {
	case 'b':
	  /* CTRL-B: Selection mode on/off */
	  if(tracker_is_in_selection_mode(tracker))
	    tracker_mark_selection(tracker, FALSE);
	  else
	    tracker_mark_selection(tracker, TRUE);
	  handled=TRUE;
	  break;
	case 'k':
	  /* CTRL-K: Clear until the end of the track */
	  block_clear(song->blocks[editor->block],
		      tracker->cursor_ch, tracker->patpos,
		      tracker->cursor_ch,
		      song->blocks[editor->block]->length-1);
	  tracker_redraw(tracker);
	  handled=TRUE;
	  break;
	case GDK_Left:
	  /* CTRL-Left: Previous instrument */
	  if(editor->instrument>0) {
	    editor->instrument--;

	    /* Make sure the instrument exists */
	    song_check_instrument(song, editor->instrument);
	    gui_instrument_refresh(gui, TRUE);
	  }
	  handled=TRUE;
	  break;
	case GDK_Right:
	  /* CTRL-Right: Previous instrument */
	  editor->instrument++;

	  /* Make sure the instrument exists */
	  song_check_instrument(song, editor->instrument);
	  gui_instrument_refresh(gui, TRUE);

	  handled=TRUE;
	  break;
	case GDK_ISO_Left_Tab:
	  /* CTRL-Shift-Tab: Next command page */
	  editor_set_commandpage(editor, editor->cmdpage+1);
	  gui_info_refresh(gui);  
	  handled=TRUE;
	  break;
	case GDK_Delete:
	  {
	    int i;

	    /* Delete command and refresh */
	    for(i=0; i<song->blocks[editor->block]->commandpages; i++)
	      block_set_command_full(song->blocks[editor->block], editor->line,
				     tracker->cursor_ch, i, 0, 0);
	    tracker_redraw_row(tracker, editor->line);
	    
	    if(editor->chord) {
	      /* Chord mode: Go to next channel */
	      tracker_step_cursor_channel(tracker, 1);
	      if(event->keyval!=chord_previous_keyval) {
		editor->chordstatus++;
		chord_previous_keyval=event->keyval;
	      }
	    } else
	      editor_set_line(editor, editor->line+editor->space);
	    handled=TRUE;
	    break;
	  }
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  /* CTRL-0-9: Set space value */
	  editor->space=event->keyval-'0';
	  gtk_spin_button_set_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_space")), editor->space);
	  handled=TRUE;
	  break;
	}
      } else if(shift) {
	switch (event->keyval) {
	case GDK_Left:
	  /* Shift-Left: Next position */
	  if(editor->position>0)
	    editor_set_position(editor, editor->position-1);
	  handled=TRUE;
	  break;
	case GDK_Right:
	  /* Shift-Right: Previous position */
	  if(editor->position<song->playseqs[editor->playseq]->length-1)
	    editor_set_position(editor, editor->position+1);
	  handled=TRUE;
	  break;
	case GDK_ISO_Left_Tab:
	  /* Shift-Tab: Previous track */
	  if((tracker)->cursor_item>0) {
	    (tracker)->cursor_item=0;
	    tracker_step_cursor_channel(tracker, 0);
	  } else
	    tracker_step_cursor_channel(tracker, -1);
	  handled=TRUE;
	  break;
	case GDK_Delete:
	  {
	    int i;

	    /* Delete note+command and refresh */
	    block_set_note(song->blocks[editor->block], editor->line,
			   tracker->cursor_ch, 0, 0, 0);
	    for(i=0; i<song->blocks[editor->block]->commandpages; i++)
	      block_set_command_full(song->blocks[editor->block], editor->line,
				     tracker->cursor_ch, i, 0, 0);
	    tracker_redraw_row(tracker, editor->line);
	    
	    if(editor->chord) {
	      /* Chord mode: Go to next channel */
	      tracker_step_cursor_channel(tracker, 1);
	      if(event->keyval!=chord_previous_keyval) {
		editor->chordstatus++;
		chord_previous_keyval=event->keyval;
	      }
	    } else
	      editor_set_line(editor, editor->line+editor->space);
	    handled=TRUE;
	    break;
	  }
	case GDK_BackSpace:
	  /* Shift-Backspace: Insert row */
	  if(editor->edit) {
	    int i;
	    struct block *b;
	    if(alt) {
	      int j;
	      b=block_copy(song->blocks[editor->block],
			   0, tracker->patpos,
			   song->blocks[editor->block]->tracks-1,
			   song->blocks[editor->block]->length-1);
	      block_paste(song->blocks[editor->block], b,
			  0, tracker->patpos+1);
	      for(j=0; j<song->blocks[editor->block]->tracks; j++) {
		block_set_note(song->blocks[editor->block],
			       tracker->patpos, j, 0, 0, 0);
		for(i=0; i<song->blocks[editor->block]->commandpages; i++)
		  block_set_command_full(song->blocks[editor->block],
					 tracker->patpos, j,
					 i, 0, 0);
	      }
	    } else {
	      b=block_copy(song->blocks[editor->block],
			   tracker->cursor_ch, tracker->patpos,
			   tracker->cursor_ch,
			   song->blocks[editor->block]->length-1);
	      block_paste(song->blocks[editor->block], b,
			  tracker->cursor_ch, tracker->patpos+1);
	      block_set_note(song->blocks[editor->block],
			     tracker->patpos, tracker->cursor_ch, 0, 0, 0);
	      for(i=0; i<song->blocks[editor->block]->commandpages; i++)
		block_set_command_full(song->blocks[editor->block],
				       tracker->patpos, tracker->cursor_ch,
				       i, 0, 0);
	    }
	    block_free(b);
	    tracker_redraw(tracker);
	  }
	  handled=TRUE;
	  break;
	default:
	  break;
	}
      } else if(alt) {
	switch (event->keyval) {
	case GDK_Left:
	  /* Alt-Left: Previous block */
	  editor_set_block(editor, editor->block-1);
	  handled=TRUE;
	  break;
	case GDK_Right:
	  /* Alt-Left: Next block */
	  editor_set_block(editor, editor->block+1);
	  handled=TRUE;
	  break;
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	case '0': {
	  /* Alt+0-9: Switch channels on/off */
	  int channel=9;
	  if(event->keyval>='1' && event->keyval<='9')
	    channel=event->keyval-'1';
	  if(channel<editor->song->maxtracks) {
	    if(editor->song->trackvolumes[channel]>127)
	      editor->song->trackvolumes[channel]&=127;
	    else
	      editor->song->trackvolumes[channel]|=128;
	    gui_trackvolumes_refresh(gui);
	  }
	  break;
	}
	case GDK_BackSpace:
	  {
	    /* Backspace: delete line */
	    int i, j;
	    struct block *b=block_copy(song->blocks[editor->block],
				       0, tracker->patpos+1,
				       song->blocks[editor->block]->tracks-1,
				       song->blocks[editor->block]->length-1);
	    block_paste(song->blocks[editor->block], b, 0, tracker->patpos);
	    block_free(b);
	    for(i=0; i<song->blocks[editor->block]->tracks; i++) {
	      block_set_note(song->blocks[editor->block],
			     song->blocks[editor->block]->length-1,
			     i, 0, 0, 0);
	      for(j=0; j<song->blocks[editor->block]->commandpages; j++)
		block_set_command_full(song->blocks[editor->block],
				       song->blocks[editor->block]->length-1,
				       i, j, 0, 0);
	    }
	    tracker_redraw(tracker);
	  }
	  handled=TRUE;
	  break;
	default:
	  break;
	}
      } else {
	char instrument=-1;
	char note=-1;
	if(editor->edit) {
	  char data=-1;
	  switch(tracker->cursor_item) {
	  case 0:
	    /* Editing notes */
	    switch(event->keyval) {
	    case 'z':
	      data=1;
	      break;
	    case 's':
	      data=2;
	      break;
	    case 'x':
	      data=3;
	      break;
	    case 'd':
	      data=4;
	      break;
	    case 'c':
	      data=5;
	      break;
	    case 'v':
	      data=6;
	      break;
	    case 'g':
	      data=7;
	      break;
	    case 'b':
	      data=8;
	      break;
	    case 'h':
	      data=9;
	      break;
	    case 'n':
	      data=10;
	      break;
	    case 'j':
	      data=11;
	      break;
	    case 'm':
	      data=12;
	      break;
	    case 'q':
	    case ',':
	      data=13;
	      break;
	    case '2':
	    case 'l':
	      data=14;
	      break;
	    case 'w':
	    case '.':
	      data=15;
	      break;
	    case '3':
	    case '':
	      data=16;
	      break;
	    case 'e':
	    case '-':
	      data=17;
	      break;
	    case 'r':
	      data=18;
	      break;
	    case '5':
	      data=19;
	      break;
	    case 't':
	      data=20;
	      break;
	    case '6':
	      data=21;
	      break;
	    case 'y':
	      data=22;
	      break;
	    case '7':
	      data=23;
	      break;
	    case 'u':
	      data=24;
	      break;
	    case 'i':
	      data=25;
	      break;
	    case '9':
	      data=26;
	      break;
	    case 'o':
	      data=27;
	      break;
	    case '0':
	      data=28;
	      break;
	    case 'p':
	      data=29;
	      break;
	    case 229:
	      data=30;
	      break;
	    case 180:
	      data=31;
	      break;
	    case GDK_Delete:
	      data=0;
	      break;
	    }
	    if(data!=-1) {
	      /* Set note and refresh */
	      block_set_note(song->blocks[editor->block], editor->line,
			     tracker->cursor_ch, editor->octave, data,
			     editor->instrument+1);
	      tracker_redraw_row(tracker, editor->line);

	      if(editor->chord) {
		/* Chord mode: Go to next channel */
		tracker_step_cursor_channel(tracker, 1);
		if(event->keyval!=chord_previous_keyval) {
		  editor->chordstatus++;
		  chord_previous_keyval=event->keyval;
		}
	      } else
		editor_set_line(editor, editor->line+editor->space);
	      handled=TRUE;
	    }
	    break;
	  case 1:
	    /* Editing instrument */
	    if(event->keyval>='0' && event->keyval<='9')
	      data=event->keyval-'0';
	    else if(event->keyval>='a' && event->keyval<='z')
	      data=10+event->keyval-'a';
	    else if(event->keyval==GDK_Delete)
	      data=0;

	    if(data!=-1) {
	      /* Set instrument and refresh */
	      block_set_instrument(song->blocks[editor->block],
				   editor->line, tracker->cursor_ch,
				   data);
	      tracker_redraw_row(tracker, editor->line);
	      editor_set_line(editor, editor->line+editor->space);
	      handled=TRUE;
	    }
	    break;
	  case 2:
	  case 3:
	  case 4:
	  case 5:
	    /* Editing effects */
	    if(event->keyval>='0' && event->keyval<='9')
	      data=event->keyval-'0';
	    else if(event->keyval>='a' && event->keyval<='f')
	      data=10+event->keyval-'a';
	    else if(event->keyval==GDK_Delete)
	      data=0;

	    if(data!=-1) {
	      /* Set effect and refresh */
	      block_set_command(song->blocks[editor->block],
				editor->line, tracker->cursor_ch,
				editor->cmdpage, tracker->cursor_item-2, data);
	      tracker_redraw_row(tracker, editor->line);
	      editor_set_line(editor, editor->line+editor->space);
	      handled=TRUE;
	    }
	    break;
	  }
	}

	switch (event->keyval) {
	case GDK_space:
	  /* If the song is playing, stop */
	  if(editor->player->mode!=MODE_IDLE) {
	    player_stop(editor->player);
	    gui_info_refresh(gui);
	  } else {
	    /* Otherwise toggle edit mode */
	    editor->edit^=1;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "checkbutton_edit")), editor->edit);
	  }
	  handled=TRUE;
	  break;
	case GDK_F1:
	case GDK_F2:
	case GDK_F3:
	case GDK_F4:
	case GDK_F5:
	case GDK_F6:
	case GDK_F7:
	case GDK_F8:
	case GDK_F9:
	case GDK_F10:
	case GDK_F11:
	  /* Set active keyboard octave */
	  gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_keyboard")), event->keyval-GDK_F1);
	  editor->octave=event->keyval-GDK_F1;
	  handled=TRUE;
	  break;
	case GDK_Num_Lock:
	  instrument=0;
	  break;
	case GDK_KP_Divide:
	  instrument=1;
	  break;
	case GDK_KP_Multiply:
	  instrument=2;
	  break;
	case GDK_KP_Subtract:
	  instrument=3;
	  break;
	case GDK_KP_7:
	case GDK_KP_Home:
	  instrument=4;
	  break;
	case GDK_KP_8:
	case GDK_KP_Up:
	  instrument=5;
	  break;
	case GDK_KP_9:
	case GDK_KP_Page_Up:
	  instrument=6;
	  break;
	case GDK_KP_Add:
	  instrument=7;
	  break;
	case GDK_KP_4:
	case GDK_KP_Left:
	  instrument=8;
	  break;
	case GDK_KP_5:
	case GDK_KP_Begin:
	  instrument=9;
	  break;
	case GDK_KP_6:
	case GDK_KP_Right:
	  instrument=10;
	  break;
	case GDK_KP_1:
	case GDK_KP_End:
	  instrument=11;
	  break;
	case GDK_KP_2:
	case GDK_KP_Down:
	  instrument=12;
	  break;
	case GDK_KP_3:
	case GDK_KP_Page_Down:
	  instrument=13;
	  break;
	case GDK_KP_Enter:
	  instrument=14;
	  break;
	case GDK_KP_0:
	case GDK_KP_Insert:
	  instrument=15;
	  break;
	case GDK_KP_Decimal:
	case GDK_KP_Delete:
	  instrument=16;
	  break;
	case 'z':
	  note=0;
	  break;
	case 's':
	  note=1;
	  break;
	case 'x':
	  note=2;
	  break;
	case 'd':
	  note=3;
	  break;
	case 'c':
	  note=4;
	  break;
	case 'v':
	  note=5;
	  break;
	case 'g':
	  note=6;
	  break;
	case 'b':
	  note=7;
	  break;
	case 'h':
	  note=8;
	  break;
	case 'n':
	  note=9;
	  break;
	case 'j':
	  note=10;
	  break;
	case 'm':
	  note=11;
	  break;
	case 'q':
	case ',':
	  note=12;
	  break;
	case '2':
	case 'l':
	  note=13;
	  break;
	case 'w':
	case '.':
	  note=14;
	  break;
	case '3':
	case '':
	  note=15;
	  break;
	case 'e':
	case '-':
	  note=16;
	  break;
	case 'r':
	  note=17;
	  break;
	case '5':
	  note=18;
	  break;
	case 't':
	  note=19;
	  break;
	case '6':
	  note=20;
	  break;
	case 'y':
	  note=21;
	  break;
	case '7':
	  note=22;
	  break;
	case 'u':
	  note=23;
	  break;
	case 'i':
	  note=24;
	  break;
	case '9':
	  note=25;
	  break;
	case 'o':
	  note=26;
	  break;
	case '0':
	  note=27;
	  break;
	case 'p':
	  note=28;
	  break;
	case 229:
	  note=29;
	  break;
	case 180:
	  note=30;
	  break;
	case GDK_Down:
	  /* Down: Go down */
	  editor_set_line(editor, editor->line+1);
	  handled=TRUE;
	  break;
	case GDK_Up:
	  /* Up: Go up */
	  editor_set_line(editor, editor->line-1);
	  handled=TRUE;
	  break;
	case GDK_Left:
	  /* Left: Go left */
	  tracker_step_cursor_item(tracker, -1);
	  handled=TRUE;
	  break;
	case GDK_Right:
	  /* Right: Go right */
	  tracker_step_cursor_item(tracker, 1);
	  handled=TRUE;
	  break;
	case GDK_Tab:
	  /* Tab: Next track */
	  (tracker)->cursor_item=0;
	  tracker_step_cursor_channel(tracker, 1);
	  handled=TRUE;
	  break;
	case GDK_Home:
	  /* End: Go to the beginning of block */
	  editor_set_line(editor, 0);
	  handled=TRUE;
	  break;
	case GDK_End:
	  /* End: Go to the end of block */
	  editor_set_line(editor, song->blocks[editor->block]->length-1);
	  handled=TRUE;
	  break;
	case GDK_Page_Down:
	  /* Page down: Go down 8 lines */
	  editor_set_line(editor, editor->line+8);
	  handled=TRUE;
	  break;
	case GDK_Page_Up:
	  /* Page up: Go up 8 lines */
	  editor_set_line(editor, editor->line-8);
	  handled=TRUE;
	  break;
	case GDK_BackSpace:
	  {
	    /* Backspace: delete line */
	    int i;
	    struct block *b=block_copy(song->blocks[editor->block],
				       tracker->cursor_ch,
				       tracker->patpos+1,
				       tracker->cursor_ch,
				       song->blocks[editor->block]->length-1);
	    block_paste(song->blocks[editor->block], b,
			tracker->cursor_ch, tracker->patpos);
	    block_free(b);
	    block_set_note(song->blocks[editor->block],
			   song->blocks[editor->block]->length-1,
			   tracker->cursor_ch, 0, 0, 0);
	    for(i=0; i<song->blocks[editor->block]->commandpages; i++)
	      block_set_command_full(song->blocks[editor->block],
				     song->blocks[editor->block]->length-1,
				     tracker->cursor_ch, i, 0, 0);
	    tracker_redraw(tracker);
	  }
	  handled=TRUE;
	  break;
	}

	/* Select an instrument if an instrument selection key was pressed */
	if(instrument!=-1) {
	  editor->instrument=instrument;

	  /* Make sure the instrument exists */
	  song_check_instrument(song, instrument);
	  gui_instrument_refresh(gui, TRUE);
	  handled=TRUE;
	}

	/* Play note if a key was pressed but not if cursor is in cmd pos */
	if(note!=-1 && tracker->cursor_item==0) {
	  player_play_note(editor->player, editor->instrument,
			   editor->octave*12+note, 127,
			   tracker->cursor_ch);
	  handled=TRUE;
	}
      }

      if(handled)
	g_signal_stop_emission_by_name(G_OBJECT(widget), "key_press_event");
    } else {
      /* Key has been released */
      if(!ctrl && !shift) {
	if(editor->edit && editor->chord) {
	  switch(event->keyval) {
	  case 'z':
	  case 's':
	  case 'x':
	  case 'd':
	  case 'c':
	  case 'v':
	  case 'g':
	  case 'b':
	  case 'h':
	  case 'n':
	  case 'j':
	  case 'm':
	  case 'q':
	  case ',':
	  case '2':
	  case 'l':
	  case 'w':
	  case '.':
	  case '3':
	  case '':
	  case 'e':
	  case '-':
	  case 'r':
	  case '5':
	  case 't':
	  case '6':
	  case 'y':
	  case '7':
	  case 'u':
	  case 'i':
	  case '9':
	  case 'o':
	  case '0':
	  case 'p':
	  case 229:
	  case 180:
	    tracker_step_cursor_channel(tracker, -1);
	    editor->chordstatus--;
	    if(editor->chordstatus==0)
	      editor_set_line(editor, editor->line+editor->space);
	    handled=TRUE;
	    break;
	  }
	}
      }

      if(handled)
	g_signal_stop_emission_by_name(G_OBJECT(widget), "key_release_event");
    }
  } else {
    /* An entry widget was active */
    if(pressed) {
      switch(event->keyval) {
      case GDK_Tab:
      case GDK_Return:
	/* Activate the window widget so the user can play again */
	gtk_window_set_focus(GTK_WINDOW(widget), NULL);
	handled=TRUE;
	break;
      }
    }
  }

  return handled;
}
