/* midi.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "midi.h"

/* Opens a MIDI device */
struct midi *midi_open(char *filename) {
  struct midi *midi=(struct midi *)calloc(1, sizeof(struct midi));

  /* Open MIDI device:read/write, do not block (for input), auto flush */
  midi->fd=open(filename, O_RDWR|O_NONBLOCK);
  if(midi->fd==-1) {
    fprintf(stderr, "Could not open MIDI device %s.\n", filename);
    free(midi);
    midi=NULL;
  }

  return midi;
}

/* Closes a MIDI device */
void midi_close(struct midi *midi) {
  if(midi==NULL) {
    fprintf(stderr, "midi_close() called with null midi\n");
    return;
  }

  if(midi->fd!=-1) {
    fsync(midi->fd);
    close(midi->fd);
  }
  free(midi);
}

/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(struct midi *midi, unsigned char channel, unsigned char note,
		   unsigned char velocity) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_note_off() called with null midi\n");
    return;
  }

  data[0]=0x80|channel;
  data[1]=note&0x7f;
  data[2]=velocity&0x7f;
  
  write(midi->fd, data, 3);
  fsync(midi->fd);
}

/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(struct midi *midi, unsigned char channel, unsigned char note,
		   unsigned char velocity) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_note_on() called with null midi\n");
    return;
  }

  data[0]=0x90|channel;
  data[1]=note&0x7f;
  data[2]=velocity&0x7f;
  
  write(midi->fd, data, 3);
  fsync(midi->fd);
}

/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(struct midi *midi, unsigned char channel, unsigned char note,
		     unsigned char pressure) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_aftertouch() called with null midi\n");
    return;
  }

  data[0]=0xa0|channel;
  data[1]=note&0x7f;
  data[2]=pressure&0x7f;
  
  write(midi->fd, data, 3);
  fsync(midi->fd);
}

/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(struct midi *midi, unsigned char channel, 
		     unsigned char controller, unsigned char value) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_controller() called with null midi\n");
    return;
  }

  data[0]=0xb0|channel;
  data[1]=controller&0x7f;
  data[2]=value&0x7f;
  
  write(midi->fd, data, 3);
  fsync(midi->fd);
}

/* Send a program change on a MIDI channel */
void midi_program_change(struct midi *midi, unsigned char channel,
			 unsigned char program) {
  unsigned char data[2];

  if(midi==NULL) {
    fprintf(stderr, "midi_program_change() called with null midi\n");
    return;
  }

  data[0]=0xc0|channel;
  data[1]=program&0x7f;
  
  write(midi->fd, data, 2);
  fsync(midi->fd);
}

/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(struct midi *midi, unsigned char channel,
			   unsigned char pressure) {
  unsigned char data[2];

  if(midi==NULL) {
    fprintf(stderr, "midi_channel_pressure() called with null midi\n");
    return;
  }

  data[0]=0xd0|channel;
  data[1]=pressure&0x7f;
  
  write(midi->fd, data, 2);
  fsync(midi->fd);
}

/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(struct midi *midi, unsigned char channel,
		      unsigned short value) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_pitch_wheel() called with null midi\n");
    return;
  }

  data[0]=0xe0|channel;
  data[1]=(value>>7)&0x7f;
  data[2]=value&0x7f;
  
  write(midi->fd, data, 3);
  fsync(midi->fd);
}

/* Sends an arbitrary MIDI message */
void midi_write(struct midi *midi, unsigned char *message, unsigned short length) {
  if(midi==NULL) {
    fprintf(stderr, "midi_write() called with null midi\n");
    return;
  }

  if(length>0) {
    write(midi->fd, message, length);
    fsync(midi->fd);
  }
}

/* Send a clock message */
void midi_clock(struct midi *midi) {
  unsigned char data[]={ 0xf8 };

  if(midi==NULL) {
    fprintf(stderr, "midi_clock() called with null midi\n");
    return;
  }

  write(midi->fd, data, 1);
  fsync(midi->fd);
}

/* Receives a MIDI message */
unsigned char *midi_read(struct midi *midi) {
  int status=1;
  unsigned char c=0, *d=NULL;

  if(midi==NULL) {
    fprintf(stderr, "midi_read() called with null midi\n");
    return NULL;
  }

  while((status=read(midi->fd, &c, 1))==1 && c<0x80);

  if(status!=1)
    return NULL;
  
  switch(c&0xf0) {
  case 0x80:
  case 0x90:
  case 0xa0:
  case 0xb0:
  case 0xe0: {
    d=(unsigned char *)calloc(3, sizeof(unsigned char));
    d[0]=c;
    status=read(midi->fd, d+1, 2);

    if(status==2)
      return d;
    else {
      free(d);
      return NULL;
    }
  }
  default:
    break;
  }
  return NULL;
}

/* Receives a MIDI message */
void midi_read_system_exclusive(struct midi *midi, struct message *message,
				int autostop) {
  int status=0, i=0;
  unsigned char c=0;

  if(midi==NULL) {
    fprintf(stderr, "midi_read_system_exclusive() called with null midi\n");
    return;
  }

  if(message==NULL) {
    fprintf(stderr, "midi_read_system_exclusive() called with null message\n");
    return;
  }

  if(autostop==1) {
    /* Read until 0xf7 (end of SysEx) */
    c=0;
    message->length=0;

    while(c!=0xf7) {
      status=read(midi->fd, &c, 1);
      
      if(status==1) {
	if(i>=message->length) {
	  message->length+=65536;
	  message->data=(unsigned char *)realloc(message->data, message->length);
	}

	if(c!=0xf1 && c!=0xf8 && c!=0xf9 && c!=0xfe)
	  message->data[i++]=c;
      }
    }
  } else {
    /* Read until the buffer is full */
    while(i<message->length) {
      status=read(midi->fd, &c, 1);
      
      if(status==1) {
	if(c!=0xf1 && c!=0xf8 && c!=0xf9 && c!=0xfe)
	  message->data[i++]=c;
      }
    }
  }
  
  /* Remove unnecessary bytes if necessary */
  if(i<message->length) {
    message->length=i;
    message->data=(unsigned char *)realloc(message->data, message->length);
  }
}

/* Sets the length of a MIDI message */
void message_set_length(struct message *message, unsigned int length) {
  if(message==NULL) {
    fprintf(stderr, "message_set_length() called with null message\n");
    return;
  }

  message->length=length;
  message->data=(unsigned char *)realloc(message->data, message->length);
}

/* Sets the autosend flag of a MIDI message */
void message_set_autosend(struct message *message, unsigned int autosend) {
  if(message==NULL) {
    fprintf(stderr, "message_set_autosend() called with null message\n");
    return;
  }

  message->autosend=autosend;
}

/* Parses a message element in an XML file */
struct message *message_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct message *message=NULL;
  
  if((!xmlStrcmp(cur->name, "message")) && (cur->ns==ns)) {
    xmlNodePtr temp=cur->xmlChildrenNode;
    char c[3], *prop;
    int i;
    unsigned int d;

    /* Temporary string for hexadecimal parsing */
    c[2]=0;

    /* Allocate message */
    message=(struct message *)calloc(1, sizeof(struct message));
    message->name=xmlGetProp(cur, "name");
    message->length=strlen(temp->content)/2;
    prop=xmlGetProp(cur, "autosend");
    if(prop!=NULL)
      message->autosend=atoi(prop);
    message->data=(unsigned char *)calloc(message->length, sizeof(unsigned char));
    for(i=0; i<message->length; i++) {
      c[0]=temp->content[i*2];
      c[1]=temp->content[i*2+1];
      sscanf(c, "%X", &d);
      message->data[i]=d;
    }
  } else if(cur->type!=XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected message, got %s\n", cur->name);

  return message;
}

/* Saves a message to an XML file */
void message_save(struct message *message, int number, xmlNodePtr parent) {
  char *c, t[10];
  xmlNodePtr node;
  int i;

  if(message==NULL) {
    fprintf(stderr, "message_save() called with null message\n");
    return;
  }

  c=(char *)calloc(2*message->length+1, sizeof(char));
  for(i=0; i<message->length; i++)
    sprintf(c+(i*2), "%.2X", message->data[i]);

  node=xmlNewChild(parent, NULL, "message", c);
  snprintf(t, 10, "%d", number);
  xmlSetProp(node, "number", t);
  xmlSetProp(node, "name", message->name);
  snprintf(t, 10, "%d", message->autosend);
  xmlSetProp(node, "autosend", t);

  xmlAddChild(parent, xmlNewText("\n"));

  free(c);
}

/* Loads a message from a file */
void message_load_binary(struct message *message, char *filename) {
  unsigned char *data;
  FILE *file;

  if(message==NULL) {
    fprintf(stderr, "message_load_binary() called with null message\n");
    return;
  }
  
  if(filename==NULL) {
    fprintf(stderr, "message_load_binary() called with null filename\n");
    return;
  }

  file=fopen(filename, "r");

  if(file!=NULL) {
    int length;

    fseek(file, 0, SEEK_END);
    length=ftell(file);
    data=(unsigned char *)calloc(length, sizeof(unsigned char));
    fseek(file, 0, SEEK_SET);
    fread(data, sizeof(unsigned char), length, file);
    fclose(file);

    if(message->data!=NULL)
      free(message->data);

    message->data=data;
    message->length=length;
  }
}

/* Saves a message to a file */
void message_save_binary(struct message *message, char *filename) {
  FILE *file;

  if(message==NULL) {
    fprintf(stderr, "message_save_binary() called with null message\n");
    return;
  }
  
  if(filename==NULL) {
    fprintf(stderr, "message_save_binary() called with null filename\n");
    return;
  }

  file=fopen(filename, "w");

  if(file!=NULL) {
    fwrite(message->data, sizeof(unsigned char), message->length, file);
    fclose(file);
  }
}
