/* midi.h
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _MIDI_H
#define _MIDI_H

#include <libxml/tree.h>
#include <alsa/asoundlib.h>

#define MIDI_BUFFER_BLOCK_SIZE 4096

enum {
  MIDI_NULL,
  MIDI_OSS,
  MIDI_ALSA,
  MIDI_BUFFER
};

enum {
  DIRECTION_NONE = 0,
  DIRECTION_IN = 1,
  DIRECTION_OUT = 2,
  DIRECTION_BOTH = 3
};

/* MIDI data structure */
struct midi {
  /* Output device name */
  char *devicename;
  /* User readable name */
  char *name;
  /* Output device type */
  unsigned int type;
  /* Output device direction */
  unsigned int direction;
  /* Output device structure */
  void *data;
  /* Current tick and tick of last event */
  unsigned int tick, oldtick;
};

struct midi_oss {
  /* OSS input MIDI device file descriptor */
  int in;
  /* OSS output MIDI device file descriptor */
  int out;
};

struct midi_alsa {
  snd_rawmidi_t *in;
  snd_rawmidi_t *out;
};

struct midi_buffer {
  /* MIDI outbut buffer */
  unsigned char *data;
  /* Current position */
  unsigned char *ptr;
  /* Number of bytes currently allocated */
  unsigned int allocated;
};

struct message {
  /* Name */
  char *name;
  /* Length in bytes */
  unsigned int length;
  /* Dump data */
  unsigned char *data;
  /* Automatically send after loading? */
  unsigned int autosend;
};

/* Gets the number of available ALSA MIDI devices */
unsigned int midi_get_num_alsa_devices();
/* Returns MIDI data structures for available ALSA MIDI devices */
struct midi **midi_get_alsa_devices();
/* Allocates a MIDI interface */
struct midi *midi_alloc(unsigned int, char *, char *);
/* Frees a MIDI structure */
void midi_free(struct midi *);
/* Opens the MIDI interfaces defined in a string */
void midi_open_defined(struct midi **, unsigned int, char *, char *);
/* Opens a MIDI interface */
unsigned int midi_open(struct midi *, unsigned int);
/* Closes a MIDI interface */
void midi_close(struct midi *, unsigned int);
/* Sets the current tick */
void midi_set_tick(struct midi *, unsigned int);
/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(struct midi *, unsigned char, unsigned char, unsigned char);
/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(struct midi *, unsigned char, unsigned char, unsigned char);
/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(struct midi *, unsigned char, unsigned char, unsigned char);
/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(struct midi *, unsigned char, unsigned char, unsigned char);
/* Send a program change on a MIDI channel */
void midi_program_change(struct midi *, unsigned char, unsigned char);
/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(struct midi *, unsigned char, unsigned char);
/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(struct midi *, unsigned char, unsigned short);
/* Sends a MIDI message */
void midi_write_raw(struct midi *, unsigned char *, unsigned short);
/* Send a clock message */
void midi_clock(struct midi *);
/* Set the tempo (used when exporting) */
void midi_tempo(struct midi *, unsigned int);
/* Receives a MIDI message */
unsigned char *midi_read_raw(struct midi *);
/* Receives a MIDI message */
void midi_read_system_exclusive(struct midi *, struct message *, int);
/* Returns the data written to the MIDI buffer */
unsigned char *midi_get_buffer(struct midi *);
/* Returns the length of data written to the MIDI buffer */
int midi_get_buffer_length(struct midi *);
/* Sets the length of a MIDI message */
void message_set_length(struct message *, unsigned int);
/* Sets the auto send flag of a MIDI message */
void message_set_autosend(struct message *, unsigned int);
/* Parses a element in an XML file */
struct message *message_parse(xmlDocPtr, xmlNsPtr, xmlNodePtr);
/* Saves a message to an XML file */
void message_save(struct message *, int, xmlNodePtr);
/* Loads a message from a file */
void message_load_binary(struct message *, char *);
/* Saves a message to a file */
void message_save_binary(struct message *, char *);

#endif
