/* player.h
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _PLAYER_H
#define _PLAYER_H

#include <glib.h>
#include "song.h"
#include "editor.h"
#include "midi.h"

/* Player mode */
enum {
  MODE_IDLE,
  MODE_PLAY_SONG,
  MODE_PLAY_BLOCK
};

struct player_interface_status {
  char midichannel;
  char volume;
  char note;
  char hold;
};

/* Track status values */
struct player_track_status {
  char instrument;
  char previouscommand;
  struct player_interface_status **interfaces;
};

/* 128 MIDI controllers plus aftertouch, channel pressure and pitch wheel */
enum {
  VALUES = (128 + 3),
  VALUES_AFTERTOUCH = 128,
  VALUES_CHANNEL_PRESSURE = 129,
  VALUES_PITCH_WHEEL = 130
};

enum {
  SCHED_NONE,
  SCHED_RTC,
  SCHED_NANOSLEEP
};

struct player {
  /* Current location in song */
  unsigned int section, playseq, position, block, line, tick;
  /* The song currently being played */
  struct song *song;
  /* Player mode */
  unsigned int mode;
  /* Player scheduling mode */
  unsigned int sched;
  /* Status of tracks; notes playing */
  struct player_track_status **trackstatus;
  /* MIDI controller values; one for each controller on each channel */
  unsigned char **midicontrollervalues;
  /* For measuring how long the song has been playing */
  struct timeval playingstarted, playedsofar;
  /* Ticks passed after playing started */
  unsigned int tickssofar;
  /* Player thread pointer */
  GThread *thread;
  /* Mutex for the player thread */
  GMutex *mutex;
  /* Kill player flag (the mutex must be used when accessing) */
  unsigned int killthread;
  /* MIDI data structures */
  struct midi **midiinterfaces;
  /* Number of MIDI data structures */
  unsigned int nummidiinterfaces;
  /* RTC device file descriptor */
  int rtc;
  /* Editor for this player */
  struct editor *editor;
  /* Indicates whether some tracks are soloed or not */
  unsigned int solo;
};

/* Plays a song to a MIDI interface witout any scheduling (for export) */
void player_midi_export(struct song *, struct midi *);
/* Creates a player for a song */
struct player *player_open(struct song *, struct editor *, struct midi **, unsigned int, unsigned int);
/* Closes a player for a song */
void player_close(struct player *);

/* Starts the player thread */
void player_start(struct player *, unsigned int, int, int, int, int);
/* Kills the player thread */
void player_stop(struct player *);

/* Plays a note using given instrument on a given channel */
void player_play_note(struct player *, unsigned int, unsigned char, unsigned char, unsigned char);
/* Stops notes playing at the moment */
void player_stop_notes(struct player *);
/* Stops all notes */
void player_stop_all_notes(struct player *);
/* Resets the pitch wheel on all channels */
void player_reset_pitch(struct player *);
/* Reallocate track status array */
void player_trackstatus_create(struct player *, int);
/* Resets the player time */
void player_reset_time(struct player *, gboolean);

/* Set player position */
void player_set_section(struct player *, int);
void player_set_playseq(struct player *, int);
void player_set_position(struct player *, int);
void player_set_block(struct player *, int);
void player_set_line(struct player *, int);
  
/* Sets whether some tracks are considered soloed or not */
void player_set_solo(struct player *, unsigned int);
/* Checks whether some tracks are soloed or not */
void player_check_solo(struct player *);

#endif
