/* midi.h
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _MIDI_H
#define _MIDI_H

#include <libxml/tree.h>
#include <alsa/asoundlib.h>
#include <glib.h>

#define MIDI_BUFFER_BLOCK_SIZE 4096

enum {
  MIDI_NULL,
  MIDI_ALSA,
  MIDI_BUFFER
};

enum {
  DIRECTION_OUT = 0,
  DIRECTION_IN = 1
};

/* MIDI data */
struct midi {
  /* ALSA MIDI sequencer interface */
  snd_seq_t *seq;
  /* Sequencer client ID */
  int client;
  /* Sequencer port ID */
  int port;
  /* MIDI event encoder */
  snd_midi_event_t *encoder;
  /* MIDI event decoder */
  snd_midi_event_t *decoder;
  /* Data structures of subscribed MIDI outputs */
  struct midi_interface **outputs;
  /* Number of available MIDI outputs */
  unsigned int numoutputs;
  /* Data structures of subscribed MIDI inputs */
  struct midi_interface **inputs;
  /* Number of available MIDI inputs */
  unsigned int numinputs;
  /* Character string representing successfully subscribed outputs */
  char *outputs_ok;
  /* Character string representing successfully subscribed inputs */
  char *inputs_ok;
  /* Character string representing failed outputs */
  char *outputs_error;
  /* Character string representing failed inputs */
  char *inputs_error;
  /* Incoming data */
  unsigned char *incoming;
  /* Number of incoming bytes */
  int incominglength;
  /* Position in incoming buffer */
  int incomingposition;
  /* Whether the MIDI interface list has changed */
  int changed;
};

/* MIDI interface data structure */
struct midi_interface {
  /* Parent structure */
  struct midi *midi;
  /* Name */
  char *name;
  /* Device type (MIDI_NULL/ALSA/BUFFER) */
  unsigned int type;
  /* Current tick and tick of last event */
  unsigned int tick, oldtick;
  /* Type specific data structure */
  void *data;
};

struct midi_alsa {
  /* Client ID */
  int client;
  /* Port ID */
  int port;
  /* Subscription */
  snd_seq_port_subscribe_t *subs;
};

struct midi_buffer {
  /* MIDI outbut buffer */
  unsigned char *data;
  /* Current position */
  unsigned char *ptr;
  /* Number of bytes currently allocated */
  unsigned int allocated;
};

struct message {
  /* Name */
  char *name;
  /* Length in bytes */
  unsigned int length;
  /* Dump data */
  unsigned char *data;
  /* Automatically send after loading? */
  unsigned int autosend;
};

/* Open the MIDI subsystem */
struct midi *midi_open(char *, char *, char *, char *);
/* Close the MIDI subsystem */
void midi_close(struct midi *);
/* Subscribe to a client/port */
unsigned int midi_subscribe(struct midi *, int, int, unsigned int);
/* Unsubscribe from a client/port */
unsigned int midi_unsubscribe(struct midi *, int, int, unsigned int);
/* Allocate a MIDI interface */
struct midi_interface *midi_interface_alloc(unsigned int, char *);
/* Free a MIDI interface */
void midi_interface_free(struct midi_interface *);
/* Get MIDI interface from available interfaces by client and port */
struct midi_interface *midi_interface_get_by_client_and_port(struct midi *, int, int, unsigned int);
/* Gets an MIDI interface number by name from the available interfaces */
int midi_interface_number_get_by_name(struct midi *, char *, unsigned int);
/* Gets an MIDI interface name by number from the available interfaces */
char *midi_interface_name_get_by_number(struct midi *, unsigned int, unsigned int);
/* Opens the MIDI interfaces defined in a string */
void midi_subscribe_all(struct midi *);
/* Unsubscribe all MIDI interfaces and free data */
void midi_unsubscribe_all(struct midi *midi);
/* Gets the list of available interfaces */
void midi_interfaces_get(struct midi *, unsigned int);
/* Refreshes the list of available interfaces */
void midi_interfaces_refresh(struct midi *);
/* Sets the current tick */
void midi_set_tick(struct midi_interface *, unsigned int);
/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(struct midi_interface *, unsigned char, unsigned char, unsigned char);
/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(struct midi_interface *, unsigned char, unsigned char, unsigned char);
/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(struct midi_interface *, unsigned char, unsigned char, unsigned char);
/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(struct midi_interface *, unsigned char, unsigned char, unsigned char);
/* Send a program change on a MIDI channel */
void midi_program_change(struct midi_interface *, unsigned char, unsigned char);
/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(struct midi_interface *, unsigned char, unsigned char);
/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(struct midi_interface *, unsigned char, unsigned short);
/* Sends a MIDI message */
void midi_write_raw(struct midi_interface *, unsigned char *, unsigned short);
/* Send a clock message */
void midi_clock(struct midi_interface *);
/* Set the tempo (used when exporting) */
void midi_tempo(struct midi_interface *, unsigned int);
/* Receives a MIDI message */
unsigned char *midi_read_raw(struct midi *);
/* Receives a MIDI message */
int midi_read_system_exclusive(struct midi *, struct message *, unsigned int, int);
/* Returns the data written to the MIDI buffer */
unsigned char *midi_get_buffer(struct midi_interface *);
/* Returns the length of data written to the MIDI buffer */
int midi_get_buffer_length(struct midi_interface *);
/* Sets the length of a MIDI message */
void message_set_length(struct message *, unsigned int);
/* Sets the auto send flag of a MIDI message */
void message_set_autosend(struct message *, unsigned int);
/* Parses a element in an XML file */
struct message *message_parse(xmlDocPtr, xmlNsPtr, xmlNodePtr);
/* Saves a message to an XML file */
void message_save(struct message *, int, xmlNodePtr);
/* Loads a message from a file */
void message_load_binary(struct message *, char *);
/* Saves a message to a file */
void message_save_binary(struct message *, char *);

#endif
