/* gui.c
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <gmodule.h>
#include <gnome.h>
#include <glade/glade.h>
#include "gui.h"
#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"
#include "cellrendererspin.h"

#define new_column gtk_tree_view_column_new_with_attributes

gboolean gui_quit = FALSE;

/* Return the length of an integer in string representation */
static int intlen(int i)
{
  if (i == 0)
    return 1;

  if (i < 0)
    return (int)log10((double)-i) + 2;

  return (int)log10((double)i) + 1;
}

/* Connects signals to objects and passes the gui struct on as user_data */
void gui_gladexmlconnectfunc(const gchar * handler_name, GObject * object, const gchar * signal_name, const gchar * signal_data, GObject * connect_object, gboolean after, gpointer user_data)
{
  GCallback func;

  /* Use the gmodule interface to resolve the function name to a pointer */
  if (!g_module_symbol(g_module_open(NULL, 0), handler_name, (gpointer *) & func))
    g_warning("could not find signal handler '%s'.", handler_name);

  /* Connect the signal */
  if (after)
    g_signal_connect_after(object, signal_name, func, user_data);
  else
    g_signal_connect(object, signal_name, func, user_data);

};

void gtk_main_quit()
{
  gui_quit = TRUE;
}

/* Initializes a GUI */
struct gui *gui_open(struct editor *editor)
{
  struct gui *gui = (struct gui *)calloc(1, sizeof(struct gui));
  GtkCellRenderer *renderer;
  GtkTreeView *treeview;
  GtkTreeSelection *select;
  GtkListStore *store;
  GdkPixbuf *pixbuf;
  GtkWidget *w;
  char *songpath, *messagepath;

  gui->editor = editor;

  /* Mutex for the refreshing function */
  gui->refresh_mutex = g_mutex_new();

  /* Load the GUI from a GLADE file and connect signals */
  gui->xml = glade_xml_new(PACKAGE_DATA_DIR "/glade/tutka.glade", NULL, NULL);
  if (gui->xml == NULL) {
    gui_close(gui);
    return NULL;
  }
  glade_xml_signal_autoconnect_full(gui->xml, gui_gladexmlconnectfunc, gui);
  gui->window_main = glade_xml_get_widget(gui->xml, "window_main");
  gtk_widget_add_events(gui->window_main, GDK_KEY_RELEASE_MASK);

  /* Create the MVC classes for various list view widgets */
  /* Section list */
  store = gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_sectionlist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Section", renderer, "text", GUI_SECTIONLIST_COLUMN_SECTION, NULL));
  if (gui->numplayseqs == NULL)
    renderer = gui_cell_renderer_spin_new(1, 1, 1, 2, 2, 1, 0, &gui->numplayseqs);
  else
    renderer = gui_cell_renderer_spin_new(1, *gui->numplayseqs, 1, 2, 2, 1, 0, &gui->numplayseqs);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_SECTIONLIST_COLUMN_PLAYSEQ_NUMBER));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_sectionlist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Playing Sequence", renderer, "text", GUI_SECTIONLIST_COLUMN_PLAYSEQ_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Playing Sequence Name", renderer, "text", GUI_SECTIONLIST_COLUMN_PLAYSEQ_NAME, NULL));

  /* Playseq List */
  store = gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseqlist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_playseqlist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Number", renderer, "text", GUI_PLAYSEQLIST_COLUMN_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PLAYSEQLIST_COLUMN_NAME));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_playseqlist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_PLAYSEQLIST_COLUMN_NAME, NULL));

  /* Playing Sequence */
  store = gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_playseq_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Position", renderer, "text", GUI_PLAYSEQ_COLUMN_POSITION, NULL));
  if (gui->numblocks == NULL)
    renderer = gui_cell_renderer_spin_new(1, 1, 1, 2, 2, 1, 0, &gui->numblocks);
  else
    renderer = gui_cell_renderer_spin_new(1, *gui->numblocks, 1, 2, 2, 1, 0, &gui->numblocks);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PLAYSEQ_COLUMN_BLOCK_NUMBER));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_playseq_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Block Number", renderer, "text", GUI_PLAYSEQ_COLUMN_BLOCK_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Block Name", renderer, "text", GUI_PLAYSEQ_COLUMN_BLOCK_NAME, NULL));

  /* Block List */
  store = gtk_list_store_new(5, G_TYPE_INT, G_TYPE_STRING, G_TYPE_INT, G_TYPE_INT, G_TYPE_INT);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_blocklist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Number", renderer, "text", GUI_BLOCKLIST_COLUMN_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_NAME));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_BLOCKLIST_COLUMN_NAME, NULL));
  renderer = gui_cell_renderer_spin_new(1, 256, 1, 2, 2, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_TRACKS));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Tracks", renderer, "text", GUI_BLOCKLIST_COLUMN_TRACKS, NULL));
  renderer = gui_cell_renderer_spin_new(1, 4096, 1, 2, 2, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_LENGTH));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Length", renderer, "text", GUI_BLOCKLIST_COLUMN_LENGTH, NULL));
  renderer = gui_cell_renderer_spin_new(1, 64, 1, 2, 2, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_COMMAND_PAGES));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Command pages", renderer, "text", GUI_BLOCKLIST_COLUMN_COMMAND_PAGES, NULL));

  /* Message List */
  store = gtk_list_store_new(4, G_TYPE_INT, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_messagelist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Number", renderer, "text", GUI_MESSAGELIST_COLUMN_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_MESSAGELIST_COLUMN_NAME));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_messagelist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_MESSAGELIST_COLUMN_NAME, NULL));
  renderer = gui_cell_renderer_spin_new(0, 2097152, 1, 512, 512, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_MESSAGELIST_COLUMN_LENGTH));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_messagelist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Length", renderer, "text", GUI_MESSAGELIST_COLUMN_LENGTH, NULL));
  renderer = gtk_cell_renderer_toggle_new();
  g_object_set(renderer, "activatable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_MESSAGELIST_COLUMN_AUTOSEND));
  g_signal_connect(renderer, "toggled", G_CALLBACK(on_cell_messagelist_toggled), gui);
  gtk_tree_view_append_column(treeview, new_column("Automatically Send Message After Loading Song", renderer, "active", GUI_MESSAGELIST_COLUMN_AUTOSEND, NULL));

  /* Preferences */
  store = gtk_list_store_new(4, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_INT, G_TYPE_INT);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_preferences_midi_interfaces_output"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_NONE);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_NAME));
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_NAME, NULL));
  renderer = gtk_cell_renderer_toggle_new();
  g_object_set(renderer, "activatable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_ENABLED));
  g_object_set_data(G_OBJECT(renderer), "direction", GUINT_TO_POINTER(DIRECTION_OUT));
  g_signal_connect(renderer, "toggled", G_CALLBACK(on_cell_preferences_midi_interfaces_toggled), gui);
  gtk_tree_view_append_column(treeview, new_column("Enabled", renderer, "active", GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_ENABLED, NULL));

  store = gtk_list_store_new(4, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_INT, G_TYPE_INT);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_preferences_midi_interfaces_input"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_NONE);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_NAME));
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_NAME, NULL));
  renderer = gtk_cell_renderer_toggle_new();
  g_object_set(renderer, "activatable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_ENABLED));
  g_object_set_data(G_OBJECT(renderer), "direction", GUINT_TO_POINTER(DIRECTION_IN));
  g_signal_connect(renderer, "toggled", G_CALLBACK(on_cell_preferences_midi_interfaces_toggled), gui);
  gtk_tree_view_append_column(treeview, new_column("Enabled", renderer, "active", GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_ENABLED, NULL));

  /* Set the keyboard octave */
  gui->editor->octave = 3;
  gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_keyboard")), gui->editor->octave);

  /* Set default choices in combo boxes */
  gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area")), 0);
  gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_instrument")), 0);
  gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_mode")), 0);
  gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")), 0);
  gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")), 0);

  /* Create the MIDI interface selectors (libglade doesn't do this if no items are defined) */
  w = glade_xml_get_widget(gui->xml, "combobox_instrumentproperties_midiinterface");
  store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
  gtk_combo_box_set_model(GTK_COMBO_BOX(w), GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(w), renderer, TRUE);
  gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(w), renderer, "text", 0, NULL);

  /* Set default path in file dialogs */
  songpath = editor_song_path_get(gui->editor);
  if (songpath != NULL) {
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(glade_xml_get_widget(gui->xml, "filechooserdialog_file_open")), songpath);
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(glade_xml_get_widget(gui->xml, "filechooserdialog_file_saveas")), songpath);
  }
  messagepath = editor_message_path_get(gui->editor);
  if (messagepath != NULL) {
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(glade_xml_get_widget(gui->xml, "filechooserdialog_message_load")), messagepath);
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(glade_xml_get_widget(gui->xml, "filechooserdialog_message_save")), messagepath);
  }

  /* Add icons */
  pixbuf = gdk_pixbuf_new_from_file(PACKAGE_PIXMAP_DIR "/tutka.png", NULL);
  if (pixbuf != NULL)
    gtk_window_set_default_icon_list(g_list_append(NULL, pixbuf));

  return gui;
}

/* Closes a GUI */
void gui_close(struct gui *gui)
{
  g_mutex_free(gui->refresh_mutex);

  free(gui);
}

/* Sets the current filename to the save dialog */
void gui_set_saveas_filename(struct gui *gui, char *filename)
{
  /* Set default path in save as dialog */
  if (filename != NULL) {
    gchar *path, *name;

    path = g_path_get_dirname(filename);
    name = g_path_get_basename(filename);
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(glade_xml_get_widget(gui->xml, "filechooserdialog_file_saveas")), path);
    gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(glade_xml_get_widget(gui->xml, "filechooserdialog_file_saveas")), name);
    g_free(path);
    g_free(name);
  }
}

/* Refreshes all parts of the GUI */
void gui_refresh_all(struct gui *gui)
{
  /* Reset the tracker */
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  tracker_reset(tracker);
  tracker_set_pattern(tracker, editor_song_block_current_get(gui->editor));
  tracker_set_num_channels(tracker, editor_song_block_current_tracks_get(gui->editor));

  /* Refresh the GUI */
  gui_info_refresh(gui);
  gui_tempo_refresh(gui);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_playseqlist_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
  gui_instrument_refresh(gui);
  gui_messagelist_refresh(gui, TRUE);
  gui_trackvolumes_refresh(gui);
  gui_song_refresh(gui);
  gui_window_title_refresh(gui);

  /* Show the main window */
  gtk_widget_show(gui->window_main);
  gtk_widget_show(GTK_WIDGET(tracker));

  /* FIX: Tracker widget kludge */
  tracker->cursor_ch = 0;

  /* The current CellRendererSpin implementation needs a pointer to the upper limit values */
  /* FIX: this is not the way this should be done */
  if (gui->editor->song == NULL) {
    gui->numplayseqs = NULL;
    gui->numblocks = NULL;
  } else {
    gui->numplayseqs = &gui->editor->song->numplayseqs;
    gui->numblocks = &gui->editor->song->numblocks;
  }
}

Tracker *gui_get_tracker(struct gui *gui)
{
  return (Tracker *) glade_xml_get_widget(gui->xml, "tracker");
}

int gui_get_tracker_position(struct gui *gui)
{
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  return tracker->patpos;
}

/* Set the refresh mask */
void gui_set_refresh(struct gui *gui, unsigned int mask)
{
  if (g_thread_self() != gui->editor->thread)
    g_mutex_lock(gui->refresh_mutex);
  gui->refresh_mask |= mask;
  if (g_thread_self() != gui->editor->thread)
    g_mutex_unlock(gui->refresh_mutex);
}

/* Refresh all those parts of the GUI that need to be refreshed */
void gui_refresh(struct gui *gui)
{
  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_INSTRUMENT) {
    gui_instrument_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_INSTRUMENT;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TEMPO) {
    gui_tempo_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TEMPO;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_INFO) {
    gui_info_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_INFO;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_WINDOW_TITLE) {
    gui_window_title_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_WINDOW_TITLE;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_STATUSBAR) {
    gui_statusbar_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_STATUSBAR;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TIMER) {
    gui_timer_refresh(gui, gui->editor->time);
    gui->refresh_mask ^= GUI_REFRESH_TIMER;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_BLOCKLIST) {
    gui_blocklist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_BLOCKLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQ) {
    gui_playseq_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQ;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQ_RECREATE) {
    gui_playseq_refresh(gui, TRUE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQ_RECREATE;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQLIST) {
    gui_playseqlist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SECTIONLIST) {
    gui_sectionlist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_SECTIONLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_MESSAGELIST) {
    gui_messagelist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_MESSAGELIST;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TRACKVOLUMES) {
    gui_trackvolumes_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TRACKVOLUMES;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SONG) {
    gui_song_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_SONG;
  }
  g_mutex_unlock(gui->refresh_mutex);

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TRACKER) {
    gui_tracker_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TRACKER;
  }
  g_mutex_unlock(gui->refresh_mutex);
}

/* Refresh the tracker */
void gui_tracker_refresh(struct gui *gui)
{
  Tracker *tracker = gui_get_tracker(gui);
  unsigned int length = editor_song_block_current_length_get(gui->editor);
  unsigned int cmdpages = editor_song_block_current_commandpages_get(gui->editor);

  /* Check that the line is valid */
  if (gui->editor->line >= length)
    gui->editor->line = length - 1;
  /* Check that the command page is valid */
  if (gui->editor->cmdpage > cmdpages)
    gui->editor->cmdpage = cmdpages - 1;

  tracker_set_num_channels(tracker, editor_song_block_current_tracks_get(gui->editor));
  tracker_set_pattern(tracker, editor_song_block_current_get(gui->editor));
  tracker_set_patpos(tracker, gui->editor->line);
  tracker_set_cmdpage(tracker, gui->editor->cmdpage);
}

/* Refresh the song properties window widgets */
void gui_song_refresh(struct gui *gui)
{
  GtkEntry *entry = GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_songname"));
  GtkToggleButton *togglebutton = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "checkbutton_sendsync"));
  char *name;

  g_signal_handlers_block_by_func(entry, on_entry_songname_changed, gui);
  g_signal_handlers_block_by_func(togglebutton, on_checkbutton_sendsync_toggled, gui);

  name = editor_song_name_get(gui->editor);
  /* Refresh the song name widget */
  if (name != NULL)
    gtk_entry_set_text(entry, name);
  else
    gtk_entry_set_text(entry, "");

  /* Refresh the send sync widget */
  if (editor_song_sendsync_get(gui->editor))
    gtk_toggle_button_set_active(togglebutton, TRUE);
  else
    gtk_toggle_button_set_active(togglebutton, FALSE);

  g_signal_handlers_unblock_by_func(entry, on_entry_songname_changed, gui);
  g_signal_handlers_unblock_by_func(togglebutton, on_checkbutton_sendsync_toggled, gui);
}

/* Refresh widgets related to the properties of the current instrument */
void gui_instrument_refresh(struct gui *gui)
{
  GtkSpinButton *spinbutton = GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_instrument"));
  GtkEntry *entry = GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrumentproperties_name"));
  GtkComboBox *combobox = GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_instrumentproperties_midiinterface"));
  GtkRange *range_volume = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_volume"));
  GtkRange *range_transpose = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_transpose"));
  GtkRange *range_hold = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_hold"));
  GtkRange *range_midich = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_midich"));
  char midiinterface[256], *name;
  int i, n = 0, current, selected = 0;

  current = editor_song_instrument_current_midiinterface_get(gui->editor);

  /* Refresh the main window widgets */
  g_signal_handlers_block_by_func(spinbutton, on_spinbutton_instrument_changed, gui);
  g_signal_handlers_block_by_func(entry, on_entry_instrumentproperties_name_changed, gui);
  g_signal_handlers_block_by_func(combobox, on_combobox_instrumentproperties_midiinterface_changed, gui);
  g_signal_handlers_block_by_func(range_volume, on_scale_volume_value_changed, gui);
  g_signal_handlers_block_by_func(range_transpose, on_scale_transpose_value_changed, gui);
  g_signal_handlers_block_by_func(range_hold, on_scale_hold_value_changed, gui);
  g_signal_handlers_block_by_func(range_midich, on_scale_midich_value_changed, gui);
  gtk_spin_button_set_value(spinbutton, gui->editor->instrument + 1);

  name = editor_song_instrument_current_name_get(gui->editor);
  if (name != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrument")), name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrument")), "");

  /* Refresh the instrument window widgets */
  if (name != NULL)
    gtk_entry_set_text(entry, name);
  else
    gtk_entry_set_text(entry, "");

  GtkListStore *store = GTK_LIST_STORE(gtk_combo_box_get_model(combobox));
  gtk_list_store_clear(store);
  for (i = 0; i < gui->editor->midi->numoutputs; i++) {
    GtkTreeIter iter;

    if (gui->editor->midi->outputs[i]->type != MIDI_ALSA) {
      if (gui->editor->midi->outputs[i]->name != NULL)
	snprintf(midiinterface, 256, "%s", gui->editor->midi->outputs[i]->name);
      else
	snprintf(midiinterface, 256, "Unknown");
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, 0, midiinterface, 1, i, -1);
      if (i == current)
	selected = n;
      n++;
    } else {
      struct midi_alsa *alsa = (struct midi_alsa *)gui->editor->midi->outputs[i]->data;
      if (alsa->subs != NULL) {
	if (gui->editor->midi->outputs[i]->name != NULL)
	  snprintf(midiinterface, 256, "%s (%d:%d)", gui->editor->midi->outputs[i]->name, alsa->client, alsa->port);
	else
	  snprintf(midiinterface, 256, "Unknown (%d:%d)", alsa->client, alsa->port);
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 0, midiinterface, 1, i, -1);
	if (i == current)
	  selected = n;
	n++;
      }
    }
  }
  gtk_combo_box_set_active(combobox, selected);
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_volume), editor_song_instrument_current_defaultvelocity_get(gui->editor));
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_transpose), editor_song_instrument_current_transpose_get(gui->editor));
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_hold), editor_song_instrument_current_hold_get(gui->editor));
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_midich), editor_song_instrument_current_midichannel_get(gui->editor) + 1);

  g_signal_handlers_unblock_by_func(spinbutton, on_spinbutton_instrument_changed, gui);
  g_signal_handlers_unblock_by_func(entry, on_entry_instrumentproperties_name_changed, gui);
  g_signal_handlers_unblock_by_func(combobox, on_combobox_instrumentproperties_midiinterface_changed, gui);
  g_signal_handlers_unblock_by_func(range_volume, on_scale_volume_value_changed, gui);
  g_signal_handlers_unblock_by_func(range_transpose, on_scale_transpose_value_changed, gui);
  g_signal_handlers_unblock_by_func(range_hold, on_scale_hold_value_changed, gui);
  g_signal_handlers_unblock_by_func(range_midich, on_scale_midich_value_changed, gui);
}

/* Refresh tempo related widgets */
void gui_tempo_refresh(struct gui *gui)
{
  GtkRange *scale_tempo = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_tempo"));
  GtkRange *scale_tpl = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_tpl"));

  /* Refresh the song properties window widgets */
  g_signal_handlers_block_by_func(scale_tempo, on_scale_tempo_value_changed, gui);
  g_signal_handlers_block_by_func(scale_tpl, on_scale_tpl_value_changed, gui);

  gtk_adjustment_set_value(gtk_range_get_adjustment(scale_tempo), editor_song_tempo_get(gui->editor));
  gtk_adjustment_set_value(gtk_range_get_adjustment(scale_tpl), editor_song_ticksperline_get(gui->editor));

  g_signal_handlers_unblock_by_func(scale_tempo, on_scale_tempo_value_changed, gui);
  g_signal_handlers_unblock_by_func(scale_tpl, on_scale_tpl_value_changed, gui);
}

/* Refresh the main window title */
void gui_window_title_refresh(struct gui *gui)
{
  gtk_window_set_title(GTK_WINDOW(gui->window_main), gui->editor->song->name);
}

/* Refresh the statusbar */
void gui_statusbar_refresh(struct gui *gui)
{
  GtkLabel *status = GTK_LABEL(glade_xml_get_widget(gui->xml, "label_status"));

  /* Status bar */
  switch (editor_player_mode_get(gui->editor)) {
  case MODE_PLAY_SONG:
    gtk_label_set_text(status, "Playing song");
    break;
  case MODE_PLAY_BLOCK:
    gtk_label_set_text(status, "Playing block");
    break;
  case MODE_IDLE:
    gtk_label_set_text(status, "Stopped");
    break;
  }
}

/* Refresh the main window information */
void gui_info_refresh(struct gui *gui)
{
  char *section, *playseq, *position, *block, *cmdpage;
  /* Calculate lengths of the info strings and allocate memory for them */
  int sectionlength = 8 + intlen(gui->editor->section + 1) + 1 + intlen(editor_song_numsections_get(gui->editor)) + 1;
  int playseqlength;
  int positionlength = 9 + intlen(gui->editor->position + 1) + 1 + intlen(editor_song_playseq_current_length_get(gui->editor)) + 1;
  int blocklength;
  int cmdpagelength = 13 + intlen(gui->editor->cmdpage + 1) + 1 + intlen(editor_song_block_current_commandpages_get(gui->editor)) + 1;

  if (editor_song_playseq_current_name_get(gui->editor) != NULL)
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(editor_song_numplayseqs_get(gui->editor)) + 2 + strlen(editor_song_playseq_current_name_get(gui->editor)) + 1;
  else
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(editor_song_numplayseqs_get(gui->editor)) + 1;

  if (editor_song_block_current_name_get(gui->editor) != NULL)
    blocklength = 6 + intlen(gui->editor->block + 1) + 1 + intlen(editor_song_numblocks_get(gui->editor)) + 2 + strlen(editor_song_block_current_name_get(gui->editor)) + 1;
  else
    blocklength = 6 + intlen(gui->editor->block + 1) + 1 + intlen(editor_song_numblocks_get(gui->editor)) + 2;

  section = malloc(sectionlength);
  playseq = malloc(playseqlength);
  position = malloc(positionlength);
  block = malloc(blocklength);
  cmdpage = malloc(cmdpagelength);

  /* Print the info strings */
  snprintf(section, sectionlength, "Section %d/%d", gui->editor->section + 1, editor_song_numsections_get(gui->editor));
  if (editor_song_playseq_current_name_get(gui->editor) != NULL)
    snprintf(playseq, playseqlength, "Playing Sequence %d/%d: %s", gui->editor->playseq + 1, editor_song_numplayseqs_get(gui->editor), editor_song_playseq_current_name_get(gui->editor));
  else
    snprintf(playseq, playseqlength, "Playing Sequence %d/%d", gui->editor->playseq + 1, editor_song_numplayseqs_get(gui->editor));
  snprintf(position, playseqlength, "Position %d/%d", gui->editor->position + 1, editor_song_playseq_current_length_get(gui->editor));
  if (editor_song_block_current_name_get(gui->editor) != NULL)
    snprintf(block, blocklength, "Block %d/%d: %s", gui->editor->block + 1, editor_song_numblocks_get(gui->editor), editor_song_block_current_name_get(gui->editor));
  else
    snprintf(block, blocklength, "Block %d/%d", gui->editor->block + 1, editor_song_numblocks_get(gui->editor));
  snprintf(cmdpage, cmdpagelength, "Command Page %d/%d", gui->editor->cmdpage + 1, editor_song_block_current_commandpages_get(gui->editor));
  /* Refresh the info string widgets */
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_section")), section);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_playseq")), playseq);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_position")), position);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_block")), block);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_commandpage")), cmdpage);

  /* Free the info strings */
  free(section);
  free(playseq);
  free(position);
  free(block);
  free(cmdpage);
}

/* Refresh the timer */
void gui_timer_refresh(struct gui *gui, unsigned int seconds)
{
  char time[6];

  /* Refresh the main window widgets if the window exists */
  if (gui->window_main != NULL && GTK_IS_WINDOW(gui->window_main)) {
    snprintf(time, 6, "%.2d:%.2d", seconds / 60, seconds % 60);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_timer")), time);
  }
}

/* Recreate and refresh the block list window contents */
void gui_blocklist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char block[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(block, 10, "%d", gui->editor->block);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));

  if (recreate) {
    int n = editor_song_numblocks_get(gui->editor);

    /* Recreate the block list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_blocklist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < n; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_BLOCKLIST_COLUMN_NUMBER, i + 1, GUI_BLOCKLIST_COLUMN_NAME, editor_song_block_name_get(gui->editor, i), GUI_BLOCKLIST_COLUMN_TRACKS, editor_song_block_tracks_get(gui->editor, i), GUI_BLOCKLIST_COLUMN_LENGTH, editor_song_block_length_get(gui->editor, i), GUI_BLOCKLIST_COLUMN_COMMAND_PAGES, editor_song_block_commandpages_get(gui->editor, i), -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_blocklist_changed, gui);
  }

  /* Move back to the previously selected block */
  treepath = gtk_tree_path_new_from_string(block);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Make the delete button sensitive/insensitive */
  if (editor_song_numblocks_get(gui->editor) > 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_blocklist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_blocklist_delete"), FALSE);
}

/* Refresh the playing sequence list window widgets */
void gui_playseqlist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char playseq[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(playseq, 10, "%d", gui->editor->playseq);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseqlist"));

  if (recreate) {
    int n = editor_song_numplayseqs_get(gui->editor);

    /* Recreate the playing sequence list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_playseqlist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < n; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_PLAYSEQLIST_COLUMN_NUMBER, i + 1, GUI_PLAYSEQLIST_COLUMN_NAME, editor_song_playseq_name_get(gui->editor, i), -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_playseqlist_changed, gui);
  }

  /* Move to the correct position */
  treepath = gtk_tree_path_new_from_string(playseq);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Make the delete playing sequence button sensitive/insensitive */
  if (editor_song_numplayseqs_get(gui->editor) > 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseqlist_delete")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseqlist_delete")), FALSE);
}

/* Refresh the playing sequence window widgets */
void gui_playseq_refresh(struct gui *gui, gboolean recreate)
{
  int i, playseqlength;
  char playseqpos[10], *title;
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  char *name;

  snprintf(playseqpos, 10, "%d", gui->editor->position);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));

  name = editor_song_playseq_current_name_get(gui->editor);
  /* Calculate length of the window title and allocate memory for it */
  if (name != NULL)
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(editor_song_numplayseqs_get(gui->editor)) + 2 + strlen(name) + 1;
  else
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(editor_song_numplayseqs_get(gui->editor)) + 1;

  title = malloc(playseqlength);

  /* Print the window title */
  if (name != NULL)
    snprintf(title, playseqlength, "Playing Sequence %d/%d: %s", gui->editor->playseq + 1, editor_song_numplayseqs_get(gui->editor), name);
  else
    snprintf(title, playseqlength, "Playing Sequence %d/%d", gui->editor->playseq + 1, editor_song_numplayseqs_get(gui->editor));

  if (recreate) {
    int n = editor_song_playseq_current_length_get(gui->editor);

    /* Recreate the playing sequence */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_playseq_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < n; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_PLAYSEQ_COLUMN_POSITION, i + 1, GUI_PLAYSEQ_COLUMN_BLOCK_NUMBER, editor_song_playseq_current_block_get(gui->editor, i) + 1, GUI_PLAYSEQ_COLUMN_BLOCK_NAME, editor_song_block_name_get(gui->editor, editor_song_playseq_current_block_get(gui->editor, i)), -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_playseq_changed, gui);
  }

  /* Move to the correct position */
  treepath = gtk_tree_path_new_from_string(playseqpos);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Set window title */
  gtk_window_set_title(GTK_WINDOW(glade_xml_get_widget(gui->xml, "dialog_playseq")), title);

  /* Make the delete position button sensitive/insensitive */
  if (editor_song_playseq_current_length_get(gui->editor) > 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_delete")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_delete")), FALSE);

  free(title);
}

/* Refresh all section list widgets */
void gui_sectionlist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char section[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(section, 10, "%d", gui->editor->section);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));

  if (recreate) {
    int n = editor_song_numsections_get(gui->editor);

    /* Recreate the section list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_sectionlist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < n; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_SECTIONLIST_COLUMN_SECTION, i + 1, GUI_SECTIONLIST_COLUMN_PLAYSEQ_NUMBER, editor_song_section_get(gui->editor, i) + 1, GUI_SECTIONLIST_COLUMN_PLAYSEQ_NAME, editor_song_playseq_name_get(gui->editor, editor_song_section_get(gui->editor, i)), -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_sectionlist_changed, gui);
  }

  /* Move back to the previously selected section */
  treepath = gtk_tree_path_new_from_string(section);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Make the delete button sensitive/insensitive */
  if (editor_song_numsections_get(gui->editor) > 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_delete"), FALSE);
}

/* Refresh the message list widgets */
void gui_messagelist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char oldmessage[10];
  GtkTreeView *treeview;

  if (editor_song_nummessages_get(gui->editor) == 0)
    gui->messagelist_message = -1;

  snprintf(oldmessage, 10, "%d", gui->messagelist_message);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));

  if (recreate) {
    int n = editor_song_nummessages_get(gui->editor);

    /* Recreate the Message list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_messagelist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < n; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_MESSAGELIST_COLUMN_NUMBER, i, GUI_MESSAGELIST_COLUMN_NAME, editor_song_message_name_get(gui->editor, i), GUI_MESSAGELIST_COLUMN_LENGTH, editor_song_message_length_get(gui->editor, i), GUI_MESSAGELIST_COLUMN_AUTOSEND, editor_song_message_autosend_get(gui->editor, i), -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_messagelist_changed, gui);
  }

  /* Move back to the previously selected message and make widgets sensitive */
  if (gui->messagelist_message != -1) {
    gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"))), gtk_tree_path_new_from_string(oldmessage));

    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_send"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_receive"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_load"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_save"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_delete"), TRUE);
  } else {
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_send"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_receive"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_load"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_save"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_delete"), FALSE);
  }
}

void gui_trackvolumes_refresh(struct gui *gui)
{
  /* Refresh the track volumes window widgets */
  int i;
  GtkWidget *table = glade_xml_get_widget(gui->xml, "table_trackvolumes");
  int n = editor_song_maxtracks_get(gui->editor);
  char track[10];

  /* Destroy all children */
  gtk_container_foreach(GTK_CONTAINER(table), (GtkCallback) gtk_widget_destroy, NULL);
  gtk_table_resize(GTK_TABLE(table), 5, n);

  /* Create widgets for each channel */
  for (i = 0; i < n; i++) {
    unsigned int volume = editor_song_track_volume_get(gui->editor, i);
    unsigned int mute = editor_song_track_mute_get(gui->editor, i);
    unsigned int solo = editor_song_track_solo_get(gui->editor, i);
    char *name = editor_song_track_name_get(gui->editor, i);
    GtkWidget *w, *x;

    /* Track numbers */
    snprintf(track, 10, "Track %d", i + 1);
    x = gtk_label_new(track);
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_FILL, 2, 4);

    /* Volume sliders */
    w = (GtkWidget *) gtk_adjustment_new(volume, 0, 127, 1, 8, 0);
    g_object_set_data(G_OBJECT(w), "track", (gpointer) i);
    g_signal_connect(G_OBJECT(w), "value_changed", G_CALLBACK(on_adjustment_trackvolumes_volume_changed), gui);
    x = gtk_vscale_new(GTK_ADJUSTMENT(w));
    gtk_range_set_inverted(GTK_RANGE(x), TRUE);
    gtk_scale_set_digits(GTK_SCALE(x), 0);
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 1, 2, GTK_EXPAND, GTK_EXPAND | GTK_FILL, 2, 4);

    /* Mute buttons */
    w = gtk_check_button_new_with_label("Mute");
    g_object_set_data(G_OBJECT(w), "track", (gpointer) i);
    if (mute)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
    g_signal_connect(G_OBJECT(w), "toggled", G_CALLBACK(on_togglebutton_trackvolumes_mute_toggled), gui);
    gtk_table_attach(GTK_TABLE(table), w, i, i + 1, 2, 3, GTK_EXPAND, GTK_FILL, 2, 4);

    /* Solo buttons */
    w = gtk_check_button_new_with_label("Solo");
    g_object_set_data(G_OBJECT(w), "track", (gpointer) i);
    if (solo)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
    g_signal_connect(G_OBJECT(w), "toggled", G_CALLBACK(on_togglebutton_trackvolumes_solo_toggled), gui);
    gtk_table_attach(GTK_TABLE(table), w, i, i + 1, 3, 4, GTK_EXPAND, GTK_FILL, 2, 4);

    /* Track names */
    x = gtk_entry_new();
    gtk_entry_set_width_chars(GTK_ENTRY(x), 7);
    g_object_set_data(G_OBJECT(x), "track", (gpointer) i);
    g_signal_connect(G_OBJECT(x), "changed", G_CALLBACK(on_entry_trackvolumes_name_changed), gui);
    if (name != NULL)
      gtk_entry_set_text(GTK_ENTRY(x), name);
    else
      gtk_entry_set_text(GTK_ENTRY(x), "");
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 4, 5, GTK_EXPAND | GTK_FILL, GTK_FILL, 2, 4);
  }

  gtk_widget_show_all(table);
}

void gui_preferences_refresh(struct gui *gui)
{
  GtkTreeView *treeview_output = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_preferences_midi_interfaces_output"));
  GtkTreeView *treeview_input = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_preferences_midi_interfaces_input"));
  GtkComboBox *combobox = GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_preferences_scheduling_mode"));
  int i;

  /* Refresh the preferences window widgets */
  g_signal_handlers_block_by_func(treeview_output, on_cell_preferences_midi_interfaces_toggled, gui);
  g_signal_handlers_block_by_func(treeview_input, on_cell_preferences_midi_interfaces_toggled, gui);
  g_signal_handlers_block_by_func(combobox, on_combobox_preferences_scheduling_mode_changed, gui);

  /* Recreate the interfaces lists */
  GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview_output));
  gtk_list_store_clear(store);
  for (i = 1; i < gui->editor->midi->numoutputs; i++) {
    GtkTreeIter iter;
    char name[256];
    gboolean enabled = FALSE;
    int client = 0, port = 0;

    if (gui->editor->midi->outputs[i]->type == MIDI_ALSA) {
      struct midi_alsa *alsa = (struct midi_alsa *)gui->editor->midi->outputs[i]->data;

      if (alsa->subs != NULL)
	enabled = TRUE;

      client = alsa->client;
      port = alsa->port;

      if (gui->editor->midi->outputs[i]->name != NULL)
	snprintf(name, 256, "%s (%d:%d)", gui->editor->midi->outputs[i]->name, client, port);
      else
	snprintf(name, 256, "Unknown (%d:%d)", client, port);
    } else {
      if (gui->editor->midi->outputs[i]->name != NULL)
	snprintf(name, 256, "%s", gui->editor->midi->outputs[i]->name);
      else
	snprintf(name, 256, "Unknown");
    }

    gtk_list_store_append(store, &iter);
    gtk_list_store_set(store, &iter, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_NAME, name, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_ENABLED, enabled, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_CLIENT, client, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_PORT, port, -1);
  }

  store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview_input));
  gtk_list_store_clear(store);
  for (i = 0; i < gui->editor->midi->numinputs; i++) {
    GtkTreeIter iter;
    char name[256];
    gboolean enabled = FALSE;
    int client = 0, port = 0;

    if (gui->editor->midi->inputs[i]->type == MIDI_ALSA) {
      struct midi_alsa *alsa = (struct midi_alsa *)gui->editor->midi->inputs[i]->data;

      if (alsa->subs != NULL)
	enabled = TRUE;

      client = alsa->client;
      port = alsa->port;

      if (gui->editor->midi->inputs[i]->name != NULL)
	snprintf(name, 256, "%s (%d:%d)", gui->editor->midi->inputs[i]->name, client, port);
      else
	snprintf(name, 256, "Unknown (%d:%d)", client, port);
    } else {
      if (gui->editor->midi->inputs[i]->name != NULL)
	snprintf(name, 256, "%s", gui->editor->midi->inputs[i]->name);
      else
	snprintf(name, 256, "Unknown");
    }

    gtk_list_store_append(store, &iter);
    gtk_list_store_set(store, &iter, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_NAME, name, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_ENABLED, enabled, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_CLIENT, client, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_PORT, port, -1);
  }

  gtk_combo_box_set_active(combobox, editor_scheduler_get(gui->editor) - SCHED_RTC);
  g_signal_handlers_unblock_by_func(treeview_output, on_cell_preferences_midi_interfaces_toggled, gui);
  g_signal_handlers_unblock_by_func(treeview_input, on_cell_preferences_midi_interfaces_toggled, gui);
  g_signal_handlers_unblock_by_func(combobox, on_combobox_preferences_scheduling_mode_changed, gui);
}

void gui_receive_refresh(struct gui *gui, int bytes, int total)
{
  GtkProgressBar *pbar = GTK_PROGRESS_BAR(glade_xml_get_widget(gui->xml, "progressbar_receive"));
  char b[40];

  /* Update bytes received count: if total is 0 it is unknown and not shown */
  if (total == 0) {
    if (bytes >= 0) {
      snprintf(b, 40, "Bytes received: %d", bytes);
      gtk_progress_bar_pulse(pbar);
    } else
      snprintf(b, 40, "Bytes received: 0");
  } else {
    if (bytes >= 0) {
      snprintf(b, 40, "Bytes received: %d/%d", bytes, total);
      gtk_progress_bar_set_fraction(pbar, (gdouble) bytes / total);
    } else
      snprintf(b, 40, "Bytes received: 0/0");
  }

  gtk_progress_bar_set_text(pbar, b);
}
