/* player.c
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <glib.h>
#include <sched.h>
#include <linux/rtc.h>
#include "tutka.h"
#include "song.h"
#include "midi.h"
#include "player.h"
#include "editor.h"

#define RTC_FREQ 512

/* Refreshes playseq from section and block from position */
static void player_refresh_playseq_and_block(struct player *player)
{
  nullcheck_void(player, player_refresh_playseq_and_block);

  struct song *song;

  song = player->song;

  if (player->section >= song->numsections)
    player->section = 0;
  player->playseq = song->sections[player->section];

  if (player->position >= song->playseqs[player->playseq]->length) {
    player->position = 0;
    player_refresh_playseq_and_block(player);
  } else
    player->block = song->playseqs[player->playseq]->blocknumbers[player->position];
}

/* Advances in section and jumps to the beginning if necessary */
static int player_next_section(struct player *player)
{
  nullcheck_int(player, player_next_section);

  player->section++;
  if (player->section >= player->song->numsections) {
    player->section = 0;
    return 1;
  }
  return 0;
}

/* Advances in playing sequence and jumps to next section if necessary */
static int player_next_playseq(struct player *player)
{
  nullcheck_int(player, player_next_playseq);

  player->position++;
  if (player->position >= player->song->playseqs[player->playseq]->length) {
    player->position = 0;
    return player_next_section(player);
  }
  return 0;
}

/* Plays a note using given instrument on a given track */
void player_play_note(struct player *player, unsigned int instrument, unsigned char note, unsigned char volume, unsigned char track)
{
  nullcheck_void(player, player_play_note);

  int i;
  struct song *song = player->song;

  /* Stop currently playing note */
  for (i = 0; i < player->trackstatus[track]->numinterfaces; i++) {
    struct player_interface_status *status = player->trackstatus[track]->interfaces[i];
    if (status->note != -1) {
      midi_note_off(player->midi->outputs[i], status->midichannel, status->note, 127);
      status->note = -1;
    }
  }

  /* Don't play a note if the instrument does not exist */
  if (instrument < song->numinstruments && song->instruments[instrument] != NULL) {
    /* Set new track status (channel, note, volume, time) of each interface defined in the instrument */
    player->trackstatus[track]->instrument = instrument;
    for (i = 0; i < song->instruments[instrument]->numoutputs; i++) {
      struct instrument_output *output = song->instruments[instrument]->outputs[i];
      struct player_interface_status *status = player->trackstatus[track]->interfaces[output->midiinterface];

      /* Apply note transpose */
      char realnote = note + output->transpose;

      /* Update track status for the selected output */
      status->note = realnote;
      status->midichannel = output->midichannel;
      /* If channels are soloed solo must be taken into account, otherwise not */
      /* NOTE: Current behaviour does NOT ignore muted or non-soloed channels. The channels are played with volume 0 causing note offs. This is by design. Is it relevant? */
      if (player->solo)
	status->volume = output->defaultvelocity * volume / 127 * song->tracks[track]->volume / 127 * song->tracks[track]->solo * (1 - song->tracks[track]->mute) * song->mastervolume / 127;
      else
	status->volume = output->defaultvelocity * volume / 127 * song->tracks[track]->volume / 127 * (1 - song->tracks[track]->mute) * song->mastervolume / 127;
      if (output->hold > 0)
	status->hold = output->hold;
      else
	status->hold = -1;

      /* Make sure the volume isn't too large */
      if (status->volume < 0)
	status->volume = 127;

      /* Play note */
      midi_note_on(player->midi->outputs[output->midiinterface], status->midichannel, status->note, status->volume);
    }
  }
}

/* Stops notes playing at the moment */
void player_stop_notes(struct player *player)
{
  nullcheck_void(player, player_stop_notes);

  int i, j;

  for (i = 0; i < player->song->maxtracks; i++) {
    for (j = 0; j < player->trackstatus[i]->numinterfaces; j++) {
      if (player->trackstatus[i]->interfaces[j]->note != -1) {
	struct player_interface_status *status = player->trackstatus[i]->interfaces[j];

	midi_note_off(player->midi->outputs[j], status->midichannel, status->note, 127);

	status->midichannel = -1;
	status->note = -1;
	status->volume = -1;
	status->hold = -1;
      }
    }
  }
}

/* Stops all notes playing at the moment */
void player_stop_all_notes(struct player *player)
{
  nullcheck_void(player, player_stop_all_notes);

  int i, j, k;

  for (i = 0; i < 16; i++)
    for (j = 0; j < 128; j++)
      for (k = 0; k < player->midi->numoutputs; k++)
	midi_note_off(player->midi->outputs[k], i, j, 127);
}

/* Resets the pitch wheel on all channels */
void player_reset_pitch(struct player *player)
{
  nullcheck_void(player, player_reset_pitch);

  int i, j;

  for (i = 0; i < 16; i++)
    for (j = 0; j < player->midi->numoutputs; j++)
      midi_pitch_wheel(player->midi->outputs[j], i, 64);
}

/* Player thread */
static void player_thread(struct player *player)
{
  nullcheck_void(player, player_thread);

  struct timespec req, rem;
  struct timeval next, now;

  player->tick = 0;
  player->tickssofar = 0;

  if (player->sched == SCHED_RTC) {
    if (ioctl(player->rtc, RTC_UIE_OFF, 0) == -1 || ioctl(player->rtc, RTC_IRQP_SET, RTC_FREQ) == -1 || ioctl(player->rtc, RTC_PIE_ON, 0) == -1)
      player->sched = SCHED_NANOSLEEP;
  }

  next.tv_sec = player->playingstarted.tv_sec;
  next.tv_usec = player->playingstarted.tv_usec;

  while (1) {
    struct timeval tod;
    int i, j, k, changetempo = 0;
    struct song *song;
    struct block *b;
    unsigned char postcommand = 0, postvalue = 0;
    int looped = 0;

    /* Calculate time of next tick (tick every 1000000/((BPM/60)*24) usecs) */
    next.tv_sec += (25 / player->song->tempo) / 10;
    next.tv_usec += ((2500000 / player->song->tempo) % 1000000);
    while (next.tv_usec > 1000000) {
      next.tv_sec++;
      next.tv_usec -= 1000000;
    }

    if (player->sched != SCHED_NONE) {
      gettimeofday(&now, NULL);

      /* Select scheduling type */
      if (player->sched == SCHED_NANOSLEEP) {
	/* Nanosleep: calculate difference between now and the next tick */
	req.tv_sec = next.tv_sec - now.tv_sec;
	req.tv_nsec = (next.tv_usec - now.tv_usec) * 1000;
	while (req.tv_nsec < 0) {
	  req.tv_sec--;
	  req.tv_nsec += 1000000000;
	}

	/* Sleep until the next tick if necessary */
	if (req.tv_sec >= 0)
	  while (nanosleep(&req, &rem) == -1);
      } else {
	/* Rtc: read RTC interrupts until it's time to play again */
	while ((next.tv_sec - now.tv_sec) > 0 || ((next.tv_sec - now.tv_sec) == 0 && (next.tv_usec - now.tv_usec) > 1000000 / RTC_FREQ)) {
	  unsigned long l;
	  read(player->rtc, &l, sizeof(unsigned long));
	  gettimeofday(&now, NULL);
	}
	while ((next.tv_sec - now.tv_sec) > 0 || ((next.tv_sec - now.tv_sec) == 0 && (next.tv_usec - now.tv_usec) > 0)) {
	  gettimeofday(&now, NULL);
	}
      }
    }

    /* Lock */
    if (player->mutex != NULL)
      g_mutex_lock(player->mutex);

    /* Handle this tick */
    for (i = 0; i < player->midi->numoutputs; i++)
      midi_set_tick(player->midi->outputs[i], player->tickssofar);

    if (player->editor != NULL)
      editor_song_lock(player->editor);
    song = player->song;
    b = song->blocks[player->block];

    /* Send MIDI sync if requested */
    if (song->sendsync)
      for (i = 0; i < player->midi->numoutputs; i++)
	midi_clock(player->midi->outputs[i]);

    /* Pass 1: Stop notes if there are new notes about to be played */
    for (i = 0; i < b->tracks; i++) {
      struct player_track_status *trackstatus = player->trackstatus[i];
      int delay = 0;
      unsigned char note = b->notes[(b->tracks * player->line + i) * 2];

      if (note != 0) {
	for (j = 0; j < b->commandpages; j++) {
	  unsigned char command = b->commands[j * b->tracks * b->length * 2 + (b->tracks * player->line + i) * 2];
	  unsigned char value = b->commands[j * b->tracks * b->length * 2 + (b->tracks * player->line + i) * 2 + 1];

	  if (command != 0 || value != 0) {
	    /* Check for previous command if any */
	    if (command == COMMAND_PREVIOUS_COMMAND_VALUE) {
	      if (value != 0)
		command = trackstatus->previouscommand;
	    } else
	      trackstatus->previouscommand = command;

	    switch (command) {
	    case COMMAND_RETRIGGER:
	      delay = -value;
	      break;
	    case COMMAND_DELAY:
	      delay = value;
	      break;
	    }
	  }
	}

	/* Stop currently playing note */
	if ((delay >= 0 && player->tick == delay) || (delay < 0 && player->tick % (-delay) == 0)) {
	  for (j = 0; j < trackstatus->numinterfaces; j++) {
	    struct player_interface_status *interfacestatus = trackstatus->interfaces[j];
	    if (interfacestatus->note != -1) {
	      midi_note_off(player->midi->outputs[j], interfacestatus->midichannel, interfacestatus->note, 127);
	      interfacestatus->note = -1;
	    }
	  }
	}
      }
    }

    /* Pass 2: Play notes scheduled to be played */
    for (i = 0; i < b->tracks; i++) {
      struct player_track_status *trackstatus = player->trackstatus[i];
      unsigned int volume = 127;
      int delay = 0, hold = -1;
      unsigned char note = b->notes[(b->tracks * player->line + i) * 2];
      unsigned char instrument = b->notes[(b->tracks * player->line + i) * 2 + 1];

      /* Handle commands on all command pages */
      for (j = 0; j < b->commandpages; j++) {
	unsigned char command = b->commands[j * b->tracks * b->length * 2 + (b->tracks * player->line + i) * 2];
	unsigned char value = b->commands[j * b->tracks * b->length * 2 + (b->tracks * player->line + i) * 2 + 1];

	if (command != 0 || value != 0) {
	  unsigned int *outputs;
	  unsigned int numoutputs;

	  /* Check which MIDI interface/channel pairs the command will affect */
	  if (instrument != 0) {
	    /* Instrument number defines MIDI interfaces/channels */
	    numoutputs = song->instruments[instrument - 1]->numoutputs;
	    outputs = (unsigned int *)calloc(numoutputs * 2, sizeof(unsigned int));
	    for (k = 0; k < numoutputs; k++) {
	      outputs[k * 2 + 0] = song->instruments[instrument - 1]->outputs[k]->midiinterface;
	      outputs[k * 2 + 1] = song->instruments[instrument - 1]->outputs[k]->midichannel;
	    }
	  } else {
	    /* Note playing defines MIDI interfaces/channels */
	    numoutputs = player->trackstatus[i]->numinterfaces;
	    outputs = (unsigned int *)calloc(numoutputs * 2, sizeof(unsigned int));
	    for (k = 0; k < numoutputs; k++) {
	      outputs[k * 2 + 0] = k;
	      outputs[k * 2 + 1] = player->trackstatus[i]->interfaces[k]->midichannel;
	    }
	  }

	  /* Handle all relevant MIDI interfaces/channels */
	  for (k = 0; k < numoutputs; k++) {
	    int midiinterface = outputs[k * 2 + 0];
	    int midichannel = outputs[k * 2 + 1];
	    struct midi_interface *midi = player->midi->outputs[midiinterface];
	    struct player_interface_status *interfacestatus = trackstatus->interfaces[midiinterface];

	    /* Check for previous command if any */
	    if (command == COMMAND_PREVIOUS_COMMAND_VALUE) {
	      if (value != 0)
		command = trackstatus->previouscommand;
	    } else
	      trackstatus->previouscommand = command;

	    switch (command) {
	    case COMMAND_PITCH_WHEEL:
	      /* Program change can be sent if the MIDI channel is known */
	      if (midichannel != -1) {
		if (value < 0x80) {
		  if (player->tick == 0) {
		    midi_pitch_wheel(midi, midichannel, value);
		    player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_PITCH_WHEEL] = value;
		  }
		} else {
		  if (player->tick < song->ticksperline - 1) {
		    int delta = (value - 0x80 - player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_PITCH_WHEEL]) / ((int)song->ticksperline - 1);

		    midi_pitch_wheel(midi, midichannel, player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_PITCH_WHEEL] + player->tick * delta);
		  } else {
		    midi_pitch_wheel(midi, midichannel, value - 0x80);
		    player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_PITCH_WHEEL] = value - 0x80;
		  }
		}
	      }
	      break;
	    case COMMAND_PROGRAM_CHANGE:
	      /* Pitch wheel can be set if the MIDI channel is known */
	      if (midichannel != -1 && player->tick == 0)
		midi_program_change(midi, midichannel, value & 0x7f);
	      break;
	    case COMMAND_END_BLOCK:
	      /* Only on last tick */
	      if (player->tick == song->ticksperline - 1) {
		postcommand = COMMAND_END_BLOCK;
		postvalue = value;
	      }
	      break;
	    case COMMAND_PLAYSEQ_POSITION:
	      /* Only on last tick */
	      if (player->tick == song->ticksperline - 1) {
		postcommand = COMMAND_PLAYSEQ_POSITION;
		postvalue = value;
	      }
	      break;
	    case COMMAND_SEND_MESSAGE:
	      /* Only on first tick */
	      if (player->tick == 0 && value < song->nummessages)
		midi_write_raw(midi, song->messages[value]->data, song->messages[value]->length);
	      break;
	    case COMMAND_HOLD:
	      hold = value;
	      break;
	    case COMMAND_RETRIGGER:
	      delay = -value;
	      break;
	    case COMMAND_DELAY:
	      delay = value;
	      break;
	    case COMMAND_VELOCITY:
	      if (note != 0) {
		volume = value;
		if (midichannel != -1)
		  player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_AFTERTOUCH] = value;
	      } else {
		/* Note playing defines MIDI channel */
		midichannel = interfacestatus->midichannel;

		if (midichannel != -1) {
		  if (value < 0x80) {
		    if (player->tick == 0) {
		      if (value > 0) {
			midi_aftertouch(midi, midichannel, interfacestatus->note, value);
			player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_AFTERTOUCH] = value;
		      } else {
			midi_note_off(midi, midichannel, interfacestatus->note, 127);
			interfacestatus->note = -1;
		      }
		    }
		  } else {
		    if (player->tick < song->ticksperline - 1) {
		      int delta = (value - 0x80 - player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_AFTERTOUCH]) / ((int)song->ticksperline - 1);

		      midi_aftertouch(midi, midichannel, interfacestatus->note, player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_AFTERTOUCH] + player->tick * delta);
		    } else {
		      midi_aftertouch(midi, midichannel, interfacestatus->note, value - 0x80);
		      player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_AFTERTOUCH] = value - 0x80;
		    }
		  }
		}
	      }
	      break;
	    case COMMAND_CHANNEL_PRESSURE:
	      /* Channel pressure can be set if the MIDI channel is known */
	      if (midichannel != -1) {
		if (value < 0x80) {
		  if (player->tick == 0) {
		    midi_channel_pressure(midi, midichannel, value);
		    player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_CHANNEL_PRESSURE] = value;
		  }
		} else {
		  if (player->tick < song->ticksperline - 1) {
		    int delta = (value - 0x80 - player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_CHANNEL_PRESSURE]) / ((int)song->ticksperline - 1);

		    midi_channel_pressure(midi, midichannel, player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_CHANNEL_PRESSURE] + player->tick * delta);
		  } else {
		    midi_channel_pressure(midi, midichannel, value - 0x80);
		    player->midicontrollervalues[midiinterface][midichannel * VALUES + VALUES_CHANNEL_PRESSURE] = value - 0x80;
		  }
		}
	      }
	      break;
	    case COMMAND_TPL:
	      if (value == 0) {
		/* Only on last tick */
		if (player->tick == song->ticksperline - 1)
		  postcommand = COMMAND_TPL;
	      } else {
		song_set_tpl(song, value);
		changetempo = 1;
	      }
	      break;
	    case COMMAND_TEMPO:
	      if (value == 0) {
		/* Only on last tick */
		if (player->tick == song->ticksperline - 1)
		  postcommand = COMMAND_TEMPO;
	      } else {
		song_set_tempo(song, value);
		midi_tempo(midi, value);
		changetempo = 1;
	      }
	      break;
	    }

	    /* Handle MIDI controllers */
	    if (command >= COMMAND_MIDI_CONTROLLERS) {
	      /* MIDI controllers can be set if the MIDI channel is known */
	      if (midichannel != -1) {
		if (value < 0x80) {
		  if (player->tick == 0) {
		    midi_controller(midi, midichannel, command - COMMAND_MIDI_CONTROLLERS, value);
		    player->midicontrollervalues[midiinterface][midichannel * VALUES + command - COMMAND_MIDI_CONTROLLERS] = value;
		  }
		} else {
		  if (player->tick < song->ticksperline - 1) {
		    int delta = (value - 0x80 - player->midicontrollervalues[midiinterface][midichannel * VALUES + command - COMMAND_MIDI_CONTROLLERS]) / ((int)song->ticksperline - 1);

		    midi_controller(midi, midichannel, command - COMMAND_MIDI_CONTROLLERS, player->midicontrollervalues[midiinterface][midichannel * VALUES + command - COMMAND_MIDI_CONTROLLERS] + player->tick * delta);
		  } else {
		    midi_controller(midi, midichannel, command - COMMAND_MIDI_CONTROLLERS, value - 0x80);
		    player->midicontrollervalues[midiinterface][midichannel * VALUES + command - COMMAND_MIDI_CONTROLLERS] = value - 0x80;
		  }
		}
	      }
	    }
	  }
	  free(outputs);
	}
      }

      /* Is there a note to play? */
      if (note != 0 && ((delay >= 0 && player->tick == delay) || (delay < 0 && player->tick % (-delay) == 0))) {
	struct instrument *instr = NULL;

	note--;

	/* Use previous instrument if none defined */
	if (instrument == 0)
	  instrument = trackstatus->instrument + 1;

	/* Play note if instrument is defined */
	if (instrument != 0)
	  player_play_note(player, instrument - 1, note, volume, i);

	if (instrument <= song->numinstruments)
	  instr = song->instruments[instrument - 1];

	if (instr != NULL) {
	  for (j = 0; j < instr->numoutputs; j++) {
	    struct instrument_output *output = instr->outputs[j];
	    struct player_interface_status *interfacestatus = trackstatus->interfaces[output->midiinterface];

	    /* If no hold value was defined use the instrument's hold value */
	    if (hold == -1)
	      hold = output->hold;

	    if (hold == 0)
	      interfacestatus->hold = -1;
	    else
	      interfacestatus->hold = hold;
	  }
	}
      }

      /* First tick, no note but instrument defined? */
      for (j = 0; j < trackstatus->numinterfaces; j++) {
	struct player_interface_status *interfacestatus = trackstatus->interfaces[j];
	if (player->tick == 0 && note == 0 && instrument > 0 && interfacestatus->hold >= 0)
	  for (k = 0; k < song->instruments[instrument - 1]->numoutputs; k++)
	    if (song->instruments[instrument - 1]->outputs[k]->midiinterface == j)
	      interfacestatus->hold += song->instruments[instrument - 1]->outputs[k]->hold;
      }
    }

    /* Decrement hold times of notes and stop notes that should be stopped */
    for (i = 0; i < song->maxtracks; i++) {
      struct player_track_status *trackstatus = player->trackstatus[i];
      for (j = 0; j < trackstatus->numinterfaces; j++) {
	struct player_interface_status *interfacestatus = trackstatus->interfaces[j];
	if (interfacestatus->hold >= 0) {
	  interfacestatus->hold--;
	  if (interfacestatus->hold < 0 && interfacestatus->note != -1) {
	    midi_note_off(player->midi->outputs[j], interfacestatus->midichannel, interfacestatus->note, 127);
	    interfacestatus->note = -1;
	  }
	}
      }
    }

    /* Next tick */
    player->tickssofar++;
    player->tick++;
    player->tick %= song->ticksperline;

    /* Advance and handle post commands if ticksperline ticks have passed */
    if (player->tick == 0) {
      int changeblock = 0;

      player->line++;

      switch (postcommand) {
      case COMMAND_END_BLOCK:
	player->line = postvalue;
	if (player->mode == MODE_PLAY_SONG) {
	  looped = player_next_playseq(player);
	  changeblock = 1;
	}
	break;
      case COMMAND_PLAYSEQ_POSITION:
	player->line = 0;
	player->position = postvalue;
	if (player->position >= song->playseqs[player->playseq]->length) {
	  player->position = 0;
	  looped = player_next_section(player);
	}
	changeblock = 1;
	break;
      case COMMAND_TEMPO:
	/* COMMAND_TPL and COMMAND_TEMPO can only mean "stop" as stop cmds */
	player->killthread = 1;
	return;
	break;
      default:
	/* Advance in block */
	if (player->line >= song->blocks[player->block]->length) {
	  player->line = 0;
	  if (player->mode == MODE_PLAY_SONG) {
	    looped = player_next_playseq(player);
	    changeblock = 1;
	  }
	}
	break;
      }

      if (changeblock == 1)
	player_refresh_playseq_and_block(player);

      if (player->sched != SCHED_NONE)
	gettimeofday(&tod, NULL);

      /* Tell the editor about the new position if an editor exists */
      if (player->editor != NULL) {
	editor_set_section(player->editor, player->section);
	editor_set_playseq(player->editor, player->playseq);
	editor_set_position(player->editor, player->position);
	editor_set_block(player->editor, player->block);
	editor_set_line(player->editor, player->line);
	if (player->sched != SCHED_NONE)
	  editor_set_time(player->editor, (int)(player->playedsofar.tv_sec * 1000 + player->playedsofar.tv_usec / 1000 + (tod.tv_sec * 1000 + tod.tv_usec / 1000) - (player->playingstarted.tv_sec * 1000 + player->playingstarted.tv_usec / 1000)) / 1000);
	if (changetempo)
	  editor_set_tempo(player->editor);
      }
    }
    if (player->editor != NULL)
      editor_song_unlock(player->editor);

    /* Check whether this thread should be killed */
    if (player->killthread || (player->sched == SCHED_NONE && looped))
      break;
    if (player->mutex != NULL)
      g_mutex_unlock(player->mutex);
  }

  if (player->sched != SCHED_NONE) {
    /* Calculate how long the song has been playing */
    gettimeofday(&now, NULL);
    now.tv_sec -= player->playingstarted.tv_sec;
    if (now.tv_usec >= player->playingstarted.tv_usec)
      now.tv_usec -= player->playingstarted.tv_usec;
    else {
      now.tv_usec += 1000000 - player->playingstarted.tv_usec;
      now.tv_sec--;
    }
    player->playedsofar.tv_sec += now.tv_sec;
    player->playedsofar.tv_usec += now.tv_usec;
    if (player->playedsofar.tv_usec > 1000000) {
      player->playedsofar.tv_sec++;
      player->playedsofar.tv_usec -= 1000000;
    }

    if (player->sched == SCHED_RTC)
      ioctl(player->rtc, RTC_PIE_OFF, 0);

    /* Stop all playing notes */
    player_stop_notes(player);
  }

  /* The mutex is locked if the thread was killed and loop broken */
  if (player->mutex != NULL)
    g_mutex_unlock(player->mutex);

  /* Tell the editor playing has stopped */
  if (player->editor != NULL)
    editor_stop(player->editor);
}

/* Plays a song to a MIDI device witout any scheduling (for export) */
void player_midi_export(struct song *song, struct midi_interface *midi)
{
  int i, j;
  struct player *player = (struct player *)calloc(1, sizeof(struct player));

  /* Check how many MIDI interfaces are used in the song */
  for (i = 0; i < song->numinstruments; i++) {
    struct instrument *instrument = song->instruments[i];
    for (j = 0; j < instrument->numoutputs; j++) {
      struct instrument_output *output = instrument->outputs[j];
      if (output->midiinterface > player->midi->numoutputs)
	player->midi->numoutputs = output->midiinterface;
    }
  }
  player->midi->numoutputs++;

  player->song = song;
  player->midi->outputs = (struct midi_interface **)calloc(player->midi->numoutputs, sizeof(struct midi_interface *));
  for (i = 0; i < player->midi->numoutputs; i++)
    player->midi->outputs[i] = midi;
  player->sched = SCHED_NONE;
  player->mode = MODE_PLAY_SONG;
  player_refresh_playseq_and_block(player);

  /* Initialize track status array */
  player_trackstatus_create(player, 0);

  /* Create MIDI controller values array */
  player->midicontrollervalues = (unsigned char **)calloc(player->midi->numoutputs, sizeof(unsigned char *));
  for (i = 0; i < player->midi->numoutputs; i++)
    player->midicontrollervalues[i] = (unsigned char *)calloc(16 * VALUES, sizeof(unsigned char));

  /* Send messages to be autosent */
  for (i = 0; i < player->song->nummessages; i++) {
    if (player->song->messages[i]->autosend)
      midi_write_raw(player->midi->outputs[0], player->song->messages[i]->data, player->song->messages[i]->length);
  }

  /* Set tempo */
  midi_tempo(player->midi->outputs[0], player->song->tempo);

  player_thread(player);
  player_stop_notes(player);

  /* Free the track status array */
  if (player->trackstatus != NULL) {
    for (i = 0; i < player->song->maxtracks; i++) {
      for (j = 0; j < player->trackstatus[i]->numinterfaces; j++)
	free(player->trackstatus[i]->interfaces[j]);
      free(player->trackstatus[i]->interfaces);
      free(player->trackstatus[i]);
    }
    free(player->trackstatus);
  }

  /* Free MIDI interface array */
  free(player->midi->outputs);
  free(player);
}

/* Creates a player for a song */
struct player *player_open(struct song *song, struct editor *editor, struct midi *midi, unsigned int scheduler)
{
  struct player *player = (struct player *)calloc(1, sizeof(struct player));
  int i, j;

  player->song = song;
  player->editor = editor;
  player->midi = midi;
  player->sched = scheduler;
  if (player->sched == SCHED_RTC) {
    player->rtc = open("/dev/rtc", O_RDONLY);
    if (player->rtc != -1)
      player->sched = SCHED_RTC;
    else
      player->sched = SCHED_NANOSLEEP;
  } else
    player->rtc = -1;
  player->mutex = g_mutex_new();

  /* Initialize track status array */
  player_trackstatus_create(player, 0);

  /* Create MIDI controller values array */
  player->nummidicontrollervalues = player->midi->numoutputs;
  player->midicontrollervalues = (unsigned char **)calloc(player->nummidicontrollervalues, sizeof(unsigned char *));
  for (i = 0; i < player->nummidicontrollervalues; i++)
    player->midicontrollervalues[i] = (unsigned char *)calloc(16 * VALUES, sizeof(unsigned char));

  /* Check solo status */
  player_check_solo(player);

  /* Send messages to be autosent */
  for (i = 0; i < player->song->nummessages; i++) {
    if (player->song->messages[i]->autosend)
      for (j = 0; j < player->midi->numoutputs; j++)
	midi_write_raw(player->midi->outputs[j], player->song->messages[i]->data, player->song->messages[i]->length);
  }

  return player;
}

/* Closes a player for a song */
void player_close(struct player *player)
{
  nullcheck_void(player, player_close);

  int i, j;

  /* Stop the player */
  player_stop(player);

  /* Free the mutex */
  if (player->mutex != NULL)
    g_mutex_free(player->mutex);

  /* Free MIDI controller values array */
  if (player->midicontrollervalues) {
    for (i = 0; i < player->nummidicontrollervalues; i++)
      free(player->midicontrollervalues[i]);
    free(player->midicontrollervalues);
  }

  /* Free the track status array */
  if (player->trackstatus != NULL) {
    for (i = 0; i < player->song->maxtracks; i++) {
      for (j = 0; j < player->trackstatus[i]->numinterfaces; j++)
	free(player->trackstatus[i]->interfaces[j]);
      free(player->trackstatus[i]->interfaces);
      free(player->trackstatus[i]);
    }
    free(player->trackstatus);
  }

  /* Close RTC */
  if (player->rtc != -1)
    close(player->rtc);

  free(player);
}

/* Starts the player thread */
void player_start(struct player *player, unsigned int mode, int section, int position, int block, int cont)
{
  nullcheck_void(player, player_start);

  player_stop(player);

  player->mode = mode;
  player->tick = 0;
  player->tickssofar = 0;

  switch (mode) {
  case MODE_PLAY_SONG:
    if (cont) {
      player->section = section;
      player->position = position;
    } else {
      player->section = 0;
      player->position = 0;
      player->line = 0;
    }
    player_refresh_playseq_and_block(player);
    break;
  case MODE_PLAY_BLOCK:
    player->block = block;
    if (!cont)
      player->line = 0;
    break;
  }

  /* Get the starting time */
  if (cont)
    player_reset_time(player, FALSE);
  else
    player_reset_time(player, TRUE);

  /* Create a new thread only if there isn't one already */
  if (player->thread == NULL) {
    player->thread = g_thread_create((GThreadFunc) player_thread, player, TRUE, NULL);
    g_thread_set_priority(player->thread, G_THREAD_PRIORITY_URGENT);
  }
}

/* Kills the player thread */
void player_stop(struct player *player)
{
  nullcheck_void(player, player_stop);

  if (player->mode != MODE_IDLE)
    player->mode = MODE_IDLE;

  if (player->thread != NULL) {
    g_mutex_lock(player->mutex);
    player->killthread = 1;
    g_mutex_unlock(player->mutex);
    g_thread_join(player->thread);

    player->thread = NULL;
    player->killthread = 0;
  } else
    player_stop_notes(player);
}

/* Reallocate track status array */
void player_trackstatus_create(struct player *player, int oldmax)
{
  nullcheck_void(player, player_trackstatus_create);

  int i, j;

  /* Free the extraneous status structures */
  if (player->song->maxtracks < oldmax)
    for (i = player->song->maxtracks; i < oldmax; i++) {
      for (j = 0; j < player->trackstatus[i]->numinterfaces; j++)
	free(player->trackstatus[i]->interfaces[j]);
      free(player->trackstatus[i]->interfaces);
      free(player->trackstatus[i]);
    }

  player->trackstatus = (struct player_track_status **)realloc(player->trackstatus, player->song->maxtracks * sizeof(struct player_track_status *));

  /* Set new tracks to -1 */
  for (i = oldmax; i < player->song->maxtracks; i++) {
    player->trackstatus[i] = (struct player_track_status *)calloc(1, sizeof(struct player_track_status));
    player->trackstatus[i]->numinterfaces = player->midi->numoutputs;
    player->trackstatus[i]->interfaces = (struct player_interface_status **)calloc(player->midi->numoutputs, sizeof(struct player_interface_status *));
    player->trackstatus[i]->instrument = -1;
    player->trackstatus[i]->previouscommand = 0;
    for (j = 0; j < player->trackstatus[i]->numinterfaces; j++) {
      player->trackstatus[i]->interfaces[j] = (struct player_interface_status *)calloc(1, sizeof(struct player_interface_status));
      player->trackstatus[i]->interfaces[j]->note = -1;
      player->trackstatus[i]->interfaces[j]->midichannel = -1;
      player->trackstatus[i]->interfaces[j]->volume = -1;
      player->trackstatus[i]->interfaces[j]->hold = -1;
    }
  }
}

/* Set current section */
void player_set_section(struct player *player, int section)
{
  nullcheck_void(player, player_set_section);

  /* Bounds checking */
  if (section < 0)
    section = 0;
  if (section >= player->song->numsections)
    section = player->song->numsections - 1;

  g_mutex_lock(player->mutex);
  player->section = section;
  g_mutex_unlock(player->mutex);
}

/* Set current playseq */
void player_set_playseq(struct player *player, int playseq)
{
  nullcheck_void(player, player_set_playseq);

  /* Bounds checking */
  if (playseq < 0)
    playseq = 0;
  if (playseq >= player->song->numplayseqs)
    playseq = player->song->numplayseqs - 1;

  g_mutex_lock(player->mutex);
  player->playseq = playseq;
  g_mutex_unlock(player->mutex);
}

/* Set current position */
void player_set_position(struct player *player, int position)
{
  nullcheck_void(player, player_set_position);

  /* Bounds checking */
  if (position < 0)
    position = 0;
  if (position >= player->song->playseqs[player->playseq]->length)
    position = player->song->playseqs[player->playseq]->length - 1;

  g_mutex_lock(player->mutex);
  player->position = position;
  g_mutex_unlock(player->mutex);
}

/* Set current block */
void player_set_block(struct player *player, int block)
{
  nullcheck_void(player, player_set_block);

  /* Bounds checking */
  if (block < 0)
    block = 0;
  if (block >= player->song->numblocks)
    block = player->song->numblocks - 1;

  g_mutex_lock(player->mutex);
  player->block = block;
  g_mutex_unlock(player->mutex);
}

/* Set current line */
void player_set_line(struct player *player, int line)
{
  nullcheck_void(player, player_set_line);

  /* Bounds checking */
  while (line < 0)
    line += player->song->blocks[player->block]->length;
  while (line >= player->song->blocks[player->block]->length)
    line -= player->song->blocks[player->block]->length;

  g_mutex_lock(player->mutex);
  player->line = line;
  g_mutex_unlock(player->mutex);
}

/* Resets the player time */
void player_reset_time(struct player *player, gboolean reset_sofar)
{
  nullcheck_void(player, player_reset_time);
  gettimeofday(&player->playingstarted, NULL);

  if (reset_sofar) {
    player->playedsofar.tv_sec = 0;
    player->playedsofar.tv_usec = 0;
  }
}

/* Sets whether some tracks are considered soloed or not */
void player_set_solo(struct player *player, unsigned int solo)
{
  nullcheck_void(player, player_set_solo);

  player->solo = solo;
}

/* Checks whether some tracks are soloed or not */
void player_check_solo(struct player *player)
{
  nullcheck_void(player, player_check_solo);

  int i, solo = 0;

  for (i = 0; i < player->song->maxtracks; i++)
    solo |= player->song->tracks[i]->solo;

  player->solo = solo;
}

/* Notifies the player that MIDI interfaces have changed */
void player_midi_changed(struct player *player)
{
  nullcheck_void(player, player_midi_changed);

  int i, j;

  /* Recreate the track status array */
  if (player->trackstatus != NULL) {
    for (i = 0; i < player->song->maxtracks; i++) {
      /* Remove extraneous interfaces */
      if (player->midi->numoutputs < player->trackstatus[i]->numinterfaces)
	for (j = player->midi->numoutputs; j < player->trackstatus[i]->numinterfaces; j++)
	  free(player->trackstatus[i]->interfaces[j]);

      player->trackstatus[i]->interfaces = (struct player_interface_status **)realloc(player->trackstatus[i]->interfaces, player->midi->numoutputs * sizeof(struct player_interface_status *));

      /* Create new interfaces */
      for (j = player->trackstatus[i]->numinterfaces; j < player->midi->numoutputs; j++) {
	player->trackstatus[i]->interfaces[j] = (struct player_interface_status *)calloc(1, sizeof(struct player_interface_status));
	player->trackstatus[i]->interfaces[j]->note = -1;
	player->trackstatus[i]->interfaces[j]->midichannel = -1;
	player->trackstatus[i]->interfaces[j]->volume = -1;
	player->trackstatus[i]->interfaces[j]->hold = -1;
      }

      player->trackstatus[i]->numinterfaces = player->midi->numoutputs;
    }
  }

  /* Recreate MIDI controller values array */
  /* Remove extraneous controller values */
  if (player->midi->numoutputs < player->nummidicontrollervalues)
    for (i = player->midi->numoutputs; i < player->nummidicontrollervalues; i++)
      free(player->midicontrollervalues[i]);

  player->midicontrollervalues = (unsigned char **)realloc(player->midicontrollervalues, player->midi->numoutputs * sizeof(unsigned char *));

  /* Create new controller values */
  for (i = player->nummidicontrollervalues; i < player->midi->numoutputs; i++)
    player->midicontrollervalues[i] = (unsigned char *)calloc(16 * VALUES, sizeof(unsigned char));
  player->nummidicontrollervalues = player->midi->numoutputs;
}

/* Lock the player */
void player_lock(struct player *player) {
  nullcheck_void(player, player_lock);

  if (player->mutex != NULL)
    g_mutex_lock(player->mutex);
}

/* Unlock the player */
void player_unlock(struct player *player) {
  nullcheck_void(player, player_unlock);

  if (player->mutex != NULL)
    g_mutex_unlock(player->mutex);
}
