/* callbacks.c
 *
 * Copyright 2002-2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include <glade/glade.h>

#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"
#include "gui.h"
#include "conversion.h"

void on_menuitem_file_new_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_open(gui->editor, NULL);
}

void on_menuitem_file_open_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_file_open");

  if (!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show_all(fs);

  gtk_window_present(GTK_WINDOW(fs));
}

void on_menuitem_file_save_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  /* If no filename has been specified open the save as dialog */
  if (gui->editor->filename == NULL) {
    on_menuitem_file_saveas_activate(menuitem, user_data);
    return;
  }

  editor_song_save(gui->editor, NULL);
}

void on_menuitem_file_saveas_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_file_saveas");

  if (!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);

  gtk_window_present(GTK_WINDOW(fs));
}

void on_menuitem_file_print_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  /* Fix: A file requester is needed here */
  editor_song_print(gui->editor);
}

void on_menuitem_file_exit_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_quit");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_edit_cut_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  if (tracker_is_in_selection_mode(tracker))
    tracker_mark_selection(tracker, FALSE);

  editor_song_block_current_copyarea(gui->editor, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
  editor_song_block_current_clear(gui->editor, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_menuitem_edit_copy_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  if (tracker_is_in_selection_mode(tracker))
    tracker_mark_selection(tracker, FALSE);

  editor_song_block_current_copyarea(gui->editor, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
}

void on_menuitem_edit_paste_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  editor_song_block_current_copyarea_paste(gui->editor, tracker->cursor_ch, tracker->patpos);
  tracker_redraw(tracker);
}

void on_menuitem_edit_clear_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  editor_song_block_current_clear(gui->editor, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
  tracker_redraw(tracker);
}

void on_menuitem_edit_transpose_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_transpose");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area"))) == GUI_AREA_SELECTION) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area")), GUI_AREA_SELECTION);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_edit_changeinstrument_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area"))) == GUI_AREA_SELECTION) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")), GUI_AREA_SELECTION);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_edit_expandshrink_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area"))) == GUI_AREA_SELECTION) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")), GUI_AREA_SELECTION);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_scale_tempo_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_tempo_set(gui->editor, (unsigned int)gtk_range_get_value(range));
  editor_set_tempo(gui->editor);
}

void on_scale_tpl_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_ticksperline_set(gui->editor, (unsigned int)gtk_range_get_value(range));
  editor_set_tempo(gui->editor);
}

void on_menuitem_song_trackvolumes_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_trackvolumes");

  if (!GTK_WIDGET_VISIBLE(window)) {
    if (editor_song_maxtracks_check(gui->editor))
      gui_trackvolumes_refresh(gui);
    gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_song_transpose_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_transpose");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area"))) == GUI_AREA_SONG) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area")), GUI_AREA_SONG);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_song_expandshrink_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area"))) == GUI_AREA_SONG) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")), GUI_AREA_SONG);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_song_changeinstrument_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area"))) == GUI_AREA_SONG) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")), GUI_AREA_SONG);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_song_properties_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_songproperties");

  if (!GTK_WIDGET_VISIBLE(window)) {
    gui_song_refresh(gui);
    gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_block_cut_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_block_current_copyblock(gui->editor);
  editor_song_block_current_clear(gui->editor, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
  tracker_redraw((Tracker *) glade_xml_get_widget(gui->xml, "tracker"));
}

void on_menuitem_block_copy_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_block_current_copyblock(gui->editor);
}

void on_menuitem_block_paste_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_block_current_copyblock_paste(gui->editor);
  tracker_redraw((Tracker *) glade_xml_get_widget(gui->xml, "tracker"));
}

void on_menuitem_block_selectall_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  tracker->sel_start_ch = 0;
  tracker->sel_start_row = 0;
  tracker->sel_end_ch = editor_song_block_current_tracks_get(gui->editor) - 1;
  tracker->sel_end_row = editor_song_block_current_length_get(gui->editor) - 1;
  tracker->inSelMode = FALSE;
  tracker_redraw(tracker);
}

void on_menuitem_block_clear_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_block_current_clear(gui->editor, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
  tracker_redraw((Tracker *) glade_xml_get_widget(gui->xml, "tracker"));
}

void on_menuitem_block_transpose_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_transpose");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area"))) == GUI_AREA_BLOCK) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area")), GUI_AREA_BLOCK);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_block_expandshrink_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area"))) == GUI_AREA_BLOCK) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")), GUI_AREA_BLOCK);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_block_changeinstrument_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area"))) == GUI_AREA_BLOCK) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")), GUI_AREA_BLOCK);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_track_cut_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  editor_song_block_current_copytrack(gui->editor, tracker->cursor_ch);
  editor_song_block_current_clear(gui->editor, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
  tracker_redraw(tracker);
}

void on_menuitem_track_copy_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  editor_song_block_current_copytrack(gui->editor, tracker->cursor_ch);
}

void on_menuitem_track_paste_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  editor_song_block_current_copytrack_paste(gui->editor, tracker->cursor_ch);
  tracker_redraw(tracker);
}

void on_menuitem_track_selectall_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  tracker->sel_start_ch = tracker->cursor_ch;
  tracker->sel_start_row = 0;
  tracker->sel_end_ch = tracker->cursor_ch;
  tracker->sel_end_row = editor_song_block_current_length_get(gui->editor) - 1;
  tracker->inSelMode = FALSE;
  tracker_redraw(tracker);
}

void on_menuitem_track_clear_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  editor_song_block_current_clear(gui->editor, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
  tracker_redraw(tracker);
}

void on_menuitem_track_transpose_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_transpose");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area"))) == GUI_AREA_TRACK) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area")), GUI_AREA_TRACK);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_track_expandshrink_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area"))) == GUI_AREA_TRACK) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")), GUI_AREA_TRACK);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_track_changeinstrument_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if (gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area"))) == GUI_AREA_TRACK) {
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  } else {
    gtk_combo_box_set_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")), GUI_AREA_TRACK);
    if (!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_song_messagelist_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_messagelist");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_midi_killallnotes_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_all_notes_stop(gui->editor);
}

void on_menuitem_midi_resetpitch_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_pitch_reset(gui->editor);
}

void on_menuitem_settings_sendmidisync_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_sendsync_set(gui->editor, gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) ? 1 : 0);
}

void on_menuitem_settings_recordcontrollers_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  gui->editor->recordcontrollers = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) ? 1 : 0;
}

void on_menuitem_settings_externalsync_off_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_set_external_sync(gui->editor, EXTERNAL_SYNC_OFF);
}

void on_menuitem_settings_externalsync_midi_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_set_external_sync(gui->editor, EXTERNAL_SYNC_MIDI);
}

void on_menuitem_settings_externalsync_jack_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_set_external_sync(gui->editor, EXTERNAL_SYNC_JACK);
}

void on_menuitem_settings_preferences_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_preferences");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_about_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  gchar *authors[] = { "Vesa Halttunen <vesuri@jormas.com>", NULL };

  if (gui->window_about != NULL) {
    gdk_window_show(gui->window_about->window);
    gdk_window_raise(gui->window_about->window);
    return;
  }

  gui->window_about = gnome_about_new("Tutka", VERSION, "Copyright \xc2\xa9 2002-2005 Vesa Halttunen", "http://www.nongnu.org/tutka/", (const gchar **)authors, (const gchar **)authors, NULL, NULL);

  gtk_window_set_destroy_with_parent(GTK_WINDOW(gui->window_about), TRUE);
  g_signal_connect(G_OBJECT(gui->window_about), "destroy", G_CALLBACK(gtk_widget_destroyed), &gui->window_about);
  gtk_widget_show(gui->window_about);
}


GtkWidget *create_tracker(gchar * widget_name, gchar * string1, gchar * string2, gint int1, gint int2)
{
  return tracker_new();
}

void on_button_playsong_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_start(gui->editor, MODE_PLAY_SONG, 0);
}

void on_button_playblock_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_start(gui->editor, MODE_PLAY_BLOCK, 0);
}

void on_button_contsong_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  if (editor_player_mode_get(gui->editor) != MODE_PLAY_SONG)
    editor_player_start(gui->editor, MODE_PLAY_SONG, 1);
}

void on_button_contblock_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  if (editor_player_mode_get(gui->editor) != MODE_PLAY_BLOCK)
    editor_player_start(gui->editor, MODE_PLAY_BLOCK, 1);
}

void on_button_stop_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_player_stop(gui->editor);
}

void on_scale_volume_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_instrument_current_defaultvelocity_set(gui->editor, (unsigned char)gtk_range_get_value(range));
}

void on_scale_transpose_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_instrument_current_transpose_set(gui->editor, (char)gtk_range_get_value(range));
}

void on_scale_hold_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_instrument_current_hold_set(gui->editor, (unsigned char)gtk_range_get_value(range));
}

void on_scale_midich_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_instrument_current_midichannel_set(gui->editor, (unsigned char)gtk_range_get_value(range) - 1);
}

void on_button_properties_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_instrumentproperties");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_checkbutton_edit_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  if (gtk_toggle_button_get_active(togglebutton))
    gui->editor->edit = 1;
  else
    gui->editor->edit = 0;
}

void on_checkbutton_chord_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  if (gtk_toggle_button_get_active(togglebutton))
    gui->editor->chord = 1;
  else
    gui->editor->chord = 0;
}

void on_spinbutton_space_changed(GtkSpinButton * spinbutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  gui->editor->space = gtk_spin_button_get_value(spinbutton);

  /* Activate another widget to prevent typing in the spin button */
  gtk_widget_grab_focus(glade_xml_get_widget(gui->xml, "button_stop"));
}

void on_menuitem_song_sectionlist_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_sectionlist");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_playseq_list_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_playseqlist");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_playseq_edit_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_playseq");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_menuitem_block_list_activate(GtkMenuItem * menuitem, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_blocklist");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gtk_window_present(GTK_WINDOW(window));
}

void on_button_trackvolumes_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_trackvolumes");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_selection_messagelist_changed(GtkTreeSelection * selection, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;

  /* Get the selected row */
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    gtk_tree_model_get(model, &iter, 0, &gui->messagelist_message, -1);

  gui_messagelist_refresh(gui, FALSE);
}

void on_button_message_appendnew_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_message_insert(gui->editor, editor_song_nummessages_get(gui->editor));

  /* If no message was selected select the new message */
  if (gui->messagelist_message == -1)
    gui->messagelist_message = editor_song_nummessages_get(gui->editor) - 1;

  gui_messagelist_refresh(gui, TRUE);
}

void on_button_message_delete_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  if (gui->messagelist_message != -1) {
    int n = editor_song_nummessages_get(gui->editor);

    editor_song_message_delete(gui->editor, gui->messagelist_message);

    n--;
    if (gui->messagelist_message >= n)
      gui->messagelist_message = n - 1;

    gui_messagelist_refresh(gui, TRUE);
  }
}

void on_button_message_insertnew_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_message_insert(gui->editor, gui->messagelist_message);

  /* If no message was selected select the new message */
  if (gui->messagelist_message == -1)
    gui->messagelist_message = 0;

  gui_messagelist_refresh(gui, TRUE);
}

void on_button_message_send_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  if (gui->messagelist_message != -1)
    editor_song_message_send(gui->editor, gui->messagelist_message);
}

void on_button_message_receive_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_receive");

  if (!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);

  gui->editor->receivemessage = gui->messagelist_message;
  
  gtk_window_present(GTK_WINDOW(window));
  gui_receive_refresh(gui, -1, 0);
}

void on_button_receive_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWindow *window = GTK_WINDOW(glade_xml_get_widget(gui->xml, "dialog_receive"));

  if (!gui->editor->receiveactive) {
    /* Start receiving */
    gtk_window_set_modal(window, TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "checkbutton_receive_autostop"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_receive"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_receive_close"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_receive_stop"), TRUE);

    /* Reset the progress bar */
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(glade_xml_get_widget(gui->xml, "progressbar_receive")), 0);

    gui->editor->receivedbytes = 0;
    gui->editor->receiveactive = 1;
  }
}

void on_button_receive_stop_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWindow *window = GTK_WINDOW(glade_xml_get_widget(gui->xml, "dialog_receive"));

  if (gui->editor->receiveactive) {
    /* Stop receiving */
    gtk_window_set_modal(window, FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "checkbutton_receive_autostop"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_receive"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_receive_close"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_receive_stop"), FALSE);

    /* Reset the progress bar */
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(glade_xml_get_widget(gui->xml, "progressbar_receive")), 0);

    /* Remove unnecessary bytes if necessary */
    if (gui->editor->receivedbytes < editor_song_message_length_get(gui->editor, gui->editor->receivemessage))
      editor_song_message_length_set(gui->editor, gui->editor->receivemessage, gui->editor->receivedbytes);

    gui_messagelist_refresh(gui, TRUE);

    gui->editor->receiveactive = 0;
  }
}

void on_button_receive_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_receive");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_checkbutton_message_autostop_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  if (gtk_toggle_button_get_active(togglebutton))
    gui->editor->receiveautostop = 1;
  else
    gui->editor->receiveautostop = 0;

}

void on_selection_sectionlist_changed(GtkTreeSelection * selection, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value = 0;

  /* Get the selected row */
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_section(gui->editor, value);
}


void on_button_sectionlist_delete_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_section_current_delete(gui->editor);
  editor_refresh_playseq_and_block(gui->editor);
  gui_sectionlist_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_sectionlist_append_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_section_insert(gui->editor, editor_song_numsections_get(gui->editor));
  gui_sectionlist_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_sectionlist_insert_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_section_insert(gui->editor, gui->editor->section);
  gui_sectionlist_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_selection_playseqlist_changed(GtkTreeSelection * selection, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value = 0;

  /* Get the selected row */
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_playseq(gui->editor, value);
}

void on_selection_playseq_changed(GtkTreeSelection * selection, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value = 0;

  /* Get the selected row */
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_position(gui->editor, value);
}

void on_button_playseq_delete_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_playseq_current_delete_current(gui->editor);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_playseq_append_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_playseq_current_insert(gui->editor, editor_song_playseq_current_length_get(gui->editor));
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_playseq_insert_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_playseq_current_insert(gui->editor, gui->editor->position);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_playseqlist_insertnew_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_playseq_insert(gui->editor, gui->editor->playseq);
  gui_info_refresh(gui);
  gui_playseqlist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_button_playseqlist_appendnew_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_playseq_insert(gui->editor, editor_song_numplayseqs_get(gui->editor));
  gui_info_refresh(gui);
  gui_playseqlist_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_button_playseqlist_delete_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_playseq_current_delete(gui->editor);
  gui_info_refresh(gui);
  gui_playseqlist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_selection_blocklist_changed(GtkTreeSelection * selection, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value = 0;

  /* Get the selected row */
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_block(gui->editor, value);
}

void on_button_blocklist_insertnew_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_block_insert(gui->editor, gui->editor->block, gui->editor->block);
  gui_tracker_refresh(gui);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_blocklist_appendnew_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_block_insert(gui->editor, editor_song_numblocks_get(gui->editor), gui->editor->block);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_blocklist_delete_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  unsigned int line = gui->editor->line;

  editor_song_block_current_delete(gui->editor);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);

  /* Go back to the line the editor was on */
  gui->editor->line = line;
  gui_tracker_refresh(gui);
}

void on_entry_instrumentproperties_name_changed(GtkEditable * widget, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_instrument_current_name_set(gui->editor, (char *)gtk_entry_get_text(GTK_ENTRY(widget)));
  gui_instrument_refresh(gui);
}

void on_button_messagelist_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_messagelist");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_file_saveas_ok_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_file_saveas");

  editor_song_save(gui->editor, (char *)gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fs)));
  gtk_widget_hide(fs);
}

void on_button_file_open_ok_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_file_open");

  editor_song_open(gui->editor, (char *)gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fs)));
  gtk_widget_hide(fs);
}

void on_button_file_open_cancel_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_file_open");

  if (GTK_WIDGET_VISIBLE(fs))
    gtk_widget_hide(fs);
}

void on_button_file_saveas_cancel_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_file_saveas");

  if (GTK_WIDGET_VISIBLE(fs))
    gtk_widget_hide(fs);
}

void on_entry_songname_changed(GtkEditable * widget, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_name_set(gui->editor, (char *)gtk_entry_get_text(GTK_ENTRY(widget)));
  gui_window_title_refresh(gui);
}

void on_button_songproperties_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_songproperties");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_spinbutton_instrument_changed(GtkEditable * editable, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  /* Select an instrument if an instrument selection key was pressed */
  gui->editor->instrument = (unsigned char)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value - 1;

  /* Make sure the instrument exists */
  editor_song_instrument_current_check(gui->editor);
  gui_instrument_refresh(gui);

  /* Activate another widget to prevent typing in the spin button */
  gtk_widget_grab_focus(glade_xml_get_widget(gui->xml, "button_stop"));
}

gint on_spinbutton_instrument_output(GtkSpinButton * spin_button, gpointer user_data)
{
  gchar v[3];
  int i = (int)spin_button->adjustment->value;

  /* Convert the instrument number to hex */
  snprintf(v, 3, "%.2X", i);

  /* Only update the GtkEntry if the value was changed */
  if (strcmp(v, gtk_entry_get_text(GTK_ENTRY(spin_button))))
    gtk_entry_set_text(GTK_ENTRY(spin_button), v);

  return TRUE;
}

gint on_spinbutton_instrument_input(GtkSpinButton * spin_button, gpointer arg1, gpointer user_data)
{
  const gchar *c = gtk_entry_get_text(GTK_ENTRY(spin_button));
  gdouble *new_val = arg1;
  char *err;
  
  if (c == NULL)
    return GTK_INPUT_ERROR;

  /* Convert the hex string to an integer */
  *new_val = strtol(c, &err, 16);
  if (*err != 0)
    return GTK_INPUT_ERROR;
  
  return TRUE;
}

void on_adjustment_trackvolumes_volume_changed(GtkAdjustment * adjustment, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  int track = (int)g_object_get_data(G_OBJECT(adjustment), "track");

  editor_song_track_volume_set(gui->editor, track, (unsigned char)adjustment->value);
}

void on_togglebutton_trackvolumes_mute_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  int track = (int)g_object_get_data(G_OBJECT(togglebutton), "track");

  editor_song_track_mute_set(gui->editor, track, gtk_toggle_button_get_active(togglebutton) ? 1 : 0);
}

void on_togglebutton_trackvolumes_solo_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  int track = (int)g_object_get_data(G_OBJECT(togglebutton), "track");

  editor_song_track_solo_set(gui->editor, track, gtk_toggle_button_get_active(togglebutton) ? 1 : 0);
}

void on_entry_trackvolumes_name_changed(GtkEditable * widget, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  int track = (int)g_object_get_data(G_OBJECT(widget), "track");

  editor_song_track_name_set(gui->editor, track, (char *)gtk_entry_get_text(GTK_ENTRY(widget)));
}

void on_combobox_keyboard_changed(GtkComboBox * combobox, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  gui->editor->octave = gtk_combo_box_get_active(combobox);
}

void on_button_transpose_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_transpose");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_transpose_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  int area = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_area")));
  int instruments = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_instrument")));
  int mode = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_transpose_mode")));

  int instrument = -1, halfnotes = 0;

  switch (mode) {
  case 0:
    halfnotes = 12;
    break;
  case 1:
    halfnotes = 1;
    break;
  case 2:
    halfnotes = -1;
    break;
  case 3:
    halfnotes = -12;
    break;
  default:
    break;
  }

  if (instruments == 1)
    instrument = gui->editor->instrument;

  switch (area) {
  case GUI_AREA_SONG:
    editor_song_transpose(gui->editor, instrument, halfnotes);
    break;
  case GUI_AREA_BLOCK:
    editor_song_block_current_transpose(gui->editor, instrument, halfnotes, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_TRACK:
    editor_song_block_current_transpose(gui->editor, instrument, halfnotes, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_SELECTION:
    if (tracker->sel_start_ch >= 0 && tracker->sel_end_ch >= 0 && tracker->sel_start_row >= 0 && tracker->sel_end_row >= 0)
      editor_song_block_current_transpose(gui->editor, instrument, halfnotes, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_button_expandshrink_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_expandshrink_shrink_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  int area = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")));
  int factor = gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_expandshrink_factor")));

  if (factor < 2)
    return;

  switch (area) {
  case GUI_AREA_SONG:
    editor_song_expandshrink(gui->editor, -factor);
    break;
  case GUI_AREA_BLOCK:
    editor_song_block_current_expandshrink(gui->editor, -factor, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_TRACK:
    editor_song_block_current_expandshrink(gui->editor, -factor, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_SELECTION:
    if (tracker->sel_start_ch >= 0 && tracker->sel_end_ch >= 0 && tracker->sel_start_row >= 0 && tracker->sel_end_row >= 0)
      editor_song_block_current_expandshrink(gui->editor, -factor, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_button_expandshrink_expand_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  int area = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_expandshrink_area")));
  int factor = gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_expandshrink_factor")));

  if (factor < 2)
    return;

  switch (area) {
  case GUI_AREA_SONG:
    editor_song_expandshrink(gui->editor, factor);
    break;
  case GUI_AREA_BLOCK:
    editor_song_block_current_expandshrink(gui->editor, factor, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_TRACK:
    editor_song_block_current_expandshrink(gui->editor, factor, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_SELECTION:
    if (tracker->sel_start_ch >= 0 && tracker->sel_end_ch >= 0 && tracker->sel_start_row >= 0 && tracker->sel_end_row >= 0)
      editor_song_block_current_expandshrink(gui->editor, factor, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_button_changeinstrument_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_changeinstrument_swap_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  int area = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")));
  int from = gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_from")));
  int to = gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_to")));

  switch (area) {
  case GUI_AREA_SONG:
    editor_song_changeinstrument(gui->editor, from, to, 1);
    break;
  case GUI_AREA_BLOCK:
    editor_song_block_current_changeinstrument(gui->editor, from, to, 1, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_TRACK:
    editor_song_block_current_changeinstrument(gui->editor, from, to, 1, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_SELECTION:
    if (tracker->sel_start_ch >= 0 && tracker->sel_end_ch >= 0 && tracker->sel_start_row >= 0 && tracker->sel_end_row >= 0)
      editor_song_block_current_changeinstrument(gui->editor, from, to, 1, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  editor_song_instrument_check(gui->editor, from);
  editor_song_instrument_check(gui->editor, to);

  tracker_redraw(tracker);
}

void on_button_changeinstrument_change_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  int area = gtk_combo_box_get_active(GTK_COMBO_BOX(glade_xml_get_widget(gui->xml, "combobox_changeinstrument_area")));
  int from = gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_from")));
  int to = gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_to")));

  switch (area) {
  case GUI_AREA_SONG:
    editor_song_changeinstrument(gui->editor, from, to, 0);
    break;
  case GUI_AREA_BLOCK:
    editor_song_block_current_changeinstrument(gui->editor, from, to, 0, 0, 0, editor_song_block_current_tracks_get(gui->editor) - 1, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_TRACK:
    editor_song_block_current_changeinstrument(gui->editor, from, to, 0, tracker->cursor_ch, 0, tracker->cursor_ch, editor_song_block_current_length_get(gui->editor) - 1);
    break;
  case GUI_AREA_SELECTION:
    if (tracker->sel_start_ch >= 0 && tracker->sel_end_ch >= 0 && tracker->sel_start_row >= 0 && tracker->sel_end_row >= 0)
      editor_song_block_current_changeinstrument(gui->editor, from, to, 0, tracker->sel_start_ch, tracker->sel_start_row, tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  editor_song_instrument_check(gui->editor, to);

  tracker_redraw(tracker);
}

void on_checkbutton_message_autosend_toggled(GtkToggleButton * togglebutton, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  if (gui->messagelist_message != -1) {
    if (gtk_toggle_button_get_active(togglebutton))
      editor_song_message_autosend_set(gui->editor, gui->messagelist_message, 1);
    else
      editor_song_message_autosend_set(gui->editor, gui->messagelist_message, 0);
  }
}

void on_button_trackvolumes_mute_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_song_mute(gui->editor);

  gui_trackvolumes_refresh(gui);
}

void on_combobox_preferences_scheduling_mode_changed(GtkComboBox * widget, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  editor_scheduler_set(gui->editor, SCHED_RTC + gtk_combo_box_get_active(widget));
}

void on_button_preferences_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_preferences");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void gconf_notify_function(GConfClient * client, guint cnxn_id, GConfEntry * entry, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  gui_preferences_refresh(gui);
}

void on_button_message_load_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_message_load");

  if (!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);

  gtk_window_present(GTK_WINDOW(fs));
}

void on_button_message_save_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_message_save");

  if (!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);

  gtk_window_present(GTK_WINDOW(fs));
}

void on_button_message_save_ok_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_message_save");

  editor_song_message_save(gui->editor, gui->messagelist_message, (char *)gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fs)));

  /* Close the file selector and refresh the GUI */
  gtk_widget_hide(fs);
  gui_messagelist_refresh(gui, FALSE);
}

void on_button_message_load_ok_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_message_load");

  editor_song_message_load(gui->editor, gui->messagelist_message, (char *)gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fs)));

  /* Close the file selector and refresh the GUI */
  gtk_widget_hide(fs);
  gui_messagelist_refresh(gui, TRUE);
}

void on_button_message_load_cancel_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_message_load");

  if (GTK_WIDGET_VISIBLE(fs))
    gtk_widget_hide(fs);
}

void on_button_message_save_cancel_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *fs = glade_xml_get_widget(gui->xml, "filechooserdialog_message_save");

  if (GTK_WIDGET_VISIBLE(fs))
    gtk_widget_hide(fs);
}

void on_button_quit_quit_clicked(GtkButton * button, gpointer user_data)
{
  gtk_main_quit();
}

void on_button_quit_cancel_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_quit");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}


void on_button_instrumentproperties_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_instrumentproperties");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_blocklist_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_blocklist");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_playseqlist_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_playseqlist");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_playseq_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_playseq");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_button_sectionlist_close_clicked(GtkButton * button, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkWidget *window = glade_xml_get_widget(gui->xml, "dialog_sectionlist");

  if (GTK_WIDGET_VISIBLE(window))
    gtk_widget_hide(window);
}

void on_window_main_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_menuitem_file_exit_activate(NULL, user_data);
}

void on_filechooserdialog_file_saveas_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_file_saveas_cancel_clicked(NULL, user_data);
}

void on_filechooserdialog_file_open_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_file_open_cancel_clicked(NULL, user_data);
}

void on_filechooserdialog_message_save_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_message_save_cancel_clicked(NULL, user_data);
}

void on_filechooserdialog_message_load_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_message_load_cancel_clicked(NULL, user_data);
}

void on_dialog_transpose_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_transpose_close_clicked(NULL, user_data);
}

void on_dialog_songproperties_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_songproperties_close_clicked(NULL, user_data);
}

void on_dialog_instrumentproperties_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_instrumentproperties_close_clicked(NULL, user_data);
}

void on_dialog_blocklist_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_blocklist_close_clicked(NULL, user_data);
}

void on_dialog_playseq_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_playseq_close_clicked(NULL, user_data);
}

void on_dialog_playseqlist_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_playseqlist_close_clicked(NULL, user_data);
}

void on_dialog_sectionlist_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_sectionlist_close_clicked(NULL, user_data);
}

void on_dialog_messagelist_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_messagelist_close_clicked(NULL, user_data);
}

void on_dialog_trackvolumes_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_trackvolumes_close_clicked(NULL, user_data);
}

void on_dialog_changeinstrument_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_changeinstrument_close_clicked(NULL, user_data);
}

void on_dialog_expandshrink_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_expandshrink_close_clicked(NULL, user_data);
}

void on_dialog_preferences_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_preferences_close_clicked(NULL, user_data);
}

void on_dialog_quit_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_quit_cancel_clicked(NULL, user_data);
}

void on_eventbox_timer_button_press_event(GtkWidget * widget, GdkEventButton * event, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;

  if (event->type == GDK_BUTTON_PRESS)
    editor_reset_time(gui->editor);
}

void on_eventbox_section_button_press_event(GtkWidget * widget, GdkEventButton * event, gpointer user_data)
{
  on_menuitem_song_sectionlist_activate(NULL, user_data);
}

void on_eventbox_playseq_button_press_event(GtkWidget * widget, GdkEventButton * event, gpointer user_data)
{
  on_menuitem_playseq_list_activate(NULL, user_data);
}

void on_eventbox_position_button_press_event(GtkWidget * widget, GdkEventButton * event, gpointer user_data)
{
  on_menuitem_playseq_edit_activate(NULL, user_data);
}

void on_eventbox_block_button_press_event(GtkWidget * widget, GdkEventButton * event, gpointer user_data)
{
  on_menuitem_block_list_activate(NULL, user_data);
}

void on_cell_sectionlist_edited(GtkCellRendererText * cell, gchar * path_string, gchar * new_text, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint column;
  int value;

  /* Column number is passed as renderer object data */
  gpointer columnptr = g_object_get_data(G_OBJECT(cell), "column");

  if (columnptr == NULL)
    return;

  column = GPOINTER_TO_UINT(columnptr);

  /* Get the iterator */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* Update the model */
  switch (column) {
  default:
    {
      char *error;
      value = (int)strtol(new_text, &error, 10);
      /* Only update if the entered value was valid */
      if (*error == 0 && value > 0 && value <= editor_song_numplayseqs_get(gui->editor)) {
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, value, -1);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, column + 1, editor_song_playseq_name_get(gui->editor, value - 1), -1);
	editor_song_section_current_set(gui->editor, value - 1);
      }
      break;
    }
  }
}

void on_cell_playseqlist_edited(GtkCellRendererText * cell, gchar * path_string, gchar * new_text, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint column;

  /* Column number is passed as renderer object data */
  gpointer columnptr = g_object_get_data(G_OBJECT(cell), "column");

  if (columnptr == NULL)
    return;

  column = GPOINTER_TO_UINT(columnptr);

  /* Get the iterator */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseqlist"));
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* Update the model */
  switch (column) {
  default:
    gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, new_text, -1);
    editor_song_playseq_current_name_set(gui->editor, (char *)new_text);
    gui_playseq_refresh(gui, TRUE);
    gui_sectionlist_refresh(gui, TRUE);
    gui_info_refresh(gui);
    break;
  }
}

void on_cell_playseq_edited(GtkCellRendererText * cell, gchar * path_string, gchar * new_text, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint column;
  int value;

  /* Column number is passed as renderer object data */
  gpointer columnptr = g_object_get_data(G_OBJECT(cell), "column");

  if (columnptr == NULL)
    return;

  column = GPOINTER_TO_UINT(columnptr);

  /* Get the iterator */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* Update the model */
  switch (column) {
  default:
    {
      char *error;
      value = (int)strtol(new_text, &error, 10);
      /* Only update if the entered value was valid */
      if (*error == 0 && value > 0 && value <= editor_song_numblocks_get(gui->editor)) {
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, value, -1);
	editor_song_playseq_current_set(gui->editor, value - 1);
	gui_playseq_refresh(gui, TRUE);
      }
      break;
    }
  }
}

void on_cell_blocklist_edited(GtkCellRendererText * cell, gchar * path_string, gchar * new_text, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint column;
  gboolean changed = FALSE;
  int value = 0;

  /* Column number is passed as renderer object data */
  gpointer columnptr = g_object_get_data(G_OBJECT(cell), "column");

  if (columnptr == NULL)
    return;

  column = GPOINTER_TO_UINT(columnptr);

  /* Get the iterator */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* Update the model */
  switch (column) {
  case GUI_BLOCKLIST_COLUMN_NAME:
    gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, new_text, -1);
    changed = TRUE;
    break;
  default:
    {
      char *error;
      value = (int)strtol(new_text, &error, 10);
      /* Only update if the entered value was valid */
      if (*error == 0 && value > 0) {
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, value, -1);
	changed = TRUE;
      }
      break;
    }
  }

  /* If the model was not changed no need to do anything */
  if (!changed)
    return;

  /* See what whas changed and act accordingly */
  switch (column) {
  case GUI_BLOCKLIST_COLUMN_NAME:
    {
      editor_song_block_current_name_set(gui->editor, new_text);
      gui_blocklist_refresh(gui, TRUE);
      gui_playseq_refresh(gui, TRUE);
      gui_info_refresh(gui);

      break;
    }
  case GUI_BLOCKLIST_COLUMN_TRACKS:
    {
      editor_song_block_current_tracks_set(gui->editor, value);

      /* No need to call editor_player_trackstatus_create since
	 editor_song_maxtracks_check does the necessary thing automatically */
      if (editor_song_maxtracks_check(gui->editor))
	gui_trackvolumes_refresh(gui);

      gui_tracker_refresh(gui);
      break;
    }
  case GUI_BLOCKLIST_COLUMN_LENGTH:
    {
      editor_song_block_current_length_set(gui->editor, value);
      gui_tracker_refresh(gui);

      break;
    }
  case GUI_BLOCKLIST_COLUMN_COMMAND_PAGES:
    {
      editor_song_block_current_commandpages_set(gui->editor, value);
      if (gui->editor->cmdpage >= editor_song_block_current_commandpages_get(gui->editor)) {
	gui->editor->cmdpage = editor_song_block_current_commandpages_get(gui->editor) - 1;
	gui_tracker_refresh(gui);
      }
      gui_info_refresh(gui);
      break;
    }
  default:
    break;
  }
}

void on_cell_messagelist_edited(GtkCellRendererText * cell, gchar * path_string, gchar * new_text, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint column;
  int value;

  /* Column number is passed as renderer object data */
  gpointer columnptr = g_object_get_data(G_OBJECT(cell), "column");

  if (columnptr == NULL)
    return;

  column = GPOINTER_TO_UINT(columnptr);

  /* Get the iterator */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* Update the model */
  switch (column) {
  case GUI_MESSAGELIST_COLUMN_NAME:
    if (gui->messagelist_message != -1) {
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, new_text, -1);
      editor_song_message_name_set(gui->editor, gui->messagelist_message, new_text);
    }
    break;
  default:
    {
      char *error;
      value = (int)strtol(new_text, &error, 10);
      /* Only update if the entered value was valid */
      if (*error == 0 && gui->messagelist_message != -1) {
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, value, -1);
	editor_song_message_length_set(gui->editor, gui->messagelist_message, value);
      }
      break;
    }
  }
}

void on_cell_messagelist_toggled(GtkCellRendererToggle * cell, gchar * path_string, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GValue message = { 0 };
  unsigned int i;

  /* Get the iterator */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* The selected message may be changed - get message number */
  gtk_tree_model_get_value(model, &iter, GUI_MESSAGELIST_COLUMN_NUMBER, &message);
  gui->messagelist_message = g_value_get_int(&message);

  /* Update the model */
  i = editor_song_message_autosend_get(gui->editor, gui->messagelist_message) ^ 1;
  editor_song_message_autosend_set(gui->editor, gui->messagelist_message, i);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, GUI_MESSAGELIST_COLUMN_AUTOSEND, i, -1);
}

void on_cell_preferences_midi_interfaces_toggled(GtkCellRendererToggle * cell, gchar * path_string, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeView *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint column, direction;
  gboolean state;
  GValue client = { 0 }, port = { 0 };
  
  /* Column number and direction are passed as renderer object data */
  gpointer columnptr = g_object_get_data(G_OBJECT(cell), "column");
  gpointer directionptr = g_object_get_data(G_OBJECT(cell), "direction");

  column = GPOINTER_TO_UINT(columnptr);
  direction = GPOINTER_TO_UINT(directionptr);

  /* Get the iterator */
  if (direction == DIRECTION_OUT)
    treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_preferences_midi_interfaces_output"));
  else
    treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_preferences_midi_interfaces_input"));    
  model = gtk_tree_view_get_model(treeview);
  gtk_tree_model_get_iter_from_string(model, &iter, path_string);

  /* Try changing the state of a device */
  g_object_get(cell, "active", &state, NULL);
  if (state)
    state = FALSE;
  else
    state = TRUE;

  gtk_tree_model_get_value(model, &iter, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_CLIENT, &client);
  gtk_tree_model_get_value(model, &iter, GUI_PREFERENCES_MIDI_INTERFACES_COLUMN_PORT, &port);
  if (editor_midi_interface_status_set(gui->editor, g_value_get_int(&client), g_value_get_int(&port), direction, state))
    state = TRUE;
  else
    state = FALSE;

  /* Update the model according to the true status of the device */
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, column, state, -1);

  g_value_unset(&client);
  g_value_unset(&port);
}

void on_combobox_instrumentproperties_midiinterface_changed(GtkComboBox * combobox, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GValue interface = { 0 };
  int i = 0;

  if (gtk_combo_box_get_active_iter(combobox, &iter)) {
    model = gtk_combo_box_get_model(combobox);
    gtk_tree_model_get_value(model, &iter, 1, &interface);
    i = g_value_get_int(&interface);
    g_value_unset(&interface);
  }

  editor_song_instrument_current_midiinterface_set(gui->editor, i);
}

void on_dialog_receive_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
  on_button_receive_close_clicked(NULL, user_data);
}

void on_vscrollbar_tracker_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");
  int line = (int)(gtk_range_get_value(range) * (editor_song_block_current_length_get(gui->editor) + tracker->disp_rows));
  editor_set_line(gui->editor, line);
}

void on_hscrollbar_tracker_value_changed(GtkRange * range, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");
  int left = (int)(gtk_range_get_value(range) * editor_song_block_current_tracks_get(gui->editor));
  tracker_set_xpanning(tracker, left);
}

void on_tracker_patpos(GtkWidget * widget, int current, int length, int visible, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
    
  GtkRange *r = GTK_RANGE(glade_xml_get_widget(gui->xml, "vscrollbar_tracker"));
  GtkAdjustment *a = gtk_range_get_adjustment(r);
  a->value = (gdouble)current / (length + visible - 1);
  a->lower = (gdouble)0;
  a->upper = (gdouble)1;
  a->step_increment = (gdouble)1 / (length + visible - 1);
  a->page_increment = (gdouble)8 / (length + visible - 1);
  a->page_size = (gdouble)visible / (length + visible - 1);
  gtk_adjustment_changed(a);
}

void on_tracker_xpanning(GtkWidget * widget, int current, int tracks, int visible, gpointer user_data)
{
  struct gui *gui = (struct gui *)user_data;
    
  GtkRange *r = GTK_RANGE(glade_xml_get_widget(gui->xml, "hscrollbar_tracker"));
  GtkAdjustment *a = gtk_range_get_adjustment(r);
  a->value = (gdouble)current / tracks;
  a->lower = (gdouble)0;
  a->upper = (gdouble)1;
  a->step_increment = (gdouble)1 / tracks;
  a->page_increment = (gdouble)4 / tracks;
  a->page_size = (gdouble)visible / tracks;
  gtk_adjustment_changed(a);
}

gboolean on_tracker_scroll_event (GtkWidget *widget, GdkEventScroll *event, gpointer user_data) {
  struct gui *gui = (struct gui *)user_data;
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  switch (event->direction) {
  case GDK_SCROLL_UP:
    editor_set_line(gui->editor, gui->editor->line - 4);
    return TRUE;
  case GDK_SCROLL_DOWN:
    editor_set_line(gui->editor, gui->editor->line + 4);
    return TRUE;
  case GDK_SCROLL_LEFT:
    tracker_step_cursor_item(tracker, -2);
    return TRUE;
  case GDK_SCROLL_RIGHT:
    tracker_step_cursor_item(tracker, 2);
    return TRUE;
  default:
    break;
  }
  return FALSE;
}
