/* jack.c
 *
 * Copyright 2005 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_JACK

#include "tutka.h"
#include "jack.h"

static int jack_sync(jack_transport_state_t state, jack_position_t * pos, void *arg)
{
  struct jack *jack = (struct jack *)arg;

  /* Jack processing is only allowed after the player has initialized itself */
  return editor_player_available(jack->editor);
}

static int jack_process(jack_nframes_t nframes, void *arg)
{
  struct jack *jack = (struct jack *)arg;
  jack_position_t pos;
  jack_transport_state_t state;
  int ticks;

  /* Do nothing if external sync is not on */
  if (editor_player_get_external_sync(jack->editor) != EXTERNAL_SYNC_JACK)
    return 0;

  state = jack_transport_query(jack->client, &pos);

  /* Calculate current position in ticks */
  ticks = (int)((double)pos.frame / pos.frame_rate * pos.beats_per_minute * 24 / 60);

  /* Check whether the state has changed */
  if (state != jack->prevstate) {
    if (state == JackTransportRolling) {
      editor_seek(jack->editor, ticks);
      editor_player_start(jack->editor, MODE_PLAY_SONG, 1);
      jack->prevticks = ticks;
    } else if (state == JackTransportStopped)
      editor_player_stop(jack->editor);
    jack->prevstate = state;
  }

  if (state == JackTransportRolling && ticks > jack->prevticks) {
    editor_player_external_sync(jack->editor, ticks - jack->prevticks);
    jack->prevticks = ticks;
  }
  return 0;
}

static void jack_shutdown(void *arg)
{

}

struct jack *jack_open(struct editor *editor)
{
  nullcheck_pointer(editor, jack_open);

  struct jack *jack = (struct jack *)calloc(1, sizeof(struct jack));

  jack->editor = editor;
  jack->client = jack_client_new("Tutka");
  if (jack->client == 0) {
    fprintf(stderr, "Unable to open Jack\n");
    free(jack);
    return NULL;
  }

  /* Set previous state to an unused state to force state checking */
  jack->prevstate = JackTransportLooping;
  jack_set_sync_callback(jack->client, jack_sync, jack);
  jack_set_process_callback(jack->client, jack_process, jack);
  jack_on_shutdown(jack->client, jack_shutdown, jack);
  jack_activate(jack->client);

  return jack;
}

void jack_close(struct jack *jack)
{
  nullcheck_void(jack, jack_close);

  if (jack->client != 0)
    jack_client_close(jack->client);

  free(jack);
}

#endif
