/* song.h
 *
 * Copyright 2002-2006 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _SONG_H
#define _SONG_H

#include "playseq.h"
#include "block.h"
#include "instrument.h"
#include "midi.h"

enum {
  COMMAND_PREVIOUS_COMMAND_VALUE = 0x00,
  COMMAND_PITCH_WHEEL = 0x01,
  COMMAND_END_BLOCK = 0x02,
  COMMAND_PLAYSEQ_POSITION = 0x03,
  COMMAND_PROGRAM_CHANGE = 0x07,
  COMMAND_SEND_MESSAGE = 0x08,
  COMMAND_HOLD = 0x09,
  COMMAND_RETRIGGER = 0x0a,
  COMMAND_DELAY = 0x0b,
  COMMAND_VELOCITY = 0x0c,
  COMMAND_CHANNEL_PRESSURE = 0x0d,
  COMMAND_TPL = 0x0e,
  COMMAND_TEMPO = 0x0f,
  COMMAND_NOT_DEFINED = 0x10,
  COMMAND_MIDI_CONTROLLERS = 0x80
};

struct track {
  /* Name of the track */
  char *name;
  /* Track volume */
  unsigned int volume;
  /* Mute toggle */
  unsigned int mute;
  /* Solo toggle */
  unsigned int solo;
};

struct song {
  /* Name of the song */
  char *name;
  /* Tempo, ticks per line */
  unsigned int tempo, ticksperline;
  /* Number of sections */
  unsigned int numsections;
  /* Section array */
  unsigned int *sections;
  /* Number of blocks */
  unsigned int numplayseqs;
  /* Playing sequence array */
  struct playseq **playseqs;
  /* Number of blocks */
  unsigned int numblocks;
  /* Block array */
  struct block **blocks;
  /* Number of instruments */
  unsigned int numinstruments;
  /* Instrument array */
  struct instrument **instruments;
  /* Maximum number of tracks */
  unsigned int maxtracks;
  /* Track volume array */
  struct track **tracks;
  /* Master volume */
  unsigned int mastervolume;
  /* Number of System Exclusive messages */
  unsigned int nummessages;
  /* System Exclusive messages */
  struct message **messages;
  /* Whether to send MIDI sync or not */
  unsigned int sendsync;
};

/* Allocates a new song structure and initializes it to default values */
struct song *song_alloc(void);
/* Frees a song structure and its contents */
void song_free(struct song *);
/* Inserts a new block in the block array in the given position */
void song_insert_block(struct song *, int, int);
/* Deletes a block from the given position of the block array */
void song_delete_block(struct song *, int);
/* Inserts a new playseq in the playseq array in the given position */
void song_insert_playseq(struct song *, int);
/* Deletes a playseq from the given position of the playseq array */
void song_delete_playseq(struct song *, int);
/* Inserts a new section in the section array in the given position */
void song_insert_section(struct song *, int);
/* Deletes a section from the given position of the section array */
void song_delete_section(struct song *, int);
/* Inserts a new Message in the Message array in the given position */
void song_insert_message(struct song *, int);
/* Deletes a Message from the given position of the Message array */
void song_delete_message(struct song *, int);
/* Sets a section in the given position to point somewhere */
void song_set_section(struct song *, int, int);
/* Sets the number of ticks per line for a song */
void song_set_tpl(struct song *, unsigned int);
/* Sets the tempo of a song */
void song_set_tempo(struct song *, unsigned int);
/* If the maximum number of tracks has changed recreate the track volumes */
int song_check_maxtracks(struct song *);
/* Make sure the instrument exists; add instruments if necessary */
void song_check_instrument(struct song *, int, unsigned short);
/* Loads a song from an XML file */
struct song *song_load(char *);
/* Parses a song element in an XML file */
struct song *song_parse(xmlDocPtr, xmlNsPtr, xmlNodePtr);
/* Saves a song to an XML file */
void song_save(struct song *, char *);
/* Transposes all blocks in a song */
void song_transpose(struct song *, int, int);
/* Expands/shrinks all blocks in a song */
void song_expandshrink(struct song *, int);
/* Changes or swaps an instrument with another in all blocks of a song */
void song_changeinstrument(struct song *, int, int, int);
/* Prints a song as a text file */
void song_print(struct song *, char *);

#endif
