/* tutkaplayer.c
 *
 * Copyright 2002-2006 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <signal.h>
#include <time.h>
#include <glib.h>
#include "tutka.h"
#include "editor.h"
#include "midi.h"
#include "player.h"
#include "mmd.h"

static struct editor *editor;
static int playing = 1, statuschanged = 0;
static int oldsection = 0, oldplayseq = 0, oldposition = 0;
static int oldblock = 0, oldline = 0, oldtime = 0;
char *nullcheck_error_string = "%s() called with null %s\n";

/* Not used */
struct MMD2 *MMD2_load(char *filename)
{
  return NULL;
}
void MMD2_free(struct MMD2 *mmd)
{
}

/* Stops the player */
void stop()
{
  playing = 0;
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_values(struct editor *editor, int newsection, int newplayseq, int newposition, int newblock, int newline, int newtime, int tempochanged)
{
  nullcheck_void(editor, editor_set_values);

  if (oldsection != newsection) {
    oldsection = newsection;
    statuschanged = 1;
  }
  if (oldplayseq != newplayseq) {
    oldplayseq = newplayseq;
    statuschanged = 1;
  }
  if (oldposition != newposition) {
    oldposition = newposition;
    statuschanged = 1;
  }
  if (oldblock != newblock) {
    oldblock = newblock;
    statuschanged = 1;
  }
  if (oldline != newline) {
    oldline = newline;
    statuschanged = 1;
  }
  if (oldtime != newtime) {
    oldtime = newtime;
    statuschanged = 1;
  }
  if (tempochanged)
    statuschanged = 1;
}

void editor_set_section(struct editor *editor, int blah)
{
}
void editor_set_playseq(struct editor *editor, int blah)
{
}
void editor_set_position(struct editor *editor, int blah)
{
}
void editor_set_block(struct editor *editor, int blah)
{
}
void editor_set_line(struct editor *editor, int blah)
{
}
void editor_set_time(struct editor *editor, int blah)
{
}
void editor_set_tempo(struct editor *editor)
{
}
void editor_stop(struct editor *editor)
{
}
void editor_song_lock(struct editor *editor)
{
}
void editor_song_unlock(struct editor *editor)
{
}
void editor_player_start(struct editor *editor, unsigned int mode, int cont)
{
}
void editor_player_stop(struct editor *editor)
{
}
void editor_player_external_sync(struct editor *editor, unsigned int ticks)
{
}
unsigned int editor_player_get_external_sync(struct editor *editor)
{
  return 0;
}
unsigned int editor_player_available(struct editor *editor)
{
  return 1;
}
void editor_seek(struct editor *editor, int ticks)
{
}

int main(int argc, char *argv[])
{
  const struct timespec nobusywait = { 0, 40000 };

  /* Check command line arguments */
  if (argc < 2) {
    fprintf(stderr, "%s: Not enough arguments\n", argv[0]);
    exit(1);
  }

  /* Threads are needed */
  if (!g_thread_supported()) {
    g_thread_init(NULL);
    gdk_threads_init();
  } else {
    fprintf(stderr, "No threads implementation. Tutka requires threads.\n");
    return -1;
  }

  /* Trap CTRL-C */
  signal(SIGINT, stop);

  /* Load a song to the "editor" */
  editor = (struct editor *)calloc(1, sizeof(struct editor));
  editor->song = song_load(argv[1]);

  /* Use GConf */
  editor->gconfclient = gconf_client_get_default();
  gconf_client_add_dir(editor->gconfclient, "/apps/tutka", GCONF_CLIENT_PRELOAD_NONE, NULL);

  if (editor->song != NULL) {
    int i, j;
    
    /* Open MIDI */
    editor->midi = midi_open(editor, gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/in", NULL), gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/out", NULL), gconf_client_get_string(editor->gconfclient, "/apps/tutka/midi_interfaces/unavailable_in", NULL), gconf_client_get_string(editor->gconfclient, "/apps/tutka/midiinterfaces/unavailable_out", NULL));
    
    /* Remap the instruments to current set of MIDI interfaces */
    for (i = 0; i < editor->song->numinstruments; i++) {
      struct instrument *instrument = editor->song->instruments[i];
      for (j = 0; j < instrument->numoutputs; j++) {
        struct instrument_output *output = instrument->outputs[j];
        output->midiinterface = midi_interface_number_get_by_name(editor->midi, output->midiinterfacename, DIRECTION_OUT);
      }
    }

    /* Open the player and start playing */
    editor->player = player_open(editor->song, editor, editor->midi);
    player_set_scheduler(editor->player, gconf_client_get_int(editor->gconfclient, "/apps/tutka/scheduling_mode", NULL));
    player_start(editor->player, MODE_PLAY_SONG, 0, 0, 0, 0);

    /* Wait while playing */
    while (playing) {
      if (statuschanged) {
	fprintf(stdout, "%c[80D%.2d:%.2d: Section %.2d/%.2d - Position %.2d/%.2d - Block %.2d/%.2d - Line %.2d/%.2d", 27, oldtime / 60, oldtime % 60, oldsection + 1, editor->song->numsections, oldposition + 1, editor->song->playseqs[oldplayseq]->length, oldblock + 1, editor->song->numblocks, oldline, editor->song->blocks[oldblock]->length);
	fflush(stdout);
	statuschanged = 0;
      }

      /* Sleep for a while */
      nanosleep(&nobusywait, 0);
    }

    /* Close the player */
    player_close(editor->player);

    /* Close MIDI */
    midi_close(editor->midi);
    
    song_free(editor->song);
  }

  /* Unref GConf client */
  if (editor->gconfclient != NULL)
    g_object_unref(editor->gconfclient);
  
  free(editor);

  fprintf(stdout, "\n");
  fflush(stdout);

  return 0;
}
