/* editor.h
 *
 * Copyright 2002-2011 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _EDITOR_H
#define _EDITOR_H

#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>
#include <pthread.h>
#include <gconf/gconf-client.h>
#include "song.h"
#include "block.h"
#include "gui.h"
#include "midi.h"
#include "player.h"
#ifdef HAVE_JACK
#include "jack.h"
#endif

#define DEFAULT_TEMPO 130

enum {
  MIDIINTERFACES_AVAILABLE,
  MIDIINTERFACES_UNAVAILABLE
};

enum {
  EXTERNAL_SYNC_OFF,
  EXTERNAL_SYNC_MIDI,
  EXTERNAL_SYNC_JACK,
  EXTERNAL_SYNC_JACK_START_ONLY
};

struct editor {
  /* The song currently being edited */
  struct song *song;
  /* The full file name of the song currently being edited */
  char *filename;
  /* Current location in song */
  unsigned int section, playseq, position, block, line, time, cmdpage;
  /* Editor modes */
  unsigned int octave, instrument, edit, chord, space, recordcontrollers;
  /* Tracks currently skipped in chord mode */
  unsigned int chordstatus;
  /* Buffers for copy & paste */
  struct block *copyblock, *copytrack, *copyarea;
  /* GUI for this editor */
  struct gui *gui;
  /* Editor thread pointer */
  GThread *thread;
  /* Mutex for the song */
  GMutex *songmutex;
  /* MIDI subsystem */
  struct midi *midi;
  /* Player for this editor */
  struct player *player;
#ifdef HAVE_JACK
  /* Jack subsystem */
  struct jack *jack;
#endif
  /* GConf client for preferences */
  GConfClient *gconfclient;
  guint gconfnotifyid;
  /* Receive MIDI message status */
  unsigned char receiveactive;
  unsigned int receivemessage;
  unsigned int receivedbytes;
  unsigned char receiveautostop;
  unsigned int externalsync;
};

/* Initializes the editor */
struct editor *editor_open();
/* Opens an existing song or creates new */
void editor_song_open(struct editor *, char *);
/* Deinitializes the editor */
void editor_close(struct editor *);
/* Remaps the instruments to current set of MIDI interfaces */
void editor_song_midi_interface_numbers_set(struct editor *);
/* Remaps the instruments to current set of MIDI interfaces */
void editor_song_midi_interface_names_set(struct editor *);
/* Checks whether there have been changes in MIDI interfaces */
void editor_midi_status_check(struct editor *);

/* Refreshes playseq from section and block from position */
void editor_refresh_playseq_and_block(struct editor *);

/* Sets the position of the editor */
void editor_set_section(struct editor *, int);
void editor_set_playseq(struct editor *, int);
void editor_set_position(struct editor *, int);
void editor_set_block(struct editor *, int);
void editor_set_line(struct editor *, int);
void editor_seek(struct editor *, int);
void editor_set_commandpage(struct editor *, int);
void editor_set_time(struct editor *, int);
void editor_set_tempo(struct editor *);
/* Refreshes the GUI when the player has stopped */
void editor_stop(struct editor *);

/* Reads MIDI input */
void editor_midi_input(struct editor *, unsigned char *);

/* Resets the timer to 00:00 */
void editor_reset_time(struct editor *);

/* Song editing functions */
void editor_song_lock(struct editor *);
void editor_song_unlock(struct editor *);
void editor_song_instrument_current_check(struct editor *);
void editor_song_instrument_current_defaultvelocity_set(struct editor *, unsigned char);
void editor_song_block_current_clear(struct editor *, int, int, int, int);
unsigned int editor_song_block_current_length_get(struct editor *);
unsigned int editor_song_block_current_tracks_get(struct editor *);
unsigned int editor_song_block_current_commandpages_get(struct editor *);
void editor_song_block_current_note_set(struct editor *, unsigned int, unsigned int, unsigned char, unsigned char, unsigned char);
unsigned char editor_song_block_current_instrument_get(struct editor *, unsigned int, unsigned int);
void editor_song_block_current_instrument_set(struct editor *, unsigned int, unsigned int, unsigned char);
void editor_song_block_current_command_set(struct editor *, unsigned int, unsigned int, unsigned int, unsigned char, unsigned char);
void editor_song_block_current_command_full_set(struct editor *, unsigned int, unsigned int, unsigned int, unsigned char, unsigned char);
unsigned int editor_song_playseq_current_length_get(struct editor *);
int editor_song_maxtracks_check(struct editor *);
unsigned int editor_song_maxtracks_get(struct editor *);
void editor_song_block_current_insert_line(struct editor *, unsigned int);
void editor_song_block_current_insert_line_track_all(struct editor *);
void editor_song_block_current_delete_line(struct editor *, unsigned int);
void editor_song_block_current_delete_line_track_all(struct editor *);
void editor_song_block_current_insert_track(struct editor *, unsigned int);
void editor_song_block_current_delete_track(struct editor *, unsigned int);
void editor_song_track_volume_set(struct editor *, unsigned int, unsigned int);
unsigned int editor_song_track_volume_get(struct editor *, unsigned int);
void editor_song_track_mute_set(struct editor *, unsigned int, unsigned int);
unsigned int editor_song_track_mute_get(struct editor *, unsigned int);
void editor_song_track_solo_set(struct editor *, unsigned int, unsigned int);
unsigned int editor_song_track_solo_get(struct editor *, unsigned int);
void editor_song_track_name_set(struct editor *, unsigned int, char *);
char *editor_song_track_name_get(struct editor *, unsigned int);
void editor_song_block_current_copyarea(struct editor *, int, int, int, int);
void editor_song_block_current_copyarea_paste(struct editor *, int, int);
void editor_song_block_current_copytrack(struct editor *, int);
void editor_song_block_current_copytrack_paste(struct editor *, int);
void editor_song_block_current_copyblock(struct editor *);
void editor_song_block_current_copyblock_paste(struct editor *);
void editor_song_instrument_current_midiinterface_set(struct editor *, unsigned short);
unsigned short editor_song_instrument_current_midiinterface_get(struct editor *);
void editor_song_instrument_current_defaultvelocity_set(struct editor *, unsigned char);
unsigned char editor_song_instrument_current_defaultvelocity_get(struct editor *);
void editor_song_instrument_current_transpose_set(struct editor *, char);
char editor_song_instrument_current_transpose_get(struct editor *);
void editor_song_instrument_current_hold_set(struct editor *, unsigned char);
unsigned char editor_song_instrument_current_hold_get(struct editor *);
void editor_song_instrument_current_midichannel_set(struct editor *, unsigned char);
unsigned char editor_song_instrument_current_midichannel_get(struct editor *);
unsigned int editor_song_nummessages_get(struct editor *);
void editor_song_message_insert(struct editor *, unsigned int);
void editor_song_message_delete(struct editor *, unsigned int);
void editor_song_message_send(struct editor *, unsigned int);
unsigned int editor_song_message_receive(struct editor *);
void editor_song_section_current_delete(struct editor *);
unsigned int editor_song_numsections_get(struct editor *);
void editor_song_section_insert(struct editor *, unsigned int);
unsigned int editor_song_section_get(struct editor *, unsigned int);
void editor_song_playseq_current_delete_current(struct editor *);
void editor_song_playseq_current_insert(struct editor *, unsigned int);
void editor_song_playseq_insert(struct editor *, unsigned int);
unsigned int editor_song_numplayseqs_get(struct editor *);
void editor_song_playseq_current_delete(struct editor *);
unsigned int editor_song_playseq_current_block_get(struct editor *, unsigned int);
void editor_song_block_insert(struct editor *, unsigned int, unsigned int);
unsigned int editor_song_numblocks_get(struct editor *);
void editor_song_block_current_delete(struct editor *);
void editor_song_instrument_current_name_set(struct editor *, char *);
char *editor_song_instrument_current_name_get(struct editor *);
char *editor_song_block_name_get(struct editor *, unsigned int);
void editor_song_save(struct editor *, char *);
void editor_song_name_set(struct editor *, char *);
void editor_song_transpose(struct editor *, int, int);
void editor_song_block_current_transpose(struct editor *, int, int, int, int, int, int);
void editor_song_expandshrink(struct editor *, int);
void editor_song_block_current_expandshrink(struct editor *, int, int, int, int, int);
void editor_song_changeinstrument(struct editor *, int, int, int);
void editor_song_block_current_changeinstrument(struct editor *, int, int, int, int, int, int, int);
void editor_song_instrument_check(struct editor *, unsigned int);
void editor_song_sendsync_set(struct editor *, unsigned int);
void editor_song_message_autosend_set(struct editor *, unsigned int, unsigned int);
void editor_song_message_save(struct editor *, unsigned int, char *);
void editor_song_message_load(struct editor *, unsigned int, char *);
void editor_song_mute(struct editor *);
void editor_song_section_current_set(struct editor *, unsigned int);
char *editor_song_playseq_name_get(struct editor *, unsigned int);
void editor_song_playseq_current_name_set(struct editor *, char *);
char *editor_song_playseq_current_name_get(struct editor *);
void editor_song_playseq_current_set(struct editor *, unsigned int);
void editor_song_block_current_name_set(struct editor *, char *);
char *editor_song_block_current_name_get(struct editor *);
void editor_song_block_current_tracks_set(struct editor *, unsigned int);
unsigned int editor_song_block_tracks_get(struct editor *, unsigned int);
void editor_song_block_current_length_set(struct editor *, unsigned int);
unsigned int editor_song_block_length_get(struct editor *, unsigned int);
void editor_song_block_current_commandpages_set(struct editor *, unsigned int);
unsigned int editor_song_block_commandpages_get(struct editor *, unsigned int);
struct block *editor_song_block_current_get(struct editor *);
void editor_song_message_name_set(struct editor *, unsigned int message, char *);
char *editor_song_message_name_get(struct editor *, unsigned int);
void editor_song_message_length_set(struct editor *, unsigned int, unsigned int);
unsigned int editor_song_message_length_get(struct editor *, unsigned int);
unsigned int editor_song_message_autosend_get(struct editor *, unsigned int);
void editor_song_print(struct editor *);
void editor_song_tempo_set(struct editor *, unsigned int);
unsigned int editor_song_tempo_get(struct editor *);
void editor_song_ticksperline_set(struct editor *, unsigned int);
unsigned int editor_song_ticksperline_get(struct editor *);
char *editor_song_name_get(struct editor *);
unsigned int editor_song_sendsync_get(struct editor *);

/* Preferences functions */
/* Sets the scheduling mode */
void editor_scheduler_set(struct editor *, unsigned int);
/* Returns the scheduling mode */
unsigned int editor_scheduler_get(struct editor *);
/* Returns the tracker font name */
char *editor_tracker_font_get(struct editor *);
/* Enables or disables a MIDI interface by name */
unsigned int editor_midi_interface_status_set(struct editor *, char *, unsigned int, unsigned int);
/* Returns the name of the MIDI interface to use */
char *editor_midi_interface_get(struct editor *, unsigned int, unsigned int);
/* Marks the enabled interfaces as enabled in the preferences */
void editor_midi_interface_set_from_current(struct editor *);
/* Returns the song path from the preferences */
char *editor_song_path_get(struct editor *);
/* Returns the message path from the preferences */
char *editor_message_path_get(struct editor *);

/* Player control functions */
void editor_player_start(struct editor *, unsigned int, int);
void editor_player_stop(struct editor *);
void editor_player_all_notes_stop(struct editor *);
void editor_player_pitch_reset(struct editor *);
void editor_player_trackstatus_create(struct editor *, unsigned int);
unsigned int editor_player_mode_get(struct editor *);
void editor_player_play_note(struct editor *, unsigned char, unsigned char);
void editor_player_set_scheduler(struct editor *);
unsigned int editor_player_get_external_sync(struct editor *);
void editor_player_set_external_sync(struct editor *, unsigned int);
void editor_player_external_sync(struct editor *, unsigned int);
unsigned int editor_player_available(struct editor *);

#endif
