/* midi.c
 *
 * Copyright 2002-2011 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include "tutka.h"
#include "midi.h"
#include "player.h"

static char *string_remove_string(char *from, char *toremove);
static char *string_append_string(char *to, char *append);
/* Not used */
/*
static char *string_append_char(char *to, char append);
*/

/* Open the MIDI subsystem */
struct midi *midi_open(struct editor *editor, char *in, char *out, char *unavailablein, char *unavailableout)
{
  struct midi *midi = (struct midi *)calloc(1, sizeof(struct midi));
  snd_seq_addr_t sender, dest;
  snd_seq_port_subscribe_t *subs;

  midi->editor = editor;

  int err = snd_seq_open(&midi->seq, "default", SND_SEQ_OPEN_DUPLEX, SND_SEQ_NONBLOCK);
  if (err < 0) {
    fprintf(stderr, "Couldn't create client: %s", snd_strerror(err));
    midi_close(midi);
    return NULL;
  }

  snd_seq_set_client_name(midi->seq, "Tutka");
  midi->client = snd_seq_client_id(midi->seq);

  /* Create a new port */
  midi->port = snd_seq_create_simple_port(midi->seq, "Tutka", SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_SUBS_READ | SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE, SND_SEQ_PORT_TYPE_MIDI_GENERIC);
  if (midi->port < 0) {
    fprintf(stderr, "Couldn't create port: %s", snd_strerror(err));
    midi_close(midi);
    return NULL;
  }

  /* Subscribe to the announce port */
  sender.client = SND_SEQ_CLIENT_SYSTEM;
  sender.port = SND_SEQ_PORT_SYSTEM_ANNOUNCE;
  dest.client = midi->client;
  dest.port = midi->port;
  snd_seq_port_subscribe_alloca(&subs);
  snd_seq_port_subscribe_set_sender(subs, &sender);
  snd_seq_port_subscribe_set_dest(subs, &dest);
  snd_seq_subscribe_port(midi->seq, subs);

  /* Create a MIDI event encoder and decoder */
  snd_midi_event_new(65536, &midi->encoder);
  snd_midi_event_new(65536, &midi->decoder);
  snd_midi_event_init(midi->encoder);
  snd_midi_event_init(midi->decoder);
  snd_midi_event_no_status(midi->encoder, 1);
  snd_midi_event_no_status(midi->decoder, 1);

  /* Set requested interfaces */
  midi->outputs_ok = out;
  midi->inputs_ok = in;
  midi->outputs_error = unavailableout;
  midi->inputs_error = unavailablein;

  /* Get MIDI interfaces */
  midi_interfaces_refresh(midi);

  return midi;
}

/* Close the MIDI subsystem */
void midi_close(struct midi *midi)
{
  nullcheck_void(midi, midi_close);

  if (midi->seq != NULL) {
    snd_seq_addr_t sender, dest;
    snd_seq_port_subscribe_t *subs;

    /* Unsubscribe the announce port */
    sender.client = SND_SEQ_CLIENT_SYSTEM;
    sender.port = SND_SEQ_PORT_SYSTEM_ANNOUNCE;
    dest.client = midi->client;
    dest.port = midi->port;
    snd_seq_port_subscribe_alloca(&subs);
    snd_seq_port_subscribe_set_sender(subs, &sender);
    snd_seq_port_subscribe_set_dest(subs, &dest);
    snd_seq_unsubscribe_port(midi->seq, subs);

    /* Unsubscribe and free all interfaces */
    midi_unsubscribe_all(midi);

    /* Free MIDI event encoder and decoder */
    if (midi->encoder != NULL)
      snd_midi_event_free(midi->encoder);
    if (midi->decoder != NULL)
      snd_midi_event_free(midi->decoder);

    /* Delete port */
    if (midi->port >= 0)
      snd_seq_delete_simple_port(midi->seq, midi->port);

    /* Delete client */
    snd_seq_close(midi->seq);
  }

  /* Free interface names */
  free(midi->outputs_ok);
  free(midi->inputs_ok);
  free(midi->outputs_error);
  free(midi->inputs_error);

  free(midi);
}

/* Subscribe to a client/port */
unsigned int midi_subscribe(struct midi *midi, char *name, unsigned int direction)
{
  nullcheck_int(midi, midi_subscribe);
  nullcheck_int(name, midi_subscribe);

  struct midi_interface *iface = midi_interface_get_by_name(midi, name, direction);
  int result = 0;
  char *temp;

  if (iface != NULL) {
    snd_seq_addr_t sender, dest;
    snd_seq_port_subscribe_t *subs;
    struct midi_alsa *alsa = (struct midi_alsa *)iface->data;

    if (direction == DIRECTION_OUT) {
      sender.client = midi->client;
      sender.port = midi->port;
      dest.client = alsa->client;
      dest.port = alsa->port;
    } else {
      sender.client = alsa->client;
      sender.port = alsa->port;
      dest.client = midi->client;
      dest.port = midi->port;
    }
    snd_seq_port_subscribe_alloca(&subs);
    snd_seq_port_subscribe_set_sender(subs, &sender);
    snd_seq_port_subscribe_set_dest(subs, &dest);

    /* Try subscribing to the port */
    if (snd_seq_subscribe_port(midi->seq, subs) == 0) {
      alsa->subs = subs;
      result = 1;
    }
  }

  temp = malloc(strlen(name) + 2);
  strcpy(temp, name);
  strcat(temp, ",");

  if (result) {
    /* Add to the subscribed strings */
    if (direction == DIRECTION_OUT)
      midi->outputs_ok = string_append_string(midi->outputs_ok, temp);
    else
      midi->inputs_ok = string_append_string(midi->inputs_ok, temp);
  } else {
    /* Add to the error strings */
    if (direction == DIRECTION_OUT)
      midi->outputs_error = string_append_string(midi->outputs_error, temp);
    else
      midi->inputs_error = string_append_string(midi->inputs_error, temp);
  }

  free(temp);

  return result;
}

/* Unsubscribe from a client/port */
unsigned int midi_unsubscribe(struct midi *midi, char *name, unsigned int direction)
{
  nullcheck_int(midi, midi_unsubscribe);

  struct midi_interface *iface = midi_interface_get_by_name(midi, name, direction);
  char *temp;

  if (iface == NULL)
    return 0;

  struct midi_alsa *alsa = (struct midi_alsa *)iface->data;
  if (alsa->subs != NULL) {
    snd_seq_addr_t sender, dest;
    snd_seq_port_subscribe_t *subs;

    if (direction == DIRECTION_OUT) {
      sender.client = midi->client;
      sender.port = midi->port;
      dest.client = alsa->client;
      dest.port = alsa->port;
    } else {
      sender.client = alsa->client;
      sender.port = alsa->port;
      dest.client = midi->client;
      dest.port = midi->port;
    }
    snd_seq_port_subscribe_alloca(&subs);
    snd_seq_port_subscribe_set_sender(subs, &sender);
    snd_seq_port_subscribe_set_dest(subs, &dest);

    /* Try to unsubscribe from the port */
    if (snd_seq_unsubscribe_port(midi->seq, subs) == 0) {
      alsa->subs = NULL;

      /* Remove from the subscribed strings */
      temp = malloc(strlen(name) + 2);
      strcpy(temp, name);
      strcat(temp, ",");

      if (direction == DIRECTION_OUT)
        midi->outputs_ok = string_remove_string(midi->outputs_ok, temp);
      else
        midi->inputs_ok = string_remove_string(midi->inputs_ok, temp);

      free(temp);

      return 1;
    }
  }
  return 0;
}

/* Allocate a MIDI interface */
struct midi_interface *midi_interface_alloc(unsigned int type, char *name)
{
  struct midi_interface *midi = (struct midi_interface *)calloc(1, sizeof(struct midi_interface));

  midi->type = type;
  switch (type) {
  case MIDI_ALSA:
    midi->data = calloc(1, sizeof(struct midi_alsa));
    break;
  case MIDI_BUFFER:
    midi->data = calloc(1, sizeof(struct midi_buffer));
    break;
  }

  if (name != NULL)
    midi->name = strdup(name);

  return midi;
}

/* Free a MIDI interface */
void midi_interface_free(struct midi_interface *midi)
{
  if (midi != NULL) {
    free(midi->name);
    free(midi->data);
    free(midi);
  }
}

/* Get MIDI interface from available interfaces by client and port */
struct midi_interface *midi_interface_get_by_client_and_port(struct midi *midi, int client, int port, unsigned int direction)
{
  nullcheck_pointer(midi, midi_interface_get_by_client_and_port);

  struct midi_interface **interfaces;
  int i, n;

  if (direction == DIRECTION_OUT) {
    interfaces = midi->outputs;
    n = midi->numoutputs;
  } else {
    interfaces = midi->inputs;
    n = midi->numinputs;
  }

  for (i = 0; i < n; i++) {
    if (interfaces[i]->type == MIDI_ALSA) {
      struct midi_alsa *alsa = (struct midi_alsa *)interfaces[i]->data;
      if (alsa->client == client && alsa->port == port)
        return interfaces[i];
    }
  }

  return NULL;
}

/* Gets an MIDI interface by name from available interfaces */
struct midi_interface *midi_interface_get_by_name(struct midi *midi, char *name, unsigned int direction)
{
  nullcheck_pointer(midi, midi_interface_get_by_name);

  /* Use the null interface if no name specified */
  if (name == NULL)
    return NULL;

  struct midi_interface **interfaces;
  int i, n;

  if (direction == DIRECTION_OUT) {
    interfaces = midi->outputs;
    n = midi->numoutputs;
  } else {
    interfaces = midi->inputs;
    n = midi->numinputs;
  }

  for (i = 0; i < n; i++) {
    struct midi_interface *iface = interfaces[i];
    if (iface->name != NULL && strcmp(iface->name, name) == 0)
      return interfaces[i];
  }

  /* If no interface is found use the null interface */
  return NULL;
}

/* Gets an MIDI interface number by name from the subscribed interfaces */
int midi_interface_number_get_by_name(struct midi *midi, char *name, unsigned int direction)
{
  nullcheck_int(midi, midi_interface_number_get_by_name);

  /* Use the null interface if no name specified */
  if (name == NULL)
    return 0;

  int i, n;
  struct midi_interface **interfaces;

  if (direction == DIRECTION_OUT) {
    n = midi->numoutputs;
    interfaces = midi->outputs;
  } else {
    n = midi->numinputs;
    interfaces = midi->inputs;
  }

  for (i = 1; i < n; i++) {
    struct midi_interface *iface = interfaces[i];
    if (iface->name != NULL && strcmp(iface->name, name) == 0)
      return i;
  }

  /* If no interface is found use the null interface */
  return 0;
}

/* Gets an MIDI interface name by number from the available interfaces */
char *midi_interface_name_get_by_number(struct midi *midi, unsigned int number, unsigned int direction)
{
  nullcheck_pointer(midi, midi_interface_name_get_by_number);

  struct midi_interface *iface;

  if (direction == DIRECTION_OUT)
    iface = midi->outputs[number];
  else
    iface = midi->inputs[number];

  if (iface->name != NULL) {
    return strdup(iface->name);
  } else
    return NULL;
}

/* Opens the MIDI interfaces defined in the strings */
void midi_subscribe_all(struct midi *midi)
{
  nullcheck_void(midi, midi_subscribe_all);

  int j, last;
  unsigned int direction = 0;
  char *current = NULL, *next = NULL;
  char *outputs_ok = midi->outputs_ok;
  char *inputs_ok = midi->inputs_ok;
  char *outputs_error = midi->outputs_error;
  char *inputs_error = midi->inputs_error;
  midi->outputs_ok = NULL;
  midi->inputs_ok = NULL;
  midi->outputs_error = NULL;
  midi->inputs_error = NULL;

  /* Handle both input and output, available and unavailable */
  for (j = 0; j < 4; j++) {
    last = 0;

    switch (j) {
    case 0:
      current = inputs_ok;
      direction = DIRECTION_IN;
      break;
    case 1:
      current = outputs_ok;
      direction = DIRECTION_OUT;
      break;
    case 2:
      current = inputs_error;
      direction = DIRECTION_IN;
      break;
    case 3:
      current = outputs_error;
      direction = DIRECTION_OUT;
      break;
    }
    next = current;

    if (current != NULL) {
      while (!last) {
        /* Find the end of the device name */
        while (*next != ',' && *next != 0)
          next++;
        if (*next == 0)
          last = 1;
        else
          *next = 0;

        /* Empty string? */
        if (strlen(current) == 0)
          break;

        /* Try to subscribe */
        midi_subscribe(midi, current, direction);

        current = ++next;
      }
    }
  }

  free(outputs_ok);
  free(inputs_ok);
  free(outputs_error);
  free(inputs_error);
}

/* Unsubscribe all MIDI interfaces and free data */
void midi_unsubscribe_all(struct midi *midi)
{
  int i;
  snd_seq_addr_t sender, dest;
  snd_seq_port_subscribe_t *subs;

  for (i = 0; i < midi->numoutputs; i++) {
    struct midi_interface *iface = midi->outputs[i];
    if (iface->type == MIDI_ALSA) {
      struct midi_alsa *alsa = iface->data;
      if (alsa->subs != NULL) {
        sender.client = midi->client;
        sender.port = midi->port;
        dest.client = alsa->client;
        dest.port = alsa->port;
        snd_seq_port_subscribe_alloca(&subs);
        snd_seq_port_subscribe_set_sender(subs, &sender);
        snd_seq_port_subscribe_set_dest(subs, &dest);
        snd_seq_unsubscribe_port(midi->seq, subs);
      }
    }
    midi_interface_free(iface);
  }

  for (i = 0; i < midi->numinputs; i++) {
    struct midi_interface *iface = midi->inputs[i];
    if (iface->type == MIDI_ALSA) {
      struct midi_alsa *alsa = iface->data;
      if (alsa->subs != NULL) {
        sender.client = alsa->client;
        sender.port = alsa->port;
        dest.client = midi->client;
        dest.port = midi->port;
        snd_seq_port_subscribe_alloca(&subs);
        snd_seq_port_subscribe_set_sender(subs, &sender);
        snd_seq_port_subscribe_set_dest(subs, &dest);
        snd_seq_unsubscribe_port(midi->seq, subs);
      }
    }
    midi_interface_free(iface);
  }

  midi->numoutputs = 0;
  midi->numinputs = 0;
  free(midi->outputs);
  free(midi->inputs);
  midi->outputs = NULL;
  midi->inputs = NULL;
}

/* Gets the list of available interfaces */
void midi_interfaces_get(struct midi *midi, unsigned int direction)
{
  nullcheck_void(midi, midi_interfaces_get);

  snd_seq_client_info_t *cinfo;
  snd_seq_port_info_t *pinfo;
  int bits;

  /* Set permissions to check based on direction */
  if (direction == DIRECTION_OUT) {
    bits = SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE;
  } else {
    bits = SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_SUBS_READ;
  }

  snd_seq_client_info_alloca(&cinfo);
  snd_seq_port_info_alloca(&pinfo);
  snd_seq_client_info_set_client(cinfo, SND_SEQ_CLIENT_SYSTEM);
  while (snd_seq_query_next_client(midi->seq, cinfo) >= 0) {
    /* Reset query info */
    snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));
    snd_seq_port_info_set_port(pinfo, -1);
    while (snd_seq_query_next_port(midi->seq, pinfo) >= 0) {
      int client = snd_seq_port_info_get_client(pinfo);
      int port = snd_seq_port_info_get_port(pinfo);
      /* Check whether the capabilities match the current direction */
      if (snd_seq_port_info_get_capability(pinfo) & bits && !(client == midi->client && port == midi->port)) {
        /* Create an interface structure for the port */
        struct midi_interface *iface;
        struct midi_alsa *alsa;
        alsa = calloc(1, sizeof(struct midi_alsa));
        alsa->client = client;
        alsa->port = port;
        iface = calloc(1, sizeof(struct midi_interface));
        iface->midi = midi;
        iface->type = MIDI_ALSA;
        iface->data = alsa;
        iface->name = strdup(snd_seq_port_info_get_name(pinfo));

        /* Append the interface to the interface array */
        if (direction == DIRECTION_OUT) {
          midi->numoutputs++;
          midi->outputs = (struct midi_interface **)realloc(midi->outputs, midi->numoutputs * sizeof(struct midi_interface *));
          midi->outputs[midi->numoutputs - 1] = iface;
        } else {
          midi->numinputs++;
          midi->inputs = (struct midi_interface **)realloc(midi->inputs, midi->numinputs * sizeof(struct midi_interface *));
          midi->inputs[midi->numinputs - 1] = iface;
        }
      }
    }
  }
}

/* Refreshes the list of available interfaces */
void midi_interfaces_refresh(struct midi *midi)
{
  nullcheck_void(midi, midi_interfaces_refresh);

  /* Unsubscribe all */
  midi_unsubscribe_all(midi);

  /* Add a null MIDI interface */
  midi->numoutputs = 1;
  midi->outputs = (struct midi_interface **)calloc(1, sizeof(struct midi_interface *));
  midi->outputs[0] = midi_interface_alloc(MIDI_NULL, "No output");

  /* Refresh lists */
  midi_interfaces_get(midi, DIRECTION_OUT);
  midi_interfaces_get(midi, DIRECTION_IN);

  /* Resubscribe */
  midi_subscribe_all(midi);
}

/* Sets the current tick */
void midi_set_tick(struct midi_interface *midi, unsigned int tick)
{
  nullcheck_void(midi, midi_set_tick);

  midi->tick = tick;
}

/* Reads data from a MIDI interface */
static inline int midi_read(struct midi *midi, unsigned char *data, unsigned int length)
{
  nullcheck_int(midi, midi_read);

  /* Get bytes from the incoming buffer if they exist */
  if (midi->incoming != NULL) {
    /* Get the amount of bytes left in the buffer */
    int n = midi->incominglength - midi->incomingposition;

    /* If less were requested use that amount instead */
    if (length < n)
      n = length;

    memcpy(data, midi->incoming + midi->incomingposition, n);
    midi->incomingposition += n;

    /* If all bytes are used clear the buffer */
    if (midi->incomingposition == midi->incominglength) {
      free(midi->incoming);
      midi->incoming = NULL;
    }

    return n;
  } else {
    snd_seq_event_t *ev;
    if (snd_seq_event_input(midi->seq, &ev) >= 0) {
      /* Check event type */
      switch (ev->type) {
      case SND_SEQ_EVENT_START:
        if (editor_player_get_external_sync(midi->editor) == EXTERNAL_SYNC_MIDI)
          editor_player_start(midi->editor, MODE_PLAY_SONG, 0);
        return 0;
      case SND_SEQ_EVENT_CONTINUE:
        if (editor_player_get_external_sync(midi->editor) == EXTERNAL_SYNC_MIDI)
          editor_player_start(midi->editor, MODE_PLAY_SONG, 1);
        return 0;
      case SND_SEQ_EVENT_STOP:
        if (editor_player_get_external_sync(midi->editor) == EXTERNAL_SYNC_MIDI)
          editor_player_stop(midi->editor);
        return 0;
      case SND_SEQ_EVENT_CLOCK:
        if (editor_player_get_external_sync(midi->editor) == EXTERNAL_SYNC_MIDI)
          editor_player_external_sync(midi->editor, 1);
        return 0;
      case SND_SEQ_EVENT_PORT_START:
      case SND_SEQ_EVENT_PORT_EXIT:
      case SND_SEQ_EVENT_PORT_CHANGE:
        /* Ports have been changed */
        midi->changed = 1;
        return 0;
      default: {
        /* Get the event to the incoming buffer */
        int n = snd_seq_event_length(ev);
        midi->incoming = malloc(n);
        midi->incominglength = snd_midi_event_decode(midi->decoder, midi->incoming, n, ev);
        if (midi->incominglength < 0) {
          /* Some sort of an error occurred */
          free(midi->incoming);
          midi->incoming = NULL;
          return 0;
        }
        midi->incomingposition = 0;

        /* Copy up to requested number of bytes from the buffer */
        n = midi->incominglength;
        if (length < n)
          n = length;

        memcpy(data, midi->incoming, n);
        midi->incomingposition += n;

        /* If all bytes are used clear the buffer */
        if (midi->incomingposition == midi->incominglength) {
          free(midi->incoming);
          midi->incoming = NULL;
        }

        return n;
      }
      }
    } else
      return 0;
  }

  return 0;
}

/* Writes data to the MIDI interface */
static inline void midi_write(struct midi_interface *midi, unsigned char *data, unsigned int length)
{
  nullcheck_void(midi, midi_write);

  unsigned int delta = midi->tick - midi->oldtick;
  midi->oldtick = midi->tick;

  switch (midi->type) {
  case MIDI_ALSA: {
    struct midi_alsa *alsa = midi->data;
    int l = 0;

    /* Create event */
    snd_seq_event_t ev;
    snd_seq_ev_clear(&ev);
    snd_seq_ev_set_source(&ev, midi->midi->port);
    snd_seq_ev_set_dest(&ev, alsa->client, alsa->port);
    snd_seq_ev_set_direct(&ev);
    /* The encoder may send the data in multiple packets */
    while (l < length) {
      l += snd_midi_event_encode(midi->midi->encoder, data + l, length - l, &ev);
      snd_seq_event_output(midi->midi->seq, &ev);
    }
    snd_seq_drain_output(midi->midi->seq);
    break;
  }
  case MIDI_BUFFER: {
    struct midi_buffer *buffer = midi->data;
    unsigned int curpos = buffer->ptr - buffer->data;
    unsigned long value = delta;
    unsigned long varlen = value & 0x7F;
    int i;

    /* Create a variable length version of the tick delta */
    while ((value >>= 7)) {
      varlen <<= 8;
      varlen |= ((value & 0x7F) | 0x80);
    }

    /* Check how many bytes the varlen version requires */
    value = varlen;
    for (i = 1; i <= 4; i++) {
      if (value & 0x80)
        value >>= 8;
      else
        break;
    }

    /* Allocate more space if required */
    if (curpos + length + i > buffer->allocated) {
      buffer->allocated += MIDI_BUFFER_BLOCK_SIZE;
      buffer->data = realloc(buffer->data, buffer->allocated);
      buffer->ptr = buffer->data + curpos;
    }

    /* Write the varlen tick delta value */
    for (i = 1; i <= 4; i++) {
      *buffer->ptr++ = (unsigned char)(varlen & 0xff);
      if (varlen & 0x80)
        varlen >>= 8;
      else
        break;
    }

    /* Write the actual payload */
    memcpy(buffer->ptr, data, length);
    buffer->ptr += length;
    break;
  }
  default:
    break;
  }
}

/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(struct midi_interface *midi, unsigned char channel, unsigned char note, unsigned char velocity)
{
  nullcheck_void(midi, midi_note_off);

  unsigned char data[3];

  data[0] = 0x80 | channel;
  data[1] = note & 0x7f;
  data[2] = velocity & 0x7f;

  midi_write(midi, data, 3);
}

/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(struct midi_interface *midi, unsigned char channel, unsigned char note, unsigned char velocity)
{
  nullcheck_void(midi, midi_note_on);

  unsigned char data[3];

  data[0] = 0x90 | channel;
  data[1] = note & 0x7f;
  data[2] = velocity & 0x7f;

  midi_write(midi, data, 3);
}

/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(struct midi_interface *midi, unsigned char channel, unsigned char note, unsigned char pressure)
{
  nullcheck_void(midi, midi_aftertouch);

  unsigned char data[3];

  data[0] = 0xa0 | channel;
  data[1] = note & 0x7f;
  data[2] = pressure & 0x7f;

  midi_write(midi, data, 3);
}

/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(struct midi_interface *midi, unsigned char channel, unsigned char controller, unsigned char value)
{
  nullcheck_void(midi, midi_controller);

  unsigned char data[3];

  data[0] = 0xb0 | channel;
  data[1] = controller & 0x7f;
  data[2] = value & 0x7f;

  midi_write(midi, data, 3);
}

/* Send a program change on a MIDI channel */
void midi_program_change(struct midi_interface *midi, unsigned char channel, unsigned char program)
{
  nullcheck_void(midi, midi_program_change);

  unsigned char data[2];

  data[0] = 0xc0 | channel;
  data[1] = program & 0x7f;

  midi_write(midi, data, 2);
}

/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(struct midi_interface *midi, unsigned char channel, unsigned char pressure)
{
  nullcheck_void(midi, midi_channel_pressure);

  unsigned char data[2];

  data[0] = 0xd0 | channel;
  data[1] = pressure & 0x7f;

  midi_write(midi, data, 2);
}

/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(struct midi_interface *midi, unsigned char channel, unsigned short value)
{
  nullcheck_void(midi, midi_pitch_wheel);

  unsigned char data[3];

  data[0] = 0xe0 | channel;
  data[1] = (value >> 7) & 0x7f;
  data[2] = value & 0x7f;

  midi_write(midi, data, 3);
}

/* Sends an arbitrary MIDI message */
void midi_write_raw(struct midi_interface *midi, unsigned char *message, unsigned short length)
{
  nullcheck_void(midi, midi_write_raw);

  if (length == 0)
    return;

  /* SysEx messages need special treatment if written to a file */
  if (midi->type == MIDI_BUFFER && message[0] == 0xf0) {
    unsigned long value = length - 1;
    unsigned long varlen = value & 0x7F;
    unsigned char *newmessage;
    int i, l;

    /* Create a variable length version of the length */
    while ((value >>= 7)) {
      varlen <<= 8;
      varlen |= ((value & 0x7F) | 0x80);
    }

    /* Check how many bytes the varlen version requires */
    value = varlen;
    for (l = 1; l <= 4; l++) {
      if (value & 0x80)
        value >>= 8;
      else
        break;
    }

    /* Allocate buffer */
    newmessage = (unsigned char *)malloc(length + l);
    newmessage[0] = 0xf0;

    /* Write the varlen length */
    for (i = 1; i <= 4; i++) {
      newmessage[i] = (unsigned char)(varlen & 0xff);
      if (varlen & 0x80)
        varlen >>= 8;
      else
        break;
    }

    /* Copy the rest of the data */
    memcpy(newmessage + i + 1, message + 1, length - 1);
    midi_write(midi, newmessage, length + l);
    free(newmessage);
  } else
    midi_write(midi, message, length);
}

/* Send a clock message */
void midi_clock(struct midi_interface *midi)
{
  nullcheck_void(midi, midi_clock);

  unsigned char data[] = { 0xf8 };

  midi_write(midi, data, 1);
}

/* Set the tempo (used when exporting) */
void midi_tempo(struct midi_interface *midi, unsigned int tempo)
{
  nullcheck_void(midi, midi_tempo);

  unsigned char data[] = { 0xff, 0x51, 0x03, 0x00, 0x00, 0x00 };
  unsigned int ms = 60000000 / tempo;

  /* Tempo is only relevant when exporting */
  if (midi->type != MIDI_BUFFER)
    return;

  data[3] = (ms >> 16) & 0xff;
  data[4] = (ms >> 8) & 0xff;
  data[5] = ms & 0xff;

  midi_write(midi, data, 6);
}

/* Receives a MIDI message */
unsigned char *midi_read_raw(struct midi *midi)
{
  nullcheck_int(midi, midi_read_raw);

  int status = 1;
  unsigned char c = 0, *d = NULL;
  const struct timespec nobusywait = { 0, 40000 };

  while ((status = midi_read(midi, &c, 1)) == 1 && c < 0x80)
    ;

  if (status != 1)
    return NULL;

  switch (c & 0xf0) {
  case 0x80:
  case 0x90:
  case 0xa0:
  case 0xb0:
  case 0xe0: {
    /* Number of bytes read */
    int r = 0;

    d = (unsigned char *)calloc(3, sizeof(unsigned char));
    d[0] = c;

    /* Read until 2 bytes have been read. If for some reason these two bytes will never come Tutka hangs here! */
    while (r < 2) {
      status = midi_read(midi, d + 1 + r, 2 - r);
      if (status >= 0)
        r += status;

      /* Sleep for a while to avoid busy looping */
      nanosleep(&nobusywait, 0);
    }

    return d;
  }
  default:
    break;
  }
  return NULL;
}

/* Receives a MIDI message */
int midi_read_system_exclusive(struct midi *midi, struct message *message, unsigned int read, int autostop)
{
  nullcheck_int(midi, midi_read_system_exclusive);
  nullcheck_int(message, midi_read_system_exclusive);

  int status, i = read;
  unsigned char c = 0;

  if (autostop == 1) {
    /* Read until 0xf7 (end of SysEx) */
    while (c != 0xf7) {
      status = midi_read(midi, &c, 1);

      if (status == 0)
        return i - read;

      if (i >= message->length) {
        message->length += 65536;
        message->data = (unsigned char *)realloc(message->data, message->length);
      }

      if (c != 0xf1 && c != 0xf8 && c != 0xf9 && c != 0xfe)
        message->data[i++] = c;
    }

    return -(i - read);
  } else {
    /* Read until the buffer is full */
    while (i < message->length) {
      status = midi_read(midi, &c, 1);

      if (status == 0)
        return i - read;

      if (c != 0xf1 && c != 0xf8 && c != 0xf9 && c != 0xfe)
        message->data[i++] = c;
    }

    return -(i - read);
  }
}

/* Returns the data written to the MIDI buffer */
unsigned char *midi_get_buffer(struct midi_interface *midi)
{
  nullcheck_pointer(midi, midi_get_buffer);

  unsigned char *data;
  struct midi_buffer *buffer = midi->data;

  if (midi->type != MIDI_BUFFER) {
    fprintf(stderr, "midi_get_buffer() called but MIDI interface not a buffer\n");
    return NULL;
  }

  data = malloc(buffer->ptr - buffer->data);
  memcpy(data, buffer->data, buffer->ptr - buffer->data);

  return data;
}

/* Returns the data written to the MIDI buffer */
int midi_get_buffer_length(struct midi_interface *midi)
{
  nullcheck_int(midi, midi_get_buffer_length);

  struct midi_buffer *buffer = midi->data;

  if (midi->type != MIDI_BUFFER) {
    fprintf(stderr, "midi_get_buffer_length() called but MIDI interface not a buffer\n");
    return 0;
  }

  return buffer->ptr - buffer->data;
}

/* Sets the length of a MIDI message */
void message_set_length(struct message *message, unsigned int length)
{
  nullcheck_void(message, message_set_length);

  message->length = length;
  message->data = (unsigned char *)realloc(message->data, message->length);
}

/* Sets the autosend flag of a MIDI message */
void message_set_autosend(struct message *message, unsigned int autosend)
{
  nullcheck_void(message, message_set_autosend);

  message->autosend = autosend;
}

/* Parses a message element in an XML file */
struct message *message_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
  struct message *message = NULL;

  if ((!xmlStrcmp(cur->name, (xmlChar *)"message")) && (cur->ns == ns)) {
    xmlNodePtr temp = cur->xmlChildrenNode;
    xmlChar c[3], *prop;
    int i;
    unsigned int d;

    /* Temporary string for hexadecimal parsing */
    c[2] = 0;

    /* Allocate message */
    message = (struct message *)calloc(1, sizeof(struct message));
    prop = xmlGetProp(cur, (xmlChar *)"name");
    if (prop != NULL) {
      message->name = strdup((char *)prop);
      xmlFree(prop);
    }

    if (temp != NULL)
      message->length = strlen((char *)temp->content) / 2;

    prop = xmlGetProp(cur, (xmlChar *)"autosend");
    if (prop != NULL) {
      message->autosend = atoi((char *)prop);
      xmlFree(prop);
    }

    message->data = (unsigned char *)calloc(message->length, sizeof(unsigned char));
    for (i = 0; i < message->length; i++) {
      c[0] = temp->content[i * 2];
      c[1] = temp->content[i * 2 + 1];
      sscanf((char *)c, "%X", &d);
      message->data[i] = d;
    }
  } else if (cur->type != XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected message, got %s\n", cur->name);

  return message;
}

/* Saves a message to an XML file */
void message_save(struct message *message, int number, xmlNodePtr parent)
{
  nullcheck_void(message, message_save);

  xmlChar *c, t[10];
  xmlNodePtr node;
  int i;

  c = (xmlChar *)calloc(2 * message->length + 1, sizeof(xmlChar));
  for (i = 0; i < message->length; i++)
    sprintf((char *)c + (i * 2), "%.2X", message->data[i]);

  node = xmlNewChild(parent, NULL, (xmlChar *)"message", c);
  snprintf((char *)t, 10, "%d", number);
  xmlSetProp(node, (xmlChar *)"number", t);
  xmlSetProp(node, (xmlChar *)"name", (xmlChar *)message->name);
  snprintf((char *)t, 10, "%d", message->autosend);
  xmlSetProp(node, (xmlChar *)"autosend", t);

  xmlAddChild(parent, xmlNewText((xmlChar *)"\n"));

  free(c);
}

/* Loads a message from a file */
void message_load_binary(struct message *message, char *filename)
{
  nullcheck_void(message, message_load_binary);
  nullcheck_void(filename, message_load_binary);

  unsigned char *data;
  FILE *file;

  file = fopen(filename, "r");

  if (file != NULL) {
    int length;

    fseek(file, 0, SEEK_END);
    length = ftell(file);
    data = (unsigned char *)calloc(length, sizeof(unsigned char));
    fseek(file, 0, SEEK_SET);
    fread(data, sizeof(unsigned char), length, file);
    fclose(file);

    free(message->data);

    message->data = data;
    message->length = length;
  }
}

/* Saves a message to a file */
void message_save_binary(struct message *message, char *filename)
{
  nullcheck_void(message, message_save_binary);
  nullcheck_void(filename, message_save_binary);

  FILE *file;

  file = fopen(filename, "w");

  if (file != NULL) {
    fwrite(message->data, sizeof(unsigned char), message->length, file);
    fclose(file);
  }
}

/* Helper functions */
static char *string_remove_string(char *from, char *toremove)
{
  int fl = strlen(from) + 1;
  int l = strlen(toremove);
  int ml;

  char *match = strstr(from, toremove);
  while (match != NULL) {
    ml = strlen(match);
    memmove(match, match + l, ml - l);
    fl -= l;

    /* Make sure the string won't completely disappear */
    if (fl < 1)
      fl = 1;

    from = realloc(from, fl);
    from[fl - 1] = 0;

    match = strstr(from, toremove);
  }

  return from;
}

static char *string_append_string(char *to, char *append)
{
  int l1 = 0;
  int l2 = strlen(append);

  if (to != NULL)
    l1 = strlen(to);

  to = realloc(to, l1 + l2 + 1);
  strcpy(to + l1, append);

  return to;
}

/* Not used */
/*
static char *string_append_char(char *to, char append)
{
  int l = 0;

  if (to != NULL)
    l = strlen(to);

  to = realloc(to, l + 2);
  to[l++] = append;
  to[l] = 0;

  return to;
}
*/
