/* gui.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include "song.h"
#include "gui.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"
#include "keyboard.h"
#include "playseq.h"

extern struct editor editor;
extern struct itimerval play_timer;

/* Main window GTK widgets */
GtkWidget *gui_main_window, *gui_main_instrument, *gui_main_instrument_name;
GtkWidget *gui_main_edit, *gui_main_space, *gui_main_octave, *gui_main_tracker;

/* File selector GTK widgets */
GtkWidget *gui_file_open, *gui_file_saveas;

/* Tempo window GTK widgets */
GtkWidget *gui_tempo_window, *gui_tempo_adjustment_tempo;
GtkWidget *gui_tempo_adjustment_tpl;

/* Information window GTK widgets */
GtkWidget *gui_info_song, *gui_info_section, *gui_info_playseq;
GtkWidget *gui_info_block1, *gui_info_block2;
GtkWidget *gui_info_status, *gui_info_timer;

/* Section List window GTK widgets */
GtkWidget *gui_sectionlist_window, *gui_sectionlist_current;
GtkWidget *gui_sectionlist_playseq, *gui_sectionlist_clist;

/* Playing Sequence window GTK widgets */
GtkWidget *gui_playseq_window, *gui_playseq_current, *gui_playseq_block;
GtkWidget *gui_playseq_clist, *gui_playseq_name;

/* Block List window GTK widgets */
GtkWidget *gui_blocklist_window, *gui_blocklist_current, *gui_blocklist_clist;
GtkWidget *gui_blocklist_name, *gui_blocklist_tracks;
GtkWidget *gui_blocklist_length, *gui_blocklist_effectpages;

/* Instrument properties window GTK widgets */
GtkWidget *gui_instrument_window, *gui_instrument_name, *gui_instrument_volume;
GtkWidget *gui_instrument_transpose, *gui_instrument_hold, *gui_instrument_midichannel;

/* Track volumes window GTK widgets */
GtkWidget *gui_trackvolumes_window, *gui_trackvolumes_hbox;

void delete_event(GtkWidget *w, GdkEvent *e, gpointer d) {
  gtk_main_quit();
}

void gui_play_song() {
  play_song();

  tracker_set_pattern((Tracker *)gui_main_tracker,
		      editor.song->blocks[editor.block]);
  tracker_set_patpos((Tracker *)gui_main_tracker, 0);

  gui_info_refresh();
  gui_blocklist_refresh();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
  gui_timer_refresh(0);
}

void gui_continue_song() {
  if(editor.mode!=MODE_PLAY_SONG) {
    continue_song();
    gui_info_refresh();
  }
}

void gui_play_block() {
  play_block();
  tracker_set_patpos((Tracker *)gui_main_tracker, 0);
  gui_info_refresh();
  gui_timer_refresh(0);
}

void gui_continue_block() {
  if(editor.mode!=MODE_PLAY_BLOCK) {
    continue_block();
    gui_info_refresh();
  }
}

void gui_stop() {
  stop();
    
  if(editor.mode!=MODE_IDLE)
    gui_info_refresh();
}

void gui_new() {
  editor_open(NULL);
}

void gui_file_open_activate() {
  gtk_widget_show(gui_file_open);
}

void gui_file_saveas_activate() {
  gtk_widget_show(gui_file_saveas);
}

void gui_file_open_clicked(GtkFileSelection *selector, gpointer user_data) {
  editor_open(gtk_file_selection_get_filename(GTK_FILE_SELECTION(gui_file_open)));
}

void gui_file_saveas_clicked(GtkFileSelection *selector, gpointer user_data) {
  song_save(editor.song, gtk_file_selection_get_filename(GTK_FILE_SELECTION(gui_file_saveas)));
}

void gui_create() {
  GtkWidget *w1, *w2, *w3, *w4, *mainvbox, *tophbox, *instrvbox;
  char *sectionlist_titles[3]={ "Section", "Playing Sequence",
				"Playing Sequence Name" };
  char *playseq_titles[3]={ "Position", "Block", "Block Name" };
  char *blocklist_titles[2]={ "Block", "Name" };

  /* Main window */
  gui_main_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_main_window), 0);
  gtk_window_set_title(GTK_WINDOW(gui_main_window), "");
  gtk_signal_connect(GTK_OBJECT(gui_main_window), "delete_event",
		     GTK_SIGNAL_FUNC(delete_event), NULL);
  gtk_widget_add_events(GTK_WIDGET(gui_main_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_main_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_main_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  /* Main vertical box */
  mainvbox=gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(gui_main_window), mainvbox);

  /* Menu bar */
  w2=gtk_handle_box_new();
  gtk_box_pack_start(GTK_BOX(mainvbox), w2, FALSE, FALSE, 0);
  w1=gtk_menu_bar_new();
  gtk_container_add(GTK_CONTAINER(w2), w1);
  w2=gtk_menu_item_new_with_label("File");
  gtk_menu_bar_append(GTK_MENU_BAR(w1), w2);
  w3=gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(w2), w3);
  w4=gtk_menu_item_new_with_label("New... Ctl+N");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_new), NULL);
  w4=gtk_menu_item_new_with_label("Open... Ctl+O");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_file_open_activate), NULL);
  w4=gtk_menu_item_new();
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("Save Ctl+S");
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("Save As...");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_file_saveas_activate), NULL);
  w4=gtk_menu_item_new();
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("Quit Ctl+Q");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

  w2=gtk_menu_item_new_with_label("Edit");
  gtk_menu_bar_append(GTK_MENU_BAR(w1), w2);
  w3=gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(w2), w3);
  w4=gtk_menu_item_new_with_label("Cut Ctl+X");
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("Copy Ctl+C");
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("Paste Ctl+V");
  gtk_menu_append(GTK_MENU(w3), w4);

  w2=gtk_menu_item_new_with_label("Song");
  gtk_menu_bar_append(GTK_MENU_BAR(w1), w2);
  w3=gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(w2), w3);
  w4=gtk_menu_item_new_with_label("Tempo... AltGr+T");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_tempo_show), NULL);
  w4=gtk_menu_item_new_with_label("Track Volumes... AltGr+V");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_trackvolumes_show), NULL);

  w2=gtk_menu_item_new_with_label("Block");
  gtk_menu_bar_append(GTK_MENU_BAR(w1), w2);
  w3=gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(w2), w3);
  w4=gtk_menu_item_new_with_label("List... AltGr+B");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_blocklist_show), NULL);

  w2=gtk_menu_item_new_with_label("MIDI");
  gtk_menu_bar_append(GTK_MENU_BAR(w1), w2);
  w3=gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(w2), w3);
  w4=gtk_menu_item_new_with_label("Kill ALL notes");
  gtk_menu_append(GTK_MENU(w3), w4);
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(stop_all_notes), NULL);

  /* Top horizontal box */
  tophbox=gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(mainvbox), tophbox, FALSE, FALSE, 4);
  /* Frame for playing controls */
  w2=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w2), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(tophbox), w2, TRUE, TRUE, 4);
  /* Vertical box for playing controls */
  w3=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(w2), w3);
  /* Horizontal box for song play/continue */
  w1=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w3), w1, TRUE, TRUE, 4);
  w2=gtk_label_new("Song");
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_button_new_with_label("Play");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_play_song), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  w2=gtk_button_new_with_label("Continue");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_continue_song), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  /* Horizontal box for block play/continue */
  w1=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w3), w1, TRUE, TRUE, 0);
  w2=gtk_label_new("Block");
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_button_new_with_label("Play");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_play_block), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  w2=gtk_button_new_with_label("Continue");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_continue_block), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  /* Horizontal box for stop button */
  w1=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w3), w1, TRUE, TRUE, 4);
  w2=gtk_button_new_with_label("Stop");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_stop), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);

  /* Vertical box for instrument controls */
  instrvbox=gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tophbox), instrvbox, TRUE, TRUE, 4);
  /* Frame for instrument controls */
  w1=gtk_frame_new("Instrument");
  gtk_frame_set_shadow_type(GTK_FRAME(w1), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(instrvbox), w1, TRUE, TRUE, 4);
  /* Vertical box for instrument controls */
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(w1), w2);
  /* Horizontal box for instrument number, name and type */
  w1=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w1, FALSE, FALSE, 4);
  gui_main_instrument=gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(w1), gui_main_instrument, FALSE, FALSE, 4);
  gui_main_instrument_name=gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(w1), gui_main_instrument_name, TRUE, TRUE, 0);
  w2=gtk_button_new_with_label("Properties");
  gtk_box_pack_start(GTK_BOX(w1), w2, FALSE, FALSE, 4);
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_instrument_show), NULL);
  /* Horizontal box for edit, space, chord and octave selector */
  w1=gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(instrvbox), w1, TRUE, TRUE, 4);
  gui_main_edit=gtk_check_button_new_with_label("Edit");
  gtk_signal_connect(GTK_OBJECT(gui_main_edit), "toggled",
		     GTK_SIGNAL_FUNC(gui_main_toggled),
		     (gpointer)&editor.edit);
  gtk_box_pack_start(GTK_BOX(w1), gui_main_edit, TRUE, TRUE, 4);
  w2=gtk_check_button_new_with_label("Chord");
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_label_new("Space");
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  gui_main_space=(GtkWidget *)gtk_adjustment_new(1, 0, 255, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_main_space), "value_changed",
		     GTK_SIGNAL_FUNC(gui_main_changed), (gpointer)&editor.space);
  w2=gtk_spin_button_new(GTK_ADJUSTMENT(gui_main_space), 1, 0);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_label_new("Keyboard Octaves");
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  gui_main_octave=gtk_option_menu_new();
  w3=gtk_menu_new();
  w4=gtk_menu_item_new_with_label("1 and 2");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)1);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("2 and 3");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)2);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("3 and 4");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)3);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("4 and 5");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)4);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("5 and 6");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)5);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("6 and 7");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)6);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("7 and 8");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)7);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("8 and 9");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)8);
  gtk_menu_append(GTK_MENU(w3), w4);
  w4=gtk_menu_item_new_with_label("9 and A");
  gtk_signal_connect(GTK_OBJECT(w4), "activate",
		     GTK_SIGNAL_FUNC(gui_main_octave_activate), (gpointer)9);
  gtk_menu_append(GTK_MENU(w3), w4);
  /* FIX: why is the menu too small? */
  gtk_option_menu_set_menu(GTK_OPTION_MENU(gui_main_octave), w3);
  gtk_box_pack_start(GTK_BOX(w1), gui_main_octave, TRUE, TRUE, 4);

  /* Tracker */
  gui_main_tracker=tracker_new();
  gtk_box_pack_start(GTK_BOX(mainvbox), gui_main_tracker, TRUE, TRUE, 0);

  /* Info */
  w1=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(mainvbox), w1, FALSE, FALSE, 0);
  w2=gtk_button_new_with_label("Section List");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_show), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_frame_new(NULL);
  gui_info_section=gtk_label_new("Section 1/1");
  gtk_frame_set_shadow_type(GTK_FRAME(w2), GTK_SHADOW_IN);
  gtk_container_add(GTK_CONTAINER(w2), gui_info_section);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  w2=gtk_button_new_with_label("Playing Sequence");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_show), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_frame_new(NULL);
  gui_info_playseq=gtk_label_new("Position 1/1");
  gtk_frame_set_shadow_type(GTK_FRAME(w2), GTK_SHADOW_IN);
  gtk_container_add(GTK_CONTAINER(w2), gui_info_playseq);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  w2=gtk_button_new_with_label("Block List");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_blocklist_show), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w2=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w2), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(w2), w3);
  gui_info_block1=gtk_label_new("Block 0/0");
  gui_info_block2=gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(w3), gui_info_block1, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(w3), gui_info_block2, TRUE, TRUE, 0);
  /* Lower horizontal box */
  w2=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(mainvbox), w2, FALSE, FALSE, 4);
  w3=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w3), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 4);
  gui_info_status=gtk_label_new("Stopped");
  gtk_container_add(GTK_CONTAINER(w3), gui_info_status);
  w3=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w3), GTK_SHADOW_OUT);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 4);
  gui_info_timer=gtk_label_new("00:00");
  gtk_container_add(GTK_CONTAINER(w3), gui_info_timer);

  /* Tempo window */
  gui_tempo_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_tempo_window), 4);
  gtk_window_set_title(GTK_WINDOW(gui_tempo_window), "Tempo");
  gtk_widget_add_events(GTK_WIDGET(gui_tempo_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_tempo_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_tempo_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  w1=gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(gui_tempo_window), w1);

  w3=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w3), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), w3, TRUE, TRUE, 0);
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(w3), w2);
  w3=gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 0);
  w4=gtk_label_new("Tempo");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_tempo_adjustment_tempo=(GtkWidget *)gtk_adjustment_new(DEFAULT_TEMPO, 1,
							 240, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_tempo_adjustment_tempo), "value_changed",
		     GTK_SIGNAL_FUNC(gui_tempo_changed), NULL);
  w4=gtk_hscale_new(GTK_ADJUSTMENT(gui_tempo_adjustment_tempo));
  gtk_scale_set_digits(GTK_SCALE(w4), 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE,4);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 0);
  w4=gtk_label_new("Ticks Per Line");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_tempo_adjustment_tpl=(GtkWidget *)gtk_adjustment_new(6, 1, 32, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_tempo_adjustment_tpl), "value_changed",
		     GTK_SIGNAL_FUNC(gui_tpl_changed), NULL);
  w4=gtk_hscale_new(GTK_ADJUSTMENT(gui_tempo_adjustment_tpl));
  gtk_scale_set_digits(GTK_SCALE(w4), 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w3=gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(w3), "clicked",
		     GTK_SIGNAL_FUNC(gui_tempo_hide), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w3, FALSE, FALSE, 0);

  /* Tempo window */
  gui_trackvolumes_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_trackvolumes_window), 4);
  gtk_window_set_title(GTK_WINDOW(gui_trackvolumes_window), "Track Volumes");
  gtk_widget_add_events(GTK_WIDGET(gui_trackvolumes_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_trackvolumes_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_trackvolumes_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  w1=gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(gui_trackvolumes_window), w1);

  w2=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w2), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  gui_trackvolumes_hbox=gtk_hbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(w2), gui_trackvolumes_hbox);

  w3=gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(w3), "clicked",
		     GTK_SIGNAL_FUNC(gui_trackvolumes_hide), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w3, FALSE, FALSE, 0);

  /* Section list window */
  gui_sectionlist_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_sectionlist_window), 4);
  gtk_window_set_title(GTK_WINDOW(gui_sectionlist_window), "Section List");
  gtk_widget_add_events(GTK_WIDGET(gui_sectionlist_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_sectionlist_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_sectionlist_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  w1=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_sectionlist_window), w1);
  w2=gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(w2), 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w2), GTK_POLICY_NEVER,
				 GTK_POLICY_AUTOMATIC);
  gui_sectionlist_clist=gtk_clist_new_with_titles(3, sectionlist_titles);
  gtk_clist_set_column_width(GTK_CLIST(gui_sectionlist_clist), 0, 30);
  gtk_clist_set_column_width(GTK_CLIST(gui_sectionlist_clist), 1, 30);
  gtk_clist_set_column_width(GTK_CLIST(gui_sectionlist_clist), 2, 140);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_sectionlist_clist), 0, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_sectionlist_clist), 1, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_sectionlist_clist), 2, TRUE);
  gtk_clist_set_selection_mode(GTK_CLIST(gui_sectionlist_clist),
			       GTK_SELECTION_BROWSE);
  gtk_signal_connect(GTK_OBJECT(gui_sectionlist_clist), "select_row",
		     GTK_SIGNAL_FUNC(gui_sectionlist_select_row), NULL);
  gtk_container_add(GTK_CONTAINER(w2), gui_sectionlist_clist);

  gui_sectionlist_current=gtk_frame_new("Section 1/1");
  gtk_frame_set_shadow_type(GTK_FRAME(gui_sectionlist_current),
			    GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), gui_sectionlist_current, FALSE, FALSE, 4);
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_sectionlist_current), w2);

  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 0);
  gui_sectionlist_playseq=gtk_label_new("Playing Sequence 1/1");
  gtk_box_pack_start(GTK_BOX(w3), gui_sectionlist_playseq, FALSE, FALSE, 4);
  w4=gtk_button_new_with_label("<");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_previous), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 0);
  w4=gtk_button_new_with_label(">");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_next), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w3=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 4);
  w4=gtk_button_new_with_label("Insert");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_insert), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w4=gtk_button_new_with_label("Append");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_append), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 0);
  w4=gtk_button_new_with_label("Delete");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_delete), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w2=gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_sectionlist_hide), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, FALSE, FALSE, 0);

  /* Playing sequence window */
  gui_playseq_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_playseq_window), 4);
  gtk_window_set_title(GTK_WINDOW(gui_playseq_window), "Playing Sequence 0/0");
  gtk_widget_add_events(GTK_WIDGET(gui_playseq_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_playseq_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_playseq_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  w1=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_playseq_window), w1);
  w2=gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(w2), 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w2), GTK_POLICY_NEVER,
				 GTK_POLICY_AUTOMATIC);
  gui_playseq_clist=gtk_clist_new_with_titles(3, playseq_titles);
  gtk_clist_set_column_width(GTK_CLIST(gui_playseq_clist), 0, 30);
  gtk_clist_set_column_width(GTK_CLIST(gui_playseq_clist), 1, 30);
  gtk_clist_set_column_width(GTK_CLIST(gui_playseq_clist), 2, 140);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_playseq_clist), 0, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_playseq_clist), 1, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_playseq_clist), 2, TRUE);
  gtk_clist_set_selection_mode(GTK_CLIST(gui_playseq_clist), GTK_SELECTION_BROWSE);
  gtk_signal_connect(GTK_OBJECT(gui_playseq_clist), "select_row",
		     GTK_SIGNAL_FUNC(gui_playseq_select_row), NULL);
  gtk_container_add(GTK_CONTAINER(w2), gui_playseq_clist);

  gui_playseq_current=gtk_frame_new("Position 1/1");
  gtk_frame_set_shadow_type(GTK_FRAME(gui_playseq_current),
			    GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), gui_playseq_current, FALSE, FALSE, 0);
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_playseq_current), w2);

  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 0);
  gui_playseq_block=gtk_label_new("Block 0/0");
  gtk_box_pack_start(GTK_BOX(w3), gui_playseq_block, FALSE, FALSE, 4);
  w4=gtk_button_new_with_label("<");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_previous), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 0);
  w4=gtk_button_new_with_label(">");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_next), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w3=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 4);
  w4=gtk_button_new_with_label("Insert");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_insert), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w4=gtk_button_new_with_label("Append");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_append), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 0);
  w4=gtk_button_new_with_label("Delete");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_delete), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

#ifdef hrr
  gui_playseq_playseq=gtk_frame_new("Playing Sequence 0/0");
  gtk_frame_set_shadow_type(GTK_FRAME(gui_playseq_playseq),
			    GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), gui_playseq_playseq, FALSE, FALSE, 4);
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_playseq_playseq), w2);
#endif
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w1), w3, FALSE, FALSE, 4);
  w4=gtk_label_new("Name");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gtk_label_set_justify(GTK_LABEL(w4), GTK_JUSTIFY_RIGHT);
  gui_playseq_name=gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(gui_playseq_name), "activate",
		     GTK_SIGNAL_FUNC(gui_playseq_activate_name), gui_playseq_name);
  gtk_box_pack_start(GTK_BOX(w3), gui_playseq_name, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w1), w3, FALSE, FALSE, 4);
  w4=gtk_button_new_with_label("Insert New");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_insertnew), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w4=gtk_button_new_with_label("Append New");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_appendnew), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 0);
  w4=gtk_button_new_with_label("Delete");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_delete), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w2=gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_playseq_hide), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, FALSE, FALSE, 0);

  /* Block list window */
  gui_blocklist_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_blocklist_window), 4);
  gtk_window_set_title(GTK_WINDOW(gui_blocklist_window), "Block List");
  gtk_widget_add_events(GTK_WIDGET(gui_blocklist_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  w1=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_blocklist_window), w1);
  w2=gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, TRUE, TRUE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(w2), 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w2), GTK_POLICY_NEVER,
				 GTK_POLICY_AUTOMATIC);
  gui_blocklist_clist=gtk_clist_new_with_titles(2, blocklist_titles);
  gtk_clist_set_column_width(GTK_CLIST(gui_blocklist_clist), 0, 30);
  gtk_clist_set_column_width(GTK_CLIST(gui_blocklist_clist), 1, 170);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_blocklist_clist), 0, TRUE);
  gtk_clist_set_column_auto_resize(GTK_CLIST(gui_blocklist_clist), 1, TRUE);
  gtk_clist_set_selection_mode(GTK_CLIST(gui_blocklist_clist), GTK_SELECTION_BROWSE);
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_clist), "select_row",
		     GTK_SIGNAL_FUNC(gui_blocklist_select_row), NULL);
  gtk_container_add(GTK_CONTAINER(w2), gui_blocklist_clist);

  gui_blocklist_current=gtk_frame_new("Block 0/0");
  gtk_frame_set_shadow_type(GTK_FRAME(gui_blocklist_current),
			    GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), gui_blocklist_current, FALSE, FALSE, 4);
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gui_blocklist_current), w2);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 4);
  w4=gtk_label_new("Name");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gtk_label_set_justify(GTK_LABEL(w4), GTK_JUSTIFY_RIGHT);
  gui_blocklist_name=gtk_entry_new();
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_name), "activate",
		     GTK_SIGNAL_FUNC(gui_blocklist_activate_name), gui_blocklist_name);
  gtk_box_pack_start(GTK_BOX(w3), gui_blocklist_name, TRUE, TRUE, 4);

  w3=gtk_hbox_new(FALSE, 4);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 0);
  w4=gtk_label_new("Tracks");
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  gui_blocklist_tracks=(GtkWidget *)gtk_adjustment_new(4, 1, 255, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_tracks), "value_changed",
		     GTK_SIGNAL_FUNC(gui_blocklist_tracks_changed), NULL);
  w4=gtk_spin_button_new(GTK_ADJUSTMENT(gui_blocklist_tracks), 1, 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w4=gtk_label_new("Length");
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  gui_blocklist_length=(GtkWidget *)gtk_adjustment_new(64, 1, 255, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_length), "value_changed",
		     GTK_SIGNAL_FUNC(gui_blocklist_length_changed), NULL);
  w4=gtk_spin_button_new(GTK_ADJUSTMENT(gui_blocklist_length), 1, 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w4=gtk_label_new("Command pages");
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  gui_blocklist_effectpages=(GtkWidget *)gtk_adjustment_new(1, 1, 255,
							    1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_blocklist_effectpages), "value_changed",
		     GTK_SIGNAL_FUNC(gui_blocklist_effectpages_changed), NULL);
  w4=gtk_spin_button_new(GTK_ADJUSTMENT(gui_blocklist_effectpages), 1, 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, FALSE, FALSE, 4);
  w4=gtk_button_new_with_label("Insert New");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_blocklist_insert), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w4=gtk_button_new_with_label("Append New");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_blocklist_append), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 0);
  w4=gtk_button_new_with_label("Delete");
  gtk_signal_connect(GTK_OBJECT(w4), "clicked",
		     GTK_SIGNAL_FUNC(gui_blocklist_delete), NULL);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w2=gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(w2), "clicked",
		     GTK_SIGNAL_FUNC(gui_blocklist_hide), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w2, FALSE, FALSE, 0);

  /* Instrument properties window */
  gui_instrument_window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(gui_instrument_window), 4);
  gtk_window_set_title(GTK_WINDOW(gui_instrument_window), "Instrument Properties");
  gtk_widget_add_events(GTK_WIDGET(gui_instrument_window), GDK_KEY_RELEASE_MASK);
  gtk_signal_connect(GTK_OBJECT(gui_instrument_window), "key_press_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)1);
  gtk_signal_connect(GTK_OBJECT(gui_instrument_window), "key_release_event",
		     GTK_SIGNAL_FUNC(keyboard_event), (gpointer)0);

  w1=gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(gui_instrument_window), w1);
  w3=gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(w3), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(w1), w3, TRUE, TRUE, 0);
  w2=gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(w3), w2);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 4);
  w4=gtk_label_new("Name");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_instrument_name=gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(w3), gui_instrument_name, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 0);
  w4=gtk_label_new("Volume");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_instrument_volume=(GtkWidget *)gtk_adjustment_new(127, 0, 127, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_instrument_volume), "value_changed",
		     GTK_SIGNAL_FUNC(gui_instrument_volume_changed), NULL);
  w4=gtk_hscale_new(GTK_ADJUSTMENT(gui_instrument_volume));
  gtk_scale_set_digits(GTK_SCALE(w4), 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 0);
  w4=gtk_label_new("Transpose");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_instrument_transpose=(GtkWidget *)gtk_adjustment_new(0, -12, 12, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_instrument_transpose), "value_changed",
		     GTK_SIGNAL_FUNC(gui_instrument_transpose_changed), NULL);
  w4=gtk_hscale_new(GTK_ADJUSTMENT(gui_instrument_transpose));
  gtk_scale_set_digits(GTK_SCALE(w4), 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 0);
  w4=gtk_label_new("Hold");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_instrument_hold=(GtkWidget *)gtk_adjustment_new(0, 0, 255, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_instrument_hold), "value_changed",
		     GTK_SIGNAL_FUNC(gui_instrument_hold_changed), NULL);
  w4=gtk_hscale_new(GTK_ADJUSTMENT(gui_instrument_hold));
  gtk_scale_set_digits(GTK_SCALE(w4), 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);
  w3=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(w2), w3, TRUE, TRUE, 4);
  w4=gtk_label_new("MIDI channel");
  gtk_box_pack_start(GTK_BOX(w3), w4, FALSE, FALSE, 4);
  gui_instrument_midichannel=(GtkWidget *)gtk_adjustment_new(1, 1, 16, 1, 1, 0);
  gtk_signal_connect(GTK_OBJECT(gui_instrument_midichannel), "value_changed",
		     GTK_SIGNAL_FUNC(gui_instrument_midichannel_changed),
		     NULL);
  w4=gtk_hscale_new(GTK_ADJUSTMENT(gui_instrument_midichannel));
  gtk_scale_set_digits(GTK_SCALE(w4), 0);
  gtk_box_pack_start(GTK_BOX(w3), w4, TRUE, TRUE, 4);

  w3=gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(w3), "clicked",
		     GTK_SIGNAL_FUNC(gui_instrument_hide), NULL);
  gtk_box_pack_start(GTK_BOX(w1), w3, FALSE, FALSE, 0);

  /* File selection windows */
  gui_file_open=gtk_file_selection_new("Open file...");
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(gui_file_open)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(gui_file_open_clicked), NULL);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(gui_file_open)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_hide),
			    (gpointer)gui_file_open);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(gui_file_open)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_hide),
			    (gpointer)gui_file_open);

  gui_file_saveas=gtk_file_selection_new("Save file as...");
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(gui_file_saveas)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(gui_file_saveas_clicked), NULL);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(gui_file_saveas)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_hide),
			    (gpointer)gui_file_saveas);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(gui_file_saveas)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_hide),
			    (gpointer)gui_file_saveas);
}

void gui_main_refresh() {
  gtk_option_menu_set_history(GTK_OPTION_MENU(gui_main_octave), 3);
  editor.octave=3;

  tracker_reset((Tracker *)gui_main_tracker);
  tracker_set_pattern((Tracker *)gui_main_tracker, editor.song->blocks[0]);
  tracker_set_num_channels((Tracker *)gui_main_tracker, editor.song->blocks[0]->tracks);
}

void gui_main_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_main_window), 640, 640);
  gtk_widget_show_all(gui_main_window);
}

void gui_tempo_refresh() {
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_tempo_adjustment_tempo),
			   editor.song->tempo);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_tempo_adjustment_tpl),
			   editor.song->ticksperline);
}

void gui_tempo_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_tempo_window), 200, 100);
  gtk_widget_show_all(gui_tempo_window);
}

void gui_tempo_hide() {
  gtk_widget_hide(gui_tempo_window);
}

void gui_trackvolumes_refresh() {
  int i;
  GList *children;
  GtkWidget *w1, *w2, *w3, *w4;
  char track[4];

  /* Destroy all children */
  children=gtk_container_children(GTK_CONTAINER(gui_trackvolumes_hbox));
  while(children!=NULL) {
    gtk_widget_destroy(GTK_WIDGET(children->data));
    children=children->next;
  }

  for(i=0; i<editor.song->maxtracks; i++) {
    w1=gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(gui_trackvolumes_hbox), w1, TRUE, TRUE, 4);

    /* Volume sliders */
    w2=(GtkWidget *)gtk_adjustment_new(editor.song->trackvolumes[i], 0, 127,
				       1, 1, 0);
    gtk_signal_connect(GTK_OBJECT(w2), "value_changed",
		       GTK_SIGNAL_FUNC(gui_trackvolumes_changed),
		       (gpointer)&editor.song->trackvolumes[i]);
    w3=gtk_vscale_new(GTK_ADJUSTMENT(w2));
    gtk_scale_set_digits(GTK_SCALE(w3), 0);
    gtk_box_pack_start(GTK_BOX(w1), w3, TRUE, TRUE, 4);

    /* Mute buttons */
    snprintf(track, 4, "%d", i+1);
    w4=gtk_check_button_new_with_label(track);
    gtk_signal_connect(GTK_OBJECT(w4), "toggled",
		       GTK_SIGNAL_FUNC(gui_trackvolumes_toggled),
		       (gpointer)&editor.song->trackvolumes[i]);
    gtk_box_pack_start(GTK_BOX(w1), w4, FALSE, FALSE, 4);
  }

  gtk_widget_show_all(gui_trackvolumes_hbox);
}

void gui_trackvolumes_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_trackvolumes_window), 200, 200);
  gtk_widget_show_all(gui_trackvolumes_window);
}

void gui_trackvolumes_hide() {
  gtk_widget_hide(gui_trackvolumes_window);
}

void gui_info_refresh() {
  char section[9+8], playseq[10+8], block[7+9], title[76];

  snprintf(section, 8+8, "Section %d/%d", editor.section+1,
	   editor.song->numsections);
  snprintf(playseq, 9+8, "Position %d/%d", editor.playseqpos+1,
	   editor.song->playseqs[editor.playseq]->length);
  snprintf(block, 6+9, "Block %d/%d:", editor.block, editor.song->numblocks-1);
  gtk_label_set_text(GTK_LABEL(gui_info_section), section);
  gtk_label_set_text(GTK_LABEL(gui_info_playseq), playseq);
  gtk_label_set_text(GTK_LABEL(gui_info_block1), block);

  if(editor.song->blocks[editor.block]->name)
    gtk_label_set_text(GTK_LABEL(gui_info_block2),
		       editor.song->blocks[editor.block]->name);

  snprintf(title, 76,
	   "Tutka: Block %d/%d - Cmd Page %d/%d - Tempo %d - TPL %d",
	   editor.block, editor.song->numblocks-1, 0, 0,
	   editor.song->tempo, editor.song->ticksperline);
  gtk_window_set_title(GTK_WINDOW(gui_main_window), title); 

  switch(editor.mode) {
  case MODE_PLAY_SONG:
    gtk_label_set_text(GTK_LABEL(gui_info_status), "Playing song");
    break;
  case MODE_PLAY_BLOCK:
    gtk_label_set_text(GTK_LABEL(gui_info_status), "Playing block");
    break;
  case MODE_IDLE:
    gtk_label_set_text(GTK_LABEL(gui_info_status), "Stopped");
    break;
  }
}

void gui_timer_refresh(unsigned int seconds) {
  char time[6];

  snprintf(time, 6, "%0.2d:%0.2d", seconds/60, seconds%60);
  gtk_label_set_text(GTK_LABEL(gui_info_timer), time);  
}

void gui_instrument_refresh() {
  char instrument[3];

  snprintf(instrument, 3, "%0.2d", editor.instrument+1);
  gtk_label_set_text(GTK_LABEL(gui_main_instrument), instrument);

  gtk_entry_set_text(GTK_ENTRY(gui_main_instrument_name),
		     editor.song->instruments[editor.instrument]->name);
  gtk_entry_set_text(GTK_ENTRY(gui_instrument_name),
		     editor.song->instruments[editor.instrument]->name);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_instrument_volume),
			   editor.song->instruments[editor.instrument]->defaultvelocity);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_instrument_transpose),
			   editor.song->instruments[editor.instrument]->transpose);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_instrument_hold),
			   editor.song->instruments[editor.instrument]->hold);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_instrument_midichannel),
			   editor.song->instruments[editor.instrument]->midichannel+1);
}

void gui_instrument_volume_changed(GtkAdjustment *adjustment,
				   gpointer user_data) {
  editor.song->instruments[editor.instrument]->defaultvelocity=(unsigned char)adjustment->value;
}

void gui_instrument_midichannel_changed(GtkAdjustment *adjustment,
					gpointer user_data) {
  editor.song->instruments[editor.instrument]->midichannel=(unsigned char)adjustment->value-1;
}

void gui_instrument_transpose_changed(GtkAdjustment *adjustment,
				      gpointer user_data) {
  editor.song->instruments[editor.instrument]->transpose=(char)adjustment->value;
}

void gui_instrument_hold_changed(GtkAdjustment *adjustment,
				 gpointer user_data) {
  editor.song->instruments[editor.instrument]->transpose=(unsigned char)adjustment->value;
}

void gui_sectionlist_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_sectionlist_window), 200, 300);
  gtk_widget_show_all(gui_sectionlist_window);
}

void gui_sectionlist_hide() {
  gtk_widget_hide(gui_sectionlist_window);
}

void gui_sectionlist_refresh() {
  int i;
  unsigned short section=editor.section;
  char position[4];
  char playseq[4];
  char *columns[3];

  gtk_clist_freeze(GTK_CLIST(gui_sectionlist_clist));
  gtk_clist_clear(GTK_CLIST(gui_sectionlist_clist));
  columns[0]=position;
  columns[1]=playseq;
  for(i=0; i<editor.song->numsections; i++) {
    snprintf(position, 4, "%0.3d", i+1);
    snprintf(playseq, 4, "%0.3d", editor.song->sections[i]+1);
    columns[2]=editor.song->playseqs[editor.song->sections[i]]->name;
    gtk_clist_append(GTK_CLIST(gui_sectionlist_clist), columns);
  }

  gtk_clist_thaw(GTK_CLIST(gui_sectionlist_clist));
  gtk_clist_select_row(GTK_CLIST(gui_sectionlist_clist), section, 0);
  gtk_clist_moveto(GTK_CLIST(gui_sectionlist_clist), section, 0, 0, 0);
}

void gui_sectionlist_select_row(GtkWidget *list, gint row, gint column,
		      GdkEventButton *event, gpointer data) {
  char section[8+8];
  char playseq[17+8];

  editor.section=row;
  refresh_playseq_and_block();

  snprintf(section, 8+8, "Section %d/%d", editor.section+1,
	   editor.song->numsections);
  snprintf(playseq, 17+8, "Playing Sequence %d/%d", editor.playseq+1,
	   editor.song->numplayseqs);

  gtk_frame_set_label(GTK_FRAME(gui_sectionlist_current), section);
  gtk_label_set_text(GTK_LABEL(gui_sectionlist_playseq), playseq);

  gui_playseq_refresh();
  gui_info_refresh();
}

void gui_playseq_activate_name(GtkWidget *widget, GtkWidget *entry) {
  char *name=gtk_entry_get_text(GTK_ENTRY(entry));

  if(editor.song->playseqs[editor.playseq]->name)
    free(editor.song->playseqs[editor.playseq]->name);
  
  editor.song->playseqs[editor.playseq]->name=(char *)strdup(name);
  
  gui_playseq_refresh();
}

void gui_sectionlist_insert() {
  song_insert_section(editor.song, editor.section);
  gui_sectionlist_refresh();
}

void gui_sectionlist_append() {
  song_insert_section(editor.song, editor.song->numsections);
  gui_sectionlist_refresh();
}

void gui_sectionlist_delete() {
  song_delete_section(editor.song, editor.section);
  if(editor.section>=editor.song->numsections)
    editor.section=editor.song->numsections-1;

  refresh_playseq_and_block();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void gui_sectionlist_previous() {
  if(editor.song->sections[editor.section]>0)
    editor.song->sections[editor.section]--;

  refresh_playseq_and_block();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void gui_sectionlist_next() {
  if(editor.song->sections[editor.section]<editor.song->numplayseqs-1)
    editor.song->sections[editor.section]++;

  refresh_playseq_and_block();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void gui_playseq_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_playseq_window), 250, 300);
  gtk_widget_show_all(gui_playseq_window);
}

void gui_playseq_hide() {
  gtk_widget_hide(gui_playseq_window);
}

void gui_playseq_refresh() {
  int i;
  unsigned short playseqpos=editor.playseqpos;
  char position[9+8];
  char block[6+8];
  char *columns[3];
  char title[24];

  gtk_clist_freeze(GTK_CLIST(gui_playseq_clist));
  gtk_clist_clear(GTK_CLIST(gui_playseq_clist));
  columns[0]=position;
  columns[1]=block;
  for(i=0; i<editor.song->playseqs[editor.playseq]->length; i++) {
    snprintf(position, 4, "%0.3d", i+1);
    snprintf(block, 4, "%0.3d",
	     editor.song->playseqs[editor.playseq]->blocknumbers[i]);
    columns[2]=editor.song->blocks[editor.song->playseqs[editor.playseq]->blocknumbers[i]]->name;
    gtk_clist_append(GTK_CLIST(gui_playseq_clist), columns);
  }

  gtk_clist_thaw(GTK_CLIST(gui_playseq_clist));
  gtk_clist_select_row(GTK_CLIST(gui_playseq_clist), playseqpos, 0);
  gtk_clist_moveto(GTK_CLIST(gui_playseq_clist), playseqpos, 0, 0, 0);

  snprintf(position, 9+8, "Position %d/%d", editor.playseqpos+1,
	   editor.song->playseqs[editor.playseq]->length);
  snprintf(block, 6+8, "Block %d/%d", editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos], editor.song->numblocks-1);
  gtk_frame_set_label(GTK_FRAME(gui_playseq_current), position);
  gtk_label_set_text(GTK_LABEL(gui_playseq_block), block);

  snprintf(title, 24, "Playing Sequence %d/%d", editor.playseq+1,
	   editor.song->numplayseqs);
  gtk_window_set_title(GTK_WINDOW(gui_playseq_window), title);
}

void gui_playseq_select_row(GtkWidget *list, gint row, gint column,
		      GdkEventButton *event, gpointer data) {
  char position[9+8];
  char block[6+8];

  editor.playseqpos=row;

  snprintf(position, 9+8, "Position %d/%d", editor.playseqpos+1,
	   editor.song->playseqs[editor.playseq]->length);
  snprintf(block, 6+8, "Block %d/%d", editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos], editor.song->numblocks-1);
  gtk_frame_set_label(GTK_FRAME(gui_playseq_current), position);
  gtk_label_set_text(GTK_LABEL(gui_playseq_block), block);

  gui_info_refresh();
}

void gui_playseq_insert() {
  if(editor.song!=NULL) {
    playseq_insert(editor.song->playseqs[editor.playseq], editor.playseqpos);
    gui_playseq_refresh();
  }
}

void gui_playseq_append() {
  if(editor.song!=NULL) {
    playseq_insert(editor.song->playseqs[editor.playseq],
		   editor.song->playseqs[editor.playseq]->length);
    gui_playseq_refresh();
  }
}

void gui_playseq_delete() {
  if(editor.song!=NULL) {
    playseq_delete(editor.song->playseqs[editor.playseq], editor.playseqpos);
    gui_playseq_refresh();
  }
}

void gui_playseq_insertnew() {
  if(editor.song!=NULL) {
    song_insert_playseq(editor.song, editor.playseq);
    gui_info_refresh();
    gui_playseq_refresh();
    gui_sectionlist_refresh();
  }
}

void gui_playseq_appendnew() {
  if(editor.song!=NULL) {
    song_insert_playseq(editor.song, editor.song->numplayseqs);
    gui_info_refresh();
    gui_playseq_refresh();
    gui_sectionlist_refresh();
  }
}

void gui_playseq_deletecurrent() {
  if(editor.song!=NULL) {
    song_delete_playseq(editor.song, editor.playseq);
    gui_info_refresh();
    gui_playseq_refresh();
    gui_sectionlist_refresh();
  }
}

void gui_playseq_previous() {
  unsigned short playseq=editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos];

  if(playseq>0)
    playseq_set(editor.song->playseqs[editor.playseq], editor.playseqpos,
		playseq-1);
  gui_playseq_refresh();
}

void gui_playseq_next() {
  unsigned short playseq=editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos];

  if(playseq<editor.song->numblocks-1)
    playseq_set(editor.song->playseqs[editor.playseq], editor.playseqpos,
		playseq+1);
  gui_playseq_refresh();
}

void gui_blocklist_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_blocklist_window), 200, 300);
  gtk_widget_show_all(gui_blocklist_window);
}

void gui_blocklist_hide() {
  gtk_widget_hide(gui_blocklist_window);
}

void gui_blocklist_refresh() {
  int i;
  unsigned short block=editor.block;
  char number[4];
  char *columns[2];

  gtk_clist_freeze(GTK_CLIST(gui_blocklist_clist));
  gtk_clist_clear(GTK_CLIST(gui_blocklist_clist));
  columns[0]=number;
  for(i=0; i<editor.song->numblocks; i++) {
    snprintf(number, 4, "%0.3d", i);
    columns[1]=editor.song->blocks[i]->name;
    gtk_clist_append(GTK_CLIST(gui_blocklist_clist), columns);
  }

  gtk_clist_thaw(GTK_CLIST(gui_blocklist_clist));
  gtk_clist_select_row(GTK_CLIST(gui_blocklist_clist), block, 0);
  gtk_clist_moveto(GTK_CLIST(gui_blocklist_clist), block, 0, 0, 0);

  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_blocklist_tracks),
			   editor.song->blocks[editor.block]->tracks);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_blocklist_length),
			   editor.song->blocks[editor.block]->length);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_blocklist_effectpages),
			   editor.song->blocks[editor.block]->effectpages);
}

void gui_blocklist_select_row(GtkWidget *list, gint row, gint column,
		      GdkEventButton *event, gpointer data) {
  char block[6+8];

  editor.block=row;

  if(editor.song->blocks[row]->name)
    gtk_entry_set_text(GTK_ENTRY(gui_blocklist_name),
		       editor.song->blocks[row]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(gui_blocklist_name), "");

  snprintf(block, 6+8, "Block %d/%d", editor.block,
	   editor.song->numblocks-1);

  gtk_frame_set_label(GTK_FRAME(gui_blocklist_current), block);

  gui_info_refresh();
  tracker_set_pattern((Tracker *)gui_main_tracker, editor.song->blocks[row]);
  tracker_set_num_channels((Tracker *)gui_main_tracker,
			   editor.song->blocks[row]->tracks);
  
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_blocklist_tracks),
			   editor.song->blocks[row]->tracks);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_blocklist_length),
			   editor.song->blocks[row]->length);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gui_blocklist_effectpages),
			   editor.song->blocks[row]->effectpages);
}

void gui_blocklist_activate_name(GtkWidget *widget, GtkWidget *entry) {
  if(editor.song!=NULL) {
    char *name=gtk_entry_get_text(GTK_ENTRY(entry));
    
    if(editor.song->blocks[editor.block]->name)
      free(editor.song->blocks[editor.block]->name);
    
    editor.song->blocks[editor.block]->name=(char *)strdup(name);
    
    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

void gui_blocklist_insert() {
  if(editor.song!=NULL) {
    song_insert_block(editor.song, editor.block);
    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

void gui_blocklist_append() {
  if(editor.song!=NULL) {
    song_insert_block(editor.song, editor.song->numblocks);
    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

void gui_blocklist_delete() {
  if(editor.song!=NULL) {
    song_delete_block(editor.song, editor.block);
    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

void gui_blocklist_tracks_changed(GtkAdjustment *adjustment,
				  gpointer user_data) {
  if(editor.song!=NULL) {
    block_set_tracks(editor.song->blocks[editor.block],
		     (unsigned short)adjustment->value);
    tracker_set_num_channels((Tracker *)gui_main_tracker,
			     (int)adjustment->value);
    tracker_redraw((Tracker *)gui_main_tracker);
    if(song_check_maxtracks(editor.song))
      gui_trackvolumes_refresh();
  }
}

void gui_blocklist_length_changed(GtkAdjustment *adjustment,
				  gpointer user_data) {
  if(editor.song!=NULL) {
    Tracker *tracker=(Tracker *)gui_main_tracker;
    
    block_set_length(editor.song->blocks[editor.block],
		     (unsigned short)adjustment->value);
    tracker->curpattern=editor.song->blocks[editor.block];
    tracker_redraw((Tracker *)gui_main_tracker);
  }
}

void gui_blocklist_effectpages_changed(GtkAdjustment *adjustment,
				       gpointer user_data) {
  if(editor.song!=NULL) {
    block_set_effectpages(editor.song->blocks[editor.block],
			  (unsigned short)adjustment->value);
    tracker_redraw((Tracker *)gui_main_tracker);
  }
}

void gui_main_octave_activate(GtkMenuItem *menuitem, gpointer user_data) {
  int i=(int)user_data;
  editor.octave=(unsigned char)i;
}

void gui_instrument_show() {
  gtk_window_set_default_size(GTK_WINDOW(gui_instrument_window), 300, 150);
  gtk_widget_show_all(gui_instrument_window);
}

void gui_instrument_hide() {
  gtk_widget_hide(gui_instrument_window);
}

void gui_main_toggled(GtkToggleButton *togglebutton, gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    *data=1;
  else
    *data=0;
}

void gui_main_changed(GtkAdjustment *adjustment, gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  *data=(unsigned char)adjustment->value;
}

void gui_tempo_changed(GtkAdjustment *adjustment, gpointer user_data) {
  song_set_tempo(editor.song, (unsigned char)adjustment->value);
  gui_info_refresh();
}

void gui_tpl_changed(GtkAdjustment *adjustment, gpointer user_data) {
  song_set_tpl(editor.song, (unsigned char)adjustment->value);
  gui_info_refresh();
}

void gui_trackvolumes_changed(GtkAdjustment *adjustment, gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  *data=((*data)&128)|((unsigned char)adjustment->value);
}

void gui_trackvolumes_toggled(GtkToggleButton *togglebutton,
			      gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    *data=((*data)&127)|128;
  else
    *data=(*data)&127;
}
