/* player.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <pthread.h>
#include "song.h"
#include "editor.h"
#include "trackerwidget.h"
#include "midi.h"
#include "player.h"

extern struct editor editor;
extern GtkWidget *gui_main_tracker;

static unsigned short tick;

/* Plays the song from the beginning */
void play_song() {
  editor.mode=MODE_PLAY_SONG;
  editor.section=0;
  editor.playseqpos=0;
  player_refresh_playseq_and_block();
  player_start(0);
}

/* Continues playing the song from the current position */
void continue_song() {
  if(editor.mode!=MODE_PLAY_SONG) {
    editor.mode=MODE_PLAY_SONG;
    player_start(1);
  }
}

/* Plays the block from the beginning */
void play_block() {
  editor.mode=MODE_PLAY_BLOCK;
  player_start(0);
}

/* Continues playing the block from the current position */
void continue_block() {
  if(editor.mode!=MODE_PLAY_BLOCK) {
    editor.mode=MODE_PLAY_BLOCK;
    player_start(1);
  }
}

/* Stops the player */
void stop() {
  player_stop_notes();
  if(editor.mode!=MODE_IDLE) {
    editor.mode=MODE_IDLE;
    player_stop();

    /* Refresh the GUI accordingly
     * (FIX: This is called both from player and GUI... dangerous!) */
    gui_info_refresh();
  } 
}

/* Handles one tick by playing all notes scheduled to be played */
void player_next_tick() {
  Tracker *tracker=(Tracker *)gui_main_tracker;
  int i, j;
  struct timeval tod;
  struct block *block=editor.song->blocks[editor.block];
  unsigned char posteffect=0, postvalue=0;

  /* Play notes scheduled to be played */
  for(i=0; i<block->tracks; i++) {
    unsigned char volume=127;
    unsigned char notedelay=0;
    unsigned char note=block->notes[(block->tracks*tracker->patpos+i)*2];
    unsigned char instrument=block->notes[(block->tracks*tracker->patpos+i)*2+1];
    /* Handle effects on all effect pages */
    for(j=0; j<block->effectpages; j++) {
      unsigned char effect=block->effects[j*block->tracks*block->length*2+(block->tracks*tracker->patpos+i)*2];
      unsigned char value=block->effects[j*block->tracks*block->length*2+(block->tracks*tracker->patpos+i)*2+1];
      char midichannel=-1;

      /* Check which MIDI channel the command will affect */
      if(instrument!=0)
	/* Instrument number defines MIDI channel */
	midichannel=editor.song->instruments[instrument-1]->midichannel;
      else {
	/* Note playing defines MIDI channel */
	if(editor.trackstatus[i*4+TRACK_STATUS_HOLD]!=-1)
	  midichannel=editor.trackstatus[i*4+TRACK_STATUS_MIDI_CHANNEL];
      }

      switch(effect) {
      case EFFECT_PITCH_WHEEL:
	/* Pitch wheel can be set if the MIDI channel is known */
	if(midichannel!=-1) {
	  if(value<0x80) {
	    if(tick==0) {
	      midi_pitch_wheel(midichannel, value);
	      editor.midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL]=value;
	    }
	  } else {
	    if(tick<editor.song->ticksperline-1) {
	      unsigned char delta=(value-0x80-editor.midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL])/(editor.song->ticksperline-1);

	      midi_pitch_wheel(midichannel, editor.midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL]+tick*delta);
	    } else {
	      midi_pitch_wheel(midichannel, value-0x80);
	      editor.midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL]=value-0x80;
	    }
	  }
	}
	break;
      case EFFECT_END_BLOCK:
	/* Only on last tick */
	if(tick==editor.song->ticksperline-1) {
	  posteffect=EFFECT_END_BLOCK;
	  postvalue=value;
	}
	break;
      case EFFECT_PLAYSEQ_POSITION:
	/* Only on last tick */
	if(tick==editor.song->ticksperline-1) {
	  posteffect=EFFECT_PLAYSEQ_POSITION;
	  postvalue=value;
	}
	break;
      case EFFECT_STOP:
	/* Only on last tick */
	if(tick==editor.song->ticksperline-1)
	  stop();
	break;
      case EFFECT_SEND_SYSEX:
	/* Only on first tick */
	if(tick==0 && value<editor.song->numsysexes)
	  midi_system_exclusive(editor.song->sysexes[value]->data,
				editor.song->sysexes[value]->length);
	break;
      case EFFECT_NOTE_DELAY:
	notedelay=value;
	break;
      case EFFECT_VELOCITY:
	if(note!=0) {
	  volume=value;
	  if(midichannel!=-1)
	    editor.midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]=value;
	} else {
	  /* Note playing defines MIDI channel */
	  if(editor.trackstatus[i*4+TRACK_STATUS_HOLD]!=-1) {
	    midichannel=editor.trackstatus[i*4+TRACK_STATUS_MIDI_CHANNEL];

	    if(value<0x80) {
	      if(tick==0) {
		midi_aftertouch(midichannel, editor.trackstatus[i*4+TRACK_STATUS_NOTE], value);
		editor.midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]=value;
	      }
	    } else {
	      if(tick<editor.song->ticksperline-1) {
		unsigned char delta=(value-0x80-editor.midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH])/(editor.song->ticksperline-1);
		
		midi_aftertouch(midichannel, editor.trackstatus[i*4+TRACK_STATUS_NOTE], editor.midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]+tick*delta);
	      } else {
		midi_aftertouch(midichannel, editor.trackstatus[i*4+TRACK_STATUS_NOTE], value-0x80);
		editor.midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]=value-0x80;
	      }
	    }
	  }
	}
	break;
      case EFFECT_CHANNEL_PRESSURE:
	/* Channel pressure can be set if the MIDI channel is known */
	if(midichannel!=-1) {
	  if(value<0x80) {
	    if(tick==0) {
	      midi_channel_pressure(midichannel, value);
	      editor.midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE]=value;
	    }
	  } else {
	    if(tick<editor.song->ticksperline-1) {
	      unsigned char delta=(value-0x80-editor.midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE])/(editor.song->ticksperline-1);

	      midi_channel_pressure(midichannel, editor.midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE]+tick*delta);
	    } else {
	      midi_channel_pressure(midichannel, value-0x80);
	      editor.midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE]=value-0x80;
	    }
	  }
	}
	break;
      case EFFECT_TPL:
	song_set_tpl(editor.song, value);
	gdk_threads_enter();
	gui_tempo_refresh();
	gdk_threads_leave();
	break;
      case EFFECT_TEMPO:
	song_set_tempo(editor.song, value);
	gdk_threads_enter();
	gui_tempo_refresh();
	gdk_threads_leave();
	break;
      }

      /* Handle MIDI controllers */
      if(effect>=EFFECT_MIDI_CONTROLLERS) {
	/* MIDI controllers can be set if the MIDI channel is known */
	if(midichannel!=-1) {
	  if(value<0x80) {
	    if(tick==0) {
	      midi_controller(midichannel, effect-EFFECT_MIDI_CONTROLLERS,
			      value);
	      editor.midicontrollervalues[midichannel*VALUES+effect-EFFECT_MIDI_CONTROLLERS]=value;
	    }
	  } else {
	    if(tick<editor.song->ticksperline-1) {
	      unsigned char delta=(value-0x80-editor.midicontrollervalues[midichannel*VALUES+effect-EFFECT_MIDI_CONTROLLERS])/(editor.song->ticksperline-1);

	      midi_controller(midichannel, effect-EFFECT_MIDI_CONTROLLERS,
			      editor.midicontrollervalues[midichannel*VALUES+effect-EFFECT_MIDI_CONTROLLERS]+tick*delta);
	    } else {
	      midi_controller(midichannel, effect-EFFECT_MIDI_CONTROLLERS,
			      value);
	      editor.midicontrollervalues[midichannel*VALUES+effect-EFFECT_MIDI_CONTROLLERS]=value;
	    }
	  }
	}
      }
    }
    
    if(note!=0 && tick==notedelay) {
      unsigned char hold=0;
      note--;

      player_play_note(instrument-1, note, volume, i);
      
      if(instrument<editor.song->numinstruments &&
	 editor.song->instruments[instrument-1]!=NULL)
	hold=editor.song->instruments[block->notes[(block->tracks*tracker->patpos+i)*2+1]-1]->hold;
      
      if(hold==0)
	editor.trackstatus[i*4+TRACK_STATUS_HOLD]=-1;
      else
	editor.trackstatus[i*4+TRACK_STATUS_HOLD]=hold;
    }
  }

  /* Decrement hold times of notes and stop notes that should be stopped */
  for(i=0; i<editor.song->maxtracks; i++) {
    if(editor.trackstatus[i*4+TRACK_STATUS_HOLD]!=-1) {
      editor.trackstatus[i*4+TRACK_STATUS_HOLD]--;
      if(editor.trackstatus[i*4+TRACK_STATUS_HOLD]==-1) {
	midi_note_off(editor.trackstatus[i*4+TRACK_STATUS_MIDI_CHANNEL],
		      editor.trackstatus[i*4+TRACK_STATUS_NOTE],
		      127);
	editor.trackstatus[i*4+TRACK_STATUS_NOTE]=-1;
      }
    }
  }

  /* Next tick */
  tick++;
  tick%=editor.song->ticksperline;

  /* Advance and handle post effects if ticksperline ticks have passed */
  if(tick==0) {
    int newpos=tracker->patpos+1, changeblock=0;
    switch(posteffect) {
    case EFFECT_END_BLOCK:
      newpos=postvalue;
      player_next_playseq();
      changeblock=1;
      break;
    case EFFECT_PLAYSEQ_POSITION:
      newpos=0;
      editor.playseqpos=postvalue;
      if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length) {
	editor.playseqpos=0;
	player_next_section();
      }
      changeblock=1;
      break;
    default:
      /* Advance in block */
      if(newpos>=tracker->curpattern->length) {
	newpos=0;
	if(editor.mode==MODE_PLAY_SONG)
	  player_next_playseq();
	changeblock=1;
      }
      break;
    }

    gdk_threads_enter();
    if(changeblock==1) {
      player_refresh_playseq_and_block();
      tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
      gui_info_refresh();
      gui_blocklist_refresh();
      gui_playseq_refresh();
    }
    tracker_set_patpos(tracker, newpos);
    gettimeofday(&tod, NULL);
    gui_timer_refresh((int)(editor.playedsofar.tv_sec+tod.tv_sec-editor.playingstarted.tv_sec));
    gdk_threads_leave();
  }
}

/* Advances in playing sequence and jumps to next section if necessary */
void player_next_playseq() {
  editor.playseqpos++;
  if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length) {
    editor.playseqpos=0;
    player_next_section();
  }
}

/* Advances in section and jumps to the beginning if necessary */
void player_next_section() {
  editor.section++;
  if(editor.section>=editor.song->numsections)
    editor.section=0;
}

/* Plays a note using given instrument on a given track */
void player_play_note(unsigned short instrument, unsigned char note,
		      unsigned char volume, unsigned char track) {
  /* Stop currently playing note */
  if(editor.trackstatus[track*4+TRACK_STATUS_NOTE]!=-1)
    midi_note_off(editor.trackstatus[track*4+TRACK_STATUS_MIDI_CHANNEL],
		  editor.trackstatus[track*4+TRACK_STATUS_NOTE],
		  127);

  /* Don't play a note if the instrument does not exist */
  if(instrument<editor.song->numinstruments &&
     editor.song->instruments[instrument]!=NULL) {
    /* Apply note transpose */
    note+=editor.song->instruments[instrument]->transpose;

    /* Set new track status (channel, note, volume, time) */
    editor.trackstatus[track*4+TRACK_STATUS_MIDI_CHANNEL]=editor.song->instruments[instrument]->midichannel;
    editor.trackstatus[track*4+TRACK_STATUS_NOTE]=note;
    editor.trackstatus[track*4+TRACK_STATUS_VOLUME]=editor.song->instruments[instrument]->defaultvelocity*volume/127*(editor.song->trackvolumes[track]&127)/127*((255-editor.song->trackvolumes[track])&128)/128*editor.song->mastervolume/127;
    if(editor.song->instruments[instrument]->hold>0)
      editor.trackstatus[track*4+TRACK_STATUS_HOLD]=editor.song->instruments[instrument]->hold;
    else
      editor.trackstatus[track*4+TRACK_STATUS_HOLD]=-1;

    /* Make sure the volume isn't too large */
    if(editor.trackstatus[track*4+TRACK_STATUS_VOLUME]<0)
      editor.trackstatus[track*4+TRACK_STATUS_VOLUME]=127;
    
    /* Play note */
    midi_note_on(editor.trackstatus[track*4+TRACK_STATUS_MIDI_CHANNEL],
		 editor.trackstatus[track*4+TRACK_STATUS_NOTE],
		 editor.trackstatus[track*4+TRACK_STATUS_VOLUME]);
  }
}

/* Stops notes playing at the moment */
void player_stop_notes() {
  int i;
  for(i=0; i<editor.song->maxtracks; i++) {
    if(editor.trackstatus[i*4+1]!=-1) {
      midi_note_off(editor.trackstatus[i*4], editor.trackstatus[i*4+1], 127);
      editor.trackstatus[i*4]=-1;
      editor.trackstatus[i*4+1]=-1;
      editor.trackstatus[i*4+2]=-1;
      editor.trackstatus[i*4+3]=-1;
    }
  }

  /* Reset MIDI controller values to 0x40 (can't be horribly wrong...) */
  for(i=0; i<16*VALUES; i++)
    editor.midicontrollervalues[i]=0x40;
}

/* Stops all notes playing at the moment */
void player_stop_all_notes() {
  int i, j;
  for(i=0; i<16; i++)
    for(j=0; j<128; j++)
      midi_note_off(i, j, 127);
}

/* Refreshes editor.playseq from editor.section and editor.block from
 * editor.playseqpos */
void player_refresh_playseq_and_block() {
  if(editor.section>=editor.song->numsections)
    editor.section=0;
  editor.playseq=editor.song->sections[editor.section];

  if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length) {
    editor.playseqpos=0;
    player_refresh_playseq_and_block();
  } else
    editor.block=editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos];
}

/* Creates the player thread */
void player_start(int cont) {
  player_stop();
  player_stop_notes();

  gettimeofday(&editor.playingstarted, NULL);
  if(cont==0) {
    editor.playedsofar.tv_sec=0;
    editor.playedsofar.tv_usec=0;
  }

  editor.threadstatus=pthread_create(&editor.thread, NULL, player_thread,
				     NULL);
}

/* Kills the player thread */
void player_stop() {
  if(editor.threadstatus==0) {
    pthread_cancel(editor.thread);
    editor.threadstatus=-1;
  }
}

/* Player thread */
void player_thread() {
  struct timeval next, now;
  struct timespec req, rem;

  tick=0;
  next.tv_sec=editor.playingstarted.tv_sec;
  next.tv_usec=editor.playingstarted.tv_usec;

  while(1) {
    /* Calculate time of next tick (tick every 1000000/((BPM/60)*24) usecs) */
    next.tv_sec+=(25/editor.song->tempo)/10;
    next.tv_usec+=((2500000/editor.song->tempo)%1000000);
    while(next.tv_usec>1000000) {
      next.tv_sec++;
      next.tv_usec-=1000000;
    }

    /* Calculate difference between now and the next tick */
    gettimeofday(&now, NULL);
    req.tv_sec=next.tv_sec-now.tv_sec;
    req.tv_nsec=(next.tv_usec-now.tv_usec)*1000;
    while(req.tv_nsec<0) {
      req.tv_sec--;
      req.tv_nsec+=1000000000;
    }

    /* Sleep until the next tick if necessary */
    if(req.tv_sec>=0)
      while(nanosleep(&req, &rem)==-1);

    /* Handle this tick */
    player_next_tick();

    /* Check whether this thread should be killed */
    pthread_testcancel();
  }
}
