/* block.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "block.h"

struct block *block_alloc(int tracks, int length, int effectpages) {
  struct block *block;

  block=(struct block *)calloc(1, sizeof(struct block));
  block->tracks=tracks;
  block->length=length;
  block->notes=(unsigned char *)calloc(2*tracks*length, sizeof(unsigned char));
  block->effectpages=effectpages;
  block->effects=(unsigned char *)calloc(effectpages*2*tracks*length,
					 sizeof(unsigned char));

  return block;
}

void block_free(struct block *block) {
  if(block) {
    if(block->name)
      free(block->name);
    
    if(block->notes)
      free(block->notes);
    
    if(block->effects)
      free(block->effects);
    
    free(block);
  } else
    fprintf(stderr, "block_free() called with null block\n");
}

void block_set_note(struct block *block, unsigned short line,
		    unsigned short track, unsigned char octave,
		    unsigned char note, unsigned char instrument) {
  if(block) {
    if(note) {
      block->notes[2*(block->tracks*line+track)]=octave*12+note;
      block->notes[2*(block->tracks*line+track)+1]=instrument;
    } else {
      block->notes[2*(block->tracks*line+track)]=0;
      block->notes[2*(block->tracks*line+track)+1]=0;
    }
  } else
    fprintf(stderr, "block_set_note() called with null block\n");
}

void block_set_instrument(struct block *block, unsigned short line,
			  unsigned short track, unsigned char instrument) {
  if(block)
    block->notes[2*(block->tracks*line+track)+1]=instrument;
  else
    fprintf(stderr, "block_set_instrument() called with null block\n");
}

void block_set_effect(struct block *block, unsigned short line,
		      unsigned short track, unsigned short effectpage,
		      unsigned char slot, unsigned char data) {
  if(block) {
    if(slot&1) {
      block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]&=0xf0;
      block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]|=data;
    } else {
      block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]&=0x0f;
      block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+slot/2]|=(data<<4);
    }
  } else
    fprintf(stderr, "block_set_effect() called with null block\n");
}
  
void block_set_effect_full(struct block *block, unsigned short line,
			   unsigned short track, unsigned short effectpage,
			   unsigned char effect, unsigned char data) {
  if(block) {
    block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)]=effect;
    block->effects[effectpage*2*block->tracks*block->length+2*(block->tracks*line+track)+1]=data;
  } else
    fprintf(stderr, "block_set_effect_full() called with null block\n");
}

void block_set_tracks(struct block *block, unsigned short tracks) {
  if(block) {
    unsigned char *notes=(unsigned char *)calloc(2*tracks*block->length,
						 sizeof(unsigned char));
    unsigned char *effects=(unsigned char *)calloc(block->effectpages*2*tracks*block->length,
						 sizeof(unsigned char));
    int l, t, ep, nl=block->length, nt=block->tracks;
    unsigned short existingtracks;

    if(tracks<nt)
      existingtracks=tracks;
    else
      existingtracks=nt;

    /* Copy the notes and the effects of the block to a new data array */
    for(l=0; l<nl; l++)
      for(t=0; t<existingtracks; t++) {
	notes[(l*tracks+t)*2]=block->notes[(l*nt+t)*2];
	notes[(l*tracks+t)*2+1]=block->notes[(l*nt+t)*2+1];
	for(ep=0; ep<block->effectpages; ep++) {
	  effects[ep*2*tracks*nl+(l*tracks+t)*2]=block->effects[ep*2*existingtracks*nl+(l*nt+t)*2];
	  effects[ep*2*tracks*nl+(l*tracks+t)*2+1]=block->effects[ep*2*existingtracks*nl+(l*nt+t)*2+1];
	}
      }

    if(block->notes)
      free(block->notes);
    if(block->effects)
      free(block->effects);

    block->notes=notes;
    block->effects=effects;
    block->tracks=tracks;
  } else
    fprintf(stderr, "block_set_tracks() called with null block\n");
}

void block_set_length(struct block *block, unsigned short length) {
  if(block) {
    unsigned char *notes=(unsigned char *)calloc(2*block->tracks*length,
						 sizeof(unsigned char));
    unsigned char *effects=(unsigned char *)calloc(block->effectpages*2*block->tracks*length,
						 sizeof(unsigned char));
    int l, t, ep, nl=block->length, nt=block->tracks;
    unsigned short existinglength;

    if(length<nl)
      existinglength=length;
    else
      existinglength=nl;

    /* Copy the notes and the effects of the block to a new data array */
    for(l=0; l<existinglength; l++)
      for(t=0; t<nt; t++) {
	notes[(l*nt+t)*2]=block->notes[(l*nt+t)*2];
	notes[(l*nt+t)*2+1]=block->notes[(l*nt+t)*2+1];
	for(ep=0; ep<block->effectpages; ep++) {
	  effects[ep*2*nt*length+(l*nt+t)*2]=block->effects[ep*2*nt*existinglength+(l*nt+t)*2];
	  effects[ep*2*nt*length+(l*nt+t)*2+1]=block->effects[ep*2*nt*existinglength+(l*nt+t)*2+1];
	}
      }

    if(block->notes)
      free(block->notes);
    if(block->effects)
      free(block->effects);

    block->notes=notes;
    block->effects=effects;
    block->length=length;
  } else
    fprintf(stderr, "block_set_length() called with null block\n");
}

void block_set_effectpages(struct block *block, unsigned short effectpages) {
  if(block) {
    unsigned char *effects=(unsigned char *)calloc(effectpages*2*block->tracks*block->length,
						 sizeof(unsigned char));
    unsigned short existingeffectpages;

    if(effectpages<block->effectpages)
      existingeffectpages=effectpages;
    else
      existingeffectpages=block->effectpages;

    memcpy(effects, block->effects,
	   existingeffectpages*2*block->tracks*block->length);

    if(block->effects)
      free(block->effects);

    block->effects=effects;
    block->effectpages=effectpages;
  } else
    fprintf(stderr, "block_set_effectpages() called with null block\n");
}

/* Copies a part of a block to a new block */
struct block *block_copy(struct block *block, int starttrack, int startline,
			 int endtrack, int endline) {
  if(block) {
    struct block *newblock=block_alloc(endtrack-starttrack+1,
				       endline-startline+1,
				       block->effectpages);
    int t, l, ep;
    int onl=block->length, nnl=newblock->length;
    int ont=block->tracks, nnt=newblock->tracks;
    
    /* Copy the given part of the block to a new block */
    for(l=startline; l<=endline; l++) {
      for(t=starttrack; t<=endtrack; t++) {
	newblock->notes[((l-startline)*nnt+(t-starttrack))*2]=block->notes[(l*ont+t)*2];
	newblock->notes[((l-startline)*nnt+(t-starttrack))*2+1]=block->notes[(l*ont+t)*2+1];
	for(ep=0; ep<block->effectpages; ep++) {
	  newblock->effects[ep*2*nnt*nnl+((l-startline)*nnt+(t-starttrack))*2]=block->effects[ep*2*ont*onl+(l*ont+t)*2];
	  newblock->effects[ep*2*nnt*nnl+((l-startline)*nnt+(t-starttrack))*2+1]=block->effects[ep*2*ont*onl+(l*ont+t)*2+1];
	}
      }
    }
    
    return newblock;
  } else {
    fprintf(stderr, "block_copy() called with null block\n");
    return NULL;
  }
}

/* Pastes a block to another block in the given position */
void block_paste(struct block *block, struct block *from, int track,int line) {
  if(block) {
    if(from) {
      int t, l, ep;
      int nl=block->length, nt=block->tracks;
      int fnl=from->length, fnt=from->tracks;
      int cnl=fnl, cnt=fnt;

      if(line+fnl>nl)
	cnl=nl-line;
      if(track+fnt>nt)
	cnt=nt-track;

      /* Copy the from block to the destination block; make sure it fits */
      for(l=0; l<cnl; l++) {
	for(t=0; t<cnt; t++) {
	  block->notes[((line+l)*nt+(track+t))*2]=from->notes[(l*fnt+t)*2];
	  block->notes[((line+l)*nt+(track+t))*2+1]=from->notes[(l*fnt+t)*2+1];
	  for(ep=0; ep<block->effectpages; ep++) {
	    block->effects[ep*2*nt*nl+((line+l)*nt+(track+t))*2]=from->effects[ep*2*fnt*fnl+(l*fnt+t)*2];
	    block->effects[ep*2*nt*nl+((line+l)*nt+(track+t))*2+1]=from->effects[ep*2*fnt*fnl+(l*fnt+t)*2+1];
	  }
	}
      }
    }
  } else
    fprintf(stderr, "block_paste() called with null block\n");
}

/* Clears a part of a block */
void block_clear(struct block *block, int starttrack, int startline,
			 int endtrack, int endline) {
  if(block) {
    int t, l, ep;
    int nl=block->length, nt=block->tracks;
    
    for(l=startline; l<=endline; l++) {
      for(t=starttrack; t<=endtrack; t++) {
	block->notes[(l*nt+t)*2]=0;
	block->notes[(l*nt+t)*2+1]=0;
	for(ep=0; ep<block->effectpages; ep++) {
	  block->effects[ep*2*nt*nl+(l*nt+t)*2]=0;
	  block->effects[ep*2*nt*nl+(l*nt+t)*2+1]=0;
	}
      }
    }
  } else
    fprintf(stderr, "block_clear() called with null block\n");
}

/* Parses a block element in an XML file */
struct block *block_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct block *block=NULL;
  
  if((!xmlStrcmp(cur->name, "block")) && (cur->ns==ns)) {
    char *prop;
    int tracks=4, length=64, effectpages=1;

    /* Get block properties */
    prop=xmlGetProp(cur, "tracks");
    if(prop!=NULL)
      tracks=atoi(prop);
    prop=xmlGetProp(cur, "length");
    if(prop!=NULL)
      length=atoi(prop);
    prop=xmlGetProp(cur, "effectpages");
    if(prop!=NULL)
      effectpages=atoi(prop);

    /* Allocate block */
    block=block_alloc(tracks, length, effectpages);
    block->name=xmlGetProp(cur, "name");

    /* Get block contents */
    cur=cur->xmlChildrenNode;
    while(cur!=NULL) {
      if((!xmlStrcmp(cur->name, "note")) && (cur->ns==ns)) {
	int line=0, track=0, note=0, instrument=0;

	/* Get note properties */
	prop=xmlGetProp(cur, "line");
	if(prop!=NULL)
	  line=atoi(prop);
	prop=xmlGetProp(cur, "track");
	if(prop!=NULL)
	  track=atoi(prop);
	prop=xmlGetProp(cur, "note");
	if(prop!=NULL)
	  note=atoi(prop);
	prop=xmlGetProp(cur, "instrument");
	if(prop!=NULL)
	  instrument=atoi(prop);

	/* Set the note */
	block_set_note(block, line, track, 0, note, instrument);
      } else if((!xmlStrcmp(cur->name, "effect")) && (cur->ns==ns)) {
	int line=0, track=0, effectpage=0, effect=0, value=0;

	/* Get effect properties */
	prop=xmlGetProp(cur, "line");
	if(prop!=NULL)
	  line=atoi(prop);
	prop=xmlGetProp(cur, "track");
	if(prop!=NULL)
	  track=atoi(prop);
	prop=xmlGetProp(cur, "effectpage");
	if(prop!=NULL)
	  effectpage=atoi(prop);
	prop=xmlGetProp(cur, "effect");
	if(prop!=NULL)
	  effect=atoi(prop);
	prop=xmlGetProp(cur, "value");
	if(prop!=NULL)
	  value=atoi(prop);

	/* Set the effect */
	block_set_effect_full(block, line, track, effectpage, effect, value);
      }
      cur=cur->next;
    }
  } else
    fprintf(stderr, "XML error: expected block, got %s\n", cur->name);

  return block;
}

/* Saves a block to an XML document */
void block_save(struct block *block, xmlNodePtr parent) {
  if(block) {
    char c[10];
    int i, j, k;
    xmlNodePtr node, subnode, lb;

    node=xmlNewChild(parent, NULL, "block", NULL);
    xmlSetProp(node, "name", block->name);
    snprintf(c, 10, "%d", block->tracks);
    xmlSetProp(node, "tracks", c);
    snprintf(c, 10, "%d", block->length);
    xmlSetProp(node, "length", c);
    snprintf(c, 10, "%d", block->effectpages);
    xmlSetProp(node, "effectpages", c);
    lb=xmlNewText("\n");
    xmlAddChild(node, lb);

    /* Notation data */
    for(i=0; i<block->tracks; i++)
      for(j=0; j<block->length; j++)
	if(block->notes[2*(block->tracks*j+i)]!=0 ||
	   block->notes[2*(block->tracks*j+i)+1]!=0) {
	  subnode=xmlNewChild(node, NULL, "note", NULL);
	  snprintf(c, 10, "%d", j);
	  xmlSetProp(subnode, "line", c);
	  snprintf(c, 10, "%d", i);
	  xmlSetProp(subnode, "track", c);
	  snprintf(c, 10, "%d", block->notes[2*(block->tracks*j+i)]);
	  xmlSetProp(subnode, "note", c);
	  snprintf(c, 10, "%d", block->notes[2*(block->tracks*j+i)+1]);
	  xmlSetProp(subnode, "instrument", c);
	  lb=xmlNewText("\n");
	  xmlAddChild(node, lb);
	}

    /* Effect data */
    for(i=0; i<block->tracks; i++)
      for(j=0; j<block->length; j++)
	for(k=0; k<block->effectpages; k++)
	  if(block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)]!=0 ||
	     block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)+1]!=0) {
	    subnode=xmlNewChild(node, NULL, "effect", NULL);
	    snprintf(c, 10, "%d", j);
	    xmlSetProp(subnode, "line", c);
	    snprintf(c, 10, "%d", i);
	    xmlSetProp(subnode, "track", c);
	    snprintf(c, 10, "%d", k);
	    xmlSetProp(subnode, "effectpage", c);
	    snprintf(c, 10, "%d", block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)]);
	    xmlSetProp(subnode, "effect", c);
	    snprintf(c, 10, "%d", block->effects[k*2*block->tracks*block->length+2*(block->tracks*j+i)+1]);
	    xmlSetProp(subnode, "value", c);
	    lb=xmlNewText("\n");
	    xmlAddChild(node, lb);
	  }
  
    lb=xmlNewText("\n");
    xmlAddChild(parent, lb);
  } else
    fprintf(stderr, "block_save() called with null block\n");
}
