/* callbacks.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"

GtkWidget *gui_window_main=NULL;
GtkWidget *gui_window_sectionlist=NULL;
GtkWidget *gui_window_playseq=NULL;
GtkWidget *gui_window_blocklist=NULL;
GtkWidget *gui_window_sysexlist=NULL;
GtkWidget *gui_window_tempo=NULL;
GtkWidget *gui_window_trackvolumes=NULL;
GtkWidget *gui_window_instrumentproperties=NULL;
GtkWidget *gui_window_songproperties=NULL;
GtkWidget *gui_window_about=NULL;
GtkWidget *gui_fileselection_open=NULL;
GtkWidget *gui_fileselection_saveas=NULL;

/* FIX: These should not be here. Current SysEx number */
int gui_sysexlist_sysex=-1;
unsigned char gui_sysexlist_autostop;

void gui_open() {
  Tracker *tracker;

  /* Open the main window */
  gui_window_main=create_gui_window_main();
  
  /* Set the keyboard octave */
  gtk_option_menu_set_history(GTK_OPTION_MENU(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_optionmenu_keyboard")), 3);
  editor.octave=3;

  /* Reset the tracker */
  tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
					 "gui_tracker");
  tracker_reset(tracker);
  tracker_set_pattern(tracker, editor.song->blocks[0]);
  tracker_set_num_channels(tracker, editor.song->blocks[0]->tracks);

  /* Refresh GUI */
  gui_info_refresh();
  gui_blocklist_refresh();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
  gui_instrument_refresh();
  gui_sysexlist_refresh();   
  gui_trackvolumes_refresh();
  gui_song_refresh();

  gtk_widget_show(gui_window_main);
}

void gui_song_refresh() {
  /* Refresh the song properties window widgets if the window exists */
  if(gui_window_songproperties!=NULL &&
     GTK_IS_WINDOW(gui_window_songproperties)) {
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_songproperties), "gui_entry_songname")), editor.song->name);
  }
}

void gui_instrument_refresh() {
  char instrument[3];
  snprintf(instrument, 3, "%.2d", editor.instrument+1);

  /* Refresh the main window widgets if the window exists */
  if(gui_window_main!=NULL && GTK_IS_WINDOW(gui_window_main)) {
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_spinbutton_instrument")), editor.instrument+1);
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_entry_instrument")), editor.song->instruments[editor.instrument]->name);
  }

  /* Refresh the instrument window widgets if the window exists */
  if(gui_window_instrumentproperties!=NULL && GTK_IS_WINDOW(gui_window_instrumentproperties)) {
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_entry_instrumentproperties_name")), editor.song->instruments[editor.instrument]->name);
    gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_volume"))), editor.song->instruments[editor.instrument]->defaultvelocity);
    gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_transpose"))), editor.song->instruments[editor.instrument]->transpose);
    gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_hold"))), editor.song->instruments[editor.instrument]->hold);
    gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_midich"))), editor.song->instruments[editor.instrument]->midichannel+1);
  }
}

void gui_tempo_refresh() {
  char title[76];

  /* Refresh the tempo window widgets if the window exists */
  if(gui_window_tempo!=NULL && GTK_IS_WINDOW(gui_window_tempo)) {
    gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_tempo), "gui_scale_tempo"))), editor.song->tempo);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_tempo), "gui_scale_tpl"))), editor.song->ticksperline);
  }

  /* Refresh the main window widgets if the window exists */
  if(gui_window_main!=NULL && GTK_IS_WINDOW(gui_window_main)) {
    snprintf(title, 76,
	     "Tutka: Block %d/%d - Cmd Page %d/%d - Tempo %d - TPL %d",
	     editor.block+1, editor.song->numblocks, 1,
	     editor.song->blocks[editor.block]->effectpages,
	     editor.song->tempo, editor.song->ticksperline);
    gtk_window_set_title(GTK_WINDOW(gui_window_main), title);
  }
}

void gui_info_refresh() {
  char section[9+8], playseq[10+8], block[7+10+32], title[76];

  /* Refresh the main window widgets if the window exists */
  if(gui_window_main!=NULL && GTK_IS_WINDOW(gui_window_main)) {
    GtkLabel *status=GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_main),
						   "gui_label_status"));

    snprintf(section, 8+8, "Section %d/%d", editor.section+1,
	     editor.song->numsections);
    snprintf(playseq, 9+8, "Position %d/%d", editor.playseqpos+1,
	     editor.song->playseqs[editor.playseq]->length);
    if(editor.song->blocks[editor.block]->name!=NULL)
      snprintf(block, 6+10+32, "Block %d/%d: %s", editor.block+1,
	       editor.song->numblocks,
	       editor.song->blocks[editor.block]->name);
    else
      snprintf(block, 6+10+32, "Block %d/%d: ", editor.block+1,
	       editor.song->numblocks);
    gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_label_section")), section);
    gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_label_position")), playseq);
    gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_label_block")), block);
    
    /* Window title bar */
    snprintf(title, 76,
	     "Tutka: Block %d/%d - Cmd Page %d/%d - Tempo %d - TPL %d",
	     editor.block+1, editor.song->numblocks,
	     1, editor.song->blocks[editor.block]->effectpages,
	     editor.song->tempo, editor.song->ticksperline);
    gtk_window_set_title(GTK_WINDOW(gui_window_main), title); 

    /* Status bar */
    switch(editor.mode) {
    case MODE_PLAY_SONG:
      gtk_label_set_text(status, "Playing song");
      break;
    case MODE_PLAY_BLOCK:
      gtk_label_set_text(status, "Playing block");
      break;
    case MODE_IDLE:
      gtk_label_set_text(status, "Stopped");
      break;
    }
  }
}

void gui_timer_refresh(unsigned int seconds) {
  char time[6];

  /* Refresh the main window widgets if the window exists */
  if(gui_window_main!=NULL && GTK_IS_WINDOW(gui_window_main)) {
    snprintf(time, 6, "%.2d:%.2d", seconds/60, seconds%60);
    gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_label_timer")), time);
  }
}

void gui_blocklist_refresh() {
  /* Refresh the block list window widgets if the window exists */
  if(gui_window_blocklist!=NULL && GTK_IS_WINDOW(gui_window_blocklist)) {
    int i;
    unsigned short block=editor.block;
    char number[4], *columns[2];
    GtkCList *clist=gtk_object_get_data(GTK_OBJECT(gui_window_blocklist),
					"gui_clist_blocklist");

    /* Recreate the block list */
    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);
    columns[0]=number;
    for(i=0; i<editor.song->numblocks; i++) {
      snprintf(number, 4, "%3d", i+1);
      columns[1]=editor.song->blocks[i]->name;
      gtk_clist_append(clist, columns);
    }
    gtk_clist_thaw(clist);
    
    /* Move back to the previously selected block */
    gtk_clist_select_row(clist, block, 0);
    gtk_clist_moveto(clist, block, 0, 0, 0);
    
    /* Make the delete button sensitive/insensitive */
    if(editor.song->numblocks>1)
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_button_blocklist_delete")), TRUE);
    else
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_button_blocklist_delete")), FALSE);
  }
}

void gui_playseq_refresh() {
  /* Refresh the playing sequence window widgets if the window exists */
  if(gui_window_playseq!=NULL && GTK_IS_WINDOW(gui_window_playseq)) {
    int i;
    unsigned short playseqpos=editor.playseqpos;
    char position[9+8], block[6+8], *columns[3], title[24];
    GtkCList *clist=gtk_object_get_data(GTK_OBJECT(gui_window_playseq),
					"gui_clist_playseq");

    /* Recreate the playing sequence */
    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);
    columns[0]=position;
    columns[1]=block;
    for(i=0; i<editor.song->playseqs[editor.playseq]->length; i++) {
      snprintf(position, 4, "%3d", i+1);
      snprintf(block, 4, "%3d",
	       editor.song->playseqs[editor.playseq]->blocknumbers[i]+1);
      columns[2]=editor.song->blocks[editor.song->playseqs[editor.playseq]->blocknumbers[i]]->name;
      gtk_clist_append(clist, columns);
    }
    gtk_clist_thaw(clist);

    /* Move back to the previously selected position */
    gtk_clist_select_row(clist, playseqpos, 0);
    gtk_clist_moveto(clist, playseqpos, 0, 0, 0);
    
    snprintf(position, 9+8, "Position %d/%d", editor.playseqpos+1,
	     editor.song->playseqs[editor.playseq]->length);
    snprintf(block, 6+8, "Block %d/%d", editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos]+1, editor.song->numblocks);
    gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_frame_playseq")), position);
    gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_label_playseq_block")), block);
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_entry_playseq_name")), editor.song->playseqs[editor.playseq]->name);
    
    snprintf(title, 24, "Playing Sequence %d/%d", editor.playseq+1,
	     editor.song->numplayseqs);
    gtk_window_set_title(GTK_WINDOW(gui_window_playseq), title);
    
    /* Make the delete button sensitive/insensitive */
    if(editor.song->playseqs[editor.playseq]->length>1)
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_button_playseq_delete")), TRUE);
    else
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_button_playseq_delete")), FALSE);

    if(editor.song->numplayseqs>1)
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_button_playseq_deletethis")), TRUE);
    else
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_button_playseq_deletethis")), FALSE);
  }
}

void gui_sectionlist_refresh() {
  /* Refresh the section list widgets if the window exists */
  if(gui_window_sectionlist!=NULL && GTK_IS_WINDOW(gui_window_sectionlist)) {
    int i;
    unsigned short section=editor.section;
    char position[4], playseq[4], *columns[3];
    GtkCList *clist=gtk_object_get_data(GTK_OBJECT(gui_window_sectionlist),
					"gui_clist_sectionlist");

    /* Recreate the section list */
    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);
    columns[0]=position;
    columns[1]=playseq;
    for(i=0; i<editor.song->numsections; i++) {
      snprintf(position, 4, "%3d", i+1);
      snprintf(playseq, 4, "%3d", editor.song->sections[i]+1);
      columns[2]=editor.song->playseqs[editor.song->sections[i]]->name;
      gtk_clist_append(clist, columns);
    }
    gtk_clist_thaw(clist);
    
    /* Move back to the previously selected section */
    gtk_clist_select_row(clist, section, 0);
    gtk_clist_moveto(clist, section, 0, 0, 0);
    
    /* Make the delete button sensitive/insensitive */
    if(editor.song->numsections>1)
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sectionlist), "gui_button_sectionlist_delete")), TRUE);
    else
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sectionlist), "gui_button_sectionlist_delete")), FALSE);
  }
}

void gui_sysexlist_refresh() {
  /* Refresh the sysex list widgets if the window exists */
  if(gui_window_sysexlist!=NULL && GTK_IS_WINDOW(gui_window_sysexlist)) {
    int i, oldsysex;
    char number[4], *columns[2], sysex[6+8];
    GtkCList *clist=gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist),
					"gui_clist_sysex");

    if(editor.song->numsysexes==0)
      gui_sysexlist_sysex=-1;
    
    oldsysex=gui_sysexlist_sysex;

    /* Recreate the SysEx list */
    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);
    columns[0]=number;
    for(i=0; i<editor.song->numsysexes; i++) {
      snprintf(number, 4, "%3d", i);
      columns[1]=editor.song->sysexes[i]->name;
      gtk_clist_append(clist, columns);
    }
    gtk_clist_thaw(clist);
    
    /* Move back to the previously selected SysEx and make widgets sensitive */
    if(gui_sysexlist_sysex!=-1) {
      gtk_clist_select_row(clist, oldsysex, 0);
      gtk_clist_moveto(clist, oldsysex, 0, 0, 0);
      /* Show the number of the selected SysEx message */
      snprintf(sysex, 6+8, "SysEx %d/%d", gui_sysexlist_sysex+1,
	       editor.song->numsysexes);
      gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_frame_sysex_controls")), sysex);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_table_sysex1")), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_send")), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_receive")), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_checkbutton_sysex_autostop")), TRUE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_delete")), TRUE);
    } else {
      gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_frame_sysex_controls")), "SysEx 0/0");
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_table_sysex1")), FALSE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_send")), FALSE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_receive")), FALSE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_checkbutton_sysex_autostop")), FALSE);
      gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_delete")), FALSE);
    }
  }
}

void gui_trackvolumes_refresh() {
  /* Refresh the sysex list widgets if the window exists */
  if(gui_window_trackvolumes!=NULL && GTK_IS_WINDOW(gui_window_trackvolumes)) {
    int i;
    char track[4];
    GList *children;
    GtkWidget *w1, *w2, *w3, *w4;
    GtkWidget *hbox=gtk_object_get_data(GTK_OBJECT(gui_window_trackvolumes),
					"gui_hbox_trackvolumes");

    /* Destroy all children */
    children=gtk_container_children(GTK_CONTAINER(hbox));
    while(children!=NULL) {
      gtk_widget_destroy(GTK_WIDGET(children->data));
      children=children->next;
    }

    for(i=0; i<editor.song->maxtracks; i++) {
      w1=gtk_vbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(hbox), w1, TRUE, TRUE, 4);

      /* Volume sliders */
      w2=(GtkWidget *)gtk_adjustment_new(editor.song->trackvolumes[i], 0, 127,
					 1, 1, 0);
      gtk_signal_connect(GTK_OBJECT(w2), "value_changed",
			 GTK_SIGNAL_FUNC(on_gui_adjustment_trackvolumes_changed),
			 (gpointer)&editor.song->trackvolumes[i]);
      w3=gtk_vscale_new(GTK_ADJUSTMENT(w2));
      gtk_scale_set_digits(GTK_SCALE(w3), 0);
      gtk_box_pack_start(GTK_BOX(w1), w3, TRUE, TRUE, 4);

      /* Mute buttons */
      snprintf(track, 4, "%d", i+1);
      w4=gtk_check_button_new_with_label(track);
      gtk_signal_connect(GTK_OBJECT(w4), "toggled",
			 GTK_SIGNAL_FUNC(on_gui_togglebutton_trackvolumes_toggled),
			 (gpointer)&editor.song->trackvolumes[i]);
      gtk_box_pack_start(GTK_BOX(w1), w4, FALSE, FALSE, 4);
    }

    gtk_widget_show_all(hbox);
  }
}

void on_gui_menuitem_file_new_activate(GtkMenuItem *menuitem,
				       gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  editor_open(NULL);

  tracker_reset(tracker);
  tracker_set_pattern(tracker, editor.song->blocks[0]);
  tracker_set_num_channels(tracker, editor.song->blocks[0]->tracks);
}

void on_gui_menuitem_file_open_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  if(gui_fileselection_open==NULL ||
     !(GTK_IS_FILE_SELECTION(gui_fileselection_open))) {
    gui_fileselection_open=create_gui_fileselection_open();
    gtk_widget_show(gui_fileselection_open);
  } else {
    gtk_widget_destroy(gui_fileselection_open);
    gui_fileselection_open=NULL;
  }
}

void on_gui_menuitem_file_save_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  song_save(editor.song, editor.filename);
}

void on_gui_menuitem_file_saveas_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  if(gui_fileselection_saveas==NULL ||
     !(GTK_IS_FILE_SELECTION(gui_fileselection_saveas))) {
    gui_fileselection_saveas=create_gui_fileselection_saveas();
    gtk_widget_show(gui_fileselection_saveas);
  } else {
    gtk_widget_destroy(gui_fileselection_saveas);
    gui_fileselection_saveas=NULL;
  }
}

void on_gui_menuitem_file_print_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  /* FIX: Implement */
  fprintf(stderr, "Printing not implemented yet\n"); 
}

void on_gui_menuitem_file_exit_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  gtk_main_quit();
}

void on_gui_menuitem_edit_cut_activate(GtkMenuItem *menuitem,
				       gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  if(editor.copyarea)
    block_free(editor.copyarea);
  editor.copyarea=block_copy(editor.song->blocks[editor.block],
			     tracker->sel_start_ch,
			     tracker->sel_start_row,
			     tracker->sel_end_ch,
			     tracker->sel_end_row);
  block_clear(editor.song->blocks[editor.block],
	      tracker->sel_start_ch, tracker->sel_start_row,
	      tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_gui_menuitem_edit_copy_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  if(editor.copyarea)
    block_free(editor.copyarea);
  editor.copyarea=block_copy(editor.song->blocks[editor.block],
			     tracker->sel_start_ch, tracker->sel_start_row,
			     tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_gui_menuitem_edit_paste_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  block_paste(editor.song->blocks[editor.block], editor.copyarea,
	      tracker->cursor_ch, tracker->patpos);
  tracker_redraw(tracker);
}

void on_gui_menuitem_edit_clear_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  block_clear(editor.song->blocks[editor.block],
	      tracker->sel_start_ch, tracker->sel_start_row,
	      tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_gui_scale_tempo_changed(GtkAdjustment *adjustment,
				gpointer user_data) {
  song_set_tempo(editor.song, (unsigned char)adjustment->value);
  gui_info_refresh();
}

void on_gui_scale_tpl_changed(GtkAdjustment *adjustment, gpointer user_data) {
  song_set_tpl(editor.song, (unsigned char)adjustment->value);
  gui_info_refresh();
}

void on_gui_menuitem_song_tempo_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  if(gui_window_tempo==NULL || !(GTK_IS_WINDOW(gui_window_tempo))) {
    GtkAdjustment *adjustment;

    gui_window_tempo=create_gui_window_tempo();
    adjustment=gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_tempo), "gui_scale_tempo")));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(on_gui_scale_tempo_changed), NULL);
    adjustment=gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_tempo), "gui_scale_tpl")));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(on_gui_scale_tpl_changed), NULL);
    gui_tempo_refresh();
    gtk_widget_show(gui_window_tempo);
  } else {
    gtk_widget_destroy(gui_window_tempo);
    gui_window_tempo=NULL;
  }
}

void on_gui_menuitem_song_trackvolumes_activate(GtkMenuItem *menuitem,
						gpointer user_data) {
  if(gui_window_trackvolumes==NULL ||
     !(GTK_IS_WINDOW(gui_window_trackvolumes))) {
    gui_window_trackvolumes=create_gui_window_trackvolumes();
    song_check_maxtracks(editor.song);
    gui_trackvolumes_refresh();
    gtk_widget_show(gui_window_trackvolumes);
  } else {
    gtk_widget_destroy(gui_window_trackvolumes);
    gui_window_trackvolumes=NULL;
  }
}

void on_gui_menuitem_song_properties_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  if(gui_window_songproperties==NULL ||
     !(GTK_IS_WINDOW(gui_window_songproperties))) {
    gui_window_songproperties=create_gui_window_songproperties();
    gui_song_refresh();
    gtk_widget_show(gui_window_songproperties);
  } else {
    gtk_widget_destroy(gui_window_songproperties);
    gui_window_songproperties=NULL;
  }
}

void on_gui_menuitem_block_cut_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  if(editor.copyblock)
    block_free(editor.copyblock);
  editor.copyblock=block_copy(editor.song->blocks[editor.block],
			      0, 0, editor.song->blocks[editor.block]->tracks-1, editor.song->blocks[editor.block]->length-1);
  block_clear(editor.song->blocks[editor.block],
	      0, 0, editor.song->blocks[editor.block]->tracks-1, editor.song->blocks[editor.block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_copy_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  if(editor.copyblock)
    block_free(editor.copyblock);
  editor.copyblock=block_copy(editor.song->blocks[editor.block],
			      0, 0, editor.song->blocks[editor.block]->tracks-1, editor.song->blocks[editor.block]->length-1);
}

void on_gui_menuitem_block_paste_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  block_paste(editor.song->blocks[editor.block], editor.copyblock,
	      0, 0);
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_selectall_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  tracker->sel_start_ch=0;
  tracker->sel_start_row=0;
  tracker->sel_end_ch=editor.song->blocks[editor.block]->tracks-1;
  tracker->sel_end_row=editor.song->blocks[editor.block]->length-1;
  tracker->inSelMode=FALSE;
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_clear_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  block_clear(editor.song->blocks[editor.block],
	      0, 0, editor.song->blocks[editor.block]->tracks-1, editor.song->blocks[editor.block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_list_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  on_gui_button_block_clicked(NULL, NULL);
}

void on_gui_menuitem_track_cut_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  if(editor.copytrack)
    block_free(editor.copytrack);
  editor.copytrack=block_copy(editor.song->blocks[editor.block],
			      tracker->cursor_ch, 0,
			      tracker->cursor_ch, editor.song->blocks[editor.block]->length-1);
  block_clear(editor.song->blocks[editor.block], tracker->cursor_ch, 0,
	      tracker->cursor_ch, editor.song->blocks[editor.block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_copy_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  if(editor.copytrack)
    block_free(editor.copytrack);
  editor.copytrack=block_copy(editor.song->blocks[editor.block],
			      tracker->cursor_ch, 0,
			      tracker->cursor_ch, editor.song->blocks[editor.block]->length-1);
}

void on_gui_menuitem_track_paste_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  block_paste(editor.song->blocks[editor.block], editor.copytrack,
	      tracker->cursor_ch, 0);
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_selectall_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  tracker->sel_start_ch=tracker->cursor_ch;
  tracker->sel_start_row=0;
  tracker->sel_end_ch=tracker->cursor_ch;
  tracker->sel_end_row=editor.song->blocks[editor.block]->length-1;
  tracker->inSelMode=FALSE;
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_clear_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  block_clear(editor.song->blocks[editor.block], tracker->cursor_ch, 0,
	      tracker->cursor_ch, editor.song->blocks[editor.block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_song_sysexlist_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  if(gui_window_sysexlist==NULL || !(GTK_IS_WINDOW(gui_window_sysexlist))) {
    gui_window_sysexlist=create_gui_window_sysex();
    gui_sysexlist_refresh();
    gtk_widget_show(gui_window_sysexlist);
  } else {
    gtk_widget_destroy(gui_window_sysexlist);
    gui_window_sysexlist=NULL;
  }
}

void on_gui_menuitem_midi_killallnotes_activate(GtkMenuItem *menuitem,
						gpointer user_data) {
  player_stop_all_notes();
}

void on_gui_menuitem_midi_resetpitch_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  player_reset_pitch();
}

void on_gui_menuitem_settings_preferences_activate(GtkMenuItem *menuitem,
						   gpointer user_data) {
  /* FIX: implement */
  fprintf(stderr, "Preferences not implemented yet\n");
}

void on_gui_menuitem_about_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  gchar *authors[] = { "Vesa Halttunen", NULL };

  if(gui_window_about==NULL || !(GTK_IS_WINDOW(gui_window_about))) {
    gui_window_about=gnome_about_new("Tutka", VERSION,
				     "(C) 2002 Vesa Halttunen",
				 (const gchar **)authors,
				     "http://www.freesoftware.fsf.org/tutka/",
				     NULL);
    gtk_widget_show(gui_window_about);
  } else {
    gtk_widget_destroy(gui_window_about);
    gui_window_about=NULL;
  }
}


GtkWidget* create_gui_tracker(gchar *widget_name, gchar *string1,
			      gchar *string2, gint int1, gint int2) {
  GtkWidget *tracker=tracker_new();

  tracker_set_pattern((Tracker *)tracker, editor.song->blocks[0]);
  tracker_set_num_channels((Tracker *)tracker,
			   editor.song->blocks[0]->tracks);

  return tracker;
}

void on_gui_button_playsong_clicked(GtkButton *button, gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  play_song();

  tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
  tracker_set_patpos(tracker, 0);

  gui_info_refresh();
  gui_timer_refresh(0);
  gui_blocklist_refresh();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void on_gui_button_playblock_clicked(GtkButton *button, gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  play_block();
  tracker_set_patpos(tracker, 0);

  gui_info_refresh();
  gui_timer_refresh(0);
}

void on_gui_button_contsong_clicked(GtkButton *button, gpointer user_data) {
  if(editor.mode!=MODE_PLAY_SONG) {
    continue_song();
    gui_info_refresh();
  }
}

void on_gui_button_contblock_clicked(GtkButton *button, gpointer user_data) {
  if(editor.mode!=MODE_PLAY_BLOCK) {
    continue_block();
    gui_info_refresh();
  }
}

void on_gui_button_stop_clicked(GtkButton *button, gpointer user_data) {
  stop();

  gui_info_refresh();
}

void on_gui_scale_volume_changed(GtkAdjustment *adjustment,
				 gpointer user_data) {
  editor.song->instruments[editor.instrument]->defaultvelocity=(unsigned char)adjustment->value;
}

void on_gui_scale_transpose_changed(GtkAdjustment *adjustment,
				 gpointer user_data) {
  editor.song->instruments[editor.instrument]->transpose=(unsigned char)adjustment->value;
}

void on_gui_scale_hold_changed(GtkAdjustment *adjustment,
				 gpointer user_data) {
  editor.song->instruments[editor.instrument]->hold=(unsigned char)adjustment->value;
}

void on_gui_scale_midich_changed(GtkAdjustment *adjustment,
				 gpointer user_data) {
  editor.song->instruments[editor.instrument]->midichannel=(unsigned char)adjustment->value-1;
}

void on_gui_button_properties_clicked(GtkButton *button, gpointer user_data) {
  if(gui_window_instrumentproperties==NULL ||
     !(GTK_IS_WINDOW(gui_window_instrumentproperties))) {
    GtkAdjustment *adjustment;

    gui_window_instrumentproperties=create_gui_window_instrumentproperties();
    adjustment=gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_volume")));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(on_gui_scale_volume_changed), NULL);
    adjustment=gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_transpose")));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(on_gui_scale_transpose_changed), NULL);
    adjustment=gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_hold")));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(on_gui_scale_hold_changed), NULL);
    adjustment=gtk_range_get_adjustment(GTK_RANGE(gtk_object_get_data(GTK_OBJECT(gui_window_instrumentproperties), "gui_scale_instrumentproperties_midich")));
    gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
		       GTK_SIGNAL_FUNC(on_gui_scale_midich_changed), NULL);
    gui_instrument_refresh();
    gtk_widget_show(gui_window_instrumentproperties);
  } else {
    gtk_widget_destroy(gui_window_instrumentproperties);
    gui_window_instrumentproperties=NULL;
  }
}

void on_gui_checkbutton_edit_toggled(GtkToggleButton *togglebutton,
				     gpointer user_data) {
  if(gtk_toggle_button_get_active(togglebutton))
    editor.edit=1;
  else
    editor.edit=0;
}

void on_gui_checkbutton_chord_toggled(GtkToggleButton *togglebutton,
				      gpointer user_data) {
  if(gtk_toggle_button_get_active(togglebutton))
    editor.chord=1;
  else
    editor.chord=0;
}

void on_gui_spinbutton_space_changed(GtkEditable *editable,
				     gpointer user_data) {
  editor.space=gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value;
}

void on_gui_button_section_clicked(GtkButton *button, gpointer user_data) {
  if(gui_window_sectionlist==NULL || !(GTK_IS_WINDOW(gui_window_sectionlist))) {
    gui_window_sectionlist=create_gui_window_sectionlist();
    gui_sectionlist_refresh();
    gtk_widget_show(gui_window_sectionlist);
  } else {
    gtk_widget_destroy(gui_window_sectionlist);
    gui_window_sectionlist=NULL;
  }
}

void on_gui_button_position_clicked(GtkButton *button,
				    gpointer user_data) {
  if(gui_window_playseq==NULL || !(GTK_IS_WINDOW(gui_window_playseq))) {
    gui_window_playseq=create_gui_window_playseq();
    gui_playseq_refresh();
    gtk_widget_show(gui_window_playseq);
  } else {
    gtk_widget_destroy(gui_window_playseq);
    gui_window_playseq=NULL;
  }
}

void on_gui_button_block_clicked(GtkButton *button, gpointer user_data) {
  if(gui_window_blocklist==NULL || !(GTK_IS_WINDOW(gui_window_blocklist))) {
    gui_window_blocklist=create_gui_window_blocklist();
    gui_blocklist_refresh();
    gtk_widget_show(gui_window_blocklist);
  } else {
    gtk_widget_destroy(gui_window_blocklist);
    gui_window_blocklist=NULL;
  }
}

gboolean keyboard_event(GtkWidget *widget, GdkEventKey *event,
			gpointer user_data) {
  gboolean pressed=(gboolean)user_data, handled=FALSE;

  if(!(GTK_IS_ENTRY(GTK_WINDOW(widget)->focus_widget))) {
    Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_tracker");
    int shift=event->state & GDK_SHIFT_MASK;
    int ctrl=event->state & GDK_CONTROL_MASK;
    /*  int alt=event->state & GDK_MOD1_MASK; */

    if(pressed) {
      if(ctrl) {
	switch (event->keyval) {
	case 'b':
	  if(tracker_is_in_selection_mode(tracker))
	    tracker_mark_selection(tracker, FALSE);
	  else
	    tracker_mark_selection(tracker, TRUE);
	  handled=TRUE;
	  break;
	case 'c':
	  if(tracker_is_in_selection_mode(tracker))
	    tracker_mark_selection(tracker, FALSE);
	  on_gui_menuitem_edit_copy_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case 'x':
	  if(tracker_is_in_selection_mode(tracker))
	    tracker_mark_selection(tracker, FALSE);
	  on_gui_menuitem_edit_cut_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case 'v':
	  on_gui_menuitem_edit_paste_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case 'q':
	  handled=TRUE;
	  gtk_main_quit();
	  break;
	case 'k':
	  block_clear(editor.song->blocks[editor.block],
		      tracker->cursor_ch, tracker->patpos,
		      tracker->cursor_ch,
		      editor.song->blocks[editor.block]->length-1);
	  tracker_redraw(tracker);
	  handled=TRUE;
	  break;
	case 's':
	  song_save(editor.song, editor.filename);
	  handled=TRUE;
	  break;
	case GDK_F3:
	  on_gui_menuitem_block_cut_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case GDK_F4:
	  on_gui_menuitem_block_copy_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case GDK_F5:
	  on_gui_menuitem_block_paste_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case GDK_Left:
	  if(editor.block>0) {
	    editor.block--;
	    tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
	    gui_info_refresh();
	    gui_blocklist_refresh();
	  }
	  handled=TRUE;
	  break;
	case GDK_Right:
	  if(editor.block<editor.song->numblocks-1) {
	    editor.block++;
	    tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
	    gui_info_refresh();
	    gui_blocklist_refresh();
	  }
	  handled=TRUE;
	  break;
	}
      } else if(shift) {
	switch (event->keyval) {
	case GDK_F3:
	  on_gui_menuitem_track_cut_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case GDK_F4:
	  on_gui_menuitem_track_copy_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case GDK_F5:
	  on_gui_menuitem_track_paste_activate(NULL, NULL);
	  handled=TRUE;
	  break;
	case GDK_ISO_Left_Tab:
	  if((tracker)->cursor_item>0) {
	(tracker)->cursor_item=0;
	    tracker_step_cursor_channel(tracker, 0);
	  } else
	    tracker_step_cursor_channel(tracker, -1);
	  handled=TRUE;
	  break;
	}
      } else {
	char instrument=-1;
	char note=-1;
	if(editor.edit) {
	  char data=-1;
	  switch(tracker->cursor_item) {
	  case 0:
	    switch(event->keyval) {
	    case 'z':
	      data=1;
	      break;
	    case 's':
	      data=2;
	      break;
	    case 'x':
	      data=3;
	      break;
	    case 'd':
	      data=4;
	      break;
	    case 'c':
	      data=5;
	      break;
	    case 'v':
	      data=6;
	      break;
	    case 'g':
	      data=7;
	      break;
	    case 'b':
	      data=8;
	      break;
	    case 'h':
	      data=9;
	      break;
	    case 'n':
	      data=10;
	      break;
	    case 'j':
	      data=11;
	      break;
	    case 'm':
	      data=12;
	      break;
	    case 'q':
	    case ',':
	      data=13;
	      break;
	    case '2':
	    case 'l':
	      data=14;
	      break;
	    case 'w':
	    case '.':
	      data=15;
	      break;
	    case '3':
	    case '':
	      data=16;
	      break;
	    case 'e':
	    case '-':
	      data=17;
	      break;
	    case 'r':
	      data=18;
	      break;
	    case '5':
	      data=19;
	      break;
	    case 't':
	      data=20;
	      break;
	    case '6':
	      data=21;
	      break;
	    case 'y':
	      data=22;
	      break;
	    case '7':
	      data=23;
	      break;
	    case 'u':
	      data=24;
	      break;
	    case 'i':
	      data=25;
	      break;
	    case '9':
	      data=26;
	      break;
	    case 'o':
	      data=27;
	      break;
	    case '0':
	      data=28;
	      break;
	    case 'p':
	      data=29;
	      break;
	    case GDK_Delete:
	      data=0;
	      break;
	    }
	    if(data!=-1) {
	      block_set_note(editor.song->blocks[editor.block], tracker->patpos,
			     tracker->cursor_ch, editor.octave, data,
			     editor.instrument+1);
	      tracker_redraw_row(tracker, tracker->patpos);
	      if(editor.chord)
		tracker_step_cursor_channel(tracker, 1);
	      tracker_step_cursor_row(tracker, editor.space);
	      handled=TRUE;
	    }
	    break;
	  case 1:
	    if(event->keyval>='0' && event->keyval<='9')
	      data=event->keyval-'0';
	    else if(event->keyval>='a' && event->keyval<='z')
	      data=10+event->keyval-'a';
	    else if(event->keyval==GDK_Delete)
	      data=0;

	    if(data!=-1) {
	      block_set_instrument(editor.song->blocks[editor.block],
				   tracker->patpos, tracker->cursor_ch,
				   data);
	      tracker_redraw_row(tracker, tracker->patpos);
	      tracker_step_cursor_row(tracker, editor.space);
	      handled=TRUE;
	    }
	    break;
	  case 2:
	  case 3:
	  case 4:
	  case 5:
	    if(event->keyval>='0' && event->keyval<='9')
	      data=event->keyval-'0';
	    else if(event->keyval>='a' && event->keyval<='f')
	      data=10+event->keyval-'a';
	    else if(event->keyval==GDK_Delete)
	      data=0;

	    if(data!=-1) {
	      block_set_effect(editor.song->blocks[editor.block],
			       tracker->patpos, tracker->cursor_ch, 0,
			       tracker->cursor_item-2, data);
	      tracker_redraw_row(tracker, tracker->patpos);
	      tracker_step_cursor_row(tracker, editor.space);
	      handled=TRUE;
	    }
	    break;
	  }
	}
	switch (event->keyval) {
	case GDK_space:
	  /* If the song is playing, stop */
	  if(editor.mode!=MODE_IDLE)
	    stop();
	  else {
	    /* Otherwise toggle edit mode */
	    editor.edit^=1;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_checkbutton_edit")), editor.edit);
	  }
	  handled=TRUE;
	  break;
	case GDK_F1:
	case GDK_F2:
	case GDK_F3:
	case GDK_F4:
	case GDK_F5:
	case GDK_F6:
	case GDK_F7:
	case GDK_F8:
	case GDK_F9:
	case GDK_F10:
	case GDK_F11:
	  gtk_option_menu_set_history(GTK_OPTION_MENU(gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_optionmenu_keyboard")), event->keyval-GDK_F1);
	  editor.octave=event->keyval-GDK_F1;
	  handled=TRUE;
	  break;
	case GDK_Num_Lock:
	  instrument=0;
	  break;
	case GDK_KP_Divide:
	  instrument=1;
	  break;
	case GDK_KP_Multiply:
	  instrument=2;
	  break;
	case GDK_KP_Subtract:
	  instrument=3;
	  break;
	case GDK_KP_7:
	case GDK_KP_Home:
	  instrument=4;
	  break;
	case GDK_KP_8:
	case GDK_KP_Up:
	  instrument=5;
	  break;
	case GDK_KP_9:
	case GDK_KP_Page_Up:
	  instrument=6;
	  break;
	case GDK_KP_Add:
	  instrument=7;
	  break;
	case GDK_KP_4:
	case GDK_KP_Left:
	  instrument=8;
	  break;
	case GDK_KP_5:
	case GDK_KP_Begin:
	  instrument=9;
	  break;
	case GDK_KP_6:
	case GDK_KP_Right:
	  instrument=10;
	  break;
	case GDK_KP_1:
	case GDK_KP_End:
	  instrument=11;
	  break;
	case GDK_KP_2:
	case GDK_KP_Down:
	  instrument=12;
	  break;
	case GDK_KP_3:
	case GDK_KP_Page_Down:
	  instrument=13;
	  break;
	case GDK_KP_Enter:
	  instrument=14;
	  break;
	case GDK_KP_0:
	case GDK_KP_Insert:
	  instrument=15;
	  break;
	case GDK_KP_Decimal:
	case GDK_KP_Delete:
	  instrument=16;
	  break;
	case 'z':
	  note=0;
	  break;
	case 's':
	  note=1;
	  break;
	case 'x':
	  note=2;
	  break;
	case 'd':
	  note=3;
	  break;
	case 'c':
	  note=4;
	  break;
	case 'v':
	  note=5;
	  break;
	case 'g':
	  note=6;
	  break;
	case 'b':
	  note=7;
	  break;
	case 'h':
	  note=8;
	  break;
	case 'n':
	  note=9;
	  break;
	case 'j':
	  note=10;
	  break;
	case 'm':
	  note=11;
	  break;
	case 'q':
	case ',':
	  note=12;
	  break;
	case '2':
	case 'l':
	  note=13;
	  break;
	case 'w':
	case '.':
	  note=14;
	  break;
	case '3':
	case '':
	  note=15;
	  break;
	case 'e':
	case '-':
	  note=16;
	  break;
	case 'r':
	  note=17;
	  break;
	case '5':
	  note=18;
	  break;
	case 't':
	  note=19;
	  break;
	case '6':
	  note=20;
	  break;
	case 'y':
	  note=21;
	  break;
	case '7':
	  note=22;
	  break;
	case 'u':
	  note=23;
	  break;
	case 'i':
	  note=24;
	  break;
	case '9':
	  note=25;
	  break;
	case 'o':
	  note=26;
	  break;
	case '0':
	  note=27;
	  break;
	case 'p':
	  note=28;
	  break;
	case '':
	  note=29;
	  break;
	case GDK_Down:
	  tracker_step_cursor_row(tracker, 1);
	  handled=TRUE;
	  break;
	case GDK_Up:
	  tracker_step_cursor_row(tracker, -1);
	  handled=TRUE;
	  break;
	case GDK_Left:
	  tracker_step_cursor_item(tracker, -1);
	  handled=TRUE;
	  break;
	case GDK_Right:
	  tracker_step_cursor_item(tracker, 1);
	  handled=TRUE;
	  break;
	case GDK_Tab:
	  (tracker)->cursor_item=0;
	  tracker_step_cursor_channel(tracker, 1);
	  handled=TRUE;
	  break;
	case GDK_Home:
	  tracker_set_patpos(tracker, 0);
	  handled=TRUE;
	  break;
	case GDK_End:
	  tracker_set_patpos(tracker,
			     editor.song->blocks[editor.block]->length-1);
	  handled=TRUE;
	  break;
	case GDK_Page_Down:
	  tracker_step_cursor_row(tracker, 8);
	  handled=TRUE;
	  break;
	case GDK_Page_Up:
	  tracker_step_cursor_row(tracker, -8);
	  handled=TRUE;
	  break;
	}

	/* Select an instrument if an instrument selection key was pressed */
	if(instrument!=-1) {
	  editor.instrument=instrument;

	  /* Make sure the instrument exists */
	  song_check_instrument(editor.song, instrument);
	  gui_instrument_refresh();
	  handled=TRUE;
	}

	/* Play note if a key was pressed but not if cursor is in cmd pos */
	if(note!=-1 && tracker->cursor_item==0) {
	  player_play_note(editor.instrument, editor.octave*12+note, 127,
			   tracker->cursor_ch);
	  handled=TRUE;
	}
      }
      
      if(handled)
	gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
    } else {
      /* Key has been released */
      if(!ctrl && !shift) {
	if(editor.edit && editor.chord) {
	  switch(event->keyval) {
	  case 'z':
	  case 's':
	  case 'x':
	  case 'd':
	  case 'c':
	  case 'v':
	  case 'g':
	  case 'b':
	  case 'h':
	  case 'n':
	  case 'j':
	  case 'm':
	  case 'q':
	  case ',':
	  case '2':
	  case 'l':
	  case 'w':
	  case '.':
	  case '3':
	  case '':
	  case 'e':
	  case '-':
	  case 'r':
	  case '5':
	  case 't':
	  case '6':
	  case 'y':
	  case '7':
	  case 'u':
	  case 'i':
	  case '9':
	  case 'o':
	  case '0':
	  case 'p':
	    tracker_step_cursor_channel(tracker, -1);
	    handled=TRUE;
	    break;
	  }
	}
      }

      if(handled)
	gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_release_event");
    }
  } else {
    /* An entry widget was active */
    if(pressed) {
      switch(event->keyval) {
      case GDK_Tab:
      case GDK_Return:
	/* Activate the window widget so the user can play again */
	gtk_window_set_focus(GTK_WINDOW(widget), NULL);
	handled=TRUE;
	break;
      }
    }
  }
  return handled;
}

void on_gui_button_tempo_close_clicked(GtkButton *button,
				       gpointer user_data) {
  on_gui_menuitem_song_tempo_activate(NULL, NULL);
}

void on_gui_button_trackvolumes_close_clicked(GtkButton *button,
					      gpointer user_data) {
  on_gui_menuitem_song_trackvolumes_activate(NULL, NULL);
}

gboolean on_gui_label_timer_button_press_event(GtkWidget *widget,
					       GdkEventButton *event,
					       gpointer user_data) {
  /* FIX: implement this */

  return FALSE;
}

void on_gui_clist_sysex_select_row(GtkCList *clist, gint row, gint column,
				   GdkEvent *event, gpointer user_data) {
  char sysex[6+8];
  gui_sysexlist_sysex=row;

  /* Show the number of the selected SysEx message */
  snprintf(sysex, 6+8, "SysEx %d/%d", gui_sysexlist_sysex+1,
	   editor.song->numsysexes);
  gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_frame_sysex_controls")), sysex);

  /* Show name of the selected SysEx message */
  if(editor.song->sysexes[gui_sysexlist_sysex]->name)
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_entry_sysex_name")),
		       editor.song->sysexes[gui_sysexlist_sysex]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_entry_sysex_name")), "");

  /* Show length of the selected SysEx message */
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_spinbutton_sysex_length")),
			   editor.song->sysexes[gui_sysexlist_sysex]->length);

  /* Make widgets sensitive */
  gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_table_sysex1")), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_send")), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_receive")), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_checkbutton_sysex_autostop")), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gui_window_sysexlist), "gui_button_sysex_delete")), TRUE);
}

void on_gui_button_sysex_appendnew_clicked(GtkButton *button,
					   gpointer user_data) {
  if(editor.song!=NULL) {
    song_insert_sysex(editor.song, editor.song->numsysexes);
    gui_sysexlist_refresh();
  }
}

void on_gui_button_sysex_delete_clicked(GtkButton *button,
                                        gpointer user_data) {
  if(editor.song!=NULL && gui_sysexlist_sysex!=-1) {
    song_delete_sysex(editor.song, gui_sysexlist_sysex);
    gui_sysexlist_refresh();
  }
}

void on_gui_button_sysex_insertnew_clicked(GtkButton *button,
					   gpointer user_data) {
  if(editor.song!=NULL) {
    song_insert_sysex(editor.song, gui_sysexlist_sysex);
    if(gui_sysexlist_sysex==-1)
      gui_sysexlist_sysex=0;
    gui_sysexlist_refresh();
  }
}

void on_gui_button_sysex_send_clicked(GtkButton *button,
				      gpointer user_data) {
  if(editor.song!=NULL && gui_sysexlist_sysex!=-1) {
    midi_system_exclusive(editor.song->sysexes[gui_sysexlist_sysex]->data,
			  editor.song->sysexes[gui_sysexlist_sysex]->length);
  }
}

void on_gui_button_sysex_receive_clicked(GtkButton *button,
					 gpointer user_data) {
  if(editor.song!=NULL && gui_sysexlist_sysex!=-1) {
    midi_read_system_exclusive(editor.song->sysexes[gui_sysexlist_sysex],
			       gui_sysexlist_autostop);
  }
}

void on_gui_checkbutton_sysex_autostop_toggled(GtkToggleButton *togglebutton,
					       gpointer user_data) {

}

void on_gui_clist_sectionlist_select_row(GtkCList *clist, gint row,
					 gint column, GdkEvent *event,
					 gpointer user_data) {
  char section[8+8], playseq[17+8];

  editor.section=row;
  player_refresh_playseq_and_block();

  snprintf(section, 8+8, "Section %d/%d", editor.section+1,
	   editor.song->numsections);
  snprintf(playseq, 17+8, "Playing Sequence %d/%d", editor.playseq+1,
	   editor.song->numplayseqs);

  gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_sectionlist), "gui_frame_sectionlist")), section);
  gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_sectionlist), "gui_label_sectionlist_playseq")), playseq);

  gui_playseq_refresh();
  gui_info_refresh();
}

void on_gui_button_sectionlist_next_clicked(GtkButton *button,
					    gpointer user_data) {
  if(editor.song->sections[editor.section]<editor.song->numplayseqs-1)
    editor.song->sections[editor.section]++;

  player_refresh_playseq_and_block();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void on_gui_button_sectionlist_delete_clicked(GtkButton *button,
					      gpointer user_data) {
  song_delete_section(editor.song, editor.section);
  if(editor.section>=editor.song->numsections)
    editor.section=editor.song->numsections-1;

  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void on_gui_button_sectionlist_append_clicked(GtkButton *button,
					      gpointer user_data) {
  song_insert_section(editor.song, editor.song->numsections);
  gui_sectionlist_refresh();
}

void on_gui_button_sectionlist_prev_clicked(GtkButton *button,
					    gpointer user_data) {
  if(editor.song->sections[editor.section]>0)
    editor.song->sections[editor.section]--;

  player_refresh_playseq_and_block();
  gui_playseq_refresh();
  gui_sectionlist_refresh();
}

void on_gui_button_sectionlist_insert_clicked(GtkButton *button,
					      gpointer user_data) {
  song_insert_section(editor.song, editor.section);
  gui_sectionlist_refresh();
}

void on_gui_clist_playseq_select_row(GtkCList *clist, gint row,
				     gint column, GdkEvent *event,
				     gpointer user_data) {
  /* Refresh the playing sequence window widgets if the window exists */
  if(gui_window_playseq!=NULL && GTK_IS_WINDOW(gui_window_playseq)) {
    char position[9+8], block[6+8];

    editor.playseqpos=row;
    
    snprintf(position, 9+8, "Position %d/%d", editor.playseqpos+1,
	     editor.song->playseqs[editor.playseq]->length);
    snprintf(block, 6+8, "Block %d/%d", editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos]+1, editor.song->numblocks);
    gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_frame_playseq")), position);
    gtk_label_set_text(GTK_LABEL(gtk_object_get_data(GTK_OBJECT(gui_window_playseq), "gui_label_playseq_block")), block);
  }
  gui_info_refresh();
}

void on_gui_button_playseq_next_clicked(GtkButton *button,
                                        gpointer user_data) {
  unsigned short playseq=editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos];

  if(playseq<editor.song->numblocks-1)
    playseq_set(editor.song->playseqs[editor.playseq], editor.playseqpos,
		playseq+1);

  gui_playseq_refresh();
}

void on_gui_button_playseq_delete_clicked(GtkButton *button,
					  gpointer user_data) {
  if(editor.song!=NULL) {
    playseq_delete(editor.song->playseqs[editor.playseq], editor.playseqpos);
 
    if(editor.playseqpos>=editor.song->playseqs[editor.playseq]->length)
      editor.playseqpos=editor.song->playseqs[editor.playseq]->length-1;

   gui_playseq_refresh();
  }
}

void on_gui_button_playseq_append_clicked(GtkButton *button,
					  gpointer user_data) {
  if(editor.song!=NULL) {
    playseq_insert(editor.song->playseqs[editor.playseq],
		   editor.song->playseqs[editor.playseq]->length);
    gui_playseq_refresh();
  }
}

void on_gui_button_playseq_prev_clicked(GtkButton *button,
                                        gpointer user_data) {
  unsigned short playseq=editor.song->playseqs[editor.playseq]->blocknumbers[editor.playseqpos];

  if(playseq>0)
    playseq_set(editor.song->playseqs[editor.playseq], editor.playseqpos,
		playseq-1);

  gui_playseq_refresh();
}

void on_gui_button_playseq_insert_clicked(GtkButton *button,
					  gpointer user_data) {
  if(editor.song!=NULL) {
    playseq_insert(editor.song->playseqs[editor.playseq], editor.playseqpos);
    gui_playseq_refresh();
  }
}

void on_gui_button_playseq_insertnew_clicked(GtkButton *button,
					     gpointer user_data) {
  if(editor.song!=NULL) {
    song_insert_playseq(editor.song, editor.playseq);
    gui_info_refresh();
    gui_playseq_refresh();
    gui_sectionlist_refresh();
  }
}

void on_gui_button_playseq_appendnew_clicked(GtkButton *button,
					     gpointer user_data) {
  if(editor.song!=NULL) {
    song_insert_playseq(editor.song, editor.song->numplayseqs);
    gui_info_refresh();
    gui_playseq_refresh();
    gui_sectionlist_refresh();
  }
}

void on_gui_button_playseq_deletethis_clicked(GtkButton *button,
					      gpointer user_data) {
  if(editor.song!=NULL) {
    song_delete_playseq(editor.song, editor.playseq);
    gui_info_refresh();
    gui_playseq_refresh();
    gui_sectionlist_refresh();
  }
}

void on_gui_clist_blocklist_select_row(GtkCList *clist, gint row,
				       gint column, GdkEvent *event,
				       gpointer user_data) {
  Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main),
						  "gui_tracker");

  editor.block=row;

  /* Refresh the block list window widgets if the window exists */
  if(gui_window_blocklist!=NULL && GTK_IS_WINDOW(gui_window_blocklist)) {
    GtkEntry *name=GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_entry_blocklist"));
    char label[6+8];
    
    /* Show the number of the selected block */
    snprintf(label, 6+8, "Block %d/%d", editor.block+1,
	     editor.song->numblocks);
    gtk_frame_set_label(GTK_FRAME(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_frame_blocklist")), label);

    /* Show the name of the selected block */
    if(editor.song->blocks[editor.block]->name)
      gtk_entry_set_text(name, editor.song->blocks[editor.block]->name);
    else
      gtk_entry_set_text(name, "");

    /* Show the properties of the selected block */
    gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_spinbutton_blocklist_tracks"))),
			     editor.song->blocks[editor.block]->tracks);
    gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_spinbutton_blocklist_length"))),
			     editor.song->blocks[editor.block]->length);
    gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(gui_window_blocklist), "gui_spinbutton_blocklist_commandpages"))),
			     editor.song->blocks[editor.block]->effectpages);
  }

  /* Refresh the rest of the GUI (info, tracker) */
  gui_info_refresh();
  tracker_set_pattern(tracker, editor.song->blocks[editor.block]);
  tracker_set_num_channels(tracker, editor.song->blocks[editor.block]->tracks);
}

void on_gui_spinbutton_blocklist_tracks_changed(GtkEditable *editable,
						gpointer user_data) {
  if(editor.song!=NULL) {
    Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_tracker");

    block_set_tracks(editor.song->blocks[editor.block],
		 (unsigned short)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
    tracker_set_num_channels(tracker,
			 (int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
    tracker_redraw(tracker);

    if(song_check_maxtracks(editor.song))
      gui_trackvolumes_refresh();
  }
}

void on_gui_spinbutton_blocklist_length_changed(GtkEditable *editable,
						gpointer user_data) {
  if(editor.song!=NULL) {
    Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_tracker");
    
    block_set_length(editor.song->blocks[editor.block],
		 (unsigned short)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
    tracker->curpattern=editor.song->blocks[editor.block];
    tracker_redraw(tracker);
  }
}

void on_gui_spinbutton_blocklist_commandpages_changed(GtkEditable *editable,
						      gpointer user_data) {
  if(editor.song!=NULL) {
    Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_tracker");

    block_set_effectpages(editor.song->blocks[editor.block],
			  (unsigned short)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
    tracker_redraw(tracker);
  }
}

void on_gui_button_blocklist_insertnew_clicked(GtkButton *button,
					       gpointer user_data) {
  if(editor.song!=NULL) {
    song_insert_block(editor.song, editor.block);
    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

void on_gui_button_blocklist_appendnew_clicked(GtkButton *button,
					       gpointer user_data) {
  if(editor.song!=NULL) {
    song_insert_block(editor.song, editor.song->numblocks);
    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

void on_gui_button_blocklist_delete_clicked(GtkButton *button,
					    gpointer user_data) {
  if(editor.song!=NULL) {
    song_delete_block(editor.song, editor.block);

    if(editor.block>=editor.song->numblocks)
      editor.block=editor.song->numblocks-1;

    gui_blocklist_refresh();
    gui_playseq_refresh();
  }
}

gboolean on_gui_entry_blocklist_focus_out_event(GtkWidget *widget,
						GdkEventFocus *event,
						gpointer user_data) {
  if(editor.song!=NULL) {
    char *name=gtk_entry_get_text(GTK_ENTRY(widget));
    
    if(editor.song->blocks[editor.block]->name)
      free(editor.song->blocks[editor.block]->name);
    
    editor.song->blocks[editor.block]->name=(char *)strdup(name);

    gui_blocklist_refresh();
    gui_playseq_refresh();
  }

  return FALSE;
}

gboolean on_gui_entry_sysex_name_focus_out_event(GtkWidget *widget,
						 GdkEventFocus *event,
						 gpointer user_data) {
  if(editor.song!=NULL && gui_sysexlist_sysex!=-1) {
    char *name=gtk_entry_get_text(GTK_ENTRY(widget));
    
    if(editor.song->sysexes[gui_sysexlist_sysex]->name)
      free(editor.song->sysexes[gui_sysexlist_sysex]->name);
    
    editor.song->sysexes[gui_sysexlist_sysex]->name=(char *)strdup(name);

    gui_sysexlist_refresh();
  }

  return FALSE;
}

gboolean on_gui_entry_playseq_name_focus_out_event(GtkWidget *widget,
						   GdkEventFocus *event,
						   gpointer user_data) {
  char *name=gtk_entry_get_text(GTK_ENTRY(widget));

  if(editor.song->playseqs[editor.playseq]->name)
    free(editor.song->playseqs[editor.playseq]->name);
  
  editor.song->playseqs[editor.playseq]->name=(char *)strdup(name);
  
  gui_playseq_refresh();
  gui_sectionlist_refresh();

  return FALSE;
}

gboolean on_gui_entry_instrumentproperties_name_focus_out_event(GtkWidget *widget, GdkEventFocus *event, gpointer user_data) {
  char *name=gtk_entry_get_text(GTK_ENTRY(widget));

  if(editor.song->instruments[editor.instrument]->name!=NULL)
    free(editor.song->instruments[editor.instrument]->name);

  editor.song->instruments[editor.instrument]->name=strdup(name);

  gui_instrument_refresh();

  return FALSE;
}

void on_gui_button_sysex_close_clicked(GtkButton *button, gpointer user_data) {
  on_gui_menuitem_song_sysexlist_activate(NULL, NULL);
}

void on_gui_button_saveas_ok_clicked(GtkButton *button, gpointer user_data) {
  if(gui_fileselection_saveas!=NULL &&
     GTK_IS_FILE_SELECTION(gui_fileselection_saveas)) {
    /* Store the filename */
    if(editor.filename!=NULL)
      free(editor.filename);
    editor.filename=strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(gui_fileselection_saveas)));

    /* Save... */
    song_save(editor.song, editor.filename);

    /* ...and close the file selector */
    gtk_widget_destroy(gui_fileselection_saveas);
    gui_fileselection_saveas=NULL;
  }
}

void on_gui_button_open_ok_clicked(GtkButton *button, gpointer user_data) {
  if(gui_fileselection_open!=NULL &&
     GTK_IS_FILE_SELECTION(gui_fileselection_open)) {
    Tracker *tracker=(Tracker *)gtk_object_get_data(GTK_OBJECT(gui_window_main), "gui_tracker");

    editor_open(gtk_file_selection_get_filename(GTK_FILE_SELECTION(gui_fileselection_open)));

    tracker_reset(tracker);
    tracker_set_pattern(tracker, editor.song->blocks[0]);
    tracker_set_num_channels(tracker, editor.song->blocks[0]->tracks);

    gtk_widget_destroy(gui_fileselection_open);
    gui_fileselection_open=NULL;
  }
}

void on_gui_button_open_cancel_clicked(GtkButton *button,
				       gpointer user_data) {
  if(gui_fileselection_open!=NULL &&
     GTK_IS_FILE_SELECTION(gui_fileselection_open)) {
    gtk_widget_destroy(gui_fileselection_open);
    gui_fileselection_open=NULL;
  }
}

void on_gui_button_saveas_cancel_clicked(GtkButton *button,
					 gpointer user_data) {
  if(gui_fileselection_saveas!=NULL &&
     GTK_IS_FILE_SELECTION(gui_fileselection_saveas)) {
    gtk_widget_destroy(gui_fileselection_saveas);
    gui_fileselection_saveas=NULL;
  }
}

void on_gui_spinbutton_sysex_length_changed(GtkEditable *editable,
					    gpointer user_data) {
  if(editor.song!=NULL && gui_sysexlist_sysex!=-1)
    sysex_set_length(editor.song->sysexes[gui_sysexlist_sysex],
		 (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
}

gboolean on_gui_entry_songname_focus_out_event(GtkWidget *widget,
					       GdkEventFocus *event,
					       gpointer user_data) {
  char *name=gtk_entry_get_text(GTK_ENTRY(widget));

  if(editor.song->name!=NULL)
    free(editor.song->name);

  editor.song->name=strdup(name);

  return FALSE;
}

void on_gui_button_songproperties_close_clicked(GtkButton *button,
						gpointer user_data) {
  on_gui_menuitem_song_properties_activate(NULL, NULL);
}

void on_gui_spinbutton_instrument_changed(GtkEditable*editable,
					  gpointer user_data) {
  /* Select an instrument if an instrument selection key was pressed */
  editor.instrument=(unsigned char)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value-1;

  /* Make sure the instrument exists */
  song_check_instrument(editor.song, editor.instrument);
  gui_instrument_refresh();
}

void on_gui_adjustment_trackvolumes_changed(GtkAdjustment *adjustment,
					    gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  *data=((*data)&128)|((unsigned char)adjustment->value);
}

void on_gui_togglebutton_trackvolumes_toggled(GtkToggleButton *togglebutton,
					      gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    *data=((*data)&127)|128;
  else
    *data=(*data)&127;
}
