/* midi.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include "midi.h"
#include "editor.h"

extern struct editor editor;

/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(unsigned char channel, unsigned char note,
		   unsigned char velocity) {
  unsigned char data[3];

  if(editor.midi!=NULL) {
    data[0]=0x80|channel;
    data[1]=note&0x7f;
    data[2]=velocity&0x7f;
    
    fwrite(data, 3, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(unsigned char channel, unsigned char note,
		   unsigned char velocity) {
  unsigned char data[3];

  if(editor.midi!=NULL) {
    data[0]=0x90|channel;
    data[1]=note&0x7f;
    data[2]=velocity&0x7f;
    
    fwrite(data, 3, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(unsigned char channel, unsigned char note,
		     unsigned char pressure) {
  unsigned char data[3];

  if(editor.midi!=NULL) {
    data[0]=0xa0|channel;
    data[1]=note&0x7f;
    data[2]=pressure&0x7f;
    
    fwrite(data, 3, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(unsigned char channel, unsigned char controller,
		     unsigned char value) {
  unsigned char data[3];

  if(editor.midi!=NULL) {
    data[0]=0xb0|channel;
    data[1]=controller&0x7f;
    data[2]=value&0x7f;
    
    fwrite(data, 3, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Send a program change on a MIDI channel */
void midi_program_change(unsigned char channel, unsigned char program) {
  unsigned char data[2];

  if(editor.midi!=NULL) {
    data[0]=0xc0|channel;
    data[1]=program&0x7f;
    
    fwrite(data, 2, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(unsigned char channel, unsigned char pressure) {
  unsigned char data[2];

  if(editor.midi!=NULL) {
    data[0]=0xd0|channel;
    data[1]=pressure&0x7f;
    
    fwrite(data, 2, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(unsigned char channel, unsigned short value) {
  unsigned char data[3];

  if(editor.midi!=NULL) {
    data[0]=0xe0|channel;
    data[1]=(value>>7)&0x7f;
    data[2]=value&0x7f;
    
    fwrite(data, 3, 1, editor.midi);
    fflush(editor.midi);
  }
}

/* Sends a MIDI SysEX message */
void midi_system_exclusive(unsigned char *message, unsigned short length) {
  unsigned char *data=(unsigned char *)calloc(length+2, sizeof(unsigned char));

  if(editor.midi!=NULL) {
    data[0]=0xf0;
    memcpy(data+1, message, length);
    data[length+1]=0xf7;
    
    fwrite(data, length+2, 1, editor.midi);
    fflush(editor.midi);
  }

  free(data);
}

/* Receives a MIDI SysEx message */
void midi_read_system_exclusive(struct sysex *sysex, int autostop) {
  if(sysex!=NULL) {
    int status=0, i=0;
    
    /* Read until a SysEx dump begins */
    while(status!=0xf0 && status!=EOF)
      status=fgetc(editor.midi);
    
    /* Return on error */
    if(status==EOF)
      return;
    
    /* Read until the buffer is full or to the end of the SysEx (autostop) */
    while(i<sysex->length) {
      status=fgetc(editor.midi);
      
      if(status==EOF)
	break;
      
      sysex->data[i++]=status;
      if(autostop==1 && sysex->data[i-1]==0xf7)
	break;
    }
    
    /* Remove unnecessary bytes if necessary */
    if(i<sysex->length) {
      sysex->length=i;
      sysex->data=(unsigned char *)realloc(sysex->data, sysex->length);
    }
  }    
}

/* Sets the length of a MIDI SysEx message */
void sysex_set_length(struct sysex *sysex, unsigned int length) {
  if(sysex!=NULL) {
    sysex->length=length;
    sysex->data=(unsigned char *)realloc(sysex->data, sysex->length);
  }
}

/* Parses a SysEx element in an XML file */
struct sysex *sysex_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct sysex *sysex=NULL;
  
  if((!xmlStrcmp(cur->name, "sysex")) && (cur->ns==ns)) {
    xmlNodePtr temp=cur->xmlChildrenNode;
    char c[3];
    int i;
    unsigned int d;

    /* Temporary string for hexadecimal parsing */
    c[2]=0;

    /* Allocate sysex */
    sysex=(struct sysex *)calloc(1, sizeof(struct sysex));
    sysex->name=xmlGetProp(cur, "name");
    sysex->length=strlen(temp->content)/2;
    sysex->data=(unsigned char *)calloc(sysex->length, sizeof(unsigned char));
    for(i=0; i<sysex->length; i++) {
      c[0]=temp->content[i*2];
      c[1]=temp->content[i*2+1];
      sscanf(c, "%X", &d);
      sysex->data[i]=d;
    }
  } else
    fprintf(stderr, "XML error: expected sysex, got %s\n", cur->name);

  return sysex;
}

/* Saves a SysEx message to an XML file */
void sysex_save(struct sysex *sysex, xmlNodePtr parent) {
  char *c;
  xmlNodePtr node, lb;
  int i;

  c=(char *)calloc(2*sysex->length+1, sizeof(char));
  for(i=0; i<sysex->length; i++)
    sprintf(c+(i*2), "%0.2X", sysex->data[i]);

  node=xmlNewChild(parent, NULL, "sysex", c);
  xmlSetProp(node, "name", sysex->name);

  lb=xmlNewText("\n");
  xmlAddChild(parent, lb);

  free(c);
}
