/* tutoterm_window.c -- window functions
 *
 * Copyright © 2017,2019-2020 Félicien Pillot <felicien@gnu.org>
 * This file is part of TutoTerm.
 * 
 * TutoTerm is free software: you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * TutoTerm is distributed in the hope that it will be useful,
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * with TutoTerm.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <gio/gio.h>
#include <vte/vte.h>
#include <gtk/gtk.h>
#include <string.h>
#include "../config.h"
#include <glib/gi18n-lib.h>
#include <glib/gprintf.h>

#include "../include/tutoterm.h"
#include "../include/tutoterm_window.h"
#include "../include/index_win.h"
#include "../include/input_output.h"
#include "../include/liststore.h"

/**
 * SECTION: tutoterm_window
 * @short_description: Main graphic functions for Tutoterm
 * @title: tutoterm_window
 * @section_id: tutoterm_window
 * @include: "include/index_win.h","include/input_output.h","include/liststore.h","include/tutoterm.h","include/tutoterm_window.h"
 */

G_DEFINE_TYPE (TutotermWindow,
	       tutoterm_window,
	       GTK_TYPE_APPLICATION_WINDOW);

/**
 * get_ttlabel:
 * @widget: a @GtkWidget
 *
 * return: a @GtkWidget
 */
GtkWidget*
get_ttlabel (GtkWidget* widget) {
  GList *children;
  children = gtk_container_get_children (GTK_CONTAINER (widget));
  if (!GTK_IS_LABEL (children->data))
    return get_ttlabel (children->data);
  return children->data;
}

/**
 * get_ttvtextbox:
 * @widget: a @GtkWidget
 *
 * return: a @GtkWidget
 */
GtkWidget*
get_ttvtextbox (GtkWidget* widget) {
  GList *children;
  children = gtk_container_get_children (GTK_CONTAINER (widget));
  if (!GTK_IS_BOX (children->data))
    return get_ttvtextbox (children->data);
  return children->data;
}

/**
 * key_event:
 * @ttMainWindow: a @GtkWidget
 * @event: a @GtkEventKey
 * @ttListStore: a @GtkListStore
 *
 * return: a @gboolean
 */
gboolean
key_event (GtkWidget* ttMainWindow,
	   GdkEventKey* event,
	   GtkListStore* ttListStore)
{
  /*
  (void) event;
  (void) ttListStore;
  char *input = vte_terminal_get_text_range ((VteTerminal*) ttTerm, 1, 1, 3, 3, NULL, NULL, NULL);
  g_printf ("--\n%s\n--\n", input);
  */
  GtkWidget *ttLabel;
  update_input (gdk_keyval_name (event->keyval));
  if (strlen (ttInput) != 0)
    fill_tttext_with_input_description (ttListStore);
  if (strlen (ttText) == 0)
      fill_tttext_with_context_comment (ttListStore);

  ttLabel = get_ttlabel (get_ttvtextbox (ttMainWindow));
  gtk_label_set_markup (GTK_LABEL (ttLabel), ttText);

  if (strlen (ttTextExample) >= 2) {
    gtk_label_set_text
      (GTK_LABEL (TUTOTERM_APP_WINDOW
		  (ttMainWindow)->ttLabelExample),
       ttTextExample);
    gtk_revealer_set_reveal_child (GTK_REVEALER (TUTOTERM_APP_WINDOW
						 (ttMainWindow)->ttExample),
				   TRUE);
  } else {
    gtk_revealer_set_reveal_child (GTK_REVEALER (TUTOTERM_APP_WINDOW
						 (ttMainWindow)->ttExample),
				   FALSE);
  }
  return FALSE;
}

static void
tutoterm_window_init (TutotermWindow *ttMainWindow)
{
  // Declarations
  gchar **arg, *iconFilename;
  GtkWidget
    *ttBar,
    *ttIndexButton,
    *ttLabel,
    *ttPaned,
    *ttScrollWin,
    *ttTerm,
    *ttVTextBox;
  
  GtkListStore* ttListStore;
  ttListStore = get_liststore ();
  
  // Window
  //  gtk_window_set_default_size (GTK_WINDOW (ttMainWindow), 650, 450);
  if (asprintf (&iconFilename,
		"%s/share/pixmaps/tutoterm.png",
		ROOT_DIR))
    gtk_window_set_default_icon_from_file (iconFilename, NULL);

  // HeaderBar
  ttBar = gtk_header_bar_new ();
  gtk_header_bar_set_title ((GtkHeaderBar*) ttBar, "TutoTerm");
  gtk_header_bar_set_subtitle ((GtkHeaderBar*) ttBar,
			       _("The educational terminal!"));
  gtk_header_bar_set_show_close_button ((GtkHeaderBar*) ttBar, TRUE);
  ttIndexButton = gtk_button_new_from_icon_name ("help",
						 GTK_ICON_SIZE_LARGE_TOOLBAR);
  g_signal_connect (ttIndexButton,
		    "clicked", 
		    G_CALLBACK (index_window),
		    ttListStore);
  gtk_container_add (GTK_CONTAINER (ttBar), ttIndexButton);
  gtk_window_set_titlebar (GTK_WINDOW (ttMainWindow), ttBar);
  
  // Vertical paned widget
  ttPaned = gtk_paned_new (GTK_ORIENTATION_VERTICAL);
  gtk_container_add (GTK_CONTAINER (ttMainWindow), ttPaned);

  // Tutorial area
  ttScrollWin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_min_content_height
    (GTK_SCROLLED_WINDOW (ttScrollWin), 150);
  ttVTextBox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 1);

  ttLabel = gtk_label_new (_("Welcome on TutoTerm!"));
  gtk_label_set_line_wrap (GTK_LABEL (ttLabel), TRUE);
  gtk_label_set_xalign (GTK_LABEL (ttLabel), 0);
  gtk_widget_set_margin_start (ttLabel, 10);
  gtk_box_pack_start (GTK_BOX (ttVTextBox), ttLabel, FALSE, FALSE, 10);

  ttMainWindow->ttExample = gtk_revealer_new ();
  gtk_revealer_set_transition_type
    (GTK_REVEALER (ttMainWindow->ttExample),
     GTK_REVEALER_TRANSITION_TYPE_CROSSFADE);
  gtk_revealer_set_transition_duration
    (GTK_REVEALER (ttMainWindow->ttExample),
     500);

  ttMainWindow->ttLabelExample = gtk_label_new ("");
  gtk_label_set_xalign (GTK_LABEL (ttMainWindow->ttLabelExample),
			0);
  gtk_widget_set_margin_start (ttMainWindow->ttLabelExample,
			       20);

  gtk_container_add (GTK_CONTAINER (ttMainWindow->ttExample),
		     ttMainWindow->ttLabelExample);
  gtk_box_pack_start (GTK_BOX (ttVTextBox),
		      ttMainWindow->ttExample, TRUE, TRUE, 1);
  
  gtk_container_add (GTK_CONTAINER (ttScrollWin), ttVTextBox);
  gtk_paned_add1 (GTK_PANED (ttPaned), ttScrollWin);

  // Terminal
  ttTerm = vte_terminal_new ();
  arg = g_new (gchar *, 1);
  arg[0] = g_strdup ("/bin/bash");
  arg[1] = NULL;
  vte_terminal_spawn_sync ((VteTerminal*) ttTerm, VTE_PTY_DEFAULT,
			    NULL, arg, NULL, G_SPAWN_SEARCH_PATH, NULL,
			   NULL, NULL, NULL, NULL);
  gtk_paned_add2 (GTK_PANED (ttPaned), ttTerm);
  gtk_widget_grab_focus (ttTerm);
  
  // Show widgets and wait for any key event
  gtk_widget_show_all (GTK_WIDGET (ttMainWindow));
  g_signal_connect (ttMainWindow,
		    "key-release-event",
		    G_CALLBACK(key_event),
		    ttListStore);
}

static void
tutoterm_window_class_init (TutotermWindowClass *class)
{
  (void) class;
}

TutotermWindow*
tutoterm_window_new (Tutoterm *app)
{
  return g_object_new (TUTOTERM_WINDOW_TYPE,
		       "application",
		       app,
		       NULL);
}
