/* liststore.c -- parse, store and prepare data for TutoTerm
 *
 * Copyright © 2017,2019 Félicien Pillot <felicien@gnu.org>
 * This file is part of TutoTerm.
 * 
 * TutoTerm is free software: you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * TutoTerm is distributed in the hope that it will be useful,
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * with TutoTerm.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gmodule.h>

#include <gio/gio.h>

#include "../config.h"
#include <glib/gi18n-lib.h>
#include "../include/liststore.h"

/**
 * SECTION: liststore
 * @short_description: Parse, store and prepare data
 * @title: liststore
 * @section_id: liststore
 * @include: include/liststore.h
 */

static GtkTreeIter iter;

/* Called for open tags <foo bar="baz"> */
void
start_element (GMarkupParseContext *context,
	       const gchar         *element_name,
	       const gchar        **attribute_names,
	       const gchar        **attribute_values,
	       gpointer             user_data,
	       GError             **error)
{
  (void) context;
  (void) attribute_names;
  (void) attribute_values;
  (void) error;
  GtkListStore *ttListStore = user_data;
  GValue command_name = G_VALUE_INIT;
  g_value_init (&command_name, G_TYPE_STRING);

  if (g_strcmp0 (element_name, "command") == 0)
    {
      while (*attribute_names)
	{
	  if (g_strcmp0 (*attribute_names, "name") == 0)
	    g_value_set_static_string (&command_name,
				       *attribute_values);
	  attribute_names++;
	}
      if (G_IS_VALUE (&command_name))
	{
	  gtk_list_store_append (GTK_LIST_STORE (ttListStore), &iter);
	  gtk_list_store_set_value (GTK_LIST_STORE (ttListStore),
				    &iter, 0,
				    &command_name);
	}
      else g_error ("Command name not found");
    }
}

void
text (GMarkupParseContext *context,
      const gchar         *text,
      gsize                text_len,
      gpointer             user_data,
      GError             **error)
{
  (void) error;

  GtkListStore *ttListStore = user_data;
  int column;
  const gchar *element_name = g_markup_parse_context_get_element (context);
  GValue content = G_VALUE_INIT;
  g_value_init (&content, G_TYPE_STRING);

  if (text_len != 0)
    {
      g_value_set_static_string	(&content,
				 g_strcompress (g_strchug (text)));
      column = !g_strcmp0 (element_name, "description") ? 1 :
	!g_strcmp0 (element_name, "example") ? 2 :
	!g_strcmp0 (element_name, "posttext") ? 3 : -42;
      if (column == -42)
	g_error ("%s: unknown element\n", element_name);
      
      gtk_list_store_set_value (ttListStore, &iter, column, &content);
    }
}

static GMarkupParser parser = {
  start_element,
  NULL,
  text,
  NULL,
  NULL
};

/**
 * get_liststore:
 *
 * Extract documentation strings from resource XML file, and fill a
 * @GtkListStore with it.
 *
 * Returns: a @GtkListStore containing documentation strings for the
 * GNU commands supported by Tutoterm
 */
GtkListStore*
get_liststore ()
{
  GtkListStore *ttListStore;
  ttListStore = gtk_list_store_new (4,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_STRING);

  GError *error = NULL;
  GBytes *command_descriptions;
  const gchar *file_content;

  command_descriptions = g_resources_lookup_data
    ("/org/nongnu/tutoterm/command-descriptions.xml",
     G_RESOURCE_LOOKUP_FLAGS_NONE, NULL);

  GMarkupParseContext *context =
    g_markup_parse_context_new (&parser, 0, ttListStore, NULL);

  if (!(file_content = g_bytes_get_data (command_descriptions, NULL)))
    {
      g_error (_("Couldn't load the Command Descriptions file"));
      exit(255);
    }
  
  if (g_markup_parse_context_parse (context, file_content,
				    g_bytes_get_size (command_descriptions),
				    &error) == FALSE) {
    g_error ("%s", error->message);
    exit(255);
  }

  g_markup_parse_context_free (context);

  return ttListStore;
}
