;Copyright (C) 2006 Mikalai Birukou
;
;This file is part of TwinLisp.
;
;    TwinLisp is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
;
;    TwinLisp is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with TwinLisp; if not, write to the Free Software
;    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
(in-package "TWINLISP")
;; Generic container funcs
(defgeneric iter (container &optional step))
(defgeneric len (container))
(defgeneric _getitem_ (container indexObject))
(defgeneric (setf _getitem_) (newVal container indexObject))
(defgeneric index (container elem))
(defgeneric tl-count (container elem))
(defgeneric insert (container indexObject elem))
(defgeneric tl-pop (container &optional indexObject))
(defgeneric tl-remove (container elem))
(defgeneric tl-append (seq &rest elems))
(defgeneric extend (seq &rest elems))
;; --------------------------------------------------------
;; Additional funcs
(defun _make-slice_ (start end step) (new _slice_ :start start :end end :step step))
(defun _make-hash-table_ (&rest keyValPairs) "This should make new hash-table and populate it with given
     key->value pairs" (do ((i 0 (_+_ i 2)) (tab (make-hash-table :test #'equal))) ((_>=_ i (len keyValPairs)) tab) (setf (_getitem_ tab (_getitem_ keyValPairs i)) (_getitem_ keyValPairs (_+_ i 1)))))
(defun _make-vector_ (&key ((:initContent initSeq) nil initSeqPresent) ((:elemType elemType) t) ((:initLen initDim) nil initDimPresent)) (cond (initSeqPresent (make-array (list (len initSeq)) :element-type elemType :initial-contents (coerce initSeq 'list) :adjustable t :fill-pointer t)) (t (make-array (list (cond (initDimPresent initDim) (t 0))) :element-type elemType :adjustable t :fill-pointer t))))
(defun _get-integer-slice-args_ (seqLen sl) "'start', 'end' and 'step' are returned, normalized according to provided
     _slice_ and sequence length" (values (cond ((slot-value sl 'start) (cond ((_<_ (slot-value sl 'start) 0) 0) (t (slot-value sl 'start)))) (t 0)) (cond ((slot-value sl 'end) (cond ((_>_ (slot-value sl 'end) seqLen) seqLen) (t (slot-value sl 'end)))) (t seqLen)) (slot-value sl 'step)))
(defun _positive-index_ (seqLen ind seq errMess) "Normalizes index to be positive, according to a given sequence length" (cond ((_and_ (_>=_ ind 0) (_<_ ind seqLen)) ind) ((_and_ (_<_ ind 0) (_<=_ (_unary-_ ind) seqLen)) (_+_ seqLen ind)) (t ;;----------NOT FINISHED----------------
 ;; should raise an error (continuable?)
 (error 'index-error :container seq :index ind :format-control errMess))))
(defun err-on-null (lst errMess) "Signals an error when given an empty list" (cond ((null lst) (error 'null-list-error :format-control errMess))))
;; --------------------------------------------------------
;; Sequence's class methods
(defmethod iter ((seq sequence) &optional (step 1)) (new _sequence-iterator_ :sequence seq :step step))
(defmethod len ((seq sequence)) (length seq))
(defmethod _getitem_ ((seq sequence) (intIndex integer)) (let (index) (setf index (_positive-index_ (len seq) intIndex seq "Method _getitem_ is called on sequence with out-of-range integer index")) (elt seq index)))
(defmethod (setf _getitem_) (newVal (seq sequence) (intIndex integer)) (let (index) (setf index (_positive-index_ (len seq) intIndex seq "Method setter _getitem_ is called on sequence with out-of-range integer index")) (setf (elt seq index) newVal)))
(defmethod index ((seq sequence) (elem t)) "Returns an index of the first item==elem in sequence.
     If it fails, nil is returned" (position elem seq :test '_==_))
(defmethod tl-count ((seq sequence) (elem t)) "Returns the number of times elem appears in the sequence" (count elem seq :test '_==_))
(defmethod _copy_ ((seq sequence)) (copy-seq seq))
;; --------------------------------------------------------
;; List's class methods
(defmethod iter ((lst list) &optional (step 1)) (new _list-iterator_ :list (cons nil lst) :step step))
(defmethod len ((lst list)) (list-length lst))
(defmethod _getitem_ ((lst list) (sl _slice_)) (let (start end step) (setf (values start end step) (_get-integer-slice-args_ (len lst) sl)) (cond (step (do ((i start (_+_ i step)) (newSeq nil) endCons (lstCons (nthcdr start lst) (nthcdr step lstCons))) ((_>=_ i end) newSeq) (cond ((null newSeq) (setf newSeq (cons (car lstCons) nil)) (setf endCons newSeq)) (t (setf (cdr endCons) (cons (car lstCons) nil)) (setf endCons (cdr endCons)))))) (t (cond ((_<_ start end) (subseq lst start end)) (t nil))))))
(defmethod tl-append ((lst list) &rest elems) (extend lst elems))
(defmethod extend ((lst list) &rest seqs) (err-on-null lst "Method extend should not be called on null list") (let (endCons) (setf endCons (last lst)) ;; will err if lst is circular
 (endp (cdr endCons)) ;; will err if lst was a dotted list
 (setf (cdr endCons) (cdr (tl-for (seq seqs (extLst (cons nil nil)) (lastCons extLst (last lastCons))) (extLst) (setf (cdr lastCons) (coerce seq 'list))))) nil))
(defmethod insert ((lst list) (intIndex integer) (elem t)) (err-on-null lst "Method insert should not be called on null list") (let (index) (setf index (_positive-index_ (_+_ (len lst) 1) intIndex lst "Method insert is called on list with out-of-range integer index")) (cond ((_>_ index 0) (setf (cdr (nthcdr (_-_ index 1) lst)) (cons elem (nthcdr index lst)))) (t (let (newCons) (setf newCons (cons (car lst) (cdr lst))) (setf (car lst) elem) (setf (cdr lst) newCons)))) nil))
(defmethod tl-pop ((lst list) &optional (intIndex nil indexPresent)) "Remove the item at the given position in the list, and return it.
     When index is not given, the last element is poped.
     If list initially has only one element, it will be '~(nil) after poping" (err-on-null lst "Method pop should not be called on null list") (let (lstLen) (setf lstLen (len lst)) (let (popedElem) (setf popedElem nil) (cond (indexPresent (let (index) (setf index (_positive-index_ lstLen intIndex lst "Method pop is called on list with out-of-range integer index")) (cond ((_>_ lstLen 1) (cond ((_==_ index 0) (setf popedElem (car lst)) (setf (car lst) (car (cdr lst))) (setf (cdr lst) (cdr (cdr lst)))) (t (let (cutCons) (setf cutCons (nthcdr (_-_ index 1) lst)) (setf popedElem (car (cdr cutCons))) (setf (cdr cutCons) (cdr (cdr cutCons))))))) (t (setf popedElem (car lst)) ;; index==0 anyway
 (setf (car lst) nil))))) (t (cond ((_>_ lstLen 1) (let (newEndCon) (setf newEndCon (nthcdr (_-_ lstLen 2) lst)) (setf popedElem (car (cdr newEndCon))) (setf (cdr newEndCon) nil))) (t (setf popedElem (car lst)) (setf (car lst) nil))))) popedElem)))
(defmethod tl-remove ((lst list) (elem t)) "Should remove the first item, s.t. item==elem.
     If succesful, returns index of a removed item,
     otherwise returns nil" (let (index) (setf index (index lst elem)) (cond (index (tl-pop lst index))) index))
(defmethod _copy_ ((lst list)) (copy-list lst))
;; --------------------------------------------------------
;; Hash-table's class methods
(defmethod len ((tab hash-table)) (hash-table-count tab))
(defmethod _getitem_ ((hashTab hash-table) (key t)) (gethash key hashTab))
(defmethod (setf _getitem_) (newVal (hashTab hash-table) (key t)) (setf (gethash key hashTab) newVal))
(defmethod tl-pop ((hashTab hash-table) &optional (key nil keyPresent)) (cond (keyPresent (let (popedElem) (setf popedElem (_getitem_ hashTab key)) (values popedElem (remhash key hashTab)))) (t (values nil nil))))
(defmethod _copy_ ((tab hash-table)) (let (newTab) (setf newTab (_make-hash-table_)) (maphash #'(lambda (key value) (setf (_getitem_ newTab key) value)) tab) newTab))
(defmethod has_key ((hashTab hash-table) (key t)) (nth-value 1 (gethash key hashTab)))
(defmethod tKeys ((tab hash-table)) (let (keys) (setf keys (_make-vector_)) (maphash #'(lambda (key value) (declare (ignore value)) (_<<_ keys key)) tab) keys))
(defmethod tValues ((tab hash-table)) (let (values) (setf values (_make-vector_)) (maphash #'(lambda (key value) (declare (ignore key)) (_<<_ values value)) tab) values))
(defmethod tItems ((tab hash-table)) (let (items) (setf items (_make-vector_)) (maphash #'(lambda (key value) (insert items (_unary-_ 1) (list key value))) tab) items))
(defmethod iter ((tab hash-table) &optional (step 1)) (declare (ignore step)) (iter (tItems tab)))
;; --------------------------------------------------------
;; Array's class methods
(defmethod _getitem_ ((arr array) (coord list)) (apply #'aref arr coord))
(defmethod (setf _getitem_) (newVal (arr array) (coord list)) (setf (apply #'aref arr coord) newVal))
;; --------------------------------------------------------
;; Vector's class methods
(defmethod _getitem_ ((vec vector) (sl _slice_)) (let (start end step) (setf (values start end step) (_get-integer-slice-args_ (len vec) sl)) (cond ((_<_ start end) (cond (step (do ((i start (_+_ i step)) (k 0 (_+_ k 1)) (tempVec (_make-vector_ :initLen (_+_ (_%_ (_-_ (_-_ end 1) start) step) 1)))) ((_>=_ i end) tempVec) (setf (_getitem_ tempVec k) (_getitem_ vec i)))) (t (subseq vec start end)))) (t (subseq vec 0 0)))))
(defmethod tl-append ((vec vector) &rest elems) (extend vec elems))
(defmethod extend ((vec vector) &rest seqs) (tl-for (seq seqs) (nil) (tl-for (elem seq) (nil) (vector-push-extend elem vec))))
(defmethod insert ((vec vector) (intIndex integer) (elem t)) (let (vecLen) (setf vecLen (len vec)) (let (index) (setf index (_positive-index_ (_+_ vecLen 1) intIndex vec "Method insert is called on vector with out-of-range integer index")) (cond ((_==_ index vecLen) (vector-push-extend elem vec)) (t (vector-push-extend nil vec) (do ((i (_-_ vecLen 1) (_-_ i 1))) ((_<_ i index)) (setf (_getitem_ vec (_+_ i 1)) (_getitem_ vec i))) (setf (_getitem_ vec index) elem))) nil)))
(defmethod tl-pop ((vec vector) &optional (intIndex nil indexPresent)) (let (popedElem) (setf popedElem nil) (cond (indexPresent (let (vecLen) (setf vecLen (len vec)) (let (index) (setf index (_positive-index_ vecLen intIndex vec "Method pop is called on vector with out-of-range integer index")) (setf popedElem (_getitem_ vec index)) (do ((i index (_+_ i 1))) ((_>=_ i (_-_ vecLen 1))) (setf (_getitem_ vec i) (_getitem_ vec (_+_ i 1))))))) (t (setf popedElem (_getitem_ vec (_unary-_ 1))))) (setf (_getitem_ vec (_unary-_ 1)) nil) ; do not leave elements above fill-pointer
 (vector-pop vec) popedElem))
(defmethod tl-remove ((vec vector) (elem t)) (let (index) (setf index (index vec elem)) (cond (index (tl-pop vec index))) index))
;; --------------------------------------------------------
;; iterator method
(defmethod iter ((it _iterator_) &optional (step nil stepPresent)) (cond (stepPresent (setf (slot-value it 'step) step))) it)
;; --------------------------------------------------------
;; List-iterator methods
(defmethod next ((iter _list-iterator_) &optional (step nil stepPresent)) "Returns two values: 1) 'next' element, which is reached by a step,
     and 2) boolean value of a success of this operation" (let (currStep) (setf currStep (cond (stepPresent step) (t (slot-value iter 'step)))) (cond ((_>_ currStep 0) (cond ((_not_ (slot-value iter 'iterStarted)) (setf (slot-value iter 'iterStarted) t))) (setf (slot-value iter 'lastCons) (nthcdr currStep (slot-value iter 'lastCons))) (values (car (slot-value iter 'lastCons)) (_not_ (null (slot-value iter 'lastCons))))) ((_==_ currStep 0) (cond ((slot-value iter 'iterStarted) (values (car (slot-value iter 'lastCons)) (_not_ (endp (slot-value iter 'lastCons))))) (t (values nil nil)))) (t (values nil nil)))))
;; --------------------------------------------------------
;; Sequence-iterator methods
(defmethod next ((iter _sequence-iterator_) &optional (step nil stepPresent)) "Returns two values: 1) 'next' element, which is reached by a step,
     and 2) boolean value of a success of this operation" (let (nextIndex) (setf nextIndex (_+_ (slot-value iter 'lastIndex) (cond (stepPresent step) (t (slot-value iter 'step))))) (cond ((_and_ (_>=_ nextIndex 0) (_<_ nextIndex (len (slot-value iter 'seq)))) (cond ((_not_ (slot-value iter 'iterStarted)) (setf (slot-value iter 'iterStarted) t))) (setf (slot-value iter 'lastIndex) nextIndex) (values (_getitem_ (slot-value iter 'seq) nextIndex) t)) (t (values nil nil)))))
;; --------------------------------------------------------
