/* txt2xhtml - convert plain text to XHTML document
   Copyright (C) 2004  Krzysztof Jurewicz

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   You can contact the author by sending email to kjurewicz@gazeta.pl */

#include <fstream>
#include <iostream>

using namespace std;

string replace(const char&, bool&, const bool&);

int main (int argc, char *argv[])
{
  // were command line parameters specified?
  bool help_flag=false;
  bool version_flag=false;
  bool output_file_flag=false;
  bool background_color_flag=false;
  bool color_flag=false;
  bool a_link_color_flag=false;
  bool a_visited_color_flag=false;
  bool a_hover_color_flag=false;
  bool a_active_color_flag=false;
  bool external_stylesheet_flag=false;
  bool stylesheet_flag=false;

  // Shall we convert url's and www addresses?
  bool url_flag=true;
  bool www_flag=true;

  // Shall we replace tags '<' and '>' signs?
  bool replace_tags_signs=true;

  // Shall we output complete XHTML document or only content of the <body> tag?
  bool complete_document=true;

  // Shall we output the <div> tag?
  bool div_tag=true;

  // Have we defined any styles?
  bool styles_defined=false;
  bool body_styles=false;
  bool a_link_styles=false;
  bool a_visited_styles=false;
  bool a_hover_styles=false;
  bool a_active_styles=false;

  // "help" and "version" texts
  string help="Usage: txt2xhtml [options] file...\nOptions:\n  -o <file>\t\t\tset output file as <file>\n  --help\t\t\tdisplay this help\n  --version\t\t\tdisplay information about version\n  --stylesheet <file>\timport stylesheet <file> to document\n  --external-stylesheet <file>\tset external stylesheet as <file>  --background-color <color>\tset background color as '#<color>'\n  --color <color>\t\tset text color as '#<color>'\n  --a-link-color <color>\tset a:link color as #<color>\n  --a-visited-color <color>\tset a:visited color as #<color>\n  --a-hover-color <color>\tset a:hover color as #<color>\n  --a-active-color <color>\tset a:active color as #<color>\n  -t, --title <title>\t\tset page title as <title>\n  --disable-www-links\t\tdisable converting to links url's starting with 'www.'\n  --disable-links\t\tdisable converting to links all url's\n  --body-only\t\t\toutput only the <body> tag content\n  --without-div\t\t\tdo not output the <div> tag when the '--body-only' option is used\n";
  string version="txt2xhtml 0.01\nCopyright (C) 2004 Krzysztof Jurewicz\ntxt2xhtml comes with ABSOLUTELY NO WARRANTY, not even with implied\nwarranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE\nYou can redistribute and/or modify txt2xhtml\nunder the terms of the GNU General Public License\nFor more information about these matters, see the file named COPYING\n";

  // page title
  string title;

  // encoding, utf-8 by default
  string encoding="utf-8";

  // background color, white by default
  string background_color="fff";

  // text color, black by default
  string color="000";

  // link colors
  string a_link_color="00f";
  string a_visited_color="f00";
  string a_hover_color="00f";
  string a_active_color="00f";

  // external stylesheet, a.css by default
  char external_stylesheet[80];

  // imported stylesheet
  char stylesheet[80];

  // output filename
  char output_filename[80];

  // default output filename
  char default_output_filename[]="a.xhtml";

  // Let's check command line parameters.
  if (argc==1)
    {
      cout << "Please specify input filename\n";
      return 0;
    }
  else
    {
      title=argv[argc-1];
      string parameter;
      for (int i=1; i<argc; i++)
	{
	  parameter=argv[i];
	  if (parameter=="-o")
	    {
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '-o' missing\n";
		  return 0;
		}
	      else
		{
		  strcpy(output_filename, argv[i]);
		}
	      output_file_flag=true;
	    }
	  if (parameter=="--help")
	    help_flag=true;
	  if (parameter=="--version")
	    {
	      version_flag=true;
	      break; // If we've found the '--version' parameter, there is no need to check the other.
	    }
	  if (parameter=="--background-color")
	    {
	      styles_defined=true;
	      body_styles=true;
	      background_color_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--background-color' missing\n";
		  return 0;
		}
	      else
		{
		  background_color=argv[i];
		}
	    }
	  if (parameter=="-t" || parameter=="--title")
	    {
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--title' missing\n";
		  return 0;
		}
	      else
		{
		  title=argv[i];
		}
	    }
	  if (parameter=="--disable-www-links")
	    www_flag=false;
	  if (parameter=="--disable-links")
	    {
	      url_flag=false;
	      www_flag=false;
	    }
	  if (parameter=="--body-only")
	    complete_document=false;
	  if (parameter=="--without-div")
	    div_tag=false;
	  if (parameter=="--color")
	    {
	      styles_defined=true;
	      body_styles=true;
	      color_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--color' missing\n";
		  return 0;
		}
	      else
		{
		  color=argv[i];
		}
	    }
	  if (parameter=="--a-link-color")
	    {
	      styles_defined=true;
	      a_link_styles=true;
	      a_link_color_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--a-link-color' missing\n";
		  return 0;
		}
	      else
		{
		  a_link_color=argv[i];
		}
	    }
	  if (parameter=="--a-visited-color")
	    {
	      styles_defined=true;
	      a_visited_styles=true;
	      a_visited_color_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--a-visited-color' missing\n";
		  return 0;
		}
	      else
		{
		  a_visited_color=argv[i];
		}
	    }
	  if (parameter=="--a-hover-color")
	    {
	      styles_defined=true;
	      a_hover_styles=true;
	      a_hover_color_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--a-hover-color' missing\n";
		  return 0;
		}
	      else
		{
		  a_hover_color=argv[i];
		}
	    }
	  if (parameter=="--a-active-color")
	    {
	      styles_defined=true;
	      a_link_styles=true;
	      a_link_color_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--a-active-color' missing\n";
		  return 0;
		}
	      else
		{
		  a_active_color=argv[i];
		}
	    }
	  if (parameter=="--stylesheet")
	    {
	      styles_defined=true;
	      stylesheet_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--stylesheet' missing\n";
		  return 0;
		}
	      else
		{
		  strcpy(stylesheet, argv[i]);
		}
	    }
	  if (parameter=="--external-stylesheet")
	    {
	      external_stylesheet_flag=true;
	      i++;
	      if (i>=argc-1)
		{
		  cout << "Argument to '--external-stylesheet' missing\n";
		  return 0;
		}
	      else
		{
		  strcpy(external_stylesheet, argv[i]);
		}
	    }
	}
    }

  // Parameter '--version' has higher priority than '--help'.
  if (help_flag || version_flag)
    {
      if (version_flag)
	cout << version;
      else
	cout << help;
      return 0;
    }

  // setting output filename as default if the user didn't specify his own
  if (!output_file_flag)
    strcpy (output_filename, default_output_filename);

  // opens output file
  ofstream fout(output_filename);

  if (complete_document)
    {
      // inserts information about xml version
      fout << "<?xml version=\"1.0\" encoding=\"";

      // inserts encoding information
      fout << encoding;

      // inserts doctype and first part of <head>
      fout << "\"?>\n<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n<head>\n<title>";

      // inserts title
      fout << title;

      // inserts rest of <head> and starts <body>
      fout << "</title>\n";

      // inserts external stylesheet
      if (external_stylesheet_flag)
	{
	  fout << "<link rel=\"stylesheet\" type=\"text/css\" href=\"";
	  fout << external_stylesheet;
	  fout << "\"/>\n";
	}

      // Do we need to specify any styles?
      if (styles_defined)
	{
	  fout << "<style type=\"text/css\">\n";
	  if (stylesheet_flag)
	    {
	      char c;
	      ifstream fin(stylesheet);
		while(fin.get(c))
		  fout << c;
		fout << "\n\n";
	      fin.close();
	    }
	  if (body_styles)
	    {
	      fout << "body\n{\n";
	      if (background_color_flag)
		{
		  fout << "\tbackground-color: #" << background_color << ";\n";
		}
	      if (color_flag)
		{
		  fout << "color: #" << color << ";\n";
		}
	      fout << "}\n\n";
	    }
	  if (a_link_styles)
	    {
	      fout << "a:link\n{\n";
	      if (a_link_color_flag)
		fout << "\tcolor: #" << a_link_color << ";\n";
	      fout << "}\n\n";
	    }
	  if (a_visited_styles)
	    {
	      fout << "a:visited\n{\n";
	      if (a_link_color_flag)
		fout << "\tcolor: #" << a_visited_color << ";\n";
	      fout << "}\n\n";
	    }
	  if (a_hover_styles)
	    {
	      fout << "a:hover\n{\n";
	      if (a_hover_color_flag)
		fout << "\tcolor: #" << a_hover_color << ";\n";
	      fout << "}\n\n";
	    }
	  if (a_active_styles)
	    {
	      fout << "a:active\n{\n";
	      if (a_active_color_flag)
		fout << "\tcolor: #" << a_active_color << ";\n";
	      fout << "}\n\n";
	    }
	  fout << "</style>\n";
	}
      fout << "</head>\n<body>\n";
    }

  // If we output a complete document, a <div> tag must be present.
  if (div_tag || complete_document)
  fout << "<div>\n";

  // table of chars used only to make space and dot char variables and to get some singular chars
  char test[]=" .\"<";

  // chars of space, new line and dot
  char space=test[0];
  char newline=char(10);
  char dot=test[1];
  
  // data used while converting to XHTML is below

  // current letter
  char ch;

  // Was previous character a space?
  bool previous_space=false;

  // string used to store current letter
  string temp;
  
  // current word
  string word;

  // length of current word
  unsigned long int length=0;

  // is current word an url?
  bool is_url=false;

  // does current word end with a dot (set as true only if the word is a url)?
  bool ends_dot=false;

  // is current word a www address without information about protocol (without "http://")?
  bool is_www=false;

  // opens input file
  ifstream fin(argv[argc-1]);

  // Let's convert it to XHTML and save it!
  for (int i=0; i<2; )
    {
      if (!fin.get(ch))
	  i++;
      if (ch==newline || (ch==space && !previous_space) || i==1)
	{
	  if (!is_url)
	    {
	      fout << word;
	    }
	  else
	    {
	      if (word[length-1]==dot)
		{
		  word[length-1]=test[2];
		  ends_dot=true;
		}
	      fout << "<a href=\"";
	      if (is_www)
		fout << "http://";
	      fout << word;
	      if (!ends_dot)
		fout << "\"";
	      else
		word[length-1]=test[3];
	      fout << ">";
	      fout << word;
	      if (!ends_dot)
		fout << "<";
	      fout << "/a>";
	      if (ends_dot)
		fout << ".";
	      is_url=false;
	      ends_dot=false;
	      is_www=false;
	    }
	  word="";
	  if (ch==newline)
	    fout << "<br/>\n";
	  if (ch==space)
	    {
	      fout << " ";
	      previous_space=true;
	    }
	  length=0;
	}
      else
	{
	  string temp=replace(ch, previous_space, replace_tags_signs);
	  word+=temp;
	  length++;
	}
      if ((word=="http://" || word=="ftp://" || word=="news://") && url_flag)
	is_url=true;
      if (word=="www." && www_flag)
	{
	  is_www=true;
	  is_url=true;
	}
      if (ch!=space)
	previous_space=false;
    }

  // input file isn't needed anymore
  fin.close();

  if (div_tag || complete_document)
    fout << "\n</div>";

  if (complete_document)
    fout << "\n</body>\n</html>";

  // Finished!
  fout.close();
  return 0;
}

// replace signs by strings that represent them in XHTML
string replace(const char &sign, bool &previous_space, const bool &replace_tags_signs)
{
  char signs[]="\"&<>' \t";
  if (sign==signs[0])
    return "&quot";
  if (sign==signs[1])
    return "&amp;";
  if (sign==signs[2] && replace_tags_signs)
    return "&lt;";
  if (sign==signs[3] && replace_tags_signs)
    return "&gt;";
  if (sign==signs[4])
    return "&apos;";
  if (sign==signs[5])
    return "&nbsp;";
  else
    previous_space=false;
  if (sign==signs[6])
      {
	previous_space=true;
	return "&nbsp;&nbsp;&nbsp;";
      }
  char temp[]=" ";
  temp[0]=sign;
  string temp2=temp;
  return temp2;
}
