/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   script.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: wraps the LUA api to C++ and vice-versa
 *              all the interaction with lua is contained in this file
 *              it might be splited some day since it's getting *big*
 *              lua scripting is fundamental in u61. in fact all the
 *              rules of the game are coded in lua. this allows a 
 *              player/hacker  to write his own rules and test them
 *              right away without recompiling, and be quite
 *              "protection fault" safe
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/core.h>
#include <ClanLib/lua.h>
#include <stdio.h>
#include <string.h>

#include "script.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * the following pointers allow tradionnal C functions, called by Lua,
 * to act directly on a specified object
 * this way, when one writes lua code, one only has to assume that
 * functions always operate on the right block, the right map etc...
 */
static U61_Block *current_block=NULL;
static U61_Map *current_map=NULL;

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * initializes the script module with a string
 */
bool U61_Script::do_string(char *str)
{
    bool bSuccess;

    bSuccess=false;

    return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * initializes the script module with a file
 */
bool U61_Script::do_file(char *name)
{
    bool bSuccess;
    int iResult;

    bSuccess=false;

    //lua_open();
    //cout<<"Loading script from \""<<name<<"\".\n";
    iResult=CL_Lua::dofile(name);
    switch (iResult)
    {
    case 0:
        bSuccess=true;
        break;
    default:
    //    cout<<"Lua error "<<iResult<<".\n";
        break;
    }

    return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as a string
 */
char *U61_Script::get_global_str(char *name,char *def)
{
    CL_LuaObject obj;
    char *result=def;

    obj=CL_Lua::getglobal(name);
    if (obj.isString())
    {
        result=obj;
    }    

    return result;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as a string
 */
int U61_Script::get_global_int(char *name,int def)
{
    CL_LuaObject obj;
    float result=def;

    obj=CL_Lua::getglobal(name);
    if (obj.isNumber())
    {
        result=obj;
    }    

    return (int) result;
}

/*--------------------------------------------------------------------------*/
/*
 * registers all C functions that can be used from Lua
 */
bool U61_Script::init()
{
    bool bSuccess=true;

    CL_Lua::mathlibopen();
    record_all();

    return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * cancels previous calls to prepare_for_???
 * this way, if an lua script tries to access data it is not allowed to,
 * no protection fault will occur (at least that's what we hope...)
 */
void U61_Script::secure_pointers()
{
    current_block=NULL;
    current_map=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that creates shapes
 * it does not perform clear nor center functions
 */
void U61_Script::block_shape(U61_Block *b, int type)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_block=b;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"(%d)",type);
    strcpy(command,LUA_BLOCK_SHAPE);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the right
 * the Lua function may or may not do a geometrical rotation, it
 * may for instance be color cycling
 */
void U61_Script::block_rotate_right(U61_Block *b)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_block=b;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"()");
    strcpy(command,LUA_BLOCK_ROTATE_RIGHT);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the left
 * the Lua function may or may not do a geometrical rotation, it
 * may for instance be color cycling
 */
void U61_Script::block_rotate_left(U61_Block *b)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_block=b;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"()");
    strcpy(command,LUA_BLOCK_ROTATE_LEFT);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that moves a block on the right
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::block_move_right(U61_Block *b)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_block=b;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"()");
    strcpy(command,LUA_BLOCK_MOVE_RIGHT);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the left
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::block_move_left(U61_Block *b)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_block=b;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"()");
    strcpy(command,LUA_BLOCK_MOVE_LEFT);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block down
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::block_move_down(U61_Block *b)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_block=b;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"()");
    strcpy(command,LUA_BLOCK_MOVE_DOWN);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that finds patterns in a map
 * and deletes the approprated blocks
 * this function should also update the score
 */
void U61_Script::map_match_pattern(U61_Map *m)
{
    CL_LuaObject func;
    char command[1000];
    char arguments[1000];

    secure_pointers();
    current_map=m;

    //func=CL_Lua::getglobal(LUA_NEW_BLOCK);
    //lua_callfunction(func);

    sprintf(arguments,"()");
    strcpy(command,LUA_MAP_MATCH_PATTERN);
    strcat(command,arguments);
    CL_Lua::dostring(command);
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it completely resets a block, ie after it the block has no squares left...
 */
static void block_reset()
{
    if (current_block!=NULL)
    {
        current_block->reset();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it centers a block, ie the square of coord 0,0 will be in the middle
 */
static void block_center()
{
    if (current_block!=NULL)
    {
        current_block->center();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the x position of the block
 */
static void block_get_x()
{
    CL_LuaValue *x;

    if (current_block!=NULL)
    {
        x=new CL_LuaValue(current_block->get_x());
        x->push();
        delete x;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the y position of the block
 */
static void block_get_y()
{
    CL_LuaValue *y;

    if (current_block!=NULL)
    {
        y=new CL_LuaValue(current_block->get_y());
        y->push();
        delete y;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x position of the block
 */
static void block_set_x()
{
    int x;

    if (current_block!=NULL)
    {
        x=CL_Lua::getparam(1);

        current_block->set_x(x);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the y position of the block
 */
static void block_set_y()
{
    int y;

    if (current_block!=NULL)
    {
        y=CL_Lua::getparam(1);

        current_block->set_y(y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds a block item (ie a square) to the current block
 */
static void block_add_item()
{
    int type,x,y;

    if (current_block!=NULL)
    {
        x=CL_Lua::getparam(1);        
        y=CL_Lua::getparam(2);        
        type=CL_Lua::getparam(3);        

        current_block->add_item(x,y,type);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of items (ie of squares) in a block
 */
static void block_get_nb_items()
{
    CL_LuaValue *nb;

    if (current_block!=NULL)
    {
        nb=new CL_LuaValue(current_block->get_nb_items());
        nb->push();
        delete nb;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the x coordinate of a block item (a square)
 */
static void block_get_item_x()
{
    CL_LuaValue *x;
    int i;

    if (current_block!=NULL)
    {
        i=CL_Lua::getparam(1);

        x=new CL_LuaValue(current_block->get_item_x(i));
        x->push();
        delete x;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the y coordinate of a block item (a square)
 */
static void block_get_item_y()
{
    CL_LuaValue *y;
    int i;

    if (current_block!=NULL)
    {
        i=CL_Lua::getparam(1);

        y=new CL_LuaValue(current_block->get_item_y(i));
        y->push();
        delete y;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the color of a block item (a square)
 */
static void block_get_item_color()
{
    CL_LuaValue *color;
    int i;

    if (current_block!=NULL)
    {
        i=CL_Lua::getparam(1);

        color=new CL_LuaValue(current_block->get_item_color(i));
        color->push();
        delete color;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static void block_set_item_x()
{
    int i,x;

    if (current_block!=NULL)
    {
        i=CL_Lua::getparam(1);
        x=CL_Lua::getparam(2);

        current_block->set_item_x(i,x);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static void block_set_item_y()
{
    int i,y;

    if (current_block!=NULL)
    {
        i=CL_Lua::getparam(1);
        y=CL_Lua::getparam(2);

        current_block->set_item_y(i,y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static void block_set_item_color()
{
    int i,color;

    if (current_block!=NULL)
    {
        i=CL_Lua::getparam(1);
        color=CL_Lua::getparam(2);

        current_block->set_item_color(i,color);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the width of the map
 */
static void map_get_width()
{
    CL_LuaValue *width;

    if (current_map!=NULL)
    {
        width=new CL_LuaValue(U61_Map::WIDTH);
        width->push();
        delete width;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the height of the map
 */
static void map_get_height()
{
    CL_LuaValue *height;

    if (current_map!=NULL)
    {
        height=new CL_LuaValue(U61_Map::HEIGHT);
        height->push();
        delete height;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it gets the color of a given square, color -1 means no square
 */
static void map_get_square_color()
{
    int x,y;
    CL_LuaValue *color;

    if (current_map!=NULL)
    {
        x=CL_Lua::getparam(1);
        y=CL_Lua::getparam(2);

        color=new CL_LuaValue(current_map->get_square_color(x,y));
        color->push();
        delete color;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the color of a given square, color -1 means no square
 */
static void map_set_square_color()
{
    int x,y,color;

    if (current_map!=NULL)
    {
        x=CL_Lua::getparam(1);
        y=CL_Lua::getparam(2);
        color=CL_Lua::getparam(3);

        current_map->set_square_color(x,y,color);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the score associated to a map
 */
static void map_get_score()
{
    CL_LuaValue *score;

    if (current_map!=NULL)
    {
        score=new CL_LuaValue(current_map->get_score());
        score->push();
        delete score;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the value of the socre associated to a map
 */
static void map_set_score()
{
    int score;

    if (current_map!=NULL)
    {
        score=CL_Lua::getparam(1);

        current_map->set_score(score);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * registers all C functions that can be used from Lua
 */
void U61_Script::record_all()
{
    CL_Lua::record(C_BLOCK_RESET,block_reset);
    CL_Lua::record(C_BLOCK_CENTER,block_center);
    CL_Lua::record(C_BLOCK_GET_X,block_get_x);
    CL_Lua::record(C_BLOCK_GET_Y,block_get_y);
    CL_Lua::record(C_BLOCK_SET_X,block_set_x);
    CL_Lua::record(C_BLOCK_SET_Y,block_set_y);
    CL_Lua::record(C_BLOCK_ADD_ITEM,block_add_item);
    CL_Lua::record(C_BLOCK_GET_NB_ITEMS,block_get_nb_items);
    CL_Lua::record(C_BLOCK_GET_ITEM_X,block_get_item_x);
    CL_Lua::record(C_BLOCK_GET_ITEM_Y,block_get_item_y);
    CL_Lua::record(C_BLOCK_GET_ITEM_COLOR,block_get_item_color);
    CL_Lua::record(C_BLOCK_SET_ITEM_X,block_set_item_x);
    CL_Lua::record(C_BLOCK_SET_ITEM_Y,block_set_item_y);
    CL_Lua::record(C_BLOCK_SET_ITEM_COLOR,block_set_item_color);
    CL_Lua::record(C_MAP_GET_WIDTH,map_get_width);
    CL_Lua::record(C_MAP_GET_HEIGHT,map_get_height);
    CL_Lua::record(C_MAP_GET_SQUARE_COLOR,map_get_square_color);
    CL_Lua::record(C_MAP_SET_SQUARE_COLOR,map_set_square_color);
    CL_Lua::record(C_MAP_GET_SCORE,map_get_score);
    CL_Lua::record(C_MAP_SET_SCORE,map_set_score);
}



