/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   block.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: this class is used to handle and draw blocks 
 *              a block is basically a set of squares put together
 *              it usually has a recognizable shape 
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "block.h"
#include "global.h"
#include "map.h"
#include "script.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default block 
 */ 
U61_Block::U61_Block()
{
    reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * frees the block 
 */ 
U61_Block::~U61_Block()
{
    
}

/*--------------------------------------------------------------------------*/
/*
 * sets the shape of a block, according to Lua script parameters
 */
void U61_Block::shape(int s)
{
    reset();
    U61_Script::block_shape(this,s);
    center();
}

/*--------------------------------------------------------------------------*/
/*
 * performs right rotation
 * the behavior of this function depends on Lua scripting parameters
 * it may or may not do a geometrical rotation...
 */
void U61_Block::rotate_right()
{
    U61_Script::block_rotate_right(this);
    center();
}

/*--------------------------------------------------------------------------*/
/*
 * performs left rotation
 * the behavior of this function depends on Lua scripting parameters
 * it may or may not do a geometrical rotation...
 */
void U61_Block::rotate_left()
{
    U61_Script::block_rotate_left(this);
    center();
}

/*--------------------------------------------------------------------------*/
/*
 * performs right move
 * the behavior of this function depends on Lua scripting parameters
 * it may or may not do a geometrical move...
 */
void U61_Block::move_right()
{
    U61_Script::block_move_right(this);
    center();
}

/*--------------------------------------------------------------------------*/
/*
 * performs left move
 * the behavior of this function depends on Lua scripting parameters
 * it may or may not do a geometrical move...
 */
void U61_Block::move_left()
{
    U61_Script::block_move_left(this);
    center();
}

/*--------------------------------------------------------------------------*/
/*
 * performs "move down" action
 * the behavior of this function depends on Lua scripting parameters
 * it may or may not do a geometrical translation...
 */
void U61_Block::move_down()
{
    U61_Script::block_move_down(this);
    center();
}

/*--------------------------------------------------------------------------*/
/*
 * empties a block from its squares 
 */
void U61_Block::reset()
{
    int i;

    for (i=0;i<MAX_ITEMS;++i)
    {
        items[i].reset();
    }
    nb_items=0;
    pos_x=0;
    pos_y=0;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the x position of a block
 */
int U61_Block::get_x()
{
    return pos_x;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the y position of a block
 */
int U61_Block::get_y()
{
    return pos_y;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the x position of a block
 */
void U61_Block::set_x(int x)
{
    pos_x=x;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the y position of a block
 */
void U61_Block::set_y(int y)
{
    pos_y=y;
}

/*--------------------------------------------------------------------------*/
/*
 * adds an item to the block (ie adds a square)
 * this function is normally called by Lua scripts
 */
void U61_Block::add_item(int x,int y,int color)
{
    U61_Square sq;

    if (nb_items<MAX_ITEMS)
    {
        items[nb_items].square.enable();
        items[nb_items].square.set_color(color);
        items[nb_items].set_x(x);
        items[nb_items].set_y(y);
        nb_items++;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of items in the block
 */
int U61_Block::get_nb_items()
{
    return nb_items;    
}

/*--------------------------------------------------------------------------*/
/*
 * returns the x coordinate of a square of the block
 * this function is normally called by Lua scripts
 */
int U61_Block::get_item_x(int i)
{
    return (i>=0 && i<nb_items) ? items[i].get_x() : 0;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the y coordinate of a square of the block
 * this function is normally called by Lua scripts
 */
int U61_Block::get_item_y(int i)
{
    return (i>=0 && i<nb_items) ? items[i].get_y() : 0;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the x coordinate of a square of the block
 * this function is normally called by Lua scripts
 */
int U61_Block::get_item_color(int i)
{
    return (i>=0 && i<nb_items) ? items[i].square.get_color() : 0;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the x value of a block square
 * this function is normally called by Lua scripts
 */
void U61_Block::set_item_x(int i,int x)
{
    if (i>=0 && i<nb_items)
    {
        items[i].set_x(x);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the x value of a block square
 * this function is normally called by Lua scripts
 */
void U61_Block::set_item_y(int i,int y)
{
    if (i>=0 && i<nb_items)
    {
        items[i].set_y(y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the x value of a block square
 * this function is normally called by Lua scripts
 */
void U61_Block::set_item_color(int i,int color)
{
    if (i>=0 && i<nb_items)
    {
        items[i].square.set_color(color);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns a pointer on a given block (used for display and internal
 * management)
 * most functions in this file are "secure" since checks are performed
 * on the index of the square because they are supposed to be called
 * from lua and we must prevent from crashing because of an luas script
 * error
 * still this one has no checking for it should be used only by C
 * functions where it's possible to assume the index is correct
 */
U61_BlockItem *U61_Block::get_item(int i)
{
    return items+i;
}

/*--------------------------------------------------------------------------*/
/*
 * centers the block
 * this function should be called after each potential changement on
 * the block. It "centers" the block so that the square which is at
 * x=0 and y=0 looks like it's in the center of the block
 */
void U61_Block::center()
{
    int i;
    int x_max;
    int y_max;
    int x_min;
    int y_min;
    int x_offset;
    int y_offset;
 
    /*
     * if there are no squares, there's nothing to do...
     */
    if (nb_items>0)
    {
        /*
         * initialization of min & max values
         */
        x_max=x_min=items[0].get_x();
        y_max=y_min=items[0].get_y();

        /*
         * we seek for greater or lower values
         */
        for (i=1;i<nb_items;++i)
        {
             x_max=max(x_max,items[i].get_x());    
             x_min=min(x_min,items[i].get_x());    
             y_max=max(y_max,items[i].get_y());    
             y_min=min(y_min,items[i].get_y());    
        }

        /*
         * we calculate the offsets
         */
        x_offset=(x_max+x_min+1)/2;
        y_offset=(y_max+y_min+1)/2;
        
        /*
         * we substract the offset from all the coordinates,
         * so that the block is centedred
         */
        for (i=0;i<nb_items;++i)
        {
            items[i].set_x(items[i].get_x()-x_offset);
            items[i].set_y(items[i].get_y()-y_offset);
        } 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * draws a block onto the screen
 */
void U61_Block::draw(int x,int y,int size)
{
    int i;
    int x2,y2;

    for (i=0;i<nb_items;++i)
    {
        x2=pos_x+items[i].get_x();
        y2=pos_y+items[i].get_y();
        if (x2>=0 && y2>=0 && x2<U61_Map::WIDTH && y2<U61_Map::HEIGHT)
        {
            x2=x+x2*U61_Global::data.square_w[size];
            y2=y+y2*U61_Global::data.square_h[size];
            items[i].square.draw(x2,y2,size);
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * draws the frame of a block onto the screen (used for previsions)
 */
void U61_Block::draw_prevision(int x,int y,int size)
{
    int i;
    int x2,y2;

    for (i=0;i<nb_items;++i)
    {
        x2=pos_x+items[i].get_x();
        y2=pos_y+items[i].get_y();
        if (x2>=0 && y2>=0 && x2<U61_Map::WIDTH && y2<U61_Map::HEIGHT)
        {
            x2=x+x2*U61_Global::data.square_w[size];
            y2=y+y2*U61_Global::data.square_h[size];
            items[i].square.draw_prevision(x2,y2,size);
        }
    }
}
