/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   map.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: I called this class map for I can't imagine a game without
 *              a "map" class 8) . seriously, a map is where the information
 *              about where the blocks are is stored. it's basically an array
 *              of squares.
 *              it also provides an API to move the player's falling block
 *              and handle any event.
 *              it's important to note that most of the behaviors of a map
 *              are determined by embedded LUA scripts
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "global.h"
#include "map.h"
#include "script.h"
#include "time.h"
#include "sound.h"
#include "utils.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * these arrays contain constants used for speeds, this way one can
 * have something more subtle than just a linear behavior
 */ 
const int U61_Map::speeds[MAX_SYSTEM_SPEED+1]=
    {
       2,  5, 10, 15, 20, 30, 40, 50, 60, 70, 80,
          85,90,95,100,105,110,115,120,125,130,
         135,140,145,150,155,160,170,180,190,200,
         210,220,230,240,250,260,270,280,290,300,
         310,320,330,340,350,360,370,380,390,400
    };
const int U61_Map::accels[MAX_SYSTEM_ACCEL+1]=
    {
       0,  2,  5, 10, 15, 20,
          30, 40, 60, 80,100,
         120,140,160,180,200
    };

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default map
 */ 
U61_Map::U61_Map()
{
    active=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a map
 */ 
U61_Map::~U61_Map()
{

}

/*--------------------------------------------------------------------------*/
/*
 * clears a map
 */
void U61_Map::clear()
{
    int i,j;

    for (j=0;j<HEIGHT;++j)
    {
        for (i=0;i<WIDTH;++i)
        {
            squares[i][j].disable();
            //squares[i][j].enable();
            squares[i][j].set_color((i+j)%8);
        }
    }
}


/*--------------------------------------------------------------------------*/
/*
 * begin functino, to call each time the player has lost 
 */
void U61_Map::begin()
{
    int i;

    auto_events.clear();
    
    clear();
 
    block.reset();
    block_requested=false;
    matching=false;
    speed_counter=0;
    accel_counter=0;
    speed=base_speed;
    accel=base_accel; 
    background=0;
    prevision_state=true;
    preview_state=true;
    match_count=0;
    if (score>best_score)
    {
        best_score=score;
    }
    score=0;
    for (i=0;i<NB_GLOBAL;++i)
    {
        global_val[i]=0;
    }        
    //block.shape(0);
}

/*--------------------------------------------------------------------------*/
/*
 * resets completely a map
 */
void U61_Map::reset(int s,int a)
{
    map_time=0;
    active=false;
    best_score=0;
    score=0;
    games=0; 
    base_speed=s;
    base_accel=a;
    silent=false;

    begin();
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the map is activated
 * by activated map, we mean a map where blocks are falling and someone
 * is actually playing. a player can have a disactivated map,in this
 * case u61 is waiting for him to press the start key and activate the map
 */
bool U61_Map::is_active()
{
    return active;
}

/*--------------------------------------------------------------------------*/
/*
 * activates a map
 */
void U61_Map::set_active(int time)
{
    //reset();
    map_time=time;
    active=true;
}

/*--------------------------------------------------------------------------*/
/*
 * disactivates a map
 */
void U61_Map::set_inactive()
{
    active=false;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the background of the map
 */
void U61_Map::set_background(int bg)
{
    background=bg;

    if (background<0)
    {
        background=0;
    }
    while (background>=U61_Global::data.nb_map)
    {
        background-=U61_Global::data.nb_map;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the background of the map
 */
int U61_Map::get_background()
{
    return background;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
void U61_Map::set_square_color(int x,int y,int color)
{
    if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
    {
        if (color>=0 && color<U61_Square::NB_COLORS)
        {
            squares[x][y].enable();
            squares[x][y].set_color(color);
        }
        else
        {
            squares[x][y].disable();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
int U61_Map::get_square_color(int x,int y)
{
    int color=-1;

    if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
    {
        if (squares[x][y].is_enabled())
        {
            color=squares[x][y].get_color();
        }
    }
    return color;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the score of the map, negative scores are forbidden
 */
void U61_Map::set_score(int s)
{
    if (s>=0)
    {
        score=s;
    }
    else
    {
        score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the score associated to the map
 */
int U61_Map::get_score()
{
    return score;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the best score associated to the map
 */
int U61_Map::get_best_score()
{
    return best_score;
}

/*--------------------------------------------------------------------------*/
/*
 * adds the given value to the current score
 */
void U61_Map::add_score(int s)
{
    score+=s;
    if (score<0)
    {
        score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the time associated to the map
 */
int U61_Map::get_time()
{
    return map_time;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of games associated to the map
 */
int U61_Map::get_nb_games()
{
    return games;
}

/*--------------------------------------------------------------------------*/
/*
 * returns a number which is used by the lua match function
 */
int U61_Map::get_match_count()
{
    return match_count;
}

/*--------------------------------------------------------------------------*/
/*
 * sets a global integer into the map
 * used for lua scripts to store global permanent values
 */
void U61_Map::set_global(int i, int glob)
{
    if (i>=0 && i<NB_GLOBAL)
    {
        global_val[i]=glob;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets a global integer from the map
 * used for lua scripts to store global permanent values
 */
int U61_Map::get_global(int i)
{
    int glob=0;

    if (i>=0 && i<NB_GLOBAL)
    {
        glob=global_val[i];
    }

    return glob;
}

/*--------------------------------------------------------------------------*/
/*
 * compute the map to the next instant
 */
void U61_Map::compute_next()
{
    //cout<<"Compute next\n";
    map_time++;
    if (!(map_time%TIME_CALLBACK_DELAY))
    {
        time_callback();
        if (matching)
        {
            match_pattern();
        }
    }
    update_speed();
}

/*--------------------------------------------------------------------------*/
/* 
 * updates the current speed
 */
void U61_Map::update_speed()
{
    check_speed();
    check_accel();

    speed_counter+=speeds[speed];
    if (speed_counter>SPEED_SCALE)
    {
        speed_counter-=SPEED_SCALE;
        block_move_down(); 
    }    
    accel_counter+=accels[accel];
    if (accel_counter>ACCEL_SCALE)
    {
        accel_counter-=ACCEL_SCALE;
        speed++;
    }    

    check_speed();
    check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * computes the level up to the givent time, no events are handled
 */
void U61_Map::compute_up_to(int time)
{
    if (active)
    {
        while (map_time<time)
        {
            compute_next();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles an event, ie does anything that has to be done for this event
 */
void U61_Map::handle_event(U61_Event event)
{
    if (active)
    {
        if (int(event.time)<map_time)
        {
             cout<<"Inconsitency in event dates detected event_time="
                 <<event.time<<" event_code="<<event.code<<" map_time="
                 <<map_time<<"\n";
        }  
        else
        {
            compute_up_to(event.time);
            switch (event.code)
            {
            case U61_Event::ROTATE_LEFT:
                block_rotate_left();
                break;
            case U61_Event::ROTATE_RIGHT:
                block_rotate_right();
                break;
            case U61_Event::MOVE_RIGHT:
                block_move_right();
                break;
            case U61_Event::MOVE_LEFT:
                block_move_left();
                break;
            case U61_Event::MOVE_DOWN:
                block_move_down();
                break;
            case U61_Event::DROP:
                block_drop();
                break;
            case U61_Event::NEW_BLOCK:
                new_block(event.par);
                break;
            case U61_Event::LOOSE:
                loose();
                break;
            }
        }
    }

    //cout<<"Event low level computed code="<<event.code<<"\n";
}

/*--------------------------------------------------------------------------*/
/*
 * anticipates the level to the given time, it's usefull to display the
 * "last" map. the computer just supposes no events have been generated
 * and anticipates the behavior of the map
 * but careful, since no events are computed, themap may become incoherent
 * it's not a problem for in U61 we keep 2 copies of the map, a sure and
 * an anticipated one, called "last"
 * BTW, this function deletes all events from the auto_events list
 * since they are of no use when the map has been anticipated and/or
 * we don't want the list to grow and explose system memory...
 */
void U61_Map::anticipate(int time)
{
    compute_up_to(time);
    auto_events.clear();
}

/*--------------------------------------------------------------------------*/
/*
 * polls some auto generated events, such as requests for new blocks
 */
void U61_Map::poll_auto_events()
{
    U61_Event event;

    if (active)
    {
        if ((!is_block_active()) && (!block_requested) && (!matching))
        {
            block_requested=true;
            event.code=U61_Event::NEW_BLOCK;
            event.par=U61_Utils::random(U61_Block::MAX_SHAPE);
            put_auto_event(event);
            //cout<<"new block event sent\n";
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next auto generated event
 */
U61_Event U61_Map::get_auto_event()
{
    U61_Event event;

    event=auto_events.front();
    auto_events.pop_front();

    return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event
 */
bool U61_Map::exists_auto_event()
{
    return !auto_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the generated event queue
 */
void U61_Map::put_auto_event(U61_Event event)
{
    event.time=U61_Time::for_event();
    auto_events.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map onto the screen
 */
void U61_Map::draw(int x,int y, int size, bool prev)
{
    int i,j;
    int x_square_offset;
    int y_square_offset;

    if (active)
    {
        x_square_offset=
            (U61_Global::data.map_w[size]
            -WIDTH*U61_Global::data.square_w[size])/2;
        y_square_offset=
            (U61_Global::data.map_h[size]
            -HEIGHT*U61_Global::data.square_h[size])/2;
        
        x_square_offset+=x;
        y_square_offset+=y;

        U61_Global::data.map[background][size]->put_screen(x,y);

        for (j=0;j<HEIGHT;++j)
        {
            for (i=0;i<WIDTH;++i)
            {
                squares[i][j].draw(x_square_offset
                                   +i*U61_Global::data.square_w[size],
                                   y_square_offset
                                   +j*U61_Global::data.square_h[size],
                                   size);
            }
        }
        if (prevision_state && prev)
        {
            draw_prevision(x_square_offset,y_square_offset,size);
        }
        block.draw(x_square_offset,y_square_offset,size);
    }
//    cout<<"draw square\n";
}

/*--------------------------------------------------------------------------*/
/*
 * sets the map in a mute state (does not generate sounds any more)
 */
void U61_Map::mute()
{
    silent=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the prevision state
 */
void U61_Map::set_prevision_state(bool state)
{
    prevision_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the prevision state
 */
bool U61_Map::get_prevision_state()
{
    return prevision_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the preview state
 */
void U61_Map::set_preview_state(bool state)
{
    preview_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the preview state
 */
bool U61_Map::get_preview_state()
{
    return preview_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the speed 
 */
void U61_Map::set_speed(int s)
{
    speed=s; 
    check_speed();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the speed 
 */
int U61_Map::get_speed()
{
    return speed; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the speed is within the correct range
 */
void U61_Map::check_speed()
{
    if (speed>MAX_SYSTEM_SPEED)
    {
        speed=MAX_SYSTEM_SPEED;
    }
    if (speed<0)
    {
        speed=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the accel
 */
void U61_Map::set_accel(int a)
{
    accel=a; 
    check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the accel
 */
int U61_Map::get_accel()
{
    return accel; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the accel is within the correct range
 */
void U61_Map::check_accel()
{
    if (accel>MAX_SYSTEM_ACCEL)
    {
        accel=MAX_SYSTEM_ACCEL;
    }
    if (accel<0)
    {
        accel=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the nest block 
 */
U61_Block U61_Map::get_next_block()
{
    return next_block; 
}

/*--------------------------------------------------------------------------*/
/*
 * display the prevision, the prevision is "where the block will land if
 * nothing happens" ie where it will land if I press drop  
 */
void U61_Map::draw_prevision(int x,int y,int size)
{
    U61_Block test;
    U61_Block prev;

    if (block.get_nb_items()>0)
    {
        test=block;
    
        while (is_block_ok(&test))
        {
            prev=test;
            test.move_down();
        }

        U61_Script::block_land(this,&prev);
        prev.draw_prevision(x,y,size); 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_left()
{
    //cout<<"rotate left\n";
    U61_Block test;

    test=block;
    test.rotate_left();
    if (is_block_ok(&test))
    {
        block=test;
    }
    else
    {
        test=block;
        test.move_left();
        test.rotate_left(); 
        if (is_block_ok(&test))
        {
            block=test;
        }
        else
        {
            test=block;
            test.move_right();
            test.rotate_left(); 
            if (is_block_ok(&test))
            {
                block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_right()
{
    //cout<<"rotate right\n";
    U61_Block test;

    test=block;
    test.rotate_right();
    if (is_block_ok(&test))
    {
        block=test;
    }
    else
    {
        test=block;
        test.move_right();
        test.rotate_right(); 
        if (is_block_ok(&test))
        {
            block=test;
        }
        else
        {
            test=block;
            test.move_left();
            test.rotate_right(); 
            if (is_block_ok(&test))
            {
                block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_left()
{
    //cout<<"move left\n";
    U61_Block test;

    test=block;
    test.move_left();
    if (is_block_ok(&test))
    {
        block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_right()
{
    //cout<<"move right\n";
    U61_Block test;

    test=block;
    test.move_right();
    if (is_block_ok(&test))
    {
        block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block down, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_down()
{
    //cout<<"move down\n";
    U61_Block test;

    test=block;
    test.move_down();
    if (is_block_ok(&test))
    {
        block=test;
    }
    else
    {
        stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * performs calls to move_down as much as possible so that the block lands
 */
void U61_Map::block_drop()
{
    //cout<<"drop\n";
    U61_Block test;

    test=block;
    if (block.get_nb_items()>0)
    {
        while (is_block_ok(&test))
        {
            block=test;
            test.move_down();
        }
    
        stabilize_block();
    }
}


/*--------------------------------------------------------------------------*/
/*
 * checks if there's an active block (size greater than 0)
 */
bool U61_Map::is_block_active()
{
    return block.get_nb_items()>0;
}

/*--------------------------------------------------------------------------*/
/*
 * checks if a position is possible, if yes, returns true
 * this function is used to check if the result of functions such
 * block_rotate_right or block_move_down are incoherent, and therefore
 * should be cancelled
 */
bool U61_Map::is_block_ok(U61_Block *test)
{
    int i,n,x,y;
    bool ok=true;

    n=test->get_nb_items();
    for (i=0;i<n && ok;++i)
    {
        x=test->get_x()+test->get_item_x(i);
        y=test->get_y()+test->get_item_y(i);
        /*
         * the square must be in the map, or over the map,
         * if the block is brand new and didn't show himself yet
         */
        if (x>=0 && x<WIDTH && y<HEIGHT)
        {
            /*
             * if y>=0, there could potentially be a stable square
             * at this location, so we check that
             */
            if (y>=0)
            {
                if (squares[x][y].is_enabled()) 
                {
                    ok=false;
                }
            }
        }
        else
        {
            ok=false;
        }
    }

    return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * gets a new block, the shape of the new block is calculated with
 * Lua user defined functions
 */
void U61_Map::new_block(int shape)
{
    block=next_block;
    block.set_x(WIDTH/2);
    block_requested=false;

    next_block.shape(shape);

    //cout<<"new block given\n";
}

/*--------------------------------------------------------------------------*/
/*
 * this function transfoem the moving block into a stable block,
 * ie it adds it to the current map
 */
void U61_Map::stabilize_block()
{
    U61_Event evt;
    int x,y,i,n;
    bool outside=false;

    U61_Script::block_land(this,&block);
    n=block.get_nb_items();
    for (i=0;i<n;++i)
    {
        x=block.get_x()+block.get_item_x(i);
        y=block.get_y()+block.get_item_y(i);
        if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
        {
            squares[x][y]=block.get_item(i)->square;
        }
        else
        {
            outside=true;
        }
    }
    if (outside)
    {
        evt.code=U61_Event::LOOSE;
        put_auto_event(evt);
    }
    else
    {
        match_count=0;
        matching=true;
    }

    block.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * what has to be done when the player looses
 */
void U61_Map::loose()
{
    //cout<<"The game is lost\n";
    begin();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function to match patterns
 * pattern match can be find a line and delete it, or find some colors
 * together, well, it can be pretty much anything...
 * it is usually quite a complex function
 * it also updates the score
 */
void U61_Map::match_pattern()
{
    if (matching)
    {
        if ((U61_Script::map_match_pattern(this)))
        {
            if (!silent)
            {
                U61_Sound::play_block_pattern();
            }
            match_count++;
        }
        else
        {
            if (match_count==0 && !silent)
            {
                U61_Sound::play_block_touch();
            }
            matching=false;
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback()
{
    U61_Script::map_time_callback(this,&block);
}
