/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   menu.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: abstract class, parent of all menus. it maintains the menu
 *              state (the selected item) and provides functions to center
 *              it and handle menuitems in a rational way
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "menu.h"
#include "time.h"
#include "sound.h"
#include "global.h"
#include "menuinput.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * messages displayed at the bottom of the screen in the menus
 */
char *U61_Menu::bottom_message[BOTTOM_MESSAGE_NB]=
{
    "U61, another block-game by U-Foot",
    "Copyright (C) 2000 Christian Mauduit",
    "Home page : http://www.ufoot.org/u61",
    "Contact author : ufoot@ufoot.org",
    "Use the arrows to navigate through the menus",
    "Press escape to return to the previous menu",
    "Game rules can be changed easily by writing new lua scripts, try it!",
    "This is only an alpha version...",
    "U61 uses ClanLib http://clanlib.org",
    "Graphics by U-Woman http://www.adlerhome.com"
};

/*--------------------------------------------------------------------------*/
/*
 * used to display the messages at the bottom of the screen as a moving banner
 */
int U61_Menu::current_bottom_message=0;
int U61_Menu::bottom_message_start_time=0;
int U61_Menu::bottom_message_width=-1000000;

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of an empty menu
 */ 
U61_Menu::U61_Menu(U61_Menu *parent,char *top)
{
    parent_menu=parent;
    top_message=top;
    message_active=true;
}

/*--------------------------------------------------------------------------*/
/* 
 * frees the menu,one must free all the items
 */ 
U61_Menu::~U61_Menu()
{
    int i;
    int nb_item;

    nb_item=items.size();
    for (i=0;i<nb_item;++i)
    {
        delete items[i];
    }
}

/*--------------------------------------------------------------------------*/
/*
 * performs an action, since a key press has been recorded
 */
U61_Menu *U61_Menu::action(int key)
{
    U61_Menu *next_menu;
    int old_selection;

    next_menu=this;
    old_selection=selection;

    if (!(items[selection]->is_modal()))
    {
        switch (key)
        {
        case U61_MenuInput::KEY_UP:
            select_up();
            break;
        case U61_MenuInput::KEY_DOWN:
            select_down();
            break;
        case U61_MenuInput::KEY_ESCAPE:
            next_menu=back();
            break;
        case U61_MenuInput::KEY_ENTER:
            next_menu=validate();
            break;
        }
    }

    if (items[selection]->action(key))
    {
        U61_Sound::play_menu_move();
    }

    items[selection]->update();
    items[selection]->select();

    if (next_menu!=this)
    {
        U61_Sound::play_menu_validate();
    }
    else
    {
        if (old_selection!=selection)
        {
            U61_Sound::play_menu_move();
        }
    }

    return next_menu;
}

/*--------------------------------------------------------------------------*/
/*
 * validation of the menu (ie click on ENTER)
 */
U61_Menu *U61_Menu::validate()
{
    return this;
}

/*--------------------------------------------------------------------------*/
/*
 * fired when the user presses escape to go back in the menu history
 */
U61_Menu *U61_Menu::back()
{
    return get_parent();
}

/*--------------------------------------------------------------------------*/
/*
 * draws the current menu
 */
void U61_Menu::draw()
{
    int y;
    int i;
    int nb_item;
    int last_item_height2;
    int item_height;

    y=U61_Global::data.screen_h/2;

    y-=get_height()/2;
    last_item_height2=0;

    nb_item=items.size();
    for (i=0;i<nb_item;++i)
    {
        item_height=items[i]->get_height();

        y+=last_item_height2+item_height/2;
        items[i]->draw(y);
        last_item_height2=item_height-item_height/2;
    }

    if (message_active)
    {
        draw_top_message();
        draw_bottom_message();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the height of the menu
 */
int U61_Menu::get_height()
{
    int i;
    int nb_item;
    int height=0;

    nb_item=items.size();
    for (i=0;i<nb_item;++i)
    {
        height+=items[i]->get_height();
    }
    return height;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the width of the menu
 */
int U61_Menu::get_width()
{
    int i;
    int nb_item;
    int width=0;

    nb_item=items.size();
    for (i=0;i<nb_item;++i)
    {
        width=max(items[i]->get_width(),width);
    }
    return width;
}

/*--------------------------------------------------------------------------*/
/*
 * selects the given item
 */
void U61_Menu::select(int num_item)
{
    int nb_item;
    int i;

    nb_item=items.size();
    for (i=0;i<nb_item;++i)
    {
        if (i==num_item)
        {
            items[i]->select();
        }
        else
        {
            items[i]->unselect();
        }
    }
    selection=num_item;
}

/*--------------------------------------------------------------------------*/
/*
 * selects the previous item
 */
void U61_Menu::select_up()
{
    if (selection>0)
    {
        select(selection-1);
    }
    else
    {
        select(items.size()-1);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * selects the next item
 */
void U61_Menu::select_down()
{
    if (selection<int(items.size())-1)
    {
        select(selection+1);
    }
    else
    {
        select(0);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the parent menu
 */
U61_Menu *U61_Menu::get_parent()
{
    return parent_menu;
}

/*--------------------------------------------------------------------------*/
/*
 * adds an item to the menu
 * be careful to use add_item(new U61_WhateverMenuItemClassYouWant());
 * for items are deleted when the menu is deleted itself
 */
void U61_Menu::add_item(U61_MenuItem *item)
{
    items.push_back(item);
}

/*--------------------------------------------------------------------------*/
/*
 * draws the top message only
 */
void U61_Menu::draw_top_message()
{
    int x,y;

    x=(U61_Global::data.screen_w-
       U61_Global::data.font_menu->get_text_width(top_message))/2;
    y=0;

    U61_Global::data.font_menu->print_left(x,y,top_message);
}

/*--------------------------------------------------------------------------*/
/*
 * draws the bottom message only
 */
void U61_Menu::draw_bottom_message()
{
    int x,y=0,dx;

    dx=(U61_Time::for_effect()-bottom_message_start_time)
        /BOTTOM_MESSAGE_DELAY;
    if (dx>=U61_Global::data.screen_w+bottom_message_width)
    {
        init_bottom_message(); 
        dx=0;
    }

    x=U61_Global::data.screen_w-dx;
    y=U61_Global::data.screen_h-U61_Global::data.font_menu->get_height();

    U61_Global::data.font_menu->print_left
        (x,y,bottom_message[current_bottom_message]);
}

/*--------------------------------------------------------------------------*/
/*
 * inits the bottom message
 */
void U61_Menu::init_bottom_message()
{
    current_bottom_message++;
    if (current_bottom_message>=BOTTOM_MESSAGE_NB)
    {
        current_bottom_message=0;
    }

    bottom_message_start_time=U61_Time::for_effect();

    bottom_message_width=U61_Global::data.font_menu->get_text_width
        (bottom_message[current_bottom_message]);
}

