/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   player.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: a player is a bundle of
 *              - an event source
 *              - an event copy
 *              - a "sure" map
 *              - a "last" map
 *              - a API to handle all that...
 *              it's fundamental to understand the difference between the
 *              last map and the sure map. 
 *              the sure map is a map freezed at the time of the latest
 *              event received. since events are ordered by time, we can
 *              assume this is the right map at this givent instant
 *              the last map is an anticipated map. from the suremap state,
 *              it simulates game cycles so that the map is "as it would have
 *              been now (which may be later than the latest event) if no
 *              event occured". This way, in a network game, map never freeze
 *              waiting for network events, and this gives an impression of
 *              speed even with poor bandwidth
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <string.h>

#include "player.h"
#include "time.h"
#include "localsource.h"
#include "global.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default player
 */ 
U61_Player::U61_Player()
{
    source=NULL;
    copy=NULL;

    reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a player
 */ 
U61_Player::~U61_Player()
{

}

/*--------------------------------------------------------------------------*/
/*
 * resets a player to default
 */
void U61_Player::reset()
{
    available=false;
    if (source!=NULL)
    {
        delete source;
        source=NULL;
    }
    if (copy!=NULL)
    {
        delete copy;
        copy=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the id of a player
 */
void U61_Player::set_id(int player_id)
{
    id=player_id;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the id of a player
 */
int U61_Player::get_id()
{
    return id;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of a player
 */
char *U61_Player::get_name()
{
    return name;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the score of a player
 */
int U61_Player::get_score()
{
    return last.get_score();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the best score of a player
 */
int U61_Player::get_best_score()
{
    return sure.get_best_score();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next block of a player
 */
U61_Block U61_Player::get_next_block()
{
    return last.get_next_block();
}

/*--------------------------------------------------------------------------*/
/*
 * sets the player as a local player
 */
void U61_Player::init_local(U61_Input *input,int time,
                           U61_PlayerConfig *conf,
                           bool start)
{
    U61_LocalSource *keybd; 

    reset();
    sure.reset(U61_Global::config.initial_speed,
               U61_Global::config.acceleration);
    copy_sure_to_last();
    name=conf->name;
    prevision=&(conf->prevision);
    //cout<<"init local with "<<name<<"\n";

    keybd=new U61_LocalSource(input,&sure);
    if (start)
    {
        keybd->send_start_stop(time);
    }
    source=keybd;
    available=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the player as a network player
 */
void U61_Player::init_network()
{
    reset();
    //available=true;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is available
 */
bool U61_Player::is_available()
{
    return available;
}

/*--------------------------------------------------------------------------*/
/*
 * this function pumps the events for a player, ie it:
 * read the events
 * interprets them
 * makes a copy of them if necessary
 */
void U61_Player::pump_events()
{
    bool changed=false;

    source->poll();

    while((!source->empty()) && 
        ((int)(source->check().time))<=U61_Time::for_logic())
    {
        changed=true;
        U61_Player::handle_event(source->get());
    }

    if (changed)
    {
        copy_sure_to_last();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * pumps an event
 */
void U61_Player::handle_event(U61_Event evt)
{
//    cout<<"Event pumped code="<<evt.code<<"\n";
    if (sure.is_active())
    {
        if (evt.code==U61_Event::START_STOP)
        {
	    stop();
        }
        else
        {
            sure.handle_event(evt);
        }
    }
    else
    {
        if (evt.code==U61_Event::START_STOP)
        {
            start(evt.time);
        } 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * called to start a player, ie enter him in the game
 */
void U61_Player::start(int time)
{
    sure.set_active(time);
    copy_sure_to_last();
    //cout<<"Player "<<id<<" started\n";
}

/*--------------------------------------------------------------------------*/
/*
 * called to stop a player, ie remove him from the game
 */
void U61_Player::stop()
{
    sure.set_inactive();
    copy_sure_to_last();
    //cout<<"Player "<<id<<" stopped\n";
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map & infos of the player
 */
void U61_Player::draw(int x,int y,int size)
{
    last.anticipate(U61_Time::for_event());
    //last.draw(10,10);
    last.draw(x,y,size,*prevision);
}

/*--------------------------------------------------------------------------*/
/*
 * copies the sure map to the last map
 */
void U61_Player::copy_sure_to_last()
{
    last=sure;
    last.mute();
}
