/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   protocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: contains the network code which is needed to initialize the
 *              maps on the client. this can be quite a complex process
 *              since we can not interrupt the game when someone logs in.
 *              and the amount of data to transmit might be high 
 *              (especially if the lua scripts are big)
 *              So we use lots of small messages and this forms what
 *              I call the "login protocol" and is maybe the most complex
 *              thing in U61...
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "protocol.h"
#include "packet.h"
#include "version.h"
#include "const.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a protocol
 */
U61_Protocol::U61_Protocol(int p,int c)
{
  port=p;
  channel=c;
  game=NULL;
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * deletes a protocol
 */
U61_Protocol::~U61_Protocol()
{
  if (game!=NULL)
    {
      delete game; 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet (to be defined in a sub-class)
 */
int U61_Protocol::send_next_packet()
{
  stage=PLAY;
  return LATER;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet (to be defined in a sub-class)
 */
int U61_Protocol::recv_next_packet()
{
  stage=PLAY;
  return LATER;
}

/*--------------------------------------------------------------------------*/
/*
 * sends all the pending packets
 */
int U61_Protocol::send_all()
{
  int result;

  while ((result=send_next_packet())==DONE);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives all the pending packets
 */
int U61_Protocol::recv_all()
{
  int result;

  while ((result=recv_next_packet())==DONE);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * resets the protocol 
 */
void U61_Protocol::reset()
{
  stage=BEGIN;
  step=0;
  diagnostic="Everything's fine";
}

/*--------------------------------------------------------------------------*/
/*
 * returns true is the protocol is ended and successful
 */
bool U61_Protocol::done()
{
  return (stage==END);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the current stage of the protocol (the higher, the more advanded
 */
int U61_Protocol::get_stage()
{
  return stage;
}

/*--------------------------------------------------------------------------*/
/*
 * sends the ID for game authentification
 * this is in theory useless since ClanLib does this checking but well,
 * I just enjoy checking 8-)
 */
int U61_Protocol::send_id(U61_Connection *conn)
{
  int result=LATER;
  U61_Packet p;

  // cout<<"Sending ID\n";

  p.size=strlen(GAME_ID)+1;
  strncpy(p.data,GAME_ID,U61_Packet::MAX_DATA-1);

  if (conn->send(&p))
    {
      result=DONE;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the game ID (goes with send_id...)
 */
int U61_Protocol::recv_id(U61_Connection *conn)
{
  int result=LATER;
  U61_Packet p;

  // cout<<"Receiving ID\n";

  if (conn->peek())
    {
      if (conn->recv(&p))
	{
          if (strcmp(GAME_ID,p.data)==0)
	    {
	      result=DONE;
	    }
          else
	    {
              diagnostic="Wrong game id, expecting \"";
	      diagnostic+=GAME_ID;
	      diagnostic+="\" and received \"";
	      diagnostic+=p.data;
	      diagnostic+="\"";
	      result=FAILED;
	    }
	  cout<<"Received ID "<<p.data<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the game version. We need to check if the exes are the same on
 * all the machines, for game behavior might be different. Of course
 * This is easy to hack but we assume that "cracking" u61 is not so
 * thrilling...
 */
int U61_Protocol::send_version(U61_Connection *conn)
{
  int result=LATER;
  U61_Packet p;
  char version[U61_STRING_SIZE];

  // cout<<"Sending version\n";

  U61_Version::get_version_text(version);
  p.size=strlen(version)+1;
  strncpy(p.data,version,U61_Packet::MAX_DATA-1);

  if (conn->send(&p))
    {
      result=DONE;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the game verison (goes with send_version...)
 */
int U61_Protocol::recv_version(U61_Connection *conn)
{
  int result=LATER;
  U61_Packet p;
  char version[U61_STRING_SIZE];

  //  cout<<"Receiving version\n";

  U61_Version::get_version_text(version);
  if (conn->peek())
    {
      if (conn->recv(&p))
	{
          if (strcmp(version,p.data)==0)
	    {
	      result=DONE;
	    }
          else
	    {
              diagnostic="Wrong game version, expecting \"";
	      diagnostic+=version;
	      diagnostic+="\" and received \"";
	      diagnostic+=p.data;
	      diagnostic+="\"";
	      result=FAILED;
	    }
	  cout<<"Received version "<<p.data<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Prints the diagnostic of the last network error 
 */
void U61_Protocol::print_diagnostic()
{
  cout<<"Network error : "<<diagnostic<<"\n";
}

