/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   clientprotocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the client sends informations about him and retrieves
 *              information from the server, including the rules and
 *              the state of all the other players
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "clientprotocol.h"
#include "global.h"
#include "time.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a client protocol
 */
U61_ClientProtocol::U61_ClientProtocol(U61_ClientDispatcher *d, 
				       int p,int ch,
                                       char *id) : U61_Protocol(p,ch)
{
  dispatcher=d;
  server_id=id;
  connection=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the protocol object
 */
U61_ClientProtocol::~U61_ClientProtocol()
{
  if (connection!=NULL)
    {
      delete connection;
      connection=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if the server is still there...
 */
void U61_ClientProtocol::check_connection()
{
  int i;
  U61_Player *p;

  if (stage!=CLOSED && game!=NULL && game->receive_game_closed())
    {
      stage=CLOSED;
      //cout<<"stage=CLOSED\n";
      connection->close();

      /*
       * We kill all the remote players and let the game keep going locally
       */
      for (i=0;i<U61_Dispatcher::MAX_PLAYER_ID;++i)
	{
	  p=U61_Global::game.find_player_by_id(i);
          if (p!=NULL && p->is_available() && !p->is_local())
	    {
	      p->kill();
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to send to the server
 */
int U61_ClientProtocol::send_next_packet()
{
  int result=LATER;

  switch(stage)
    {
    case BEGIN:
      if ((result=start_searching())==DONE)
	{
	  stage=SEARCHING;
          //cout<<"stage=SEARCHING\n";
	}
      break;
    case CONNECTED_TO_SERVER:
      if ((result=join_delay())==DONE)
	{
	  stage=JOIN_DELAY_ELAPSED;
	  //cout<<"stage=JOIN_DELAY_ELAPSED\n";
	}
      break;
    case CHANNEL_FOUND:
      if ((result=send_id(connection))==DONE)
	{
	  stage=ID_SENT;
          //cout<<"stage=ID_SENT\n";
	}
      break;
    case ID_RECEIVED:
      if ((result=send_version(connection))==DONE)
	{
	  stage=VERSION_SENT;
          //cout<<"stage=VERSION_SENT\n";
	}
      break;
    case TIME_RECEIVED:
      if ((result=send_ready(connection))==DONE)
	{
	  stage=READY_SENT;
          //cout<<"stage=READY_SENT\n";
	}
      break;
    case READY_RECEIVED:
      start_game();
      dispatcher->open(connection);
      
      stage=PLAYING;
      //cout<<"stage=PLAYING\n";
      
      break;
    }
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to receive from the
 * server
 */
int U61_ClientProtocol::recv_next_packet()
{
  int result=LATER;

  switch(stage)
    {
    case SEARCHING:
      if ((result=connect_to_server())==DONE)
	{
	  stage=CONNECTED_TO_SERVER;
          //cout<<"stage=CONNECTED_TO_SERVER\n";
	}
      break;
    case JOIN_DELAY_ELAPSED:
      if ((result=find_channel())==DONE)
	{
	  stage=CHANNEL_FOUND;
	  //cout<<"stage=CHANNEL_FOUND\n";
	}
      break;
    case ID_SENT:
      if ((result=recv_id(connection))==DONE)
	{  
          stage=ID_RECEIVED;
          //cout<<"stage=ID_RECEIVED\n";
	}
      break;
    case VERSION_SENT:
      if ((result=recv_version(connection))==DONE)
	{
	  stage=VERSION_RECEIVED;
          //cout<<"stage=VERSION_RECEIVED\n";
	}
      break;
    case VERSION_RECEIVED:
      if ((result=recv_script_name())==DONE)
	{
	  stage=SCRIPT_NAME_RECEIVED;
	  //cout<<"stage=SCRIPT_NAME_RECEIVED\n";
	}
      break;
    case SCRIPT_NAME_RECEIVED:
      if ((result=recv_script())==DONE)
	{
	  stage=SCRIPT_RECEIVED;
	  //cout<<"stage=SCRIPT_RECEIVED\n";
	}
      break;
    case SCRIPT_RECEIVED:
      if ((result=recv_id0())==DONE)
	{
	  stage=ID0_RECEIVED;
	  //cout<<"stage=ID0_RECEIVED\n";
	}
      break;
    case ID0_RECEIVED:
      if ((result=recv_id1())==DONE)
	{
	  stage=ID1_RECEIVED;
	  //cout<<"stage=ID1_RECEIVED\n";
	}
      break;
    case ID1_RECEIVED:
      if ((result=recv_initial_speed())==DONE)
	{
	  stage=INITIAL_SPEED_RECEIVED;
	  //cout<<"stage=INITIAL_SPEED_RECEIVED\n";
	}
      break;
    case INITIAL_SPEED_RECEIVED:
      if ((result=recv_acceleration())==DONE)
	{
	  stage=ACCELERATION_RECEIVED;
	  //cout<<"stage=ACCELERATION_RECEIVED\n";
	}
      break;
    case ACCELERATION_RECEIVED:
      if ((result=recv_time())==DONE)
	{
	  stage=TIME_RECEIVED;
	  //cout<<"stage=TIME_RECEIVED\n";
	}
      break;
    case READY_SENT:
      if ((result=recv_ready(connection))==DONE)
	{
	  stage=READY_RECEIVED;
          //cout<<"stage=READY_RECEIVED\n";
	}
      break;
    }
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * enter the server search state
 */
int U61_ClientProtocol::start_searching()
{
  int result=LATER;

  try 
    {
      CL_Network::find_game_at(GAME_ID,server_id,port);
      result=DONE;
    }
  catch (CL_Error e)
    {
      cout<<"Find error : "<<e.message<<"\n";
    }

  //  cout<<"start searching\n";

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * tries to find out if a game has been found on the server
 * and obtain a connection from this server
 */
int U61_ClientProtocol::connect_to_server()
{
  int result=LATER;

  try 
    {
      if (CL_Network::peek_game_found())
	{
	  game=CL_Network::receive_game_found(TIMEOUT);
	  if (game!=NULL)
	    {
              /*
               * OK, a game has been found
               */
              join_delay_start=U61_Time::for_effect();
	      result=DONE;
	    }
          else
            {
              cout<<"...\n";
	    }
	}
    }
  catch (CL_Error e)
    {
      cout<<"Connect error : "<<e.message<<"\n";
    }
  //cout<<"find server\n";

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * This function just waits one second before trying to send stuff
 * to the server, this is for ClanLib to find the time to give
 * network access on the right channel
 */
int U61_ClientProtocol::join_delay()
{
  int result=LATER;

  if (U61_Time::for_effect()>join_delay_start+JOIN_DELAY)
    {
      channel_to_test=0;
      result=DONE;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * So now we search a free channel in order to open a connection object
 */
int U61_ClientProtocol::find_channel()
{
  int result=LATER;
  const CL_NetComputer *server;

  if (game->is_writable(channel_to_test) 
      && game->is_readable(channel_to_test))
    {
      server=game->get_server();
      connection=new U61_Connection(game,(CL_NetComputer *) server,
                                    channel_to_test,false);
      result=DONE;
    }
  else
    {
      channel_to_test++;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the script name, this value will be used to create
 * a filename for the server remote script on the client
 */
int U61_ClientProtocol::recv_script_name()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(script_name,sizeof(script_name)-1))
	{
	  result=DONE;
          // cout<<"script_name=\""<<script_name<<"\"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the script itself, this can be a long packet,
 * I have no idea wether ClanLib handles this correctly yet (I mean that
 * I fear this transmission could "block" the game for a while)
 */
int U61_ClientProtocol::recv_script()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(script,sizeof(script)-1))
	{
	  result=DONE;
          // cout<<"script=\""<<script<<"\"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the ID that will be given to the first local player
 */
int U61_ClientProtocol::recv_id0()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(&id0))
	{
	  result=DONE;
	  //cout<<"id0="<<id0<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the ID that will be given to the second local player
 */
int U61_ClientProtocol::recv_id1()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(&id1))
	{
	  result=DONE;
	  //cout<<"id1="<<id1<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the initial speed that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_initial_speed()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(&initial_speed))
	{
	  result=DONE;
	  //cout<<"initial_speed="<<initial_speed<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the acceleration that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_acceleration()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(&acceleration))
	{
	  result=DONE;
	  //cout<<"acceleration="<<acceleration<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the time that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_time()
{
  int result=LATER;

  if (connection->peek())
    {
      if (connection->recv(&time))
	{
	  result=DONE;
	  //cout<<"time="<<time<<"\n";
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Starts the game (at last!)
 */
void U61_ClientProtocol::start_game()
{
  U61_Global::game.do_network_join(script,
				   id0,id1,
				   initial_speed,acceleration,
				   time);
}








