/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   serverprotocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: welcomes a client, gets basic information about him
 *              and sends him all the informations about
 *              the players which are already playing
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "serverprotocol.h"
#include "global.h"
#include "time.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a server protocol
 */
U61_ServerProtocol::U61_ServerProtocol(U61_ServerDispatcher *d, int p,int ch)
  : U61_Protocol(p,ch)
{
  int i;

  dispatcher=d;
  new_connection=0;

  for (i=0;i<U61_Game::NB_NET_GROUPS-1;++i)
    {
      connection[i]=NULL;
      id0[i]=-1;
      id1[i]=-1;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the protocol object
 */
U61_ServerProtocol::~U61_ServerProtocol()
{
  int i;

  for (i=0;i<U61_Game::NB_NET_GROUPS-1;++i)
    {
      if (connection[i]!=NULL)
	{
	  delete connection[i];
          connection[i]=NULL;
	}
    }
    
  if (game!=NULL)
    {
      delete game;
      game=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if there are some dead connections, and kills the players
 * if necessary
 */
void U61_ServerProtocol::check_connection()
{
  int i;
  U61_Player *p;
  CL_NetComputer *client;
  U61_Event evt;  

  if (game!=NULL && (client=game->receive_computer_leave())!=NULL)
    {
      for (i=0;i<U61_Game::NB_NET_GROUPS-1;++i)
	{
	  if (connection[i]!=NULL
	      && connection[i]->is_associated(client))
	    {
	      if (dispatcher!=NULL)
		{
		  dispatcher->close(connection[i]);
		}
	      delete connection[i];

	      evt.code=U61_Event::KILL;
	      evt.time=0;
	      evt.author=evt.target=id0[i];
	      dispatcher->put_out(evt);   
	      evt.author=evt.target=id1[i];
	      dispatcher->put_out(evt);

              p=U61_Global::game.find_player_by_id(id0[i]);
              if (p!=NULL)
		{
                  p->kill();
		}
              p=U61_Global::game.find_player_by_id(id1[i]);
              if (p!=NULL)
		{
                  p->kill();
		}

              connection[i]=NULL;
              id0[i]=-1;
              id1[i]=-1;   
	    } 
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to send to the client
 */
int U61_ServerProtocol::send_next_packet()
{
  int result=LATER;

  switch(stage)
    {
    case BEGIN:
      if ((result=start_listening())==DONE)
	{
	  stage=LISTENING;
	  //cout<<"stage=LISTENING\n";
	}
      break;
    case CLIENT_ACCEPTED:
      if ((result=send_id(connection[new_connection]))==DONE)
	{
	  stage=ID_SENT;
	  //cout<<"stage=ID_SENT\n";
	}
      break;
    case ID_RECEIVED:
      if ((result=send_version(connection[new_connection]))==DONE)
	{
	  stage=VERSION_SENT;
	  //cout<<"stage=VERSION_SENT\n";
	}
      break;
    case VERSION_RECEIVED:
      if ((result=send_script_name())==DONE)
	{
	  stage=SCRIPT_NAME_SENT;
	  //cout<<"stage=SCRIPT_NAME_SENT\n";
	}
      break;
    case SCRIPT_NAME_SENT:
      if ((result=send_script())==DONE)
	{
	  stage=SCRIPT_SENT;
	  //cout<<"stage=SCRIPT_SENT\n";
	}
      break;
    case SCRIPT_SENT:
      if ((result=send_id0())==DONE)
	{
	  stage=ID0_SENT;
	  //cout<<"stage=ID0_SENT\n";
	}
      break;
    case ID0_SENT:
      if ((result=send_id1())==DONE)
	{
	  stage=ID1_SENT;
	  //cout<<"stage=ID1_SENT\n";
	}
      break;
    case ID1_SENT:
      if ((result=send_initial_speed())==DONE)
	{
	  stage=INITIAL_SPEED_SENT;
	  //cout<<"stage=INITIAL_SPEED_SENT\n";
	}
      break;
    case INITIAL_SPEED_SENT:
      if ((result=send_acceleration())==DONE)
	{
	  stage=ACCELERATION_SENT;
	  //cout<<"stage=ACCELERATION_SENT\n";
	}
      break;
    case ACCELERATION_SENT:
      if ((result=send_curse_delay())==DONE)
	{
	  stage=CURSE_DELAY_SENT;
	  //cout<<"stage=CURSE_DELAY_SENT\n";
	}
      break;
    case CURSE_DELAY_SENT:
      if ((result=send_time())==DONE)
	{
	  stage=TIME_SENT;
	  //cout<<"stage=TIME_SENT\n";
	}
      break;
    case TIME_SENT:
      if ((result=send_ready(connection[new_connection]))==DONE)
	{
	  stage=READY_SENT;
	  //cout<<"stage=READY_SENT\n";
	}
      break;
    case READY_RECEIVED:
      // cout<<"ok\n";
      dispatcher->open(connection[new_connection]);

      //cout<<"stage=LISTENING\n";
      stage=LISTENING;
      break;
    }

  handle_error(result);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to receive from the
 * client
 */
int U61_ServerProtocol::recv_next_packet()
{
  int result=LATER;

  switch(stage)
    {
    case LISTENING:
      if ((result=accept_client())==DONE)
	{
	  stage=CLIENT_ACCEPTED;
	  //cout<<"stage=CLIENT_ACCEPTED\n";
	}
      break;
    case ID_SENT:
      if ((result=recv_id(connection[new_connection]))==DONE)
	{
	  stage=ID_RECEIVED;
	  //cout<<"stage=ID_RECEIVED\n";
	}
      break;
    case VERSION_SENT:
      if ((result=recv_version(connection[new_connection]))==DONE)
	{
	  stage=VERSION_RECEIVED;
	  //cout<<"stage=VERSION_RECEIVED\n";
	}
      break;
    case READY_SENT:
      if ((result=recv_ready(connection[new_connection]))==DONE)
	{
	  stage=READY_RECEIVED;
	  //cout<<"stage=READY_RECEIVED\n";
	}
      break;
    }

  handle_error(result);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * starts listening what's happening on the network, in case a computer wishes
 * to connect itself
 */
int U61_ServerProtocol::start_listening()
{
  int result=LATER;

  //  cout<<"start listening\n";

  try 
    {
      game=CL_Network::create_game(GAME_ID,port);
    }
  catch (CL_Error e)
    {
      result=FAILED;
      diagnostic+=e.message;
    }

  if (game!=NULL)
    {
      result=DONE;
    };
  
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * detects a computer, ie accepts to dialog with it, it does not mean that
 * the players on this remote machine will actually be accepted in the
 * game but at least we'll dialog with them 
 */
int U61_ServerProtocol::accept_client()
{
  int result=LATER;
  const CL_NetComputer *computer;
  
  if ((computer=game->receive_computer_join())!=NULL)
    {
      if ((new_connection=get_free_connection())<0)
	{
	  result=FAILED;
	}
      else
	{  
	  /*
	   * There's a connection available to talk with the client
	   * so we can discuss with him.
	   * It should be important to check when the computer
	   * object is deleted by the way...
	   */
	  connection[new_connection]=new 
	    U61_Connection(game,(CL_NetComputer *) computer,
			   channel+new_connection,true);        
	  result=DONE;
	}
    }
  
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the index of a free connection in the connection array, if
 * no connection is available it just returns a <0 value
 */
int U61_ServerProtocol::get_free_connection()
{
  int i;
  int result=-1;

  for (i=0;i<U61_Game::NB_NET_GROUPS-1;++i)
    {
      if (connection[i]==NULL)
	{
	  result=i;
	  break;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the script name to the client, we remove the path from the
 * filename, for the full path does not interest the remote client
 */
int U61_ServerProtocol::send_script_name()
{
  int result=LATER;
  char script_name[U61_STRING_SIZE];

  strncpy(script_name,
	  U61_Global::config.script_file,
	  sizeof(script_name)-1);
  if (connection[new_connection]->send(script_name,sizeof(script_name)-1))
    {
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the whole script to the client
 */
int U61_ServerProtocol::send_script()
{
  int result=LATER;
  char script[SCRIPT_SIZE];
  char script_name[U61_STRING_SIZE];
  FILE *f;
  int c;
  int pos=0;

  strncpy(script_name,U61_Global::config.script_file,sizeof(script_name)-1);
  f=fopen(script_name,"rb");
  if (f!=NULL)
    {
      while ((c=fgetc(f))!=EOF && pos<SCRIPT_SIZE)
	{
	  script[pos++]=c;
	}
      fclose(f);
      if (connection[new_connection]->send(script,sizeof(script)-1))
	{
	  result=DONE;
	}
    }
  else
    {
      diagnostic="Could not open script file \"";
      diagnostic+=script_name;
      diagnostic+="\"";
      result=FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the first id that will be usable by the client
 */
int U61_ServerProtocol::send_id0()
{
  int result=LATER;
  int id;

  id=U61_Global::game.find_free_player_id0();
  if (connection[new_connection]->send(id))
    {
      id0[new_connection]=id;
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the second id that will be usable by the client
 */
int U61_ServerProtocol::send_id1()
{
  int result=LATER;
  int id;

  id=U61_Global::game.find_free_player_id1();
  if (connection[new_connection]->send(id))
    {
      id1[new_connection]=id;
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the initial speed that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_initial_speed()
{
  int result=LATER;

  if (connection[new_connection]->send(U61_Global::config.initial_speed))
    {
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the acceleration that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_acceleration()
{
  int result=LATER;

  if (connection[new_connection]->send(U61_Global::config.acceleration))
    {
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the curse delay that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_curse_delay()
{
  int result=LATER;

  if (connection[new_connection]->send(U61_Global::config.curse_delay))
    {
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the time that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_time()
{
  int result=LATER;

  if (connection[new_connection]->send(U61_Time::for_event()))
    {
      result=DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Checks the return value of send & recv functions
 * stops everything if something's going wrong
 */
void U61_ServerProtocol::handle_error(int result)
{
  if (result==FAILED)
    {
      cout<<"Network server protocol error on connection "
	  <<new_connection
	  <<" at stage "
	  <<stage
	  <<"\n";
      if (new_connection>=0 && connection[new_connection]!=NULL)
	{
	  delete connection[new_connection];
	  connection[new_connection]=NULL;
	}

      stage=LISTENING;
    }
}

