/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   game.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: basically contains the main game loop, the object
 *              game is created by the application object, and then
 *              controls everything. I tried to keep this file as small
 *              as possible but it's not so easy...
 *              however, if you want to see how the game works, this
 *              is *the* file to check. 
 *              it loads, the config, starts the gfx mode, loads data
 *              then launches the menu etc...
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/
  
#include "block.h"
#include "game.h"
#include "time.h"
#include "mainmenu.h"
#include "sound.h"
#include "music.h"
#include "square.h"
#include "script.h"
#include "global.h"
#include "platform.h"
#include "keychoose.h"
#include "clientprotocol.h"
#include "serverprotocol.h"
#include "clientdispatcher.h"
#include "serverdispatcher.h"
#include "localdispatcher.h"
#include "utils.h"
#include "game.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/
 

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * initialization of the game
 */ 
U61_Game::U61_Game()
{

}

/*--------------------------------------------------------------------------*/
/* 
 * de-initialization of the game
 */ 
U61_Game::~U61_Game()
{
  U61_LOG_DEBUG("Display : "<<calls_to_display);
  U61_LOG_DEBUG(" Logic : "<<calls_to_logic);
  U61_LOG_DEBUG("Menu : "<<calls_to_menu); 
  U61_LOG_DEBUG("Poll : "<<calls_to_poll);
  U61_LOG_DEBUG("Poll CL : "<<calls_to_poll_clanlib); 
  U61_LOG_DEBUG("Poll U61 : "<<calls_to_poll_u61);
}

/*--------------------------------------------------------------------------*/
/*
 * performs initialization of the game
 * this contains functions that are required for the game to work but
 * are likely to fail is the game is not correctly installed or
 * parametered
 */
bool U61_Game::init(char *filename)
{
  bool success=true;
    
  calls_to_poll=0;
  calls_to_poll_clanlib=0;
  calls_to_poll_u61=0;
  calls_to_poll_menu=0;
  calls_to_display=0;
  calls_to_menu=0;
  calls_to_logic=0;

  protocol=NULL;
  dispatcher=NULL;

  strcpy(exe_name,filename);

  U61_Global::config.read();
    
  U61_Script::init();

  if (success)
    {
      success=success && U61_Global::data.load1
	(U61_Global::config.data_file);
    }

  if (success)
    {
      success=success && 
	U61_Utils::set_videomode(U61_Global::data.screen_w,
			 	 U61_Global::data.screen_h,
				 U61_Global::config.fullscreen);
    }

  if (success)
    {
      success=success && U61_Global::data.load2();
    }  

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * closes the game and exits "cleanly" ;)
 */
void U61_Game::deinit()
{
  U61_Global::config.write();
  U61_Global::data.unload();
}

/*--------------------------------------------------------------------------*/
/*
 * the run function calls the logic and display functions
 * the display and logic functions may never be running at the same time,
 * therefore, there's no need to lock memory 
 */
void U61_Game::run()
{
  int done_for_this_display;
  U61_Menu *exit_menu;

  state=STATE_RUN;
    
  bind_keys();
  reset_players();

  /*
   * we set all the time parameters to default values
   * these may be changed afterwards for real play but they
   * still need to be set for the game not to crash...
   */
  U61_Time::reset();
  last_clanlib_time_polled=CL_System::get_time();
  last_menu_time_polled=CL_System::get_time();
  last_u61_time_polled=U61_Time::for_event();

  /*
   * we have to set this so that the keychoose item works
   */
  U61_KeyChoose::register_key_map(&menu_input);

  /*
   * the game starts on the main menu, which has the exit menu as
   * a parent, logical, eh?
   */
  exit_menu=new U61_ExitMenu();
  current_menu=new U61_MainMenu(exit_menu);

  /*
   * Plays a startup sound
   */
  U61_Sound::play_game_start();
  U61_Music::test();

  /*
   * main game loop
   */
  while (state==STATE_RUN) 
    {
      /*
       * we perform calls to the logic function, but we don't
       * call it more than "max_logic_per_display" times
       * this way, the player will always have its screen
       * updated, even if his computer is not fast enough
       */
      done_for_this_display=0;
      while (U61_Time::is_logic_needed() 
	     && done_for_this_display < MAX_LOGIC_PER_DISPLAY)
        {
	  poll();
	  logic();
	  U61_Time::acknowledge_logic();
	  done_for_this_display++;
        }

      /*
       * if "max_logic_per_display" calls to logic have been
       * done, it means the computer is slow, therefore, we
       * force the computer to slow down the game, since it's
       * not powerfull enough to run it at full speed
       */
      if (done_for_this_display>=MAX_LOGIC_PER_DISPLAY)
        {
          /*  while (U61_Time::is_logic_needed())
	      {
	      U61_Time::slow_down();
	      }*/
        }

      /*
       * call the display, menu and flip fonctions once per loop
       */
      poll();
      while (!U61_Time::is_display_needed(U61_Global::config.max_fps))
	{
	  U61_Time::idle();
	  poll();
	}
      U61_Time::acknowledge_display();
      display();
      poll();

      network();
        
      poll();
      menu();
      poll();
      CL_Display::flip_display(false);
    };
}

/*--------------------------------------------------------------------------*/
/*
 * stops the game, ie exits the main game loop, 
 */
void U61_Game::quit()
{
  state=STATE_QUIT;
}

/*--------------------------------------------------------------------------*/
/*
 * quits the game and restart it (call to "exec") 
 */
void U61_Game::restart()
{
  deinit();
  U61_Platform::exec(exe_name);
}

/*--------------------------------------------------------------------------*/
/*
 * polls the keyboard, and calls any function that has to be call
 * very often, and that is fast to execute
 */
void U61_Game::poll()
{
  int i;
  int clanlib_time;
  int u61_time;

  clanlib_time=CL_System::get_time();
  if (last_clanlib_time_polled<clanlib_time)
    {
      /*
       * required by ClanLib, polls the keyboard and does 
       * some system low level stuff
       */
      CL_System::keep_alive();
      last_clanlib_time_polled=clanlib_time;
      calls_to_poll_clanlib++;

      u61_time=U61_Time::for_event();
      /*
       * this test disables useless polling
       * polling is done at a maximum rate of 30 / sec
       */
      if (last_menu_time_polled<clanlib_time-MENU_POLL_DELAY)
        {
	  last_menu_time_polled=clanlib_time;   

	  menu_input.poll(u61_time);
          menu_input.poll_mouse();

	  calls_to_poll_menu++;     
        }
      /*
       * this test disables useless polling
       * polling is done at a maximum rate of once per game cycle
       */
      if (last_u61_time_polled<u61_time)
        {
	  last_u61_time_polled=u61_time;   

	  for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL;++i)
            {
	      player_input[i].poll(u61_time);
            }

	  calls_to_poll_u61++;     
        }
    }
  calls_to_poll++;
}
 
/*--------------------------------------------------------------------------*/
/* 
 * logic function of the game
 * the logic function is called at a fixed time rate, no matter how long
 * it takes to perform the display. this way, the game will run at the
 * same speed on any kind of machine.
 */ 
void U61_Game::logic()
{
  pump_player_events();

  //U61_Script::test();
  calls_to_logic++;
}

/*--------------------------------------------------------------------------*/
/* 
 * display function of the game
 * this function performs an entire update of the game display
 * it is called as often as possible, depending on how fast the machine is
 */ 
void U61_Game::display()
{
  CL_Display::clear_display();

  U61_Square::update_cycle();
  /*
    for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
    if (player[i].is_available())
    {
    player[i].draw(0,0);
    }
    }
  */   
  U61_Global::layout.draw();
  if (current_menu!=NULL)
    {
      current_menu->draw();
    }

  U61_Script::collect_garbage();

  calls_to_display++;  
}

/*--------------------------------------------------------------------------*/
/* 
 * controls the menu. when the game is running it does nothing but waiting
 * for a keypress
 */
void U61_Game::menu()
{
  U61_Menu *next_menu;
  U61_Event key;
  int mouse_y;

  next_menu=current_menu;

  if (!menu_input.empty())
    {
      key=menu_input.get();
      next_menu=current_menu->action(key.code);
    }

  mouse_y=menu_input.get_mouse_y();
  if (mouse_y>=0)
    {
      U61_LOG_DEBUG("mouse_y="<<mouse_y);
      next_menu=current_menu->mouse_action(mouse_y);
    }

  if (next_menu==NULL)
    {
      quit();
    }
  else
    {
      if (next_menu==current_menu->get_parent())
        {
	  delete current_menu;
        }

      current_menu=next_menu;
    }

  calls_to_menu++;
}

/*--------------------------------------------------------------------------*/
/*
 * calls the network code, on a regular basis
 */
void U61_Game::network()
{
  int i;
  int slot;

  if (protocol!=NULL)
    {
      if (protocol->send_all()==U61_Protocol::FAILED 
	  || protocol->recv_all()==U61_Protocol::FAILED)
	{
	  protocol->print_diagnostic();
	}
      protocol->check_connection();
    }
  if (dispatcher!=NULL)
    {
      dispatcher->process();
      for (i=0;i<U61_Dispatcher::MAX_PLAYER_ID;++i)
	{
          if (dispatcher->is_player_busy(i) && !exists_player_id(i))
	    {
              U61_LOG_DEBUG("Should create player "<<i);
              slot=find_free_network_slot();
              if (slot>=0)
		{
                  player[slot].init_network(dispatcher,i);
		}
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the current menu. Usefull when one wants to call pop_menu,
 * since a menu must always be created with a parent
 */
U61_Menu *U61_Game::get_current_menu()
{
  return current_menu;
}

/*--------------------------------------------------------------------------*/
/*
 * Forces the current menu to be replaced by the given menu.
 * This is usefull when on wants to prompt the user with a 
 * question. A good example (the only one?) is the 
 * menu which asks the player if he wants to save the script
 * or not, when he's connecting to a server.
 */
void U61_Game::force_menu(U61_Menu *m)
{
  current_menu=m;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves parameters from the config , which can not change during
 * the game, such as game speeds
 */
void U61_Game::configure_cold()
{
    
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves parameters from the config , which might change during
 * the game, such as key settings
 */
void U61_Game::configure_hot()
{
  int i;

  bind_keys();
  for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL;++i)
    {
    } 
}

/*--------------------------------------------------------------------------*/
/*
 * starts a local game, with only one player, without asking questions
 */
void U61_Game::start_quick()
{
  U61_LOG_DEBUG("quick start");
  start_local_alone();
}

/*--------------------------------------------------------------------------*/
/*
 * starts a local game, with only one player
 */
void U61_Game::start_local_alone()
{
  int num;

  U61_LOG_DEBUG("local alone game started");
  if (standard_init())
    {
      dispatcher=new U61_LocalDispatcher();

      num=get_local_player(0);
      U61_LOG_DEBUG("num="<<num);
      U61_LOG_DEBUG("name="<<U61_Global::config.player[num].name);
      player[0].init_local(&(player_input[num]),dispatcher,
			   0,0,
			   &(U61_Global::config.player[num]));
      U61_Global::layout.init_alone(&(player[0]));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * starts a local game, with up to 4 players
 */
void U61_Game::start_local_multiplayer()
{
  int num0,num1,num2,num3;
  U61_LOG_DEBUG("local multiplayer game started");
  if (standard_init())
    {
      dispatcher=new U61_LocalDispatcher();

      num0=get_local_player(0);
      player[0].init_local(&(player_input[num0]),dispatcher,
			   0,0,
			   &(U61_Global::config.player[num0]));

      num1=get_local_player(1);
      if (num1<0)
        {
	  U61_Global::layout.init_multiplayer_1(&(player[0]));
        }
      else
        {
	  player[1].init_local(&(player_input[num1]),dispatcher,
			       1,0,
			       &(U61_Global::config.player[num1]));

	  num2=get_local_player(2);
	  if (num2<0)
            {
	      U61_Global::layout.init_multiplayer_2(&(player[0]),
						    &(player[1]));
            }
	  else
            {
	      player[2].init_local(&(player_input[num2]),dispatcher,
				   2,0,
				   &(U61_Global::config.player[num2]));
                
	      num3=get_local_player(3);
	      if (num3<0)
                {
		  U61_Global::layout.init_multiplayer_3(&(player[0]),
							&(player[1]),
							&(player[2]));
                }
	      else
                {
		  player[3].init_local(&(player_input[num3]),dispatcher,
				       3,0,
				       &(U61_Global::config.player[num3]));
                    
		  U61_Global::layout.init_multiplayer_4(&(player[0]),
							&(player[1]),
							&(player[2]),
							&(player[3]));
                }
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * creates a new network game 
 */
void U61_Game::start_network_new()
{
  int num0,num1;
  U61_ServerDispatcher *d;

  U61_LOG_DEBUG("new network game started");
  if (standard_init())
    {
      d=new U61_ServerDispatcher();
      dispatcher=d;     
      
      protocol=new U61_ServerProtocol(d,
				      U61_Global::config.network_port,
                                      U61_Global::config.network_channel);
      
      num0=get_local_player(0);
      player[0].init_local(&(player_input[num0]),dispatcher,
			   0,0,
			   &(U61_Global::config.player[num0]));

      num1=get_local_player(1);
      if (num1<0)
        {
	  U61_Global::layout.init_network_1
	    (&(player[0]),
	     &(player[U61_Layout::NB_PLAYERS_NET_LOCAL]));
        }
      else
        {
	  player[1].init_local(&(player_input[num1]),dispatcher,
			       1,0,
			       &(U61_Global::config.player[num1]));

	 
	  U61_Global::layout.init_network_2
	    (&(player[0]),
	     &(player[1]),
	     &(player[U61_Layout::NB_PLAYERS_NET_LOCAL]));
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * joins an existing network game
 */
void U61_Game::start_network_join()
{
  U61_ClientDispatcher *d;

  U61_LOG_DEBUG("joining network game");
  
  if (standard_init())
    {
      d=new U61_ClientDispatcher();
      dispatcher=d;     

      protocol=new U61_ClientProtocol(d,
				      U61_Global::config.network_port,
                                      U61_Global::config.network_channel,
                                      U61_Global::config.server_id);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Second phase of client initialization, after this, players will be
 * playing for good (ie network initializations have been completed)
 */
void U61_Game::do_network_join(char *script,
			       int id0,int id1,
			       int speed,int accel,int delay,
			       int time)
{
  int num0,num1;

  time_reset(time);
  U61_Script::do_string(script);
  initial_speed=speed;
  acceleration=accel;
  curse_delay=delay;

  num0=get_local_player(0);
  player[0].init_local(&(player_input[num0]),dispatcher,
		       id0,time,
		       &(U61_Global::config.player[num0]));

  num1=get_local_player(1);
  if (num1<0)
    {
      U61_Global::layout.init_network_1
	(&(player[0]),
	 &(player[U61_Layout::NB_PLAYERS_NET_LOCAL]));
    }
  else
    {
      player[1].init_local(&(player_input[num1]),dispatcher,
			   id1,time,
			   &(U61_Global::config.player[num1]));

	 
      U61_Global::layout.init_network_2
	(&(player[0]),
	 &(player[1]),
	 &(player[U61_Layout::NB_PLAYERS_NET_LOCAL]));
    }

}

/*--------------------------------------------------------------------------*/
/*
 * stops all network activity
 */
void U61_Game::stop_network()
{
  if (protocol!=NULL)
    {
      delete protocol;
      protocol=NULL;
    }
  if (dispatcher!=NULL)
    {
      delete dispatcher;
      dispatcher=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * stops the current game, if existing. it does not quit u61.
 */
void U61_Game::stop()
{
  reset_players();
  U61_Global::layout.reset();
  stop_network();
  U61_LOG_DEBUG("game stopped");
}

/*--------------------------------------------------------------------------*/
/*
 * standard initialization before starting a new game
 */
bool U61_Game::standard_init()
{
  bool ok=true;

  initial_speed=U61_Global::config.initial_speed;
  acceleration=U61_Global::config.acceleration;
  curse_delay=U61_Global::config.curse_delay;
  configure_cold();
  configure_hot();

  time_reset();

  stop();

  ok=ok && U61_Global::config.load_script();
    
  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * resets the time
 */
void U61_Game::time_reset(int time)
{
  int i;

  U61_Time::reset(time);
  last_u61_time_polled=time;
  menu_input.reset();
  for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL;++i)
    {
      player_input[i].reset();
    }    
}

/*--------------------------------------------------------------------------*/
/*
 * bind the different keys defined in the config object to the input
 * objects. has to be called each time a key setting is changed
 */
void U61_Game::bind_keys()
{
  int i;

  for (i=0;i<U61_Layout::NB_PLAYERS_LOCAL && i<U61_Config::NB_PLAYERS;++i)
    {
      player_input[i].init(&(U61_Global::config.player[i]));
    }
} 

/*--------------------------------------------------------------------------*/
/*
 * resets the players, after this, no player will be active
 */
void U61_Game::reset_players()
{
  int i;

  for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
      player[i].reset();
      player[i].set_id(i);
    }
} 

/*--------------------------------------------------------------------------*/
/*
 * pumps all the events associated to players
 * ie it checks incoming events and generates events to be transfered
 */
void U61_Game::pump_player_events()
{
  int i;

  for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
      if (player[i].is_available())
        {
	  player[i].pump_events();
        }
    }
} 

/*--------------------------------------------------------------------------*/
/*
 * returns the handler of the nth local player
 * for instance, if players 1 & 3 are activated, asking for player
 * 0 will return 1, player 1 will return 3 are 2 will return -1
 * if no players are activated, player 0 is used
 */
int U61_Game::get_local_player(int order)
{
  int result=-1;
  int found=0;
  int i;

  for (i=0;i<U61_Config::NB_PLAYERS;++i)
    {
      if (U61_Global::config.player[i].activable)
        {
	  if (found==order)
            {
	      result=i;
            }
	  found++;
        }
    }
  if (order==0 && result<0)
    {
      U61_Global::config.player[0].activable=true;
      result=0;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns a free even player id
 */
int U61_Game::find_free_player_id0()
{
  return find_free_player_id(0,2);
}

/*--------------------------------------------------------------------------*/
/*
 * returns a free odd player id
 */
int U61_Game::find_free_player_id1()
{
  return find_free_player_id(1,2);
}

/*--------------------------------------------------------------------------*/
/*
 * returns a free player id
 */
int U61_Game::find_free_player_id(int start,int step)
{
  int i,j;
  int result=-1;
  bool taken;

  for (i=start;i<=U61_Dispatcher::MAX_PLAYER_ID-step;i+=step)
    {
      taken=false;
      for (j=0;j<step;++j)
	{
          if (exists_player_id(i+j))
	    {
	      taken=true;
	    }
	}
      if (!taken)
	{
	  result=i;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns a free network player slot. The slot is not hte same thing than
 * the ID. The ID is used to stamp events whereas the slot is just 
 * used as an index in the U61_Game::player array.
 */
int U61_Game::find_free_network_slot()
{
  int i;
  int result=-1;

  for (i=U61_Layout::NB_PLAYERS_NET_LOCAL;
       i<U61_Layout::NB_PLAYERS_TOTAL;
       ++i)
    {
      if (!player[i].is_available())
	{
	  result=i;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there's an available player with this ID
 */
bool U61_Game::exists_player_id(int player_id)
{
  bool result=false;
  int i;

  for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
      if (player[i].is_available() && player[i].get_id()==player_id)
	{
	  result=true;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the player associated to a given id
 * This function should never fail, if it does, it means there's something
 * weird happening.
 * Indeed, if no matching player is found, then a new player is created.
 * In fact the only reason that would cause this function to fail would
 * be more than 10 players playing together. And this should always be
 * forbidden by the server.
 */
U61_Player *U61_Game::find_player_by_id(int id)
{
  U61_Player *result=NULL;
  int i;

  for (i=0;i<U61_Layout::NB_PLAYERS_TOTAL;++i)
    {
      if (player[i].is_available() && player[i].get_id()==id)
	{
          result=&(player[i]);
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Translates a scan code, used internally to recognize keys, into
 * an ascii code usable in strings. This function is an ugly hack
 * I created because menu_input wasn't accessible from U61_Global...
 */
int U61_Game::scan_code_to_ascii(int scan_code)
{
  return menu_input.get_ascii(scan_code);
}

