/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   menuitem.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: abstract class, parent of all menu items
 *              fondamentaly, a menu item is a line in a menu
 *              this line is made of letters, stored in a array
 *              the menu item will basically handle key presses
 *              and modify
 *              - its display
 *              - some parameter in the config object
 *              special events may open another menu or go back
 *              of course the class provides a "draw" function
 *              and automatically centers itself
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "menuitem.h"
#include "global.h"
#include "letter.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * generation of a default menu item
 */ 
U61_MenuItem::U61_MenuItem()
{
  U61_MenuItem("Default menu item");
}

/*--------------------------------------------------------------------------*/
/* 
 * generation of a default menu item
 */ 
U61_MenuItem::U61_MenuItem(char *str)
{
  mouse_up_limit=0;
  mouse_down_limit=0;
  set_text(str);
  unselect();
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of the menu-item
 */ 
U61_MenuItem::~U61_MenuItem()
{

}

/*--------------------------------------------------------------------------*/
/*
 * treats a key press on a menu item
 */
bool U61_MenuItem::action(int key)
{
  U61_LOG_DEBUG("menu item action, key="<<key) 
    return false;
}

/*--------------------------------------------------------------------------*/
/*
 * updates the text so that it is up to date
 */
void U61_MenuItem::update()
{

}

/*--------------------------------------------------------------------------*/
/*
 * draws a menu item
 */
void U61_MenuItem::draw(int y)
{
  int x;
  int i;
  int nb_letter;
  int last_letter_width2;
  int letter_width;

  /*
   * The update is forced here because blinking cursors require it,
   * and in a general manner, nasry bugs can happen if there are not
   * enough updates. Should not be so slow however...
   */
  update();

  x=U61_Global::data.screen_w/2;
  x-=get_width()/2;

  mouse_up_limit=y;
  mouse_down_limit=y+get_height();  

  last_letter_width2=0;

  nb_letter=text.size();    
  for (i=0;i<nb_letter;++i)
    {
      letter_width=text[i].get_width();
        
      x+=last_letter_width2+letter_width/2;
      text[i].draw(x,y);
      last_letter_width2=letter_width-letter_width/2;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the height of the menu item
 */
int U61_MenuItem::get_height()
{
  int i;
  int nb_letter;
  int height=0;

  nb_letter=text.size();
  for (i=0;i<nb_letter;++i)
    {
      height=max(text[i].get_height(),height);
    }

  return height;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the width of the menu item
 */
int U61_MenuItem::get_width()
{
  int i;
  int nb_letter;
  int width=0;

  nb_letter=text.size();
  for (i=0;i<nb_letter;++i)
    {
      width+=text[i].get_width();
    }

  return width;
}

/*--------------------------------------------------------------------------*/
/*
 * selects a letter in the current menu item 
 */
void U61_MenuItem::select_char(int i)
{
  text[i].enable_dance();
  selected=true;
}

/*--------------------------------------------------------------------------*/
/*
 * make a letter in the current menu item blink 
 */
void U61_MenuItem::blink_char(int i)
{
  text[i].enable_blink();
}

/*--------------------------------------------------------------------------*/
/*
 * selects the current menu item, starting at a given letter
 */
void U61_MenuItem::select_right(int start)
{
  int i;
  int nb_letter;
    
  nb_letter=text.size();
  for (i=start;i<nb_letter;++i)
    {
      select_char(i);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * selects the current menu item
 */
void U61_MenuItem::select()
{
  select_right(0);
}

/*--------------------------------------------------------------------------*/
/*
 * unselects the current menu item
 */
void U61_MenuItem::unselect()
{
  int i;
  int nb_letter;
    
  nb_letter=text.size();
  for (i=0;i<nb_letter;++i)
    {
      text[i].disable_dance();
      text[i].disable_blink();
    }
  selected=false;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the text of the menu item
 */
void U61_MenuItem::set_text(char *str)
{
  char old[U61_STRING_SIZE];

  /*
   * Before we empty the queue of letters and refill it with some new,
   * we check up that this is needed, ie the strings are not the same
   * if they are the same, it's no use to bother, and moreover it can
   * become a bug since the dancing letters don't behave correctly
   */
  put_text(old);
  if (strcmp(str,old)!=0)
    {
      text.clear();
      /*
       * we fill the vector of letters with the chars of the string
       * passed in arg, this will simply create a menu item that
       * display the string
       */
      while(*str)
	{
	  text.push_back(U61_Letter(U61_Global::data.font_menu,*str));
	  str++;
	} 
      /*
       * This is not useless, it just means if the item is selected, we
       * have to re-apply the effects of the selection since emptying
       * the queue has cancelled everything...
       */
      if (selected)
	{
	  select();
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * puts the text in a string buffer
 */
void U61_MenuItem::put_text(char *buf)
{
  int i;
  int nb_letter;
  char c;
    
  nb_letter=text.size();
  for (i=0;i<nb_letter;++i)
    {
      c=text[i].get_ascii();
      buf[i]=c;
    }
  buf[i]=0;
}

/*--------------------------------------------------------------------------*/
/*
 * true if the menuitem must handle all keys, including vertical arrows
 */
bool U61_MenuItem::is_modal()
{ 
  return false;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if a mouse click at this height should infuence the item
 */
bool U61_MenuItem::is_mouse_in(int y)
{
  bool ok=false;

  if (y>=mouse_up_limit && y<mouse_down_limit)
    {
      ok=true;
    }

  return ok;
}

