/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   time.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: used to manage time. it provides a "game time" which supports
 *              pauses, resets etc... and a system time which never stops
 *              running and is used for animations
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "time.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * contains the offset that has to be substracted to get_time()
 */
int U61_Time::reset_offset;

/*--------------------------------------------------------------------------*/
/*
 * contains the offset that results from calls to alter()
 */
int U61_Time::alter_offset;

/*--------------------------------------------------------------------------*/
/*
 * contains the offset that results from the pauses
 */
int U61_Time::pause_offset;

/*--------------------------------------------------------------------------*/
/*
 * contains the system time when the pause started
 */
unsigned int U61_Time::start_pause_time;

/*--------------------------------------------------------------------------*/
/*
 * boolean which value is true if the game is paused
 */
bool U61_Time::paused;

/*--------------------------------------------------------------------------*/
/*
 * contains the last value given by the for_event function
 */
unsigned int U61_Time::last_for_event;

/*--------------------------------------------------------------------------*/
/*
 * contains the last value given by the for_logic function 
 */
unsigned int U61_Time::last_for_logic;

/*--------------------------------------------------------------------------*/
/*
 * contains the last time when the display function has been called 
 */
unsigned int U61_Time::last_for_display;

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * resets the time values, and sets logic time to the parameter given
 */
void U61_Time::reset(unsigned int time)
{
    reset_offset=CL_System::get_time()-time*TIME_UNIT;
    alter_offset=0;
    pause_offset=0;
    paused=false;
    last_for_event=time;
    last_for_logic=time;
    last_for_display=CL_System::get_time();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the time that should be used to generate events.
 * this time is always ahead of the for_logic time
 */
unsigned int U61_Time::for_event()
{
    unsigned int result;

    /*
     * an event should take the system time, converted to the U61 unit,
     * which is determined by the time_unit value (in msec)
     */
    result=system_time();

    /*
     * if result is lower that last_for_logic, this means that logic
     * operations have been performed up to last_for_logic, so
     * we should use last_for_logic+1 as an event time
     */
    if (result<last_for_logic+1)
    {
        result=last_for_logic+1;
    }	 

    /*
     * we return the current time if there's no problem, 
     * if last_for_event is greater than current time, we return
     * last_for_event, this prevent the situtation where an event
     * with time t is fired after an event with time t+1
     */
    if (result<last_for_event)
    {
        result=last_for_event;
    }
    else
    {
        last_for_event=result;
    }

    return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the time for the logic operation, this time is increased
 * by step of 1, using acknowledge_logic()
 */
unsigned int U61_Time::for_logic()
{
    return last_for_logic;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the time for the display operation, this time never stops,
 * it is not affected by the pause() or resume() functions
 * it is used for visual enhancements
 */
unsigned int U61_Time::for_effect()
{
    return CL_System::get_time()/TIME_UNIT;
}

/*--------------------------------------------------------------------------*/
/*
 * alters the time, usefull to speed up or slow down the game,
 * if some inconsistency is detected during a network game for
 * instance. the value passed as an arg is used as an offset
 */
void U61_Time::alter(int delta)
{
    alter_offset-=delta*TIME_UNIT;
}

/*--------------------------------------------------------------------------*/
/*
 * calls alter to speed up the game of one tick
 * usefull when the machine seems to be late compared to other
 * networked machines, if all machines call speed_up when
 * they feel they're late, then after some time calls to
 * speed_up should never occur any more and the game time
 * should be the same on all machines
 */
void U61_Time::speed_up()
{
    alter(1);
}

/*--------------------------------------------------------------------------*/
/*
 * calls alter to slow down the game of one tick
 * usefull when your CPU is too slow and can not perform enough calls to
 * logic. with slow_down, the amount of work given to the machine will
 * decrease, and the game should run slower but OK
 */
void U61_Time::slow_down()
{
    alter(-1);
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if a call to logic() int the main game loop is needed
 */
bool U61_Time::is_logic_needed()
{
    unsigned int time;
    bool result=false;

    /*
     * we calculate the time that would be returned by for_event,
     * except we don't use for_event since it performs useless
     * and unwanted checks, and we substract 1, so that logic is
     * always one tick behind events (safer)
     */ 
    time=system_time()-1;

    /*
     * if the time calculated is greater that the last value returned
     * by for_logic, then logic actions are late, so they should
     * be performed ASAP
     */
    if (time>last_for_logic)
    {
        result=true;

    }
    return result;
}

/*--------------------------------------------------------------------------*/
/*
 * tells the time system that a logic action has been performed
 * concretely, it increases the value returned by for_logic
 */
void U61_Time::acknowledge_logic()
{
    last_for_logic++;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if a call to display() int the main game loop is needed
 * The method is just to check if there's been enough time between the
 * last call to display and now. If this time is too short, we should not
 * display anything since it is quite useless.
 */
bool U61_Time::is_display_needed(int fps)
{
    bool result=false;
    unsigned int time;

    time=CL_System::get_time();
    if (time>=last_for_display+1000/fps)
      {
	result=true;
      }   

    return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Tells the system that a display has been performed, this is necessary
 * for is_display_needed to work correctly 
 */
void U61_Time::acknowledge_display()
{
    last_for_display=CL_System::get_time();
}

/*--------------------------------------------------------------------------*/
/*
 * enters the paused state
 * subsequent calls to system_time will return a constant value
 */
void U61_Time::pause()
{
    if (!paused)
    {
        start_pause_time=CL_System::get_time();
        paused=true;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * quits the paused state 
 * this function mainly updates the pause_offset value
 */
void U61_Time::resume()
{
    if (paused)
    {
        pause_offset+=CL_System::get_time()-start_pause_time;
        paused=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns true is the time system is paused
 */
bool U61_Time::is_paused()
{
    return paused;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the system time, but includes pause handling
 * one could say, "why not use ClanLib's pauses", well, ClanLib pause
 * are good but they stop everything and in U61, we want to have the
 * for_effect to keep on returning the current time
 * plus this function converts the machine time, in msec, into
 * the game unit (defined by time_unit)
 */
int U61_Time::system_time()
{
    int time;

    if (paused)
    {
        time=start_pause_time;        
    }
    else
    {
        time=CL_System::get_time();
    }

    return (time-reset_offset-alter_offset-pause_offset)/TIME_UNIT;
}

/*--------------------------------------------------------------------------*/
/*
 * Let the CPU go idle for some time. This is usefull because sometimes
 * there's no need for the game to get all the CPU usage. Therefore whenever
 * we feel CPU is not needed, we go idle for some time
 */
void U61_Time::idle()
{
  CL_System::sleep(IDLE_DELAY);
  U61_LOG_DEBUG("game is idle");
}
