/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   map.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: I called this class map for I can't imagine a game without
 *              a "map" class 8) . seriously, a map is where the information
 *              about where the blocks are is stored. it's basically an array
 *              of squares.
 *              it also provides an API to move the player's falling block
 *              and handle any event.
 *              it's important to note that most of the behaviors of a map
 *              are determined by embedded LUA scripts
 */


/*---------------------------------------------------------------------------
 include
 ---------------------------------------------------------------------------*/

#include "global.h"
#include "map.h"
#include "script.h"
#include "time.h"
#include "sound.h"
#include "utils.h"
#include "debug.h"
#include "music.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_MAP_SPEED_SCALE 1000
#define U61_MAP_ACCEL_SCALE 1000000
#define U61_MAP_MAX_SYSTEM_SPEED 50
#define U61_MAP_MAX_SYSTEM_ACCEL 15
#define U61_MAP_MAX_SCORE 1000000
#define U61_MAP_TIME_CALLBACK_1_DELAY 100
#define U61_MAP_TIME_CALLBACK_10_DELAY 10
#define U61_MAP_TIME_CALLBACK_100_DELAY 1
#define U61_MAP_CHECK_TARGET_DELAY 100
#define U61_MAP_CHECK_CURSE_DELAY 100
#define U61_MAP_ANTICIPATE_MAX 1000

/*--------------------------------------------------------------------------*/
/* 
 * these arrays contain constants used for speeds, this way one can
 * have something more subtle than just a linear behavior
 */ 
static int speeds[U61_MAP_MAX_SYSTEM_SPEED+1]=
{
  1,  2,  5,  10, 20, 30, 40, 50, 60, 70, 80,
  84, 88, 92, 96, 100,104,108,112,116,120,
  124,128,132,136,140,144,148,152,156,160,
  164,168,172,176,180,184,188,192,196,200,
  210,220,240,260,280,300,350,400,450,500
};

static int accels[U61_MAP_MAX_SYSTEM_ACCEL+1]=
{
  0,  10, 20, 30, 40, 50,
  60, 80, 100,140,200,
  260,320,380,440,500
};

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default map
 */ 
U61_Map::U61_Map()
{
  active=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a map
 */ 
U61_Map::~U61_Map()
{

}

/*--------------------------------------------------------------------------*/
/*
 * clears a map
 */
void U61_Map::clear()
{
  int i,j;

  for (j=0;j<U61_MAP_HEIGHT;++j)
    {
      for (i=0;i<U61_MAP_WIDTH;++i)
        {
	  squares[i][j].disable();
	  //squares[i][j].enable();
	  squares[i][j].set_color((i+j)%8);
        }
    }
  curse_chosen=false;
  curse_asked=false;
  curse_x=0;
  curse_y=0;
}


/*--------------------------------------------------------------------------*/
/*
 * begin functino, to call each time the player has lost 
 */
void U61_Map::begin()
{
  int i;
  U61_Event evt;

  auto_events.clear();
  request_events.clear();
  
  clear();

  name[0]=0; 
  block.reset();
  next_block.reset();
  block_requested=false;
  matching=false;
  speed_counter=0;
  accel_counter=0;
  speed=base_speed;
  accel=base_accel; 
  background=0;
  anticipation_state=true;
  preview_state=true;
  match_count=0;
  curse_counter=0;
  curse_chosen=false;
  curse_asked=false;
  curse_state=false;
  nb_antidote=0;
  score=0;
  curse_sent=0;
  curse_received=0;
  start_time=0;
  for (i=0;i<U61_MAP_NB_GLOBAL;++i)
    {
      global_val[i]=0;
    }        
  for (i=0;i<U61_MAP_MAX_CURSE_ID;++i)
    {
      registered_curse[i].reset();
    }        
  change_background();

  evt.code=U61_EVENT_SET_VICTIM;
  evt.par=target_id;
  put_auto_event(evt);
}

/*--------------------------------------------------------------------------*/
/*
 * initialization of a map, must be called before the map is used
 */
void U61_Map::init(int id,int s,int a,int d)
{
  player_id=id;
  base_speed=s;
  base_accel=a;
  curse_delay=d*100;

  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * resets completely a map
 */
void U61_Map::reset()
{
  target_id=player_id;
  map_time=0;
  active=false;
  score=0;
  silent=false;

  begin();
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the map is activated
 * by activated map, we mean a map where blocks are falling and someone
 * is actually playing. a player can have a disactivated map,in this
 * case u61 is waiting for him to press the start key and activate the map
 */
bool U61_Map::is_active()
{
  return active;
}

/*--------------------------------------------------------------------------*/
/*
 * activates a map
 */
void U61_Map::set_active(int time)
{
  begin();
  map_time=time;
  active=true;
}

/*--------------------------------------------------------------------------*/
/*
 * disactivates a map
 */
void U61_Map::set_inactive()
{
  active=false;
}

/*--------------------------------------------------------------------------*/
/*
 * changes the background of the map randomly
 */
void U61_Map::change_background()
{
  U61_LOG_DEBUG("Number of backgrounds"<<U61_Global::data.nb_map);
  background=U61_Utils::random(U61_Global::data.nb_map);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the background of the map
 */
int U61_Map::get_background()
{
  return background;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
void U61_Map::set_square_color(int x,int y,int color)
{
  if (x>=0 && y>=0 && x<U61_MAP_WIDTH && y<U61_MAP_HEIGHT)
    {
      if (color>=0 && color<U61_SQUARE_NB_COLORS)
        {
	  squares[x][y].enable();
	  squares[x][y].set_color(color);
        }
      else
        {
	  squares[x][y].disable();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
int U61_Map::get_square_color(int x,int y)
{
  int color=-1;

  if (x>=0 && y>=0 && x<U61_MAP_WIDTH && y<U61_MAP_HEIGHT)
    {
      if (squares[x][y].is_enabled())
        {
	  color=squares[x][y].get_color();
        }
    }
  return color;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the score of the map, negative scores are forbidden
 */
void U61_Map::set_score(int s)
{
  if (s>=0)
    {
      if (s<U61_MAP_MAX_SCORE)
	{
          score=s;
	}
      else
        {
	  score=U61_MAP_MAX_SCORE-1;
	}
    }
  else
    {
      score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the score associated to the map
 */
int U61_Map::get_score()
{
  return score;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the time of the last game
 */
int U61_Map::get_elapsed_time()
{
  int elapsed;

  if (map_time==0 || start_time==0 || map_time<start_time)
    {
      elapsed=0;
    }
  else
    {
      elapsed=map_time-start_time;
    }

  return elapsed;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of curses sent during the last game
 */
int U61_Map::get_curse_sent()
{
  return curse_sent;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of curses received during the last game
 */
int U61_Map::get_curse_received()
{
  return curse_received;
}

/*--------------------------------------------------------------------------*/
/*
 * adds the given value to the current score
 */
void U61_Map::add_score(int s)
{
  score+=s;
  if (score<0)
    {
      score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the time associated to the map
 */
int U61_Map::get_time()
{
  return map_time;
}

/*--------------------------------------------------------------------------*/
/*
 * returns a number which is used by the lua match function
 */
int U61_Map::get_match_count()
{
  return match_count;
}

/*--------------------------------------------------------------------------*/
/*
 * sets a global integer into the map
 * used for lua scripts to store global permanent values
 */
void U61_Map::set_global(int i, int glob)
{
  if (i>=0 && i<U61_MAP_NB_GLOBAL)
    {
      global_val[i]=glob;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets a global integer from the map
 * used for lua scripts to store global permanent values
 */
int U61_Map::get_global(int i)
{
  int glob=0;

  if (i>=0 && i<U61_MAP_NB_GLOBAL)
    {
      glob=global_val[i];
    }

  return glob;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the age of a persistent curse
 * If negative, it means the curse is not registered or is over
 */
int U61_Map::get_curse_age(int i)
{
  int age=false;

  if (i>=0 && i<U61_MAP_MAX_CURSE_ID)
    {
      if ((!registered_curse[i].active) 
	  || map_time>registered_curse[i].end)
	{
	  age=-1;
	}
      else
	{
          age=((map_time-registered_curse[i].begin));
	}
    }

  return age;
}

/*--------------------------------------------------------------------------*/
/*
 * Registers a persistent curse. After this, calls to get_curse_age
 * will return how long ago the curse has been registered
 */
void U61_Map::register_curse(int i, int length, bool good)
{
  if (i>=0 && i<U61_MAP_MAX_CURSE_ID)
    {
      /*
       * if length is zero, the default length is 10 hours
       * that's too say infinite...
       */
      if (length<=0)
	{
	  length=3600000;
	}
      registered_curse[i].begin=map_time;
      registered_curse[i].end=map_time+length;
      registered_curse[i].active=true;
      registered_curse[i].good=good;
      U61_LOG_DEBUG("Registering "<<i<<" begin="<<curse_begin[i]<<" end="<<curse_end[i]);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Cancels a registered curse
 */
void U61_Map::cancel_curse(int i)
{
  if (i>=0 && i<U61_MAP_MAX_CURSE_ID)
    {
      registered_curse[i].active=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the oldest curse available
 * Returns -1 if none existing
 */
int U61_Map::get_oldest_curse(bool good)
{
  int oldest;
  int i;
  int result=-1;

  oldest=map_time;
  
  for (i=0;i<U61_MAP_MAX_CURSE_ID;++i)
    {
      if (get_curse_age(i)>=0 &&
	  registered_curse[i].good==good)
	{ 
	  oldest=map_time;
          result=i;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of persisten curses which are affecting the map
 */
int U61_Map::get_nb_curse(bool good)
{
  int nb;
  int i;

  nb=0;
  for (i=0;i<U61_MAP_MAX_CURSE_ID;++i)
    {
      if (get_curse_age(i)>=0 &&
	  registered_curse[i].good==good)
	{
	  nb++;
	}
    }
  return nb;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the number of antidotes
 */
int U61_Map::get_nb_antidote()
{
  return nb_antidote;
}

/*--------------------------------------------------------------------------*/
/*
 * Adds an antidote
 */
void U61_Map::add_antidote()
{
  nb_antidote++;
}

/*--------------------------------------------------------------------------*/
/*
 * Deletes an antidote
 */
void U61_Map::delete_antidote()
{
  if (nb_antidote>0)
    {
      nb_antidote--;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the x coordinate of the curse
 */
void U61_Map::set_curse_x(int x)
{
  if (x>=0 && x<U61_MAP_WIDTH)
    {
      curse_x=x;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the y coordinate of the curse
 */
void U61_Map::set_curse_y(int y)
{
  if (y>=0 && y<U61_MAP_HEIGHT)
    {
      curse_y=y;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the state of the curse
 */
void U61_Map::set_curse_state(bool state)
{
  if (state)
    {
      curse_state=true;
    }
  else
    {
      curse_counter=0;
      curse_state=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the x coordinate of the curse
 */
int U61_Map::get_curse_x()
{
  int x=-1;

  if (curse_state && curse_chosen)
    {
      x=curse_x;
    }

  return x;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the y coordinate of the curse
 */
int U61_Map::get_curse_y()
{
  int y=-1;

  if (curse_state && curse_chosen)
    {
      y=curse_y;
    }

  return y;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the state of the curse
 */
bool U61_Map::get_curse_state()
{
  return curse_state!=false;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the curse state is true and the curse is located in
 * a square which is not empty/disabled
 */
bool U61_Map::is_curse_available()
{
  bool result=false;

  if (curse_x>=0 && curse_y>=0 && curse_x<U61_MAP_WIDTH && curse_y<U61_MAP_HEIGHT
      && curse_state && curse_chosen
      && squares[curse_x][curse_y].is_enabled())
    {
      result=true;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends a curse to a remote player
 * In fact it justs puts a REQUEST_CURSE event in the event queue, and
 * the rest is done by dispatchers
 */
void U61_Map::send_curse(int curse_id)
{
  U61_Event evt;

  if (player_id!=target_id && curse_id>=0 && curse_id<U61_MAP_MAX_CURSE_ID)
    {
      U61_LOG_DEBUG("Sending curse "<<curse_id);

      curse_sent++;
      evt.code=U61_EVENT_REQUEST_CURSE;
      evt.par=curse_id;
      /*
       * We use put_request_event instead of calling put_auto_event
       * since the target is not the player here...
       */
      put_request_event(evt);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the next curse that will be fired
 */
char *U61_Map::get_curse_name()
{
  char *name;

  if (is_curse_available())
    {
      name=U61_Script::get_curse_name(this,curse_id);
    }
  else
    {
      name="";
    }

  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * compute the map to the next instant
 */
void U61_Map::compute_next()
{
  map_time++;
  curse_counter++;
  update_explosions();
  if (!(map_time%U61_MAP_TIME_CALLBACK_1_DELAY))
    {
      time_callback_1();
    }
  if (!(map_time%U61_MAP_TIME_CALLBACK_10_DELAY))
    {
      time_callback_10();
    }
  if (!(map_time%U61_MAP_TIME_CALLBACK_100_DELAY))
    {
      time_callback_100();
    }
  if (!(map_time%U61_MAP_CHECK_TARGET_DELAY))
    {
      check_target();
    }
  if (!(map_time%U61_MAP_CHECK_CURSE_DELAY))
    {
      check_curse();
    }
  if (!exists_explosion())
    {
      match_pattern();
    }
    
  update_speed();
}

/*--------------------------------------------------------------------------*/
/* 
 * updates the current speed
 */
void U61_Map::update_speed()
{
  check_speed();
  check_accel();

  speed_counter+=speeds[speed];
  if (speed_counter>U61_MAP_SPEED_SCALE)
    {
      speed_counter-=U61_MAP_SPEED_SCALE;
      block_move_down(); 
    }    
  accel_counter+=accels[accel];
  if (accel_counter>U61_MAP_ACCEL_SCALE)
    {
      accel_counter-=U61_MAP_ACCEL_SCALE;
      speed++;
      U61_LOG_DEBUG("Next speed ("<<speed);
    }    

  check_speed();
  check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * computes the level up to the givent time, no events are handled
 */
void U61_Map::compute_up_to(int time)
{
  if (active)
    {
      while (map_time<time)
        {
	  compute_next();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles an event, ie does anything that has to be done for this event
 */
void U61_Map::handle_event(U61_Event *event)
{
  if (active)
    {
      if (start_time==0)
	{
	  start_time=map_time;
	}
      if (int(event->time)<map_time)
        {
	  U61_LOG_DEBUG("Inconsitency in event dates detected with event "<<*event<<" and map_time="<<map_time);
        }  
      else
        {
	  compute_up_to(event->time);
	  switch (event->code)
            {
            case U61_EVENT_ROTATE_LEFT:
	      block_rotate_left();
	      break;
            case U61_EVENT_ROTATE_RIGHT:
	      block_rotate_right();
	      break;
            case U61_EVENT_MOVE_RIGHT:
	      block_move_right();
	      break;
            case U61_EVENT_MOVE_LEFT:
	      block_move_left();
	      break;
            case U61_EVENT_MOVE_DOWN:
	      block_move_down();
	      break;
            case U61_EVENT_DROP:
	      block_drop();
	      break;
            case U61_EVENT_NEW_BLOCK:
	      new_block(event->par);
	      break;
            case U61_EVENT_LOOSE:
	      loose();
	      break;
            case U61_EVENT_PREV_VICTIM:
              prev_victim(event);
              break;
            case U61_EVENT_NEXT_VICTIM:
              next_victim(event);
              break;
            case U61_EVENT_SET_VICTIM:
              set_victim(event->par);
              break;
            case U61_EVENT_NAME_LETTER:
              name_add_letter(event->par);
	      break;
            case U61_EVENT_NEW_CURSE:
              new_curse(event->par);
	      break;
            case U61_EVENT_REQUEST_CURSE:
              request_curse(*event);
	      break;
            case U61_EVENT_DO_CURSE:
              do_curse(*event);
	      break;
            case U61_EVENT_USE_ANTIDOTE:
              use_antidote();
              break;
            }
        }
    }

  U61_LOG_DEBUG("Event computed "<<*event);
}

/*--------------------------------------------------------------------------*/
/*
 * anticipates the level to the given time, it's usefull to display the
 * "last" map. the computer just supposes no events have been generated
 * and anticipates the behavior of the map
 * but careful, since no events are computed, themap may become incoherent
 * it's not a problem for in U61 we keep 2 copies of the map, a sure and
 * an anticipated one, called "last"
 * BTW, this function deletes all events from the auto_events list
 * since they are of no use when the map has been anticipated and/or
 * we don't want the list to grow and explose system memory...
 */
void U61_Map::anticipate(int time)
{
  /*
   * we do not anticipate too much for this can lead to
   * deadlocks when the CPU is overloaded.
   * We limit the anticipation to 10 real seconds, knowing
   * that fake events are sent 10 times/sec on local players
   * and 1 time/sec for remote players
   */
  if (time-map_time>U61_MAP_ANTICIPATE_MAX)
    {
      time=map_time+U61_MAP_ANTICIPATE_MAX;
    }
  compute_up_to(time);
  auto_events.clear();
  request_events.clear();
}

/*--------------------------------------------------------------------------*/
/*
 * polls some auto generated events, such as requests for new blocks
 */
void U61_Map::poll_auto_events()
{
  U61_Event event;

  if (active)
    {
      if ((!is_block_active()) 
	  && (!block_requested) 
	  && (!matching))
        {
	  block_requested=true;
	  event.code=U61_EVENT_NEW_BLOCK;
	  event.par=U61_Script::new_shape(this,&block);
	  put_auto_event(event);
	  U61_LOG_DEBUG("new block event sent");
        }
      if (curse_state
          && (((!curse_asked) && (!curse_chosen)) 
	      || curse_counter>curse_delay)
	  && (!exists_explosion()))
	{
	  curse_asked=true;
	  curse_counter=0;
          event.code=U61_EVENT_NEW_CURSE;
          event.par=U61_Script::new_curse(this,&block);
          put_auto_event(event);
          U61_LOG_DEBUG("Auto curse event "<<event.par);
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next auto generated event
 */
U61_Event U61_Map::get_auto_event()
{
  U61_Event event;

  event=auto_events.front();
  auto_events.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event
 */
bool U61_Map::exists_auto_event()
{
  return !auto_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the generated event queue
 */
void U61_Map::put_auto_event(U61_Event event)
{
  auto_events.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next generated event request
 */
U61_Event U61_Map::get_request_event()
{
  U61_Event event;

  event=request_events.front();
  request_events.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event request
 */
bool U61_Map::exists_request_event()
{
  return !request_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the request event queue
 */
void U61_Map::put_request_event(U61_Event event)
{
  event.target=target_id;
  event.author=player_id;
  event.target=target_id;
  event.time=0;
  request_events.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the player playing on this map
 */
char *U61_Map::get_name()
{
  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map onto the screen
 */
void U61_Map::draw(int x,int y, int size, bool anti)
{
  int i,j;
  int draw_x,draw_y;

  if (active)
    {
      for (j=0;j<U61_MAP_HEIGHT;++j)
        {
	  for (i=0;i<U61_MAP_WIDTH;++i)
            {
	      draw_x=x+i*U61_Global::data.square_w[size];

	      draw_y=y+j*U61_Global::data.square_h[size];

              if (j==curse_y && i==curse_x && 
		  curse_state && curse_chosen &&
                  squares[i][j].is_enabled())
		{
                  if (U61_Square::should_blink())
		    {
		      squares[i][j].draw_curse(draw_x,draw_y,size);
		    }
		  else
		    {
		      squares[i][j].draw(draw_x,draw_y,size);
		    }
		}
	      else
		{
		  squares[i][j].draw(draw_x,draw_y,size);
		}
            }
        }
      if (anticipation_state && anti)
        {
	  draw_anticipation(x,y,size);
        }
      block.draw(x,y,size);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the map in a mute state (does not generate sounds any more)
 */
void U61_Map::mute()
{
  silent=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the anticipation state
 */
void U61_Map::set_anticipation_state(bool state)
{
  anticipation_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the anticipation state
 */
bool U61_Map::get_anticipation_state()
{
  return anticipation_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the preview state
 */
void U61_Map::set_preview_state(bool state)
{
  preview_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the preview state
 */
bool U61_Map::get_preview_state()
{
  return preview_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the speed 
 */
void U61_Map::set_speed(int s)
{
  speed=s; 
  check_speed();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the speed 
 */
int U61_Map::get_speed()
{
  return speed; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the speed is within the correct range
 */
void U61_Map::check_speed()
{
  if (speed>U61_MAP_MAX_SYSTEM_SPEED)
    {
      speed=U61_MAP_MAX_SYSTEM_SPEED;
    }
  if (speed<0)
    {
      speed=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the accel
 */
void U61_Map::set_accel(int a)
{
  accel=a; 
  check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the accel
 */
int U61_Map::get_accel()
{
  return accel; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the accel is within the correct range
 */
void U61_Map::check_accel()
{
  if (accel>U61_MAP_MAX_SYSTEM_ACCEL)
    {
      accel=U61_MAP_MAX_SYSTEM_ACCEL;
    }
  if (accel<0)
    {
      accel=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the nest block 
 */
U61_Block U61_Map::get_next_block()
{
  return next_block; 
}

/*--------------------------------------------------------------------------*/
/*
 * puts the square in an exploding state
 */
void U61_Map::blow_up_square(int x,int y)
{
  if (x>=0 && y>=0 && x<U61_MAP_WIDTH && y<U61_MAP_HEIGHT)
    {
      squares[x][y].begin_explosion();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the square is exploding
 */
bool U61_Map::is_square_exploding(int x,int y)
{
  bool result=false;

  if (x>=0 && y>=0 && x<U61_MAP_WIDTH && y<U61_MAP_HEIGHT)
    {
      result= squares[x][y].is_exploding();
    }
    
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there's at least one square exploding
 */
bool U61_Map::exists_explosion()
{
  bool result=false;
  int x,y;

  for (y=0;y<U61_MAP_HEIGHT;++y)
    for (x=0;x<U61_MAP_WIDTH;++x)
      {
	if (squares[x][y].is_exploding())
	  {
	    result=true;
	  } 
      }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * display the anticipation, the anticipation is "where the block will land if
 * nothing happens" ie where it will land if I press drop  
 */
void U61_Map::draw_anticipation(int x,int y,int size)
{
  U61_Block test;
  U61_Block prev;

  if (block.get_nb_items()>0)
    {
      test=block;
    
      while (is_block_ok(&test))
        {
	  prev=test;
	  U61_Script::move_down(this,&test);
          test.center();
        }

      U61_Script::land(this,&prev);
      prev.draw_anticipation(x,y,size); 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_left()
{
  U61_Block test;

  U61_LOG_DEBUG("rotate left");

  test=block;
  U61_Script::rotate_left(this,&test);
  test.center();

  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      test=block;
      U61_Script::move_left(this,&test);
      U61_Script::rotate_left(this,&test);
      test.center();

      if (is_block_ok(&test))
        {
	  block=test;
        }
      else
        {
	  test=block;
	  U61_Script::move_right(this,&test);
	  U61_Script::rotate_left(this,&test);
          test.center();
	  if (is_block_ok(&test))
            {
	      block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_right()
{
  U61_Block test;

  U61_LOG_DEBUG("rotate right");

  test=block;
  U61_Script::rotate_right(this,&test);
  test.center();
  
  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      test=block;
      U61_Script::move_right(this,&test);
      U61_Script::rotate_right(this,&test);
      test.center();

      if (is_block_ok(&test))
        {
	  block=test;
        }
      else
        {
	  test=block;
	  U61_Script::move_left(this,&test);
	  U61_Script::rotate_right(this,&test);
          test.center();
	  if (is_block_ok(&test))
            {
	      block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_left()
{
  U61_Block test;

  U61_LOG_DEBUG("move left");

  test=block;
  U61_Script::move_left(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_right()
{
  U61_Block test;

  U61_LOG_DEBUG("move right");

  test=block;
  U61_Script::move_right(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block down, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_down()
{
  U61_Block test;

  test=block;
  U61_Script::move_down(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * performs calls to move_down as much as possible so that the block lands
 */
void U61_Map::block_drop()
{
  U61_Block test;

  U61_LOG_DEBUG("drop");

  test=block;
  if (block.get_nb_items()>0)
    {
      while (is_block_ok(&test))
        {
	  block=test;
	  U61_Script::move_down(this,&test);
          test.center();
        }
    
      stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Adds a letter to the map name
 */
void U61_Map::name_add_letter(char c)
{
  int len;

  len=strlen(name);
  if (len<U61_MAP_NAME_SIZE)
    {
      name[len]=c;
      name[len+1]=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Uses an antidote
 */
void U61_Map::use_antidote()
{
  U61_LOG_DEBUG("Using antidote");

  if (nb_antidote>0)
    {
      nb_antidote--;
      U61_Script::use_antidote(this,&block);
      /*
       * The call to check_block is very important since use_antidote
       * has full rights on the map and block.
       */
      check_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * checks if there's an active block (size greater than 0)
 */
bool U61_Map::is_block_active()
{
  return block.get_nb_items()>0;
}

/*--------------------------------------------------------------------------*/
/*
 * checks if a position is possible, if yes, returns true
 * this function is used to check if the result of functions such
 * block_rotate_right or block_move_down are incoherent, and therefore
 * should be cancelled
 */
bool U61_Map::is_block_ok(U61_Block *test)
{
  int i,n,x,y;
  bool ok=true;

  n=test->get_nb_items();
  for (i=0;i<n && ok;++i)
    {
      x=test->get_x()+test->get_item_x(i);
      y=test->get_y()+test->get_item_y(i);
      /*
       * the square must be in the map, or over the map,
       * if the block is brand new and didn't show himself yet
       */
      if (x>=0 && x<U61_MAP_WIDTH && y<U61_MAP_HEIGHT)
        {
	  /*
	   * if y>=0, there could potentially be a stable square
	   * at this location, so we check that
	   */
	  if (y>=0)
            {
	      if (squares[x][y].is_enabled()) 
                {
		  ok=false;
                }
            }
        }
      else
        {
	  ok=false;
        }
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * gets a new block, the shape of the new block is calculated with
 * Lua user defined functions
 */
void U61_Map::new_block(int shape)
{
  block=next_block;
  block.set_x(U61_MAP_WIDTH/2);
  block_requested=false;

  next_block.reset();
  U61_Script::do_shape(this,&next_block,shape);
  next_block.center();
  /*
   * I used to corner the block before drawing it. Now they are drawn centered
   * right away, so this line became useless.
   */
  //next_block.corner();

  block.center();

  U61_LOG_DEBUG("new block given");
}

/*--------------------------------------------------------------------------*/
/*
 * this function transfoem the moving block into a stable block,
 * ie it adds it to the current map
 */
void U61_Map::stabilize_block()
{
  U61_Event evt;
  int x,y,i,n;
  bool outside=false;

  /*
   * Before we merge the block with the map, we move the block up
   * until there's no conflict with the map. At first sight, it seems
   * useless since usually stabilize comes after a move_down, but it's
   * just safest to do this check, since many things can happen in
   * a curse for instance.
   */
  while (!is_block_ok(&block))
    {
      block.set_y(block.get_y()-1);
    }

  U61_Script::land(this,&block);
  n=block.get_nb_items();
  for (i=0;i<n;++i)
    {
      x=block.get_x()+block.get_item_x(i);
      y=block.get_y()+block.get_item_y(i);
      if (x>=0 && y>=0 && x<U61_MAP_WIDTH && y<U61_MAP_HEIGHT)
        {
	  squares[x][y]=block.get_item(i)->square;
        }
      else
        {
	  outside=true;
        }
    }
  if (outside)
    {
      evt.code=U61_EVENT_LOOSE;
      put_auto_event(evt);
    }
  else
    {
      match_count=0;
      matching=true;
    }

  block.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * what has to be done when the player looses
 */
void U61_Map::loose()
{
  if (!silent)
    {
      U61_Sound::play_player_loose();
      U61_Music::change();
    }

  U61_LOG_DEBUG("The game is lost");

  begin();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function to match patterns
 * pattern match can be find a line and delete it, or find some colors
 * together, well, it can be pretty much anything...
 * it is usually quite a complex function
 * it also updates the score
 */
void U61_Map::match_pattern()
{
  if (matching)
    {
      if ((U61_Script::match_pattern(this)))
        {
	  if (!silent)
            {
	      U61_Sound::play_block_pattern();
            }
	  match_count++;
        }
      else
        {
	  if (match_count==0 && !silent)
            {
	      if (!silent)
		{
		  U61_Sound::play_block_touch();
		}
            }
	  matching=false;
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback_1
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback_1()
{
  U61_Script::time_callback_1(this,&block);
  /*
   * The call to check_block is very important since time_callback
   * has full rights on the map and block.
   */
  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback_10
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback_10()
{
  U61_Script::time_callback_10(this,&block);
  /*
   * The call to check_block is very important since time_callback
   * has full rights on the map and block.
   */
  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback_100
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback_100()
{
  U61_Script::time_callback_100(this,&block);
  /*
   * The call to check_block is very important since time_callback
   * has full rights on the map and block.
   */
  check_block();
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the update_explosion function for each square,
 * ie each explosion counter will be increased by 1
 */
void U61_Map::update_explosions()
{
  int x,y;
  bool curse_exploded=false;
  U61_Event event;

  for (y=0;y<U61_MAP_HEIGHT;++y)
    for (x=0;x<U61_MAP_WIDTH;++x)
      { 
	squares[x][y].update_explosion();
	if (squares[x][y].is_explosion_ended())
	  {
	    U61_Script::square_blown_up(this,x,y);
	  }
	if (squares[x][y].is_explosion_new())
	  {
            /*
	     * If we find out that the square that just blew up
             * is the special curse square, then we need to launch the
             * script associated to this curse
             */
            if (curse_state && curse_chosen && curse_x==x && curse_y==y)
	      {
                curse_exploded=true;
	      }
	  }
      }

  if (curse_exploded)
    {
      /*
       * Now we could execute the Lua script right away, but we prefer
       * to send an event for 2 reasons:
       * 1 - the curse function can be quite long so it's useless to launch
       *     it too often
       * 2 - this way there's no difference between a curse you got because
       *     someone sent it to you and one you got by yourself matching
       *     a pattern, and I just like this idea
       */
      event.code=U61_EVENT_DO_CURSE;
      event.par=curse_id;
      put_auto_event(event);

      /*
       * The curse has been used so we disable it and ask for another one
       * by forcing curse_counter to curse_delay
       */
      curse_chosen=false;
      curse_asked=false;
      curse_counter=curse_delay;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Changes the curse weapon on the map. The weapon is changed on a regular
 * basis. This is done quite randomly, but still we can garantee it will
 * be the same "random" on every machine since the algorithm is based
 * on the map time.
 */
void U61_Map::new_curse(int id)
{
  int nb_squares;
  int x,y;
  int pos,chosen_pos;
 
  U61_LOG_DEBUG("New curse");

  nb_squares=0;
  for (y=0;y<U61_MAP_HEIGHT;++y)
    for (x=0;x<U61_MAP_WIDTH;++x)
      {
        if (squares[x][y].is_enabled() && 
	    !squares[x][y].is_exploding())
	  {
	    nb_squares++;
	  }
      }
  U61_LOG_DEBUG("nb_squares="<<nb_squares);
  if (nb_squares>0)
    {
      /*
       * OK, now that we know there's at least one square, we choose to
       * put the curse square on one of these squares, the random touch
       * being added by dividing the map_time by the number of squares
       * enabled, and taking the rest. It's very important not to use
       * a random-like function since we need to garantee the map
       * will behave the same on every machine.
       */
      chosen_pos=map_time%nb_squares;
      pos=0;
      for (y=0;y<U61_MAP_HEIGHT;++y)
        for (x=0;x<U61_MAP_WIDTH;++x)
	  {
	    if (squares[x][y].is_enabled() &&
		!squares[x][y].is_exploding())
	      {
		if (pos==chosen_pos)
		  {
		    curse_x=x;
		    curse_y=y;
		    U61_LOG_DEBUG("Curse chosen x="<<x<<" y="<<y);
		  }
		pos++;
	      }	
	  }
      
      curse_id=id;
      curse_chosen=true;
      curse_asked=false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles REQUEST_CURSE events
 */
void U61_Map::request_curse(U61_Event event)
{
  /*
   * Now we have received a request for a curse so
   * we decide to issue a real curse. It's important to call
   * put_auto_event because of time stamp issues.
   */
  event.code=U61_EVENT_DO_CURSE;
  event.par+=U61_MAP_MAX_CURSE_ID;

  put_auto_event(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the lua function corresponding to the curse
 */
void U61_Map::do_curse(U61_Event evt)
{
  if (evt.par<(unsigned int) U61_MAP_MAX_CURSE_ID)
    {
      U61_LOG_DEBUG("Do own curse "<<evt);
      U61_Script::do_curse(this,&block,evt.par,false);
    }
  else
    {
      curse_received++;
      evt.par-=U61_MAP_MAX_CURSE_ID;
      U61_LOG_DEBUG("Do remote curse "<<evt);
      U61_Script::do_curse(this,&block,evt.par,true);
      if (!silent)
	{
	  U61_Sound::play_curse_receive();
	}
    }
  /*
   * The call to check_block is very important since do_curse
   * has full rights on the map and block.
   */
  check_block();
}

/*-------------------------------------------------------------------------*/
/*
 * Re-enables the curse_state if it has been inactive for too long
 * It also asks sets the curse_chosen parameter to false if there's no
 * curse available (this will cause a new curse to be chosen).
 * We use this mecanism to avoid calling is_curse_available all the time.
 */
void U61_Map::check_curse()
{
  if (curse_counter>=curse_delay)
    {
      curse_state=true;
    }
  if (curse_state==true && !is_curse_available())
    {
      curse_chosen=false;
      curse_asked=false;
    }
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the next victim
 */
void U61_Map::next_victim(U61_Event *event)
{
  target_id=U61_Global::layout.get_next_player_id(target_id,player_id);
  event->code=U61_EVENT_SET_VICTIM;
  event->par=target_id;
  U61_LOG_DEBUG("Next victim "<<target_id);
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the prev victim
 */
void U61_Map::prev_victim(U61_Event *event)
{
  target_id=U61_Global::layout.get_prev_player_id(target_id,player_id);
  event->code=U61_EVENT_SET_VICTIM;
  event->par=target_id;
  U61_LOG_DEBUG("Prev victim "<<target_id);
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the specified victim
 */
void U61_Map::set_victim(int id)
{
  target_id=id;
  U61_LOG_DEBUG("Set victim "<<target_id);
}

/*-------------------------------------------------------------------------*/
/*
 * returns the id of the current target
 */
int U61_Map::get_target_id()
{
  return target_id;
}

/*-------------------------------------------------------------------------*/
/*
 * we check that the target we have is not ourself or nobody, and send
 * an auto event if not
 */
void U61_Map::check_target()
{
  int test_id;
  U61_Event event;

  test_id=U61_Global::layout.check_player_id(target_id,player_id);
  if (test_id!=target_id)
    {
      event.code=U61_EVENT_SET_VICTIM;
      event.par=test_id;
      put_auto_event(event);
      U61_LOG_DEBUG("Correcting victim "<<test_id);
    }  
}

/*-------------------------------------------------------------------------*/
/*
 * This function is very important. It checks that the block has a correct
 * position. It is called after most of the Lua scripts which can
 * modify the block and/or the map. Indeed, it is important not to leave
 * the map in an incoherent position.
 * So this function moves the block sideways if it's too much on the left
 * or too much on the right, and then it moves it up until there's no
 * inconsistency.
 */
void U61_Map::check_block()
{
  int i,n,x;
  int min_x,max_x;
  U61_Block test;
  bool ok;
  bool moved_sideways=false;

  n=block.get_nb_items();

  /*
   * First step, we check that the block is not on the right or on
   * the left. Basically we want to put it "back in the middle"
   */
  min_x=U61_MAP_WIDTH-1;
  max_x=0;
  for (i=0;i<n;++i)
    {
      x=block.get_x()+block.get_item_x(i);
      if (x<min_x)
	{
	  min_x=x;
	}
      if (x>max_x)
	{
	  max_x=x;
	}
    }
  /*
   * we move the block to the right if needed
   */
  if (min_x<0)
    {
      block.set_x(block.get_x()-min_x);
    }
  /*
   * we move the block to the left if needed
   */
  if (max_x>U61_MAP_WIDTH-1)
    {
      block.set_x(block.get_x()-max_x+U61_MAP_WIDTH-1);
    }

  /*
   * Now we are going to move the block up until it fits correctly.
   * This should solve a very common problem, which is that when
   * you fill the map with garbage, the block may be stuck in the
   * new added blocks. Therefore we move the block up. Note that in
   * this case the user is very likely to be in trouble to place the
   * block correctly, but there's no way to solve this except make
   * the block disappear, which is not a very good solution.
   */
  ok=is_block_ok(&block);
  while (!ok)
    {
      test=block;
      /*
       * we move the block up of 1 square
       */
      block.set_y(test.get_y()-1);
      if (!is_block_ok(&block))
	{
          /*
           * Ok, now if we get here it means one step up is not enough.
           * So what we'll do is try to move the block sideways a little
           * bit. This avoid the ridiculous situation where a block "climbs"
           * a very high wall to get stuck on it while there's plenty of
           * room elsewhere in the map.
           */
	  test.set_x(test.get_x()-1);
          if (is_block_ok(&test))
	    {
	      /*
	       * The block is OK after it has been moved on the left
	       */
	      moved_sideways=true;
	      block=test;
	      ok=true;
	    }
          else
	    {
	      test.set_x(test.get_x()+2);
	      if (is_block_ok(&test))
		{
		  /*
		   * The block is OK after it has been moved on the right
		   */
		  moved_sideways=true;
		  block=test;
		  ok=true;
		}
	    }
	}
      else
	{
	  /*
	   * The block is OK after it has been moved up.
	   * We move it up of one more square to give the player a chance to
	   * place it, otherwise it would be really unfair. However, it's
	   * still hard to place it correctly, but it's the game...
	   * Note that this is only done if we had needed to move the block
	   * sideways, otherwise one gets some strange buggy boucing effects
	   * when the time callback moves the block down for instance.
	   */
	  if (moved_sideways)
	    {
	      block.set_y(block.get_y()-1);
	    }
	  ok=true;
	}
    }
}

