/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   player.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: a player is a bundle of
 *              - an event source
 *              - an event copy
 *              - a "sure" map
 *              - a "last" map
 *              - a API to handle all that...
 *              it's fundamental to understand the difference between the
 *              last map and the sure map. 
 *              the sure map is a map freezed at the time of the latest
 *              event received. since events are ordered by time, we can
 *              assume this is the right map at this givent instant
 *              the last map is an anticipated map. from the suremap state,
 *              it simulates game cycles so that the map is "as it would have
 *              been now (which may be later than the latest event) if no
 *              event occured". This way, in a network game, map never freeze
 *              waiting for network events, and this gives an impression of
 *              speed even with poor bandwidth
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <string.h>

#include "player.h"
#include "time.h"
#include "localsource.h"
#include "localcopy.h"
#include "networksource.h"
#include "networkcopy.h"
#include "global.h"
#include "debug.h"
#include "sound.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default player
 */ 
U61_Player::U61_Player()
{
  source=NULL;
  copy=NULL;

  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a player
 */ 
U61_Player::~U61_Player()
{

}

/*--------------------------------------------------------------------------*/
/*
 * resets a player to default
 */
void U61_Player::reset()
{
  available=false;
  history_done=true;

  if (source!=NULL)
    {
      delete source;
      source=NULL;
    }
  if (copy!=NULL)
    {
      delete copy;
      copy=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the id of a player
 */
void U61_Player::set_id(int player_id)
{
  id=player_id;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the id of a player
 */
int U61_Player::get_id()
{
  return id;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the id of the target of a player
 */
int U61_Player::get_target_id()
{
  return last.get_target_id();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of a player
 */
char *U61_Player::get_name()
{
  return last.get_name();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the score of a player
 */
int U61_Player::get_score()
{
  return last.get_score();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next block of a player
 */
U61_Block U61_Player::get_next_block()
{
  return last.get_next_block();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of persistent curses activated
 */
int U61_Player::get_nb_curse(bool good)
{
  return last.get_nb_curse(good);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the number of antidotes available
 */
int U61_Player::get_nb_antidote()
{
  return last.get_nb_antidote();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the current curse
 */
char *U61_Player::get_curse_name()
{
  return last.get_curse_name();
}

/*--------------------------------------------------------------------------*/
/*
 * returns the index of the background to be displayed
 */
int U61_Player::get_background()
{
  int bg=-1;

  /*
   * When the player is not active we return -1, for the background
   * does not have to be drawn, and this way we can notice the layout
   * manager that the background has changed, ie the background image
   * has to be redrawn.
   */
  if (last.is_active())
    {
      bg=last.get_background();
    }

  return bg;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the player as a local player
 * Note that even if the game is a networked game, local players are
 * initialized with this function
 */
void U61_Player::init_local(U61_Input *input,
                            U61_Dispatcher *d,
			    int id,
			    int time,
			    U61_PlayerConfig *conf,
			    bool start)
{
  U61_LocalSource *src; 
  U61_LocalCopy *cpy;

  reset();

  local=true;

  set_id(id);
  sure.init(id,
	    U61_Global::game.initial_speed,
	    U61_Global::game.acceleration,
	    U61_Global::game.curse_delay);
  copy_sure_to_last();

  anticipation=&(conf->anticipation);

  src=new U61_LocalSource(input,&sure,d,id,conf->name);
  if (start)
    {
      src->send_start_stop(time);
    }

  cpy=new U61_LocalCopy(id,d);

  source=src;
  copy=cpy;

  available=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the player as a network player
 */
void U61_Player::init_network(U61_Dispatcher *d,
                              int id)
{
  U61_NetworkSource *src; 
  U61_NetworkCopy *cpy;

  reset();

  local=false;;

  set_id(id);
  sure.init(id,
	    U61_Global::game.initial_speed,
	    U61_Global::game.acceleration,
	    U61_Global::game.curse_delay);
  sure.mute();
  copy_sure_to_last();

  anticipation=&(U61_Global::config.network_anticipation);

  src=new U61_NetworkSource(d,&sure,id);
  cpy=new U61_NetworkCopy(id,d);

  source=src;
  copy=cpy;

  U61_LOG_DEBUG("Starting network player "<<id);

  available=true;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is available
 */
bool U61_Player::is_available()
{
  return available;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is active
 */
bool U61_Player::is_active()
{
  return last.is_active();;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the player is a local player
 */
bool U61_Player::is_local()
{
  return local;
}

/*--------------------------------------------------------------------------*/
/*
 * this function pumps the events for a player, ie it:
 * read the events
 * interprets them
 * makes a copy of them if necessary
 */
void U61_Player::pump_events()
{
  bool changed=false;
  U61_Event evt;
  bool kill_needed=false;

  source->poll();

  while((!source->empty()) && 
        ((source->check().time))<=U61_Time::for_logic())
    {
      changed=true;
      evt=source->get();
      handle_event(&evt);
      copy->put(evt);
      switch (evt.code)
	{
	case U61_EVENT_KILL:
	  kill_needed=true;
	  break;
        case U61_EVENT_LOOSE:
	  register_score();
	  break;
        case U61_EVENT_HISTORY_BEGIN:
	  history_done=false;
	  break;
        case U61_EVENT_HISTORY_END:
	  history_done=true;
	  break;
	}
    }

  while (sure.exists_request_event())
    {
      evt=sure.get_request_event();
      copy->put(evt);
    }

  if (changed)
    {
      copy_sure_to_last();
    }

  copy->flush();

  if (kill_needed)
    {
      kill();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * pumps an event
 */
void U61_Player::handle_event(U61_Event *evt)
{
  if (sure.is_active())
    {
      if (evt->code==U61_EVENT_START_STOP)
	{
	  stop();
	}
      else
	{
	  sure.handle_event(evt);
	}
    }
  else
    {
      if (evt->code==U61_EVENT_START_STOP)
	{
	  start(evt->time);
	} 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * called to start a player, ie enter him in the game
 */
void U61_Player::start(int time)
{
  sure.set_active(time);
  copy_sure_to_last();

  U61_LOG_DEBUG("Player "<<id<<" started");
}

/*--------------------------------------------------------------------------*/
/*
 * called to stop a player, ie remove him from the game
 */
void U61_Player::stop()
{
  register_score();
  sure.set_inactive();
  copy_sure_to_last();
  U61_LOG_DEBUG("Player "<<id<<" stopped");
}

/*--------------------------------------------------------------------------*/
/*
 * called to kill a player. This is stronger than a stop, it should
 * happen when a remote network players quits the game.
 * In this case we kill the player to free the player slot.
 */
void U61_Player::kill()
{
  register_score();
  reset();
  //available=false;
  U61_LOG_DEBUG("Player "<<id<<" killed");
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map of the player
 */
void U61_Player::draw(int x,int y,int size)
{
  if (history_done)
    {
      last.anticipate(U61_Time::for_event());
      last.draw(x,y,size,*anticipation);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * copies the sure map to the last map
 */
void U61_Player::copy_sure_to_last()
{
  last=sure;
  last.mute();
}

/*--------------------------------------------------------------------------*/
/*
 * copies the sure map to the last map
 */
void U61_Player::register_score()
{
  if (sure.is_active() && sure.get_elapsed_time()>0)
    {
      U61_Global::score.register_game(sure.get_name(),
				      local,
				      sure.get_score(),
				      sure.get_elapsed_time(),
				      sure.get_curse_sent(),
				      sure.get_curse_received());
    }
}
