/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   script.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: wraps the LUA api to C++ and vice-versa
 *              all the interaction with lua is contained in this file
 *              it might be splited some day since it's getting *big*
 *              lua scripting is fundamental in u61. in fact all the
 *              rules of the game are coded in lua. this allows a 
 *              player/hacker  to write his own rules and test them
 *              right away without recompiling, and be quite
 *              "protection fault" safe
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/core.h>
#include <ClanLib/lua.h>
#include <stdio.h>
#include <string.h>

#include "script.h"
#include "const.h"
#include "utils.h"
#include "debug.h"
#include "global.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_SCRIPT_LUA_NEW_SHAPE "user_new_shape"
#define U61_SCRIPT_LUA_DO_SHAPE "user_do_shape"
#define U61_SCRIPT_LUA_ROTATE_LEFT "user_rotate_left"
#define U61_SCRIPT_LUA_ROTATE_RIGHT "user_rotate_right"
#define U61_SCRIPT_LUA_MOVE_LEFT "user_move_left"
#define U61_SCRIPT_LUA_MOVE_RIGHT "user_move_right"
#define U61_SCRIPT_LUA_MOVE_DOWN "user_move_down"
#define U61_SCRIPT_LUA_LAND "user_land"
#define U61_SCRIPT_LUA_MATCH_PATTERN "user_match_pattern"
#define U61_SCRIPT_LUA_TIME_CALLBACK_1 "user_time_callback_1"
#define U61_SCRIPT_LUA_TIME_CALLBACK_10 "user_time_callback_10"
#define U61_SCRIPT_LUA_TIME_CALLBACK_100 "user_time_callback_100"
#define U61_SCRIPT_LUA_SQUARE_BLOWN_UP "user_square_blown_up"
#define U61_SCRIPT_LUA_NEW_CURSE "user_new_curse"
#define U61_SCRIPT_LUA_DO_CURSE "user_do_curse"
#define U61_SCRIPT_LUA_USE_ANTIDOTE "user_use_antidote"
#define U61_SCRIPT_LUA_GET_CURSE_NAME "user_get_curse_name"

#define U61_SCRIPT_C_CLEAR_BLOCK "u61_clear_block"
#define U61_SCRIPT_C_CENTER_BLOCK "u61_center_block"
#define U61_SCRIPT_C_GET_BLOCK_X "u61_get_block_x"
#define U61_SCRIPT_C_GET_BLOCK_Y "u61_get_block_y"
#define U61_SCRIPT_C_SET_BLOCK_X "u61_set_block_x"
#define U61_SCRIPT_C_SET_BLOCK_Y "u61_set_block_y"
#define U61_SCRIPT_C_ADD_ITEM "u61_add_item"
#define U61_SCRIPT_C_GET_NB_ITEMS "u61_get_nb_items"
#define U61_SCRIPT_C_GET_ITEM_X "u61_get_item_x"
#define U61_SCRIPT_C_GET_ITEM_Y "u61_get_item_y"
#define U61_SCRIPT_C_GET_ITEM_COLOR "u61_get_item_color"
#define U61_SCRIPT_C_SET_ITEM_X "u61_set_item_x"
#define U61_SCRIPT_C_SET_ITEM_Y "u61_set_item_y"
#define U61_SCRIPT_C_SET_ITEM_COLOR "u61_set_item_color"
#define U61_SCRIPT_C_GET_WIDTH "u61_get_width"
#define U61_SCRIPT_C_GET_HEIGHT "u61_get_height"
#define U61_SCRIPT_C_GET_SQUARE_COLOR "u61_get_square_color"
#define U61_SCRIPT_C_SET_SQUARE_COLOR "u61_set_square_color"
#define U61_SCRIPT_C_GET_TIME "u61_get_time"
#define U61_SCRIPT_C_GET_SCORE "u61_get_score"
#define U61_SCRIPT_C_SET_SCORE "u61_set_score"
#define U61_SCRIPT_C_ADD_SCORE "u61_add_score"
#define U61_SCRIPT_C_GET_GLOBAL "u61_get_global"
#define U61_SCRIPT_C_SET_GLOBAL "u61_set_global"
#define U61_SCRIPT_C_GET_ANTICIPATION_STATE "u61_get_anticipation_state"
#define U61_SCRIPT_C_SET_ANTICIPATION_STATE "u61_set_anticipation_state"
#define U61_SCRIPT_C_GET_PREVIEW_STATE "u61_get_preview_state"
#define U61_SCRIPT_C_SET_PREVIEW_STATE "u61_set_preview_state"
#define U61_SCRIPT_C_BLOW_UP_SQUARE "u61_blow_up_square"
#define U61_SCRIPT_C_IS_SQUARE_EXPLODING "u61_is_square_exploding"
#define U61_SCRIPT_C_GET_CURSE_X "u61_get_curse_x"
#define U61_SCRIPT_C_GET_CURSE_Y "u61_get_curse_y"
#define U61_SCRIPT_C_GET_CURSE_STATE "u61_get_curse_state"
#define U61_SCRIPT_C_SET_CURSE_X "u61_set_curse_x"
#define U61_SCRIPT_C_SET_CURSE_Y "u61_set_curse_y"
#define U61_SCRIPT_C_SET_CURSE_STATE "u61_set_curse_state"
#define U61_SCRIPT_C_IS_CURSE_AVAILABLE "u61_is_curse_available"
#define U61_SCRIPT_C_SEND_CURSE "u61_send_curse"
#define U61_SCRIPT_C_GET_CURSE_AGE "u61_get_curse_age"
#define U61_SCRIPT_C_REGISTER_CURSE "u61_register_curse"
#define U61_SCRIPT_C_CANCEL_CURSE "u61_cancel_curse"
#define U61_SCRIPT_C_GET_OLDEST_CURSE "u61_get_oldest_curse"
#define U61_SCRIPT_C_GET_NB_CURSES "u61_get_nb_curses"
#define U61_SCRIPT_C_GET_NB_ANTIDOTES "u61_get_nb_antidotes"
#define U61_SCRIPT_C_ADD_ANTIDOTE "u61_add_antidote"
#define U61_SCRIPT_C_DELETE_ANTIDOTE "u61_delete_antidote"
#define U61_SCRIPT_C_IS_BLOCK_OK "u61_is_block_ok"

#define U61_SCRIPT_RANDOM_MAX 0x7FFFFFFF

/*--------------------------------------------------------------------------*/
/*
 * the following pointers allow tradionnal C functions, called by Lua,
 * to act directly on a specified object
 * this way, when one writes lua code, one only has to assume that
 * functions always operate on the right block, the right map etc...
 * they are not members of any class since they are used from
 * standard C function
 */
static U61_Block *current_block=NULL;
static U61_Map *current_map=NULL;

/*--------------------------------------------------------------------------*/
/*
 * the following booleans allows a full control of what exactly functions
 * called from lua have the right to read/write
 * for instance one may wish a function to be able to read the map state
 * but bot overwrite it
 * they are not members of any class since they are used from
 * standard C function
 */
static bool block_read=false;
static bool block_write=false;
static bool map_read=false;
static bool map_write=false;

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * initializes the script module with a string
 */
bool U61_Script::do_string(char *str)
{
  bool bSuccess;
  int iResult;

  bSuccess=false;

  iResult=CL_Lua::dostring(str);
  switch (iResult)
    {
    case 0:
      bSuccess=true;
      break;
    default:
      U61_LOG_WARNING("Lua error reading script, code is "<<iResult<<".");
      break;
    }

  return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * initializes the script module with a file
 */
bool U61_Script::do_file(char *name)
{
  bool bSuccess;
  int iResult;
  
  U61_Global::data.chdir_to_initial_dir();

  bSuccess=false;

  //lua_open();
  U61_LOG_DEBUG("Loading script from \""<<name<<"\".");
  iResult=CL_Lua::dofile(name);
  switch (iResult)
    {
    case 0:
      bSuccess=true;
      break;
    default:
      U61_LOG_WARNING("Lua error reading script, code is "<<iResult<<".");
      break;
    }
  
  U61_Global::data.chdir_to_data_dir();

  return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as a string
 */
char *U61_Script::get_global_str(char *name,char *def)
{
  CL_LuaObject obj;
  char *result=def;

  obj=CL_Lua::getglobal(name);
  if (obj.isString())
    {
      result=obj;
    }    

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as an integer
 */
int U61_Script::get_global_int(char *name,int def)
{
  CL_LuaObject obj;
  float result=def;

  obj=CL_Lua::getglobal(name);
  if (obj.isNumber())
    {
      result=obj;
    }    

  return (int) result;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves a global lua value as a boolean
 */
bool U61_Script::get_global_bool(char *name,bool def)
{
  CL_LuaObject obj;
  bool result=def;

  obj=CL_Lua::getglobal(name);
  if (obj.isNumber())
    {
      result=(((float) obj)!=0);
    }    

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * registers all C functions that can be used from Lua
 */
bool U61_Script::init()
{
  bool bSuccess=true;

  CL_Lua::mathlibopen();
  record_all();

  return bSuccess;
}

/*--------------------------------------------------------------------------*/
/*
 * forces the lua garbage collector
 */
void U61_Script::collect_garbage()
{
  /*  lua_Object poped;

      while (!lua_isnil(poped=lua_pop()))
      {
      U61_LOG_DEBUG("Popping lua object");
      }*/

  lua_collectgarbage(0);
}

/*--------------------------------------------------------------------------*/
/*
 * cancels previous calls to prepare_for_???
 * this way, if an lua script tries to access data it is not allowed to,
 * no protection fault will occur (at least that's what we hope...)
 */
void U61_Script::secure_pointers()
{
  current_block=NULL;
  current_map=NULL;
  block_read=false;
  block_write=false;
  map_read=false;
  map_write=false;
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the code for the next block.
 * The value returned will be passed later to block_shape
 */
int U61_Script::new_shape(U61_Map *m,U61_Block *b)
{
  float result=0.;
  lua_Object func,ret;

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;

  lua_beginblock();
  /*
   * Now I use lua API directly and not ClanLib's wrapper since I could
   * not manage to get it work...
   */
  lua_pushnumber(U61_Utils::random(U61_SCRIPT_RANDOM_MAX));

  func=lua_getglobal(U61_SCRIPT_LUA_NEW_SHAPE);
  if (lua_isfunction(func))
    {
      lua_callfunction(func);
    }

  ret=lua_getresult(1);
  if (lua_isnumber(ret))
    {
      result=lua_getnumber(ret);
    }
  lua_endblock();

  return ((int) result);
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that creates shapes
 * it does not perform clear nor center functions
 */
void U61_Script::do_shape(U61_Map *m,U61_Block *b, int type)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"(%d)",type);
  strcpy(command,U61_SCRIPT_LUA_DO_SHAPE);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the right
 * the Lua function may or may not do a geometrical rotation, it
 * may for instance be color cycling
 */
void U61_Script::rotate_right(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_ROTATE_RIGHT);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the left
 * the Lua function may or may not do a geometrical rotation, it
 * may for instance be color cycling
 */
void U61_Script::rotate_left(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_ROTATE_LEFT);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that moves a block on the right
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::move_right(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_MOVE_RIGHT);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block on the left
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::move_left(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_MOVE_LEFT);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that rotates a block down
 * the Lua function may or may not do a geometrical translation it
 * may include rotations
 */
void U61_Script::move_down(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_MOVE_DOWN);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function which is meant to be call before
 * the block is stabilized/included in the map
 * the Lua function may leave the block unchanged
 */
void U61_Script::land(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_block=b;
  block_read=true;
  block_write=true;
  current_map=m;
  map_read=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_LAND);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that finds patterns in a map
 * and deletes the approprated blocks
 * this function should also update the score
 */
bool U61_Script::match_pattern(U61_Map *m)
{
  bool result=false;
  lua_Object func,ret;

  secure_pointers();
  current_map=m;
  map_read=true;
  map_write=true;

  lua_beginblock();
  /*
   * Now I use lua API directly and not ClanLib's wrapper since I could
   * not manage to get it work...
   */
  lua_pushnumber(current_map->get_match_count());

  func=lua_getglobal(U61_SCRIPT_LUA_MATCH_PATTERN);
  if (lua_isfunction(func))
    {
      lua_callfunction(func);
    }

  ret=lua_getresult(1);
  if (lua_isnumber(ret))
    {
      result=(((float) lua_getnumber(ret))!=0.);
    }
  lua_endblock();

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * function called on a regular time basis
 * it can update or modify anything in the map or the block
 * usefull to implement some curses, and/or update the score
 * it is called about 10 times/sec, and the rate does not increase
 * with the global speed of the game
 */
void U61_Script::time_callback_1(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_TIME_CALLBACK_1);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * function called on a regular time basis
 * it can update or modify anything in the map or the block
 * usefull to implement some curses, and/or update the score
 * it is called about 10 times/sec, and the rate does not increase
 * with the global speed of the game
 */
void U61_Script::time_callback_10(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_TIME_CALLBACK_10);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * function called on a regular time basis
 * it can update or modify anything in the map or the block
 * usefull to implement some curses, and/or update the score
 * it is called about 10 times/sec, and the rate does not increase
 * with the global speed of the game
 */
void U61_Script::time_callback_100(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  //func=CL_Lua::getglobal(U61_SCRIPT_LUA_NEW_BLOCK);
  //lua_callfunction(func);

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_TIME_CALLBACK_100);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that handles the end of an explosion
 * Usually it makes the square disappear and shifts the upper squares
 * of one row down, but it can do other things, for instance blow up
 * some other blocks!
 */
void U61_Script::square_blown_up(U61_Map *m,int x,int y)
{
  CL_LuaObject func;
  CL_LuaObject obj;
  CL_LuaObject arg1;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  map_read=true;
  map_write=true;

  //  lua_pushnumber(x);
  //  lua_pushnumber(y);
  //  lua_callfunction(lua_getglobal(U61_SCRIPT_LUA_MAP_SQUARE_BLOWN_UP));

  sprintf(arguments,"(%d,%d)",x,y);
  strcpy(command,U61_SCRIPT_LUA_SQUARE_BLOWN_UP);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the code for the next curse.
 * The value returned will be used by the next call to do_curse
 */
int U61_Script::new_curse(U61_Map *m,U61_Block *b)
{
  float result=0.;
  lua_Object func,ret;

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  block_read=true;

  lua_beginblock();
  /*
   * Now I use lua API directly and not ClanLib's wrapper since I could
   * not manage to get it work...
   */
  lua_pushnumber(U61_Utils::random(U61_SCRIPT_RANDOM_MAX));

  func=lua_getglobal(U61_SCRIPT_LUA_NEW_CURSE);
  if (lua_isfunction(func))
    {
      lua_callfunction(func);
    }

  ret=lua_getresult(1);
  if (lua_isnumber(ret))
    {
      result=lua_getnumber(ret);
    }
  lua_endblock();

  return ((int) result);
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the Lua script function taht performs the action linked to a curse
 * This is were most of the curse effects should be coded
 */
void U61_Script::do_curse(U61_Map *m,U61_Block *b,int id,bool sent)
{
  CL_LuaObject func;
  CL_LuaObject obj;
  CL_LuaObject arg1;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  //  lua_pushnumber(x);
  //  lua_pushnumber(y);
  //  lua_callfunction(lua_getglobal(U61_SCRIPT_LUA_MAP_SQUARE_BLOWN_UP));

  sprintf(arguments,"(%d,%d)",id,sent ? 1 : 0);
  strcpy(command,U61_SCRIPT_LUA_DO_CURSE);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that should be called when using an 
 * antidote
 */
void U61_Script::use_antidote(U61_Map *m,U61_Block *b)
{
  CL_LuaObject func;
  char command[U61_CONST_STRING_SIZE];
  char arguments[U61_CONST_STRING_SIZE];

  secure_pointers();
  current_map=m;
  current_block=b;
  map_read=true;
  map_write=true;
  block_read=true;
  block_write=true;

  sprintf(arguments,"()");
  strcpy(command,U61_SCRIPT_LUA_USE_ANTIDOTE);
  strcat(command,arguments);
  lua_beginblock();
  CL_Lua::dostring(command);
  lua_endblock();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the user defined function that returns the name of a curse.
 * The value returned will be dispalyed in the player's status zone
 */
char *U61_Script::get_curse_name(U61_Map *m,int id)
{
  static char result[U61_CONST_STRING_SIZE];
  char *str=NULL;
  lua_Object func,ret;

  secure_pointers();
  current_map=m;
  map_read=true;

  lua_beginblock();
  /*
   * Now I use lua API directly and not ClanLib's wrapper since I could
   * not manage to get it work...
   */
  lua_pushnumber(id);
  
  func=lua_getglobal(U61_SCRIPT_LUA_GET_CURSE_NAME);
  if (lua_isfunction(func))
    {
      lua_callfunction(func);
    }

  ret=lua_getresult(1);
  if (lua_isstring(ret))
    {
      str=lua_getstring(ret);
      if(str==NULL)
	{
	  U61_LOG_WARNING("Curse name is non-string");
	}
    }

  if (str==NULL)
    {
      str="???";
    }
  strcpy(result,str);
  lua_endblock();

  return (result);
}

/*--------------------------------------------------------------------------*/
/*
 * this function is a simple tool to return an lua integer within lua
 * script. It includes some basic checks. 
 */
static void return_lua_int(int ret)
{
  CL_LuaValue *val;

  val=new CL_LuaValue(ret);

  if (val!=NULL)
    {
      val->push();
      delete val;
    }
  else
    {
      U61_LOG_WARNING("Error returning lua int");
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it completely clears a block, ie after it the block has no squares left...
 */
static void clear_block()
{
  if (current_block!=NULL && block_write)
    {
      current_block->clear();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it centers a block, ie the square of coord 0,0 will be in the middle
 */
static void center_block()
{
  if (current_block!=NULL && block_write)
    {
      current_block->center();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the x position of the block
 */
static void get_block_x()
{
  if (current_block!=NULL && block_read)
    {
      return_lua_int (current_block->get_x());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the y position of the block
 */
static void get_block_y()
{
  if (current_block!=NULL && block_read)
    {
      return_lua_int (current_block->get_y());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x position of the block
 */
static void set_block_x()
{
  int x;

  if (current_block!=NULL && block_write)
    {
      x=CL_Lua::getparam(1);

      current_block->set_x(x);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the y position of the block
 */
static void set_block_y()
{
  int y;

  if (current_block!=NULL && block_write)
    {
      y=CL_Lua::getparam(1);

      current_block->set_y(y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds a block item (ie a square) to the current block
 */
static void add_item()
{
  int type,x,y;

  if (current_block!=NULL && block_write)
    {
      x=CL_Lua::getparam(1);        
      y=CL_Lua::getparam(2);        
      type=CL_Lua::getparam(3);        

      current_block->add_item(x,y,type);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of items (ie of squares) in a block
 */
static void get_nb_items()
{
  if (current_block!=NULL && block_read)
    {
      return_lua_int (current_block->get_nb_items());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the x coordinate of a block item (a square)
 */
static void get_item_x()
{
  int i;

  if (current_block!=NULL && block_read)
    {
      i=CL_Lua::getparam(1);

      return_lua_int (current_block->get_item_x(i));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the y coordinate of a block item (a square)
 */
static void get_item_y()
{
  int i;

  if (current_block!=NULL && block_read)
    {
      i=CL_Lua::getparam(1);

      return_lua_int (current_block->get_item_y(i));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the color of a block item (a square)
 */
static void get_item_color()
{
  int i;

  if (current_block!=NULL && block_read)
    {
      i=CL_Lua::getparam(1);

      return_lua_int (current_block->get_item_color(i));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static void set_item_x()
{
  int i,x;

  if (current_block!=NULL && block_write)
    {
      i=CL_Lua::getparam(1);
      x=CL_Lua::getparam(2);

      current_block->set_item_x(i,x);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static void set_item_y()
{
  int i,y;

  if (current_block!=NULL && block_write)
    {
      i=CL_Lua::getparam(1);
      y=CL_Lua::getparam(2);

      current_block->set_item_y(i,y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the x coordinate  of a block item (a square)
 */
static void set_item_color()
{
  int i,color;

  if (current_block!=NULL && block_write)
    {
      i=CL_Lua::getparam(1);
      color=CL_Lua::getparam(2);

      current_block->set_item_color(i,color);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the width of the map
 */
static void get_width()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (U61_MAP_WIDTH);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the height of the map
 */
static void get_height()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (U61_MAP_HEIGHT);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it gets the color of a given square, color -1 means no square
 */
static void get_square_color()
{
  int x,y;

  if (current_map!=NULL && map_read)
    {
      x=CL_Lua::getparam(1);
      y=CL_Lua::getparam(2);

      return_lua_int (current_map->get_square_color(x,y));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the color of a given square, color -1 means no square
 */
static void set_square_color()
{
  int x,y,color;

  if (current_map!=NULL && map_write)
    {
      x=CL_Lua::getparam(1);
      y=CL_Lua::getparam(2);
      color=CL_Lua::getparam(3);

      current_map->set_square_color(x,y,color);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the time associated to a map
 */
static void get_time()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (current_map->get_time());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the score associated to a map
 */
static void get_score()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (current_map->get_score());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the value of the score associated to a map
 */
static void set_score()
{
  int score;

  if (current_map!=NULL && map_write)
    {
      score=CL_Lua::getparam(1);

      current_map->set_score(score);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds the value to the score associated to a map
 */
static void add_score()
{
  int score;

  if (current_map!=NULL && map_write)
    {
      score=CL_Lua::getparam(1);

      current_map->add_score(score);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get global values associated to a map
 */
static void get_global()
{
  int i;

  if (current_map!=NULL && map_read)
    {
      i=CL_Lua::getparam(1);
      return_lua_int (current_map->get_global(i));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set global values associated to a map
 */
static void set_global()
{
  int i,glob;

  if (current_map!=NULL && map_write)
    {
      i=CL_Lua::getparam(1);
      glob=CL_Lua::getparam(2);

      current_map->set_global(i,glob);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the anticipation state, ie if the anticipation is to be seen
 */
static void get_anticipation_state()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (current_map->get_anticipation_state() ? 1 : 0);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the anticipation state, ie if the anticipation is to be seen
 */
static void set_anticipation_state()
{
  bool state;

  if (current_map!=NULL && map_write)
    {
      state=(((float) CL_Lua::getparam(1))!=0.);

      current_map->set_anticipation_state(state);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the preview state, ie if the preview is to be seen
 */
static void get_preview_state()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int(current_map->get_preview_state() ? 1 : 0);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it sets the preview state, ie if the preview is to be seen
 */
static void set_preview_state()
{
  bool state;

  if (current_map!=NULL && map_write)
    {
      state=(((float) CL_Lua::getparam(1))!=0.);

      current_map->set_preview_state(state);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Blows up a square, ie puts this square in a exploding state
 */
static void blow_up_square()
{
  int x,y;

  if (current_map!=NULL && map_write)
    {
      x=CL_Lua::getparam(1);
      y=CL_Lua::getparam(2);

      current_map->blow_up_square(x,y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns != 0 if the square is exploding, and 0 otherwise
 */
static void is_square_exploding()
{
  int x,y;

  if (current_map!=NULL && map_read)
    {
      x=CL_Lua::getparam(1);
      y=CL_Lua::getparam(2);

      return_lua_int (
		      current_map->is_square_exploding(x,y)
		      ? 1 :0);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the x coordinate of the curse square
 */
static void get_curse_x()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (current_map->get_curse_x());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the y coordinate of the curse square
 */
static void get_curse_y()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (current_map->get_curse_y());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the state of the curse square
 */
static void get_curse_state()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int (current_map->get_curse_state());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set the x coordinate of the curse square
 */
static void set_curse_x()
{
  int x;

  if (current_map!=NULL && map_write)
    {
      x=CL_Lua::getparam(1);

      current_map->set_curse_x(x);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set the y coordinate of the curse square
 */
static void set_curse_y()
{
  int y;

  if (current_map!=NULL && map_write)
    {
      y=CL_Lua::getparam(1);

      current_map->set_curse_y(y);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to set the state of the curse square
 */
static void set_curse_state()
{
  bool state;

  if (current_map!=NULL && map_write)
    {
      state=(((float) CL_Lua::getparam(1))!=0.);

      current_map->set_curse_state(state);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to know if the curse square is set and
 * is set on a valid square (ie a non-empty square)
 */
static void is_curse_available()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int(current_map->is_curse_available());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to send a curse to the target player
 */
static void send_curse()
{
  int par;

  if (current_map!=NULL && map_write)
    {
      par=CL_Lua::getparam(1);

      current_map->send_curse(par);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the age of a registered curse
 */
static void get_curse_age()
{
  int i;

  if (current_map!=NULL && map_read)
    {
      i=CL_Lua::getparam(1);

      return_lua_int(current_map->get_curse_age(i));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to register a curse
 */
static void register_curse()
{
  int i;
  int length;
  bool good;

  if (current_map!=NULL && map_write)
    {
      i=CL_Lua::getparam(1);
      length=CL_Lua::getparam(2);
      good=(((float) CL_Lua::getparam(3))!=0.);

      current_map->register_curse(i,length,good);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to cancel a registered curse
 */
static void cancel_curse()
{
  int i;

  if (current_map!=NULL && map_write)
    {
      i=CL_Lua::getparam(1);

      current_map->cancel_curse(i);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it allows the user scripts to get the id of the oldest curse
 */
static void get_oldest_curse()
{
  bool good;

  if (current_map!=NULL && map_read)
    {
      good=(((float) CL_Lua::getparam(1))!=0.);

      return_lua_int(current_map->get_oldest_curse(good));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of persistent curses activated
 */
static void get_nb_curses()
{
  bool good;

  if (current_map!=NULL && map_read)
    {
      good=(((float) CL_Lua::getparam(1))!=0.);

      return_lua_int(current_map->get_nb_curse(good));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it returns the number of antidotes available
 */
static void get_nb_antidotes()
{
  if (current_map!=NULL && map_read)
    {
      return_lua_int(current_map->get_nb_antidote());
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it adds an antidote
 */
static void add_antidote()
{
  if (current_map!=NULL && map_write)
    {
      current_map->add_antidote();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it deletes an antidote
 */
static void delete_antidote()
{
  if (current_map!=NULL && map_write)
    {
      current_map->delete_antidote();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * this function is accessible from lua script 
 * it checks if the current block has a correct position
 */
static void is_block_ok()
{
  if (current_map!=NULL && map_read && current_block!=NULL && block_read)
    {
      return_lua_int(current_map->is_block_ok(current_block));
    }
}

/*--------------------------------------------------------------------------*/
/*
 * registers all C functions that can be used from Lua
 */
void U61_Script::record_all()
{
  CL_Lua::record(U61_SCRIPT_C_CLEAR_BLOCK,clear_block);
  CL_Lua::record(U61_SCRIPT_C_CENTER_BLOCK,center_block);
  CL_Lua::record(U61_SCRIPT_C_GET_BLOCK_X,get_block_x);
  CL_Lua::record(U61_SCRIPT_C_GET_BLOCK_Y,get_block_y);
  CL_Lua::record(U61_SCRIPT_C_SET_BLOCK_X,set_block_x);
  CL_Lua::record(U61_SCRIPT_C_SET_BLOCK_Y,set_block_y);
  CL_Lua::record(U61_SCRIPT_C_ADD_ITEM,add_item);
  CL_Lua::record(U61_SCRIPT_C_GET_NB_ITEMS,get_nb_items);
  CL_Lua::record(U61_SCRIPT_C_GET_ITEM_X,get_item_x);
  CL_Lua::record(U61_SCRIPT_C_GET_ITEM_Y,get_item_y);
  CL_Lua::record(U61_SCRIPT_C_GET_ITEM_COLOR,get_item_color);
  CL_Lua::record(U61_SCRIPT_C_SET_ITEM_X,set_item_x);
  CL_Lua::record(U61_SCRIPT_C_SET_ITEM_Y,set_item_y);
  CL_Lua::record(U61_SCRIPT_C_SET_ITEM_COLOR,set_item_color);
  CL_Lua::record(U61_SCRIPT_C_GET_WIDTH,get_width);
  CL_Lua::record(U61_SCRIPT_C_GET_HEIGHT,get_height);
  CL_Lua::record(U61_SCRIPT_C_GET_SQUARE_COLOR,get_square_color);
  CL_Lua::record(U61_SCRIPT_C_SET_SQUARE_COLOR,set_square_color);
  CL_Lua::record(U61_SCRIPT_C_GET_TIME,get_time);
  CL_Lua::record(U61_SCRIPT_C_GET_SCORE,get_score);
  CL_Lua::record(U61_SCRIPT_C_SET_SCORE,set_score);
  CL_Lua::record(U61_SCRIPT_C_ADD_SCORE,add_score);
  CL_Lua::record(U61_SCRIPT_C_GET_GLOBAL,get_global);
  CL_Lua::record(U61_SCRIPT_C_SET_GLOBAL,set_global);
  CL_Lua::record(U61_SCRIPT_C_GET_ANTICIPATION_STATE,get_anticipation_state);
  CL_Lua::record(U61_SCRIPT_C_SET_ANTICIPATION_STATE,set_anticipation_state);
  CL_Lua::record(U61_SCRIPT_C_GET_PREVIEW_STATE,get_preview_state);
  CL_Lua::record(U61_SCRIPT_C_SET_PREVIEW_STATE,set_preview_state);
  CL_Lua::record(U61_SCRIPT_C_BLOW_UP_SQUARE,blow_up_square);
  CL_Lua::record(U61_SCRIPT_C_IS_SQUARE_EXPLODING,is_square_exploding);
  CL_Lua::record(U61_SCRIPT_C_GET_CURSE_X,get_curse_x);
  CL_Lua::record(U61_SCRIPT_C_GET_CURSE_Y,get_curse_y);
  CL_Lua::record(U61_SCRIPT_C_GET_CURSE_STATE,get_curse_state);
  CL_Lua::record(U61_SCRIPT_C_SET_CURSE_X,set_curse_x);
  CL_Lua::record(U61_SCRIPT_C_SET_CURSE_Y,set_curse_y);
  CL_Lua::record(U61_SCRIPT_C_SET_CURSE_STATE,set_curse_state);
  CL_Lua::record(U61_SCRIPT_C_IS_CURSE_AVAILABLE,is_curse_available);
  CL_Lua::record(U61_SCRIPT_C_SEND_CURSE,send_curse);
  CL_Lua::record(U61_SCRIPT_C_GET_CURSE_AGE,get_curse_age);
  CL_Lua::record(U61_SCRIPT_C_REGISTER_CURSE,register_curse);
  CL_Lua::record(U61_SCRIPT_C_CANCEL_CURSE,cancel_curse);
  CL_Lua::record(U61_SCRIPT_C_GET_OLDEST_CURSE,get_oldest_curse);
  CL_Lua::record(U61_SCRIPT_C_GET_NB_CURSES,get_nb_curses);
  CL_Lua::record(U61_SCRIPT_C_GET_NB_ANTIDOTES,get_nb_antidotes);
  CL_Lua::record(U61_SCRIPT_C_ADD_ANTIDOTE,add_antidote);
  CL_Lua::record(U61_SCRIPT_C_DELETE_ANTIDOTE,delete_antidote);
  CL_Lua::record(U61_SCRIPT_C_IS_BLOCK_OK,is_block_ok);
}





