/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   platformunix.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: contains unix-specific code.
 *              for instance, the directory used for data storing are
 *              different under linux or windows
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/core.h>
#include <ClanLib/lua.h>
#include <ClanLib/display.h>
#include <ClanLib/sound.h>
#include <ClanLib/network.h>
#include <ClanLib/mikmod.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <signal.h>

#include "platform.h"
#include "const.h"
#include "debug.h"
#include "script.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * This file is used to dump the scores and keep a trace of what's happening.
 * On UNIX it is located in ~/.u61/score.txt
 */
static FILE *score_file=NULL;

/*--------------------------------------------------------------------------*/
/*
 * The file is used to log the debug messages, it is stored in ~/.u61 and
 * not in /var/log since it can get enormous and as it is used for
 * debugging, it is to be used by a developper (me for instance) and
 * I find it more convenient to have this file in my own directory tree
 * with my permissions
 */
static FILE *debug_file=NULL;

/*--------------------------------------------------------------------------*/
/*
 * global variables for directory scan functions
 */
static DIR *scan_dir=NULL;
static char *scan_ext=NULL;

/*--------------------------------------------------------------------------*/
/*
 * This array contains some typical paths where the game could be installed.
 * In case the path specified in the config file fails, we try all these
 * paths. This way the game with still work even if the player has a static
 * binary exe that is installed in /usr/bin and a custome built datafile
 * which is in /usr/local/share...
 */
#define NB_STANDARD_PATHS 6
static char *standard_paths[NB_STANDARD_PATHS]=
{
  "../share/u61/",
  U61_INSTALL_DATA_PATH,
  "/usr/local/share/u61/",
  "/usr/share/u61/",
  "/usr/local/share/games/u61/",
  "/usr/share/games/u61/"
};

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * used for various UNIX related global initializations
 */
void U61_Platform::init()
{
  signal(SIGPIPE,SIG_IGN);
}

/*--------------------------------------------------------------------------*/
/*
 * used for various UNIX related global de-initializations
 */
void U61_Platform::deinit()
{
  
}

/*--------------------------------------------------------------------------*/
/*
 * returns the current working directory
 */
char *U61_Platform::getcwd()
{
  static char buffer[U61_CONST_STRING_SIZE]; 

  buffer[0]='.';
  buffer[1]='\0';
  ::getcwd(buffer,sizeof(buffer));

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * Changes the current working directory.
 * This is very usefull to be able to load .scr like datafiles, since at the
 * time I'm writing these lines ClanLib requires that the .scr is in the
 * current working directory.
 */
void U61_Platform::chdir(char *dir)
{
  U61_LOG_DEBUG("chdir to "<<dir);
  ::chdir(dir);
}

/*--------------------------------------------------------------------------*/
/*
 * keeps the directory only in a /dir1/dir2/path.ext string
 */
void U61_Platform::keep_dir_only(char *path)
{
  int len;
  bool found=false;

  while ((len=strlen(path))>0 && !found)
    {
      if (path[len-1]=='/')
	{
	  found=true;
	}
      else
	{
	  path[len-1]='\0';
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * keeps the file only in a /dir1/dir2/path.ext string
 */
void U61_Platform::keep_file_only(char *path)
{
  int pos;
  bool found=false;
  char tmp[U61_CONST_STRING_SIZE];

  pos=strlen(path);
  while (pos>=0 && !found)
    {
      if (path[pos]=='/')
	{
	  strcpy(tmp,path+pos+1);
	  strcpy(path,tmp);
	  found=true;
	}
      pos--;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the user path (/home/ufoot/.u61/ for instance)
 */
char *U61_Platform::get_user_path()
{
  static char buffer[U61_CONST_STRING_SIZE]; 
  int len;

  strcpy (buffer,getenv("HOME"));
  len=strlen(buffer);
  if (len>0)
    {
      if (buffer[len-1]!='/')
        {
	  strcat(buffer,"/");
        } 
    }
  else
    {
      strcpy(buffer,"./");
    }

  strcat(buffer,".u61/");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the /usr/local/share/u61 path or something like that
 */
char *U61_Platform::get_data_path()
{
  static char buffer[U61_CONST_STRING_SIZE]; 
  DIR *d;
  int len,i;

  strcpy (buffer,U61_INSTALL_DATA_PATH);
  d=opendir(buffer);
  if (d==NULL)
    {  
      /*
       * If we are here it means the path in the config file does not
       * exist. OK, than we'll try something else!
       */
      for (i=0;i<NB_STANDARD_PATHS;++i)
	{
	  d=opendir(standard_paths[i]);
	  if (d!=NULL)
	    {
	      /*
	       * We've found an existing directory named u61, we'd be
	       * really unlucky if it wasn't the right one...
	       */
	      strcpy(buffer,standard_paths[i]);
	      i=NB_STANDARD_PATHS;
	      closedir(d);
	    }
	}
    }
  else
    {
      closedir(d);
    }

  len=strlen(buffer);
  if (len>0)
    {
      if (buffer[len-1]!='/')
        {
	  strcat(buffer,"/");
        } 
    }

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for builtin themes (/usr/local/share/u61/theme...)
 */
char *U61_Platform::get_builtin_theme_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_data_path());
  strcat(buffer,"theme/");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for builtin scripts (/usr/local/share/u61/script...)
 */
char *U61_Platform::get_builtin_script_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_data_path());
  strcat(buffer,"script/");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for user themes (/home/ufoot/.u61/theme...)
 */
char *U61_Platform::get_user_theme_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_user_path());
  strcat(buffer,"theme/");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the path for user scripts (/home/ufoot/.u61/script...)
 */
char *U61_Platform::get_user_script_path()
{
  static char buffer[U61_CONST_STRING_SIZE];

  strcpy(buffer,get_user_path());
  strcat(buffer,"script/");

  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the config file (/home/ufoot/.u61/config for instance)
 */
char *U61_Platform::get_config_file()
{
  static char buffer[U61_CONST_STRING_SIZE]; 

  strcpy (buffer,get_user_path());
  strcat(buffer,"config.lua");
 
  return buffer;
}

/*--------------------------------------------------------------------------*/
/*
 * transforms "/usr/local/share/u61/script/classic.lua" into "classic"
 */
void U61_Platform::strip_remote_script_path(char *path)
{
  char buffer[U61_CONST_STRING_SIZE];
  char *pos;

  /*
   * We remove the path (directories part)
   */
  pos=strrchr(path,'/');
  if (pos!=NULL)
    {
      strcpy(buffer,pos+1);
      strcpy(path,buffer);
    }

  /*
   * We remove the extension
   */
  pos=strrchr(path,'.');
  if (pos!=NULL)
    {
      pos[0]=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * transforms "classic" into "/home/ufoot/.u61/script/classic.lua"
 */
void U61_Platform::unstrip_remote_script_path(char *path)
{
  char buffer[U61_CONST_STRING_SIZE];

  // Gets the user script path "/home/ufoot/.u61/script/"
  strncpy(buffer,get_user_script_path(),sizeof(buffer)-1);
  // Add the filename "classic"
  strcat(buffer,path);
  // Add the extension ".lua"
  strcat(buffer,".lua");

  strncpy(path,buffer,sizeof(buffer)-1);
}

/*--------------------------------------------------------------------------*/
/*
 * opens a directory for file scanning
 */
bool U61_Platform::open_dir(char *dir, char *ext)
{
  bool ok=false;

  scan_ext=ext;
  scan_dir=opendir(dir);
  if (scan_dir!=NULL)
    {
      ok=true;
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * retrieves the next file in the directory
 */
char *U61_Platform::next_file()
{
  char *name=NULL;
  struct dirent *d;
  bool end=false;
  int len1,len2;

  do 
    {
      d=readdir(scan_dir);
      if (d==NULL)
        {
	  name=NULL;
	  end=true;
        }
      else
        {
	  name=d->d_name;
	  len1=strlen(name); 
	  len2=strlen(scan_ext); 
	  if (len2<=len1)
            {
	      if (strcasecmp(name+len1-len2,scan_ext)==0)
                {
		  end=true;
                }
            }
        }
    }
  while (!end);

  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * closes the directory
 */
bool U61_Platform::close_dir()
{
  bool ok=false;

  if (closedir(scan_dir)==0)
    {
      ok=true;
    }
 
  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * Creates the directory where user files will be stored
 * returns true if the directory exists or has been succesfully created
 */
bool U61_Platform::create_user_dir()
{
  DIR *test;
  bool ok=false;

  test=opendir(get_user_path());
  if (test!=NULL)
    {
      closedir(test);
      ok=true;
    }    
  else
    {
      if ((mkdir(get_user_path(),S_IRUSR|S_IWUSR|S_IXUSR)==0)
	  && (mkdir(get_user_theme_path(),S_IRUSR|S_IWUSR|S_IXUSR)==0)
	  && (mkdir(get_user_script_path(),S_IRUSR|S_IWUSR|S_IXUSR)==0))
	{
	  ok=true;
	}
    }

  return ok;
}


/*--------------------------------------------------------------------------*/
/*
 * Quits U61 and launches "file"
 * Used when the player changes the current theme
 */
void U61_Platform::exec(char *file)
{
  char buffer[U61_CONST_STRING_SIZE];
  int len;

  U61_LOG_MESSAGE("Restarting...");

  U61_Platform::deinit();
  U61_Script::close();
  CL_SetupMikMod::deinit();
  CL_SetupNetwork::deinit();    
  CL_SetupSound::deinit();
  CL_SetupDisplay::deinit();

  U61_Debug::deinit();
  CL_SetupCore::deinit();

  /*
   * First we try to launch the executable described
   * by argv[0]. This succeeds in case the exe is ./u61
   * or something with a full path. execl shouldn't
   * returns if it succeeds
   */
  execl(file,file,NULL);

  /*
   * now the first attempt has failed, we try again
   * by appending INSTALL_BIN_PATH before argv[0].
   * this should replace u61 by /usr/local/bin/u61
   * for instance.
   */
  strcpy(buffer,U61_INSTALL_BIN_PATH);
  len=strlen(buffer);
  if (len>0)
    {
      if (buffer[len-1]!='/')
        {
	  strcat(buffer,"/");
        } 
    }
  strcat(buffer,file);
  execl(buffer,buffer,NULL);

  /*
   * Now if we get there, it means there's an error.
   * Since we have uninitialized ClanLib, we *must*
   * quit, or there's a core dump arround the corner...
   */
  U61_Debug::init();
  U61_LOG_ERROR("Unable to execute "<<file<<" or "<<buffer);
  U61_Debug::deinit();

  exit(1);
}

/*--------------------------------------------------------------------------*/
/*
 * Opens the score file, on UNIX, it is ~/.u61/score.txt
 */
void U61_Platform::open_score_file()
{
  char buffer[U61_CONST_STRING_SIZE];

  create_user_dir();
  
  if (score_file==NULL)
    {
      strcpy (buffer,get_user_path());
      strcat(buffer,"score.txt");

      score_file=fopen(buffer,"a");
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Closes the score file, on UNIX, it is ~/.u61/score.txt
 */
void U61_Platform::close_score_file()
{
  if (score_file!=NULL)
    {
      fclose(score_file);
    }

  score_file=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * Logs an error message with syslog and in the debug log if needed
 */
void U61_Platform::dump_score(char *text)
{
  time_t curtime;
  struct tm *loctime;
  char buffer_date[U61_CONST_STRING_SIZE];

  if (score_file!=NULL)
    {     
      /*
       * Get the current time.
       */
      curtime = time (NULL);
     
      /*
       * Convert it to local time representation.
       */
      loctime = localtime (&curtime);
     
      sprintf(buffer_date,asctime(loctime));
      /*
       * We get rid of the trailing '\n'
       */
      if (strlen(buffer_date)>=24)
	{
	  buffer_date[24]=0;
	}
      fprintf(score_file,"%-24s ; %s\n",buffer_date,text);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Opens all the log files, on UNIX, this concerns only the debug log
 * file, since the rest of the log is handled by syslog
 */
void U61_Platform::open_log_files()
{
  openlog("u61",LOG_PID,LOG_USER);
  
  create_user_dir();  

#ifdef U61_DEBUG
  char buffer[U61_CONST_STRING_SIZE];

  if (debug_file==NULL)
    {
      strcpy (buffer,get_user_path());
      strcat(buffer,"debug.log");

      debug_file=fopen(buffer,"w");
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Closes all the log files, on UNIX, this concerns only the debug log
 * file, since the rest of the log is handled by syslog
 */
void U61_Platform::close_log_files()
{
#ifdef U61_DEBUG
  if (debug_file!=NULL)
    {
      fclose(debug_file);
    }
#endif

  debug_file=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * Logs an error message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_error_file(char *text)
{
  syslog(LOG_ERR,"error: %s",text);

#ifdef U61_DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file,"*;error  ;%25s;%-5d;%s\n","unknown",0,text);
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Logs a warning message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_warning_file(char *text)
{
  syslog(LOG_WARNING,"warning: %s",text);

#ifdef U61_DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file,"*;warning;%25s;%-5d;%s\n","unknown",0,text);
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Logs a standard message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_message_file(char *text)
{
#ifdef U61_DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file," ;message;%25s;%-5d;%s\n","unknown",0,text);
    }
#endif
}

/*--------------------------------------------------------------------------*/
/*
 * Logs a debug message with syslog and in the debug log if needed
 */
void U61_Platform::log_to_debug_file(char *text,char *file,int line)
{
#ifdef U61_DEBUG
  if (debug_file!=NULL)
    {
      fprintf(debug_file," ;debug  ;%25s;%-5d;%s\n",file,line,text);
    }
#endif
}

