/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   serverprotocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: welcomes a client, gets basic information about him
 *              and sends him all the informations about
 *              the players which are already playing
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "serverprotocol.h"
#include "global.h"
#include "time.h"
#include "platform.h"
#include "debug.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_SERVERPROTOCOL_LISTENING           10
#define U61_SERVERPROTOCOL_CLIENT_ACCEPTED     11
#define U61_SERVERPROTOCOL_SCRIPT_NAME_SENT    30
#define U61_SERVERPROTOCOL_SCRIPT_SENT         31
#define U61_SERVERPROTOCOL_ID0_SENT            40
#define U61_SERVERPROTOCOL_ID1_SENT            41
#define U61_SERVERPROTOCOL_INITIAL_SPEED_SENT  42
#define U61_SERVERPROTOCOL_ACCELERATION_SENT   43
#define U61_SERVERPROTOCOL_CURSE_DELAY_SENT    44
#define U61_SERVERPROTOCOL_TIME_SENT           45

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a server protocol
 */
U61_ServerProtocol::U61_ServerProtocol(U61_ServerDispatcher *d, int p,int ch)
  : U61_Protocol(p,ch)
{
  int i;

  dispatcher=d;
  new_connection=0;

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      connection[i]=NULL;
      id0[i]=-1;
      id1[i]=-1;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the protocol object
 */
U61_ServerProtocol::~U61_ServerProtocol()
{
  int i;

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      if (connection[i]!=NULL)
	{
	  delete connection[i];
          connection[i]=NULL;
	}
    }
    
  if (session!=NULL)
    {
      delete session;
      session=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if there are some dead connections, and kills the players
 * if necessary
 */
void U61_ServerProtocol::check_connection()
{
  int i;
  U61_Player *p;
  CL_NetComputer client;
  U61_Event evt;  

  if (session!=NULL)
    {
      client=(CL_NetComputer) session->receive_computer_leave();
      for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
	{
	  if (connection[i]!=NULL
	      && connection[i]->is_associated(&client))
	    {
	      if (dispatcher!=NULL)
		{
		  dispatcher->close(connection[i]);
		}
	      delete connection[i];

	      evt.code=U61_EVENT_KILL;
	      evt.time=0;
	      evt.author=evt.target=id0[i];
	      dispatcher->put_out(evt);   
	      evt.author=evt.target=id1[i];
	      dispatcher->put_out(evt);

              p=U61_Global::game.find_player_by_id(id0[i]);
              if (p!=NULL)
		{
                  p->kill();
		}
              p=U61_Global::game.find_player_by_id(id1[i]);
              if (p!=NULL)
		{
                  p->kill();
		}

              connection[i]=NULL;
              id0[i]=-1;
              id1[i]=-1;   
	    } 
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to send to the client
 */
int U61_ServerProtocol::send_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_PROTOCOL_BEGIN:
      if ((result=start_listening())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_LISTENING;
	  U61_LOG_DEBUG("stage=LISTENING");
	}
      break;
    case U61_SERVERPROTOCOL_CLIENT_ACCEPTED:
      if ((result=send_program(connection[new_connection]))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_PROGRAM_SENT;
	  U61_LOG_DEBUG("stage=PROGRAM_SENT");
	}
      break;
    case U61_PROTOCOL_PROGRAM_RECEIVED:
      if ((result=send_version(connection[new_connection]))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_SENT;
	  U61_LOG_DEBUG("stage=VERSION_SENT");
	}
      break;
    case U61_PROTOCOL_VERSION_RECEIVED:
      if ((result=send_script_name())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_SCRIPT_NAME_SENT;
	  U61_LOG_DEBUG("stage=SCRIPT_NAME_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_SCRIPT_NAME_SENT:
      if ((result=send_script())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_SCRIPT_SENT;
	  U61_LOG_DEBUG("stage=SCRIPT_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_SCRIPT_SENT:
      if ((result=send_id0())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_ID0_SENT;
	  U61_LOG_DEBUG("stage=ID0_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_ID0_SENT:
      if ((result=send_id1())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_ID1_SENT;
	  U61_LOG_DEBUG("stage=ID1_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_ID1_SENT:
      if ((result=send_initial_speed())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_INITIAL_SPEED_SENT;
	  U61_LOG_DEBUG("stage=INITIAL_SPEED_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_INITIAL_SPEED_SENT:
      if ((result=send_acceleration())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_ACCELERATION_SENT;
	  U61_LOG_DEBUG("stage=ACCELERATION_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_ACCELERATION_SENT:
      if ((result=send_curse_delay())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_CURSE_DELAY_SENT;
	  U61_LOG_DEBUG("stage=CURSE_DELAY_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_CURSE_DELAY_SENT:
      if ((result=send_time())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_TIME_SENT;
	  U61_LOG_DEBUG("stage=TIME_SENT");
	}
      break;
    case U61_SERVERPROTOCOL_TIME_SENT:
      if ((result=send_ready(connection[new_connection]))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_SENT;
	  U61_LOG_DEBUG("stage=READY_SENT");
	}
      break;
    case U61_PROTOCOL_READY_RECEIVED:
      dispatcher->open(connection[new_connection]);

      U61_LOG_DEBUG("stage=LISTENING");
      stage=U61_SERVERPROTOCOL_LISTENING;
      break;
    }

  handle_error(&result);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to receive from the
 * client
 */
int U61_ServerProtocol::recv_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_SERVERPROTOCOL_LISTENING:
      if ((result=accept_client())==U61_PROTOCOL_DONE)
	{
	  stage=U61_SERVERPROTOCOL_CLIENT_ACCEPTED;
	  U61_LOG_DEBUG("stage=CLIENT_ACCEPTED");
	}
      break;
    case U61_PROTOCOL_PROGRAM_SENT:
      if ((result=recv_program(connection[new_connection]))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_PROGRAM_RECEIVED;
	  U61_LOG_DEBUG("stage=PROGRAM_RECEIVED");
	}
      break;
    case U61_PROTOCOL_VERSION_SENT:
      if ((result=recv_version(connection[new_connection]))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_RECEIVED;
	  U61_LOG_DEBUG("stage=VERSION_RECEIVED");
	}
      break;
    case U61_PROTOCOL_READY_SENT:
      if ((result=recv_ready(connection[new_connection]))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_RECEIVED;
	  U61_LOG_DEBUG("stage=READY_RECEIVED");
	}
      break;
    }

  handle_error(&result);

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * starts listening what's happening on the network, in case a computer wishes
 * to connect itself
 */
int U61_ServerProtocol::start_listening()
{
  int result=U61_PROTOCOL_LATER;

  try 
    {
      session=new CL_NetSession(U61_CONST_PROGRAM,port);
    }
  catch (CL_Error e)
    {
      result=U61_PROTOCOL_FAILED;
      diagnostic+=e.message;
      session=NULL;
    }

  if (session!=NULL)
    {
      result=U61_PROTOCOL_DONE;
    };
  
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * detects a computer, ie accepts to dialog with it, it does not mean that
 * the players on this remote machine will actually be accepted in the
 * game but at least we'll dialog with them 
 */
int U61_ServerProtocol::accept_client()
{
  int result=U61_PROTOCOL_LATER;
  CL_NetComputer computer;
  
  try 
    {
      if (!((computer=(CL_NetComputer) 
	     session->receive_computer_join())==
	    CL_NetComputer(NULL)))
	{
	  if ((new_connection=get_free_connection())<0)
	    {
	      result=U61_PROTOCOL_FAILED;
	    }
	  else
	    {  
	      /*
	       * There's a connection available to talk with the client
	       * so we can discuss with him.
	       * It should be important to check when the computer
	       * object is deleted by the way...
	       */
	      connection[new_connection]=new 
		U61_Connection(session,computer,
			       channel+new_connection,true);        
	      result=U61_PROTOCOL_DONE;
	    }
	}
    }
  catch (CL_Error e)
    {
      /*
       * Here we trap any error raised by receive_computer_join
       */
    }
  
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the index of a free connection in the connection array, if
 * no connection is available it just returns a <0 value
 */
int U61_ServerProtocol::get_free_connection()
{
  int i;
  int result=-1;

  for (i=0;i<U61_GAME_NB_NET_GROUPS-1;++i)
    {
      if (connection[i]==NULL)
	{
	  result=i;
	  break;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the script name to the client, we remove the path from the
 * filename, for the full path does not interest the remote client
 */
int U61_ServerProtocol::send_script_name()
{
  int result=U61_PROTOCOL_LATER;
  char script_name[U61_CONST_STRING_SIZE];

  strncpy(script_name,
	  U61_Global::config.script_file,
	  sizeof(script_name)-1);
  U61_Platform::strip_remote_script_path(script_name);
  if (connection[new_connection]->send(script_name,sizeof(script_name)-1))
    {
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the whole script to the client
 */
int U61_ServerProtocol::send_script()
{
  int result=U61_PROTOCOL_LATER;
  char script[U61_CONST_SCRIPT_SIZE];
  char script_name[U61_CONST_STRING_SIZE];
  FILE *f;
  int c;
  int pos=0;

  U61_Global::data.chdir_to_initial_dir();

  strncpy(script_name,U61_Global::config.script_file,sizeof(script_name)-1);

  f=fopen(script_name,"r");
  if (f!=NULL)
    {
      while ((c=fgetc(f))!=EOF && pos<U61_CONST_SCRIPT_SIZE)
	{
	  script[pos++]=c;
	}
      fclose(f);
      if (connection[new_connection]->send(script,sizeof(script)-1))
	{
	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      U61_LOG_WARNING("Unable to open \""<<script_name<<"\".");

      diagnostic="Could not open script file \"";
      diagnostic+=script_name;
      diagnostic+="\"";
      result=U61_PROTOCOL_FAILED;
    }

  U61_Global::data.chdir_to_data_dir();

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the first id that will be usable by the client
 */
int U61_ServerProtocol::send_id0()
{
  int result=U61_PROTOCOL_LATER;
  int id;

  id=U61_Global::game.find_free_player_id0();
  if (connection[new_connection]->send(id))
    {
      id0[new_connection]=id;
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the second id that will be usable by the client
 */
int U61_ServerProtocol::send_id1()
{
  int result=U61_PROTOCOL_LATER;
  int id;

  id=U61_Global::game.find_free_player_id1();
  if (connection[new_connection]->send(id))
    {
      id1[new_connection]=id;
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the initial speed that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_initial_speed()
{
  int result=U61_PROTOCOL_LATER;

  if (connection[new_connection]->send(U61_Global::config.initial_speed))
    {
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the acceleration that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_acceleration()
{
  int result=U61_PROTOCOL_LATER;

  if (connection[new_connection]->send(U61_Global::config.acceleration))
    {
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the curse delay that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_curse_delay()
{
  int result=U61_PROTOCOL_LATER;

  if (connection[new_connection]->send(U61_Global::config.curse_delay))
    {
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the time that will be usable by the client to initialize its
 * player's maps
 */
int U61_ServerProtocol::send_time()
{
  int result=U61_PROTOCOL_LATER;

  if (connection[new_connection]->send(U61_Time::for_event()))
    {
      result=U61_PROTOCOL_DONE;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Checks the return value of send & recv functions
 * stops everything if something's going wrong
 */
void U61_ServerProtocol::handle_error(int *result)
{
  if ((*result)==U61_PROTOCOL_FAILED)
    {
      U61_LOG_WARNING("Network server protocol error on connection "
		      <<new_connection
		      <<" at stage "
		      <<stage);

      if (stage!=U61_PROTOCOL_BEGIN)
	{
	  /*
	   * For all the errors except the impossibility to bind the
	   * listening socket (which can only happen at stage 
	   * U61_PROTOCOL_BEGIN), we decide to mask the error and
	   * replace it by U61_PROTOCOL_LATER. This is because we do not
	   * want the whole server connections to be closed only because
	   * of a single little problem on one channel...
	   */
	  (*result)=U61_PROTOCOL_LATER;
	}

      if (new_connection>=0 && connection[new_connection]!=NULL)
	{
	  delete connection[new_connection];
	  connection[new_connection]=NULL;
	}

      if (session==NULL)
	{
	  /*
	   * session is NULL, this means we have not successfully passed
           * the opening of the listening socket.
           * so we restart everything from scratch
           */    
	  stage=U61_PROTOCOL_BEGIN;
	}
      else
	{
	  /*
	   * session is not NULL, this means the protocol failed somewhere
	   * on an opened socket, therefore we go back to LISTENING mode
	   * ie accept new connection on the listening socket
	   */
	  stage=U61_SERVERPROTOCOL_LISTENING;
	}
    }
}

