/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   config.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the config object holds all the values which are stored
 *              on the HD between two game sessions. it would for instance
 *              store the player names and key settings
 *              it basically provides functions to load and save all the
 *              game parameters, and makes these parameters usable by any
 *              other class.
 *              it also provides some function which help handling game
 *              settings in general
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "config.h"
#include "platform.h"
#include "script.h"
#include "debug.h"
#include "sound.h"
#include "music.h"
#include "global.h"
#include "version.h"
#include "utils.h"
#include "inputmonitor.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_CONFIG_ID_PROGRAM "cfg_program"
#define U61_CONFIG_ID_VERSION "cfg_version"

#define U61_CONFIG_ID_PLAYER "cfg_player_%d_"

#define U61_CONFIG_ID_LAUNCH_COUNT "cfg_launch_count"
#define U61_CONFIG_ID_DATA_DIR "cfg_data_dir"
#define U61_CONFIG_ID_SCRIPT_DIR "cfg_script_dir"
#define U61_CONFIG_ID_DATA_FILE "cfg_data_file"
#define U61_CONFIG_ID_SCRIPT_FILE "cfg_script_file"

#define U61_CONFIG_ID_FULLSCREEN "cfg_fullscreen"
#define U61_CONFIG_ID_SHOW_SYSTEM_INFO "cfg_show_system_info"

#define U61_CONFIG_ID_SERVER_ID "cfg_server_id"
#define U61_CONFIG_ID_NETWORK_PORT "cfg_network_port"
#define U61_CONFIG_ID_NETWORK_CHANNEL "cfg_network_channel"

#define U61_CONFIG_ID_INITIAL_SPEED "cfg_initial_speed"
#define U61_CONFIG_ID_ACCELERATION "cfg_acceleration"
#define U61_CONFIG_ID_CURSE_DELAY "cfg_curse_delay"

#define U61_CONFIG_ID_SOUND_VOLUME "cfg_sound_volume"
#define U61_CONFIG_ID_MUSIC_VOLUME "cfg_music_volume"

#define U61_CONFIG_ID_MAX_FPS "cfg_max_fps"

#define U61_CONFIG_ID_NETWORK_ANTICIPATION "cfg_network_anticipation"

#define U61_CONFIG_ID_FIRST_REPEAT_MOVE "first_repeat_move"
#define U61_CONFIG_ID_STANDARD_REPEAT_MOVE "standard_repeat_move"
#define U61_CONFIG_ID_FIRST_REPEAT_ROTATE "first_repeat_rotate"
#define U61_CONFIG_ID_STANDARD_REPEAT_ROTATE "standard_repeat_rotate"
#define U61_CONFIG_ID_FIRST_REPEAT_OTHER "first_repeat_other"
#define U61_CONFIG_ID_STANDARD_REPEAT_OTHER "standard_repeat_other"

#define U61_CONFIG_ID_ANTICIPATION "anticipation"
#define U61_CONFIG_ID_ACTIVABLE "playing"
#define U61_CONFIG_ID_NAME "name"

#define U61_CONFIG_ID_KEY_START_STOP "key_start_stop"
#define U61_CONFIG_ID_KEY_MOVE_LEFT "key_move_left"
#define U61_CONFIG_ID_KEY_MOVE_RIGHT "key_move_right"
#define U61_CONFIG_ID_KEY_ROTATE_LEFT "key_rotate_left"
#define U61_CONFIG_ID_KEY_ROTATE_RIGHT "key_rotate_right"
#define U61_CONFIG_ID_KEY_MOVE_DOWN "key_move_down"
#define U61_CONFIG_ID_KEY_DROP "key_drop"
#define U61_CONFIG_ID_KEY_NEXT_VICTIM "key_next_victim"
#define U61_CONFIG_ID_KEY_PREV_VICTIM "key_prev_victim"
#define U61_CONFIG_ID_KEY_USE_ANTIDOTE "key_use_antidote"

#define U61_CONFIG_ID_KEY "cfg_key_%d_"
#define U61_CONFIG_ID_ASCII "ascii"
#define U61_CONFIG_ID_LABEL "label"

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation with default values
 */ 
U61_Config::U61_Config()
{

}

/*--------------------------------------------------------------------------*/
/*
 * destruction
 */
U61_Config::~U61_Config()
{

}

/*--------------------------------------------------------------------------*/
/*
 * checks that the datafile is a correct U61 datafile
 */
bool U61_Config::check_id()
{
  bool success=true;  
  char program_read[U61_CONST_STRING_SIZE];
  char version_read[U61_CONST_STRING_SIZE];
  char version[U61_CONST_STRING_SIZE];

  read_str(program_read,U61_CONFIG_ID_PROGRAM,"???");
  if (strcmp(program_read,U61_CONST_PROGRAM))
    {
      success=false;
      U61_LOG_WARNING("The config file does not seem to be a valid U61 config file.");
    }
  else
    {
      read_str(version_read,U61_CONFIG_ID_VERSION,"???");
      U61_Version::get_version_text(version);
      if (strcmp(version_read,version))
	{
	  success=false;
	  U61_LOG_WARNING("The config file version is \""<<version_read<<"\" but the game version is \""<<version<<"\".");
	}
    }
  
  return success;    
}

/*--------------------------------------------------------------------------*/
/* 
 * initialization with values from the config file
 */ 
bool U61_Config::read()
{
  bool ok=true;
  char *filename;
  
  U61_Global::data.chdir_to_initial_dir();

  U61_LOG_DEBUG("read config");
  filename=U61_Platform::get_config_file();

  if (!U61_Utils::file_exists(filename))
    {
      ok=false;
      
      U61_LOG_MESSAGE("Config file \""<<filename<<"\" does not exist.");
    }
  else
    {
      U61_LOG_MESSAGE("Loading config from \""<<filename<<"\".");
      
      if (!U61_Script::do_file(filename))
	{
	  ok=false;
	  U61_LOG_WARNING("Error reading the config file.");
	}
      else
	{
	  ok=check_id();
	}
    }

  if (!ok)
    {
      /*
       * If there's been a problem, we simply reset all the script stuff
       * and then re-open it. This way we avoid keeping wrong values from
       * installs coming from a previous release
       */
      U61_Script::close();
      U61_Script::open();

      U61_LOG_MESSAGE("Using default settings.");
    }

  reset();
  
  /*
   * We close/open Lua so that reading the config file does not leave
   * any trace in the Lua environnement
   */
  U61_Script::close();
  U61_Script::open();

  U61_Global::data.chdir_to_data_dir();

  return ok;
}
   
/*--------------------------------------------------------------------------*/
/* 
 * initialization with lua config file values
 */ 
void U61_Config::reset()
{
  int pl,k;
  char default_data_dir[U61_CONST_STRING_SIZE];
  char default_script_dir[U61_CONST_STRING_SIZE];
  char default_data_file[U61_CONST_STRING_SIZE];
  char default_script_file[U61_CONST_STRING_SIZE];

  strcpy(default_data_dir,U61_Platform::get_builtin_theme_path());
  strcpy(default_script_dir,U61_Platform::get_builtin_script_path());
  strcpy(default_data_file,default_data_dir);
  strcat(default_data_file,"default.dat");
  strcpy(default_script_file,default_script_dir);
  strcat(default_script_file,"classic.lua");

  launch_count=read_int(U61_CONFIG_ID_LAUNCH_COUNT,0);
  read_str(data_dir,U61_CONFIG_ID_DATA_DIR,default_data_dir);
  read_str(script_dir,U61_CONFIG_ID_SCRIPT_DIR,default_script_dir);
  read_str(data_file,U61_CONFIG_ID_DATA_FILE,default_data_file);
  read_str(script_file,U61_CONFIG_ID_SCRIPT_FILE,default_script_file);

  fullscreen=read_bool(U61_CONFIG_ID_FULLSCREEN,false);
  show_system_info=read_bool(U61_CONFIG_ID_SHOW_SYSTEM_INFO,false);

  read_str(server_id,U61_CONFIG_ID_SERVER_ID,"127.0.0.1");
  network_port=read_int(U61_CONFIG_ID_NETWORK_PORT,8061);
  network_channel=read_int(U61_CONFIG_ID_NETWORK_CHANNEL,1);
        
  initial_speed=read_int(U61_CONFIG_ID_INITIAL_SPEED,5);
  acceleration=read_int(U61_CONFIG_ID_ACCELERATION,5);
  curse_delay=read_int(U61_CONFIG_ID_CURSE_DELAY,20);

  sound_volume=read_int(U61_CONFIG_ID_SOUND_VOLUME,U61_SOUND_VOLUME_MAX/2);
  music_volume=read_int(U61_CONFIG_ID_MUSIC_VOLUME,U61_MUSIC_VOLUME_MAX/2);

  max_fps=read_int(U61_CONFIG_ID_MAX_FPS,18);

  network_anticipation=read_bool(U61_CONFIG_ID_NETWORK_ANTICIPATION,false);

  for (pl=0;pl<U61_CONFIG_NB_PLAYERS;++pl)
    {
      player[pl].first_repeat_move=
	read_player_int(pl,U61_CONFIG_ID_FIRST_REPEAT_MOVE,10);
      player[pl].standard_repeat_move=
	read_player_int(pl,U61_CONFIG_ID_STANDARD_REPEAT_MOVE,5);
      player[pl].first_repeat_rotate=
	read_player_int(pl,U61_CONFIG_ID_FIRST_REPEAT_ROTATE,20);
      player[pl].standard_repeat_rotate=
	read_player_int(pl,U61_CONFIG_ID_STANDARD_REPEAT_ROTATE,10);
      player[pl].first_repeat_other=
	read_player_int(pl,U61_CONFIG_ID_FIRST_REPEAT_OTHER,40);
      player[pl].standard_repeat_other=
	read_player_int(pl,U61_CONFIG_ID_STANDARD_REPEAT_OTHER,20);
      player[pl].anticipation=
	read_player_bool(pl,U61_CONFIG_ID_ANTICIPATION,false);
    }

  player[0].activable=read_player_bool(0,U61_CONFIG_ID_ACTIVABLE,true);
  read_player_str(player[0].name,0,U61_CONFIG_ID_NAME,"ufoot");
  player[0].key_start_stop=read_player_int(0,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F5);
  player[0].key_move_left=read_player_int(0,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_LEFT);
  player[0].key_move_right=read_player_int(0,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_RIGHT);
  player[0].key_rotate_left=read_player_int(0,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_DOWN);
  player[0].key_rotate_right=read_player_int(0,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_UP);
  player[0].key_move_down=read_player_int(0,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_KEY_END);
  player[0].key_drop=read_player_int(0,U61_CONFIG_ID_KEY_DROP, CL_KEY_RCTRL);
  player[0].key_prev_victim=read_player_int(0,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_KEY_PAGEUP);
  player[0].key_next_victim=read_player_int(0,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_KEY_PAGEDOWN);
  player[0].key_use_antidote=read_player_int(0,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_KEY_DELETE);

  player[1].activable=read_player_bool(1,U61_CONFIG_ID_ACTIVABLE,false);
  read_player_str(player[1].name,1,U61_CONFIG_ID_NAME,"etoile");
  player[1].key_start_stop=read_player_int(1,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F6);
  player[1].key_move_left=read_player_int(1,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_Z);
  player[1].key_move_right=read_player_int(1,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_X);
  player[1].key_rotate_left=read_player_int(1,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_1);
  player[1].key_rotate_right=read_player_int(1,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_Q);
  player[1].key_move_down=read_player_int(1,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_KEY_A);
  player[1].key_drop=read_player_int(1,U61_CONFIG_ID_KEY_DROP, CL_KEY_ALT);
  player[1].key_prev_victim=read_player_int(1,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_KEY_2);
  player[1].key_next_victim=read_player_int(1,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_KEY_W);
  player[1].key_use_antidote=read_player_int(1,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_KEY_S);

  player[2].activable=read_player_bool(2,U61_CONFIG_ID_ACTIVABLE,false);
  read_player_str(player[2].name,2,U61_CONFIG_ID_NAME,"arthur");
  player[2].key_start_stop=read_player_int(2,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F8);
  player[2].key_move_left=read_player_int(2,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_N);
  player[2].key_move_right=read_player_int(2,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_M);
  player[2].key_rotate_left=read_player_int(2,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_0);
  player[2].key_rotate_right=read_player_int(2,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_O);
  player[2].key_move_down=read_player_int(2,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_KEY_K);
  player[2].key_drop=read_player_int(2,U61_CONFIG_ID_KEY_DROP, CL_KEY_ALTGR);
  player[2].key_prev_victim=read_player_int(2,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_KEY_9);
  player[2].key_next_victim=read_player_int(2,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_KEY_I);
  player[2].key_use_antidote=read_player_int(2,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_KEY_J);

  player[3].activable=read_player_bool(3,U61_CONFIG_ID_ACTIVABLE,false);
  read_player_str(player[3].name,3,U61_CONFIG_ID_NAME,"popol");
  player[3].key_start_stop=read_player_int(3,U61_CONFIG_ID_KEY_START_STOP, CL_KEY_F7);
  player[3].key_move_left=read_player_int(3,U61_CONFIG_ID_KEY_MOVE_LEFT, CL_KEY_V);
  player[3].key_move_right=read_player_int(3,U61_CONFIG_ID_KEY_MOVE_RIGHT, CL_KEY_B);
  player[3].key_rotate_left=read_player_int(3,U61_CONFIG_ID_KEY_ROTATE_LEFT, CL_KEY_4);
  player[3].key_rotate_right=read_player_int(3,U61_CONFIG_ID_KEY_ROTATE_RIGHT, CL_KEY_R);
  player[3].key_move_down=read_player_int(3,U61_CONFIG_ID_KEY_MOVE_DOWN, CL_KEY_F);
  player[3].key_drop=read_player_int(3,U61_CONFIG_ID_KEY_DROP, CL_KEY_SPACE);
  player[3].key_prev_victim=read_player_int(3,U61_CONFIG_ID_KEY_PREV_VICTIM, CL_KEY_5);
  player[3].key_next_victim=read_player_int(3,U61_CONFIG_ID_KEY_NEXT_VICTIM, CL_KEY_T);
  player[3].key_use_antidote=read_player_int(3,U61_CONFIG_ID_KEY_USE_ANTIDOTE, CL_KEY_G);

  for (k=0;k<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++k)
    {
      key[k].ascii=U61_KEYDEF_ASCII_UNKNOWN;
      strcpy(key[k].label,U61_KEYDEF_LABEL_UNKNOWN);
    }

  key[CL_KEY_A].ascii=read_key_int(CL_KEY_A,U61_CONFIG_ID_ASCII,'a');
  read_key_str(key[CL_KEY_A].label,CL_KEY_A,U61_CONFIG_ID_LABEL,"a");

  key[CL_KEY_B].ascii=read_key_int(CL_KEY_B,U61_CONFIG_ID_ASCII,'b');
  read_key_str(key[CL_KEY_B].label,CL_KEY_B,U61_CONFIG_ID_LABEL,"b");

  key[CL_KEY_C].ascii=read_key_int(CL_KEY_C,U61_CONFIG_ID_ASCII,'c');
  read_key_str(key[CL_KEY_C].label,CL_KEY_C,U61_CONFIG_ID_LABEL,"c");

  key[CL_KEY_D].ascii=read_key_int(CL_KEY_D,U61_CONFIG_ID_ASCII,'d');
  read_key_str(key[CL_KEY_D].label,CL_KEY_D,U61_CONFIG_ID_LABEL,"d");

  key[CL_KEY_E].ascii=read_key_int(CL_KEY_E,U61_CONFIG_ID_ASCII,'e');
  read_key_str(key[CL_KEY_E].label,CL_KEY_E,U61_CONFIG_ID_LABEL,"e");

  key[CL_KEY_F].ascii=read_key_int(CL_KEY_F,U61_CONFIG_ID_ASCII,'f');
  read_key_str(key[CL_KEY_F].label,CL_KEY_F,U61_CONFIG_ID_LABEL,"f");

  key[CL_KEY_G].ascii=read_key_int(CL_KEY_G,U61_CONFIG_ID_ASCII,'g');
  read_key_str(key[CL_KEY_G].label,CL_KEY_G,U61_CONFIG_ID_LABEL,"g");

  key[CL_KEY_H].ascii=read_key_int(CL_KEY_H,U61_CONFIG_ID_ASCII,'h');
  read_key_str(key[CL_KEY_H].label,CL_KEY_H,U61_CONFIG_ID_LABEL,"h");

  key[CL_KEY_I].ascii=read_key_int(CL_KEY_I,U61_CONFIG_ID_ASCII,'i');
  read_key_str(key[CL_KEY_I].label,CL_KEY_I,U61_CONFIG_ID_LABEL,"i");

  key[CL_KEY_J].ascii=read_key_int(CL_KEY_J,U61_CONFIG_ID_ASCII,'j');
  read_key_str(key[CL_KEY_J].label,CL_KEY_J,U61_CONFIG_ID_LABEL,"j");

  key[CL_KEY_K].ascii=read_key_int(CL_KEY_K,U61_CONFIG_ID_ASCII,'k');
  read_key_str(key[CL_KEY_K].label,CL_KEY_K,U61_CONFIG_ID_LABEL,"k");

  key[CL_KEY_L].ascii=read_key_int(CL_KEY_L,U61_CONFIG_ID_ASCII,'l');
  read_key_str(key[CL_KEY_L].label,CL_KEY_L,U61_CONFIG_ID_LABEL,"l");

  key[CL_KEY_M].ascii=read_key_int(CL_KEY_M,U61_CONFIG_ID_ASCII,'m');
  read_key_str(key[CL_KEY_M].label,CL_KEY_M,U61_CONFIG_ID_LABEL,"m");

  key[CL_KEY_N].ascii=read_key_int(CL_KEY_N,U61_CONFIG_ID_ASCII,'n');
  read_key_str(key[CL_KEY_N].label,CL_KEY_N,U61_CONFIG_ID_LABEL,"n");

  key[CL_KEY_O].ascii=read_key_int(CL_KEY_O,U61_CONFIG_ID_ASCII,'o');
  read_key_str(key[CL_KEY_O].label,CL_KEY_O,U61_CONFIG_ID_LABEL,"o");

  key[CL_KEY_P].ascii=read_key_int(CL_KEY_P,U61_CONFIG_ID_ASCII,'p');
  read_key_str(key[CL_KEY_P].label,CL_KEY_P,U61_CONFIG_ID_LABEL,"p");

  key[CL_KEY_Q].ascii=read_key_int(CL_KEY_Q,U61_CONFIG_ID_ASCII,'q');
  read_key_str(key[CL_KEY_Q].label,CL_KEY_Q,U61_CONFIG_ID_LABEL,"q");

  key[CL_KEY_R].ascii=read_key_int(CL_KEY_R,U61_CONFIG_ID_ASCII,'r');
  read_key_str(key[CL_KEY_R].label,CL_KEY_R,U61_CONFIG_ID_LABEL,"r");

  key[CL_KEY_S].ascii=read_key_int(CL_KEY_S,U61_CONFIG_ID_ASCII,'s');
  read_key_str(key[CL_KEY_S].label,CL_KEY_S,U61_CONFIG_ID_LABEL,"s");

  key[CL_KEY_T].ascii=read_key_int(CL_KEY_T,U61_CONFIG_ID_ASCII,'t');
  read_key_str(key[CL_KEY_T].label,CL_KEY_T,U61_CONFIG_ID_LABEL,"t");

  key[CL_KEY_U].ascii=read_key_int(CL_KEY_U,U61_CONFIG_ID_ASCII,'u');
  read_key_str(key[CL_KEY_U].label,CL_KEY_U,U61_CONFIG_ID_LABEL,"u");

  key[CL_KEY_V].ascii=read_key_int(CL_KEY_V,U61_CONFIG_ID_ASCII,'v');
  read_key_str(key[CL_KEY_V].label,CL_KEY_V,U61_CONFIG_ID_LABEL,"v");

  key[CL_KEY_W].ascii=read_key_int(CL_KEY_W,U61_CONFIG_ID_ASCII,'w');
  read_key_str(key[CL_KEY_W].label,CL_KEY_W,U61_CONFIG_ID_LABEL,"w");

  key[CL_KEY_X].ascii=read_key_int(CL_KEY_X,U61_CONFIG_ID_ASCII,'x');
  read_key_str(key[CL_KEY_X].label,CL_KEY_X,U61_CONFIG_ID_LABEL,"x");

  key[CL_KEY_Y].ascii=read_key_int(CL_KEY_Y,U61_CONFIG_ID_ASCII,'y');
  read_key_str(key[CL_KEY_Y].label,CL_KEY_Y,U61_CONFIG_ID_LABEL,"y");

  key[CL_KEY_Z].ascii=read_key_int(CL_KEY_Z,U61_CONFIG_ID_ASCII,'z');
  read_key_str(key[CL_KEY_Z].label,CL_KEY_Z,U61_CONFIG_ID_LABEL,"z");

  key[CL_KEY_0].ascii=read_key_int(CL_KEY_0,U61_CONFIG_ID_ASCII,'0');
  read_key_str(key[CL_KEY_0].label,CL_KEY_0,U61_CONFIG_ID_LABEL,"0");

  key[CL_KEY_1].ascii=read_key_int(CL_KEY_1,U61_CONFIG_ID_ASCII,'1');
  read_key_str(key[CL_KEY_1].label,CL_KEY_1,U61_CONFIG_ID_LABEL,"1");

  key[CL_KEY_2].ascii=read_key_int(CL_KEY_2,U61_CONFIG_ID_ASCII,'2');
  read_key_str(key[CL_KEY_2].label,CL_KEY_2,U61_CONFIG_ID_LABEL,"2");

  key[CL_KEY_3].ascii=read_key_int(CL_KEY_3,U61_CONFIG_ID_ASCII,'3');
  read_key_str(key[CL_KEY_3].label,CL_KEY_3,U61_CONFIG_ID_LABEL,"3");

  key[CL_KEY_4].ascii=read_key_int(CL_KEY_4,U61_CONFIG_ID_ASCII,'4');
  read_key_str(key[CL_KEY_4].label,CL_KEY_4,U61_CONFIG_ID_LABEL,"4");

  key[CL_KEY_5].ascii=read_key_int(CL_KEY_5,U61_CONFIG_ID_ASCII,'5');
  read_key_str(key[CL_KEY_5].label,CL_KEY_5,U61_CONFIG_ID_LABEL,"5");

  key[CL_KEY_6].ascii=read_key_int(CL_KEY_6,U61_CONFIG_ID_ASCII,'6');
  read_key_str(key[CL_KEY_6].label,CL_KEY_6,U61_CONFIG_ID_LABEL,"6");

  key[CL_KEY_7].ascii=read_key_int(CL_KEY_7,U61_CONFIG_ID_ASCII,'7');
  read_key_str(key[CL_KEY_7].label,CL_KEY_7,U61_CONFIG_ID_LABEL,"7");

  key[CL_KEY_8].ascii=read_key_int(CL_KEY_8,U61_CONFIG_ID_ASCII,'8');
  read_key_str(key[CL_KEY_8].label,CL_KEY_8,U61_CONFIG_ID_LABEL,"8");

  key[CL_KEY_9].ascii=read_key_int(CL_KEY_9,U61_CONFIG_ID_ASCII,'9');
  read_key_str(key[CL_KEY_9].label,CL_KEY_9,U61_CONFIG_ID_LABEL,"9");

  key[CL_KEY_F1].ascii=read_key_int(CL_KEY_F1,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F1].label,CL_KEY_F1,U61_CONFIG_ID_LABEL,"F1");

  key[CL_KEY_F2].ascii=read_key_int(CL_KEY_F2,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F2].label,CL_KEY_F2,U61_CONFIG_ID_LABEL,"F2");

  key[CL_KEY_F3].ascii=read_key_int(CL_KEY_F3,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F3].label,CL_KEY_F3,U61_CONFIG_ID_LABEL,"F3");

  key[CL_KEY_F4].ascii=read_key_int(CL_KEY_F4,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F4].label,CL_KEY_F4,U61_CONFIG_ID_LABEL,"F4");

  key[CL_KEY_F5].ascii=read_key_int(CL_KEY_F5,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F5].label,CL_KEY_F5,U61_CONFIG_ID_LABEL,"F5");

  key[CL_KEY_F6].ascii=read_key_int(CL_KEY_F6,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F6].label,CL_KEY_F6,U61_CONFIG_ID_LABEL,"F6");

  key[CL_KEY_F7].ascii=read_key_int(CL_KEY_F7,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F7].label,CL_KEY_F7,U61_CONFIG_ID_LABEL,"F7");

  key[CL_KEY_F8].ascii=read_key_int(CL_KEY_F8,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F8].label,CL_KEY_F8,U61_CONFIG_ID_LABEL,"F8");

  key[CL_KEY_F9].ascii=read_key_int(CL_KEY_F9,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F9].label,CL_KEY_F9,U61_CONFIG_ID_LABEL,"F9");

  key[CL_KEY_F10].ascii=read_key_int(CL_KEY_F10,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F10].label,CL_KEY_F10,U61_CONFIG_ID_LABEL,"F10");

  key[CL_KEY_F11].ascii=read_key_int(CL_KEY_F11,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F11].label,CL_KEY_F11,U61_CONFIG_ID_LABEL,"F11");

  key[CL_KEY_F12].ascii=read_key_int(CL_KEY_F12,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_F12].label,CL_KEY_F12,U61_CONFIG_ID_LABEL,"F12");

  key[CL_KEY_ESCAPE].ascii=read_key_int(CL_KEY_ESCAPE,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_ESCAPE].label,CL_KEY_ESCAPE,U61_CONFIG_ID_LABEL,"ESCAPE");

  key[CL_KEY_LEFT].ascii=read_key_int(CL_KEY_LEFT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_LEFT].label,CL_KEY_LEFT,U61_CONFIG_ID_LABEL,"LEFT");

  key[CL_KEY_RIGHT].ascii=read_key_int(CL_KEY_RIGHT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_RIGHT].label,CL_KEY_RIGHT,U61_CONFIG_ID_LABEL,"RIGHT");

  key[CL_KEY_UP].ascii=read_key_int(CL_KEY_UP,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_UP].label,CL_KEY_UP,U61_CONFIG_ID_LABEL,"UP");

  key[CL_KEY_DOWN].ascii=read_key_int(CL_KEY_DOWN,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_DOWN].label,CL_KEY_DOWN,U61_CONFIG_ID_LABEL,"DOWN");

  key[CL_KEY_LCTRL].ascii=read_key_int(CL_KEY_LCTRL,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_LCTRL].label,CL_KEY_LCTRL,U61_CONFIG_ID_LABEL,"LCTRL");

  key[CL_KEY_RCTRL].ascii=read_key_int(CL_KEY_RCTRL,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_RCTRL].label,CL_KEY_RCTRL,U61_CONFIG_ID_LABEL,"RCTRL");

  key[CL_KEY_LSHIFT].ascii=read_key_int(CL_KEY_LSHIFT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_LSHIFT].label,CL_KEY_LSHIFT,U61_CONFIG_ID_LABEL,"LSHIFT");

  key[CL_KEY_RSHIFT].ascii=read_key_int(CL_KEY_RSHIFT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_RSHIFT].label,CL_KEY_RSHIFT,U61_CONFIG_ID_LABEL,"RSHIFT");

  key[CL_KEY_ALT].ascii=read_key_int(CL_KEY_ALT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_ALT].label,CL_KEY_ALT,U61_CONFIG_ID_LABEL,"ALT");

  key[CL_KEY_ALTGR].ascii=read_key_int(CL_KEY_ALTGR,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_ALTGR].label,CL_KEY_ALTGR,U61_CONFIG_ID_LABEL,"ALTGR");

  key[CL_KEY_TAB].ascii=read_key_int(CL_KEY_TAB,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_TAB].label,CL_KEY_TAB,U61_CONFIG_ID_LABEL,"TAB");

  key[CL_KEY_ENTER].ascii=read_key_int(CL_KEY_ENTER,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_ENTER].label,CL_KEY_ENTER,U61_CONFIG_ID_LABEL,"ENTER");

  key[CL_KEY_SPACE].ascii=read_key_int(CL_KEY_SPACE,U61_CONFIG_ID_ASCII,' ');
  read_key_str(key[CL_KEY_SPACE].label,CL_KEY_SPACE,U61_CONFIG_ID_LABEL,"SPACE");

  key[CL_KEY_BACKSPACE].ascii=read_key_int(CL_KEY_BACKSPACE,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_BACKSPACE].label,CL_KEY_BACKSPACE,U61_CONFIG_ID_LABEL,"BACKSPACE");

  key[CL_KEY_INSERT].ascii=read_key_int(CL_KEY_INSERT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_INSERT].label,CL_KEY_INSERT,U61_CONFIG_ID_LABEL,"INSERT");

  key[CL_KEY_DELETE].ascii=read_key_int(CL_KEY_DELETE,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_DELETE].label,CL_KEY_DELETE,U61_CONFIG_ID_LABEL,"DELETE");

  key[CL_KEY_HOME].ascii=read_key_int(CL_KEY_HOME,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_HOME].label,CL_KEY_HOME,U61_CONFIG_ID_LABEL,"HOME");

  key[CL_KEY_END].ascii=read_key_int(CL_KEY_END,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_END].label,CL_KEY_END,U61_CONFIG_ID_LABEL,"END");

  key[CL_KEY_PAGEUP].ascii=read_key_int(CL_KEY_PAGEUP,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_PAGEUP].label,CL_KEY_PAGEUP,U61_CONFIG_ID_LABEL,"PAGEUP");

  key[CL_KEY_PAGEDOWN].ascii=read_key_int(CL_KEY_PAGEDOWN,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_PAGEDOWN].label,CL_KEY_PAGEDOWN,U61_CONFIG_ID_LABEL,"PAGEDOWN");

  key[CL_KEY_CAPSLOCK].ascii=read_key_int(CL_KEY_CAPSLOCK,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_CAPSLOCK].label,CL_KEY_CAPSLOCK,U61_CONFIG_ID_LABEL,"CAPSLOCK");

  key[CL_KEY_NUMLOCK].ascii=read_key_int(CL_KEY_NUMLOCK,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_NUMLOCK].label,CL_KEY_NUMLOCK,U61_CONFIG_ID_LABEL,"NUMLOCK");

  key[CL_KEY_SCRLOCK].ascii=read_key_int(CL_KEY_SCRLOCK,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_SCRLOCK].label,CL_KEY_SCRLOCK,U61_CONFIG_ID_LABEL,"SCRLOCK");

  key[CL_KEY_PRINT].ascii=read_key_int(CL_KEY_PRINT,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_PRINT].label,CL_KEY_PRINT,U61_CONFIG_ID_LABEL,"PRINT");

  key[CL_KEY_PAUSE].ascii=read_key_int(CL_KEY_PAUSE,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_PAUSE].label,CL_KEY_PAUSE,U61_CONFIG_ID_LABEL,"PAUSE");

  key[CL_KEY_KP_DIV].ascii=read_key_int(CL_KEY_KP_DIV,U61_CONFIG_ID_ASCII,'/');
  read_key_str(key[CL_KEY_KP_DIV].label,CL_KEY_KP_DIV,U61_CONFIG_ID_LABEL,"KP_DIV");

  key[CL_KEY_KP_MULT].ascii=read_key_int(CL_KEY_KP_MULT,U61_CONFIG_ID_ASCII,'*');
  read_key_str(key[CL_KEY_KP_MULT].label,CL_KEY_KP_MULT,U61_CONFIG_ID_LABEL,"KP_MULT");

  key[CL_KEY_KP_MINUS].ascii=read_key_int(CL_KEY_KP_MINUS,U61_CONFIG_ID_ASCII,'-');
  read_key_str(key[CL_KEY_KP_MINUS].label,CL_KEY_KP_MINUS,U61_CONFIG_ID_LABEL,"KP_MINUS");

  key[CL_KEY_KP_PLUS].ascii=read_key_int(CL_KEY_KP_PLUS,U61_CONFIG_ID_ASCII,'+');
  read_key_str(key[CL_KEY_KP_PLUS].label,CL_KEY_KP_PLUS,U61_CONFIG_ID_LABEL,"KP_PLUS");

  key[CL_KEY_KP_ENTER].ascii=read_key_int(CL_KEY_KP_ENTER,U61_CONFIG_ID_ASCII,U61_KEYDEF_ASCII_UNKNOWN);
  read_key_str(key[CL_KEY_KP_ENTER].label,CL_KEY_KP_ENTER,U61_CONFIG_ID_LABEL,"KP_ENTER");

  key[CL_KEY_KP_DECIMAL].ascii=read_key_int(CL_KEY_KP_DECIMAL,U61_CONFIG_ID_ASCII,'.');
  read_key_str(key[CL_KEY_KP_ENTER].label,CL_KEY_KP_ENTER,U61_CONFIG_ID_LABEL,"KP .");

  key[CL_KEY_KP_0].ascii=read_key_int(CL_KEY_KP_0,U61_CONFIG_ID_ASCII,'0');
  read_key_str(key[CL_KEY_KP_0].label,CL_KEY_KP_0,U61_CONFIG_ID_LABEL,"KP 0");

  key[CL_KEY_KP_1].ascii=read_key_int(CL_KEY_KP_1,U61_CONFIG_ID_ASCII,'1');
  read_key_str(key[CL_KEY_KP_1].label,CL_KEY_KP_1,U61_CONFIG_ID_LABEL,"KP 1");

  key[CL_KEY_KP_2].ascii=read_key_int(CL_KEY_KP_2,U61_CONFIG_ID_ASCII,'2');
  read_key_str(key[CL_KEY_KP_2].label,CL_KEY_KP_2,U61_CONFIG_ID_LABEL,"KP 2");

  key[CL_KEY_KP_3].ascii=read_key_int(CL_KEY_KP_3,U61_CONFIG_ID_ASCII,'3');
  read_key_str(key[CL_KEY_KP_3].label,CL_KEY_KP_3,U61_CONFIG_ID_LABEL,"KP 3");

  key[CL_KEY_KP_4].ascii=read_key_int(CL_KEY_KP_4,U61_CONFIG_ID_ASCII,'4');
  read_key_str(key[CL_KEY_KP_4].label,CL_KEY_KP_4,U61_CONFIG_ID_LABEL,"KP 4");

  key[CL_KEY_KP_5].ascii=read_key_int(CL_KEY_KP_5,U61_CONFIG_ID_ASCII,'5');
  read_key_str(key[CL_KEY_KP_5].label,CL_KEY_KP_5,U61_CONFIG_ID_LABEL,"KP 5");

  key[CL_KEY_KP_6].ascii=read_key_int(CL_KEY_KP_6,U61_CONFIG_ID_ASCII,'6');
  read_key_str(key[CL_KEY_KP_6].label,CL_KEY_KP_6,U61_CONFIG_ID_LABEL,"KP 6");

  key[CL_KEY_KP_7].ascii=read_key_int(CL_KEY_KP_7,U61_CONFIG_ID_ASCII,'7');
  read_key_str(key[CL_KEY_KP_7].label,CL_KEY_KP_7,U61_CONFIG_ID_LABEL,"KP 7");

  key[CL_KEY_KP_8].ascii=read_key_int(CL_KEY_KP_8,U61_CONFIG_ID_ASCII,'8');
  read_key_str(key[CL_KEY_KP_8].label,CL_KEY_KP_8,U61_CONFIG_ID_LABEL,"KP 8");

  key[CL_KEY_KP_9].ascii=read_key_int(CL_KEY_KP_9,U61_CONFIG_ID_ASCII,'9');
  read_key_str(key[CL_KEY_KP_9].label,CL_KEY_KP_9,U61_CONFIG_ID_LABEL,"KP 9");
}

/*--------------------------------------------------------------------------*/
/* 
 * saves the values to the config file
 */ 
bool U61_Config::write()
{
  bool ok=false;
  char *filename;
  FILE *file;
  int pl,k;
  char version[U61_CONST_STRING_SIZE];
  
  U61_Global::data.chdir_to_initial_dir();

  U61_LOG_DEBUG("write config");
  if (U61_Platform::create_user_dir())
    {
      filename=U61_Platform::get_config_file();
      U61_LOG_MESSAGE("Saving config to \""<<filename<<"\".");
      file=fopen(filename,"w");

      if (file!=NULL)
	{
	  ok=true;

	  U61_Version::get_version_text(version);

	  ok=ok && write_str(file,U61_CONFIG_ID_PROGRAM, U61_CONST_PROGRAM);
	  ok=ok && write_str(file,U61_CONFIG_ID_VERSION, version);
	  ok=ok && write_int(file,U61_CONFIG_ID_LAUNCH_COUNT, launch_count+1);

	  ok=ok && write_str(file,U61_CONFIG_ID_DATA_DIR, data_dir);
	  ok=ok && write_str(file,U61_CONFIG_ID_SCRIPT_DIR, script_dir);
	  ok=ok && write_str(file,U61_CONFIG_ID_DATA_FILE, data_file);
	  ok=ok && write_str(file,U61_CONFIG_ID_SCRIPT_FILE, script_file);

	  ok=ok && write_int(file,U61_CONFIG_ID_FULLSCREEN, fullscreen);
	  ok=ok && write_int(file,U61_CONFIG_ID_SHOW_SYSTEM_INFO, show_system_info);

	  ok=ok && write_str(file,U61_CONFIG_ID_SERVER_ID, server_id);
	  ok=ok && write_int(file,U61_CONFIG_ID_NETWORK_PORT, network_port);
	  ok=ok && write_int(file,U61_CONFIG_ID_NETWORK_CHANNEL, network_channel);

	  ok=ok && write_int(file,U61_CONFIG_ID_INITIAL_SPEED, initial_speed);
	  ok=ok && write_int(file,U61_CONFIG_ID_ACCELERATION, acceleration);
	  ok=ok && write_int(file,U61_CONFIG_ID_CURSE_DELAY, curse_delay);

	  ok=ok && write_int(file,U61_CONFIG_ID_SOUND_VOLUME, sound_volume);
	  ok=ok && write_int(file,U61_CONFIG_ID_MUSIC_VOLUME, music_volume);

	  ok=ok && write_int(file,U61_CONFIG_ID_MAX_FPS, max_fps);

	  ok=ok && write_int(file,U61_CONFIG_ID_NETWORK_ANTICIPATION, network_anticipation);

	  for (pl=0;pl<U61_CONFIG_NB_PLAYERS;++pl)
	    {
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_FIRST_REPEAT_MOVE,
					player[pl].first_repeat_move);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_STANDARD_REPEAT_MOVE,
					player[pl].standard_repeat_move);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_FIRST_REPEAT_ROTATE,
					player[pl].first_repeat_rotate);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_STANDARD_REPEAT_ROTATE,
					player[pl].standard_repeat_rotate);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_FIRST_REPEAT_OTHER,
					player[pl].first_repeat_other);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_STANDARD_REPEAT_OTHER,
					player[pl].standard_repeat_other);
 
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_ANTICIPATION,
					player[pl].anticipation);

	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_ACTIVABLE,
					player[pl].activable);
	      ok=ok && write_player_str(pl,file,U61_CONFIG_ID_NAME,
					player[pl].name);

	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_START_STOP,
					player[pl].key_start_stop);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_MOVE_LEFT,
					player[pl].key_move_left);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_MOVE_RIGHT,
					player[pl].key_move_right);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_ROTATE_LEFT,
					player[pl].key_rotate_left);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_ROTATE_RIGHT,
					player[pl].key_rotate_right);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_MOVE_DOWN,
					player[pl].key_move_down);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_DROP,
					player[pl].key_drop);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_NEXT_VICTIM,
					player[pl].key_next_victim);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_PREV_VICTIM,
					player[pl].key_prev_victim);
	      ok=ok && write_player_int(pl,file,U61_CONFIG_ID_KEY_USE_ANTIDOTE,
					player[pl].key_use_antidote);
	    } 

	  for (k=0;k<U61_INPUTMONITOR_NUM_KEYS_SCANCODE;++k)
	    {
	      /*
	       * We do not save informations about keys which seem
	       * to be unknown/not properly regognized
	       */
	      if (key[k].ascii!=U61_KEYDEF_ASCII_UNKNOWN &&
		  strcmp(key[k].label,U61_KEYDEF_LABEL_UNKNOWN)!=0)
		{
		  ok=ok && write_key_int(k,file,U61_CONFIG_ID_ASCII,key[k].ascii);
		  ok=ok && write_key_str(k,file,U61_CONFIG_ID_LABEL,key[k].label);
		}
	    }

	  fclose(file);
	}
    }
  
  U61_Global::data.chdir_to_data_dir();

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the user scripts that will be used by the script module
 */
bool U61_Config::load_script()
{
  bool ok=false;

  U61_LOG_MESSAGE("Loading script from \""<<script_file<<"\".");
  ok=U61_Script::do_file(script_file);
    
  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * loads a string entry from the config file
 */
void U61_Config::read_str(char *buf, char *name,char *def)
{
  strcpy(buf,U61_Script::get_global_str(name,def));
}

/*--------------------------------------------------------------------------*/
/*
 * loads a number entry from the config file
 */
int U61_Config::read_int(char *name,int def)
{
  return U61_Script::get_global_int(name,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a boolean entry from the config file
 */
bool U61_Config::read_bool(char *name,bool def)
{
  return U61_Script::get_global_bool(name,def);
}

/*--------------------------------------------------------------------------*/
/*
 * saves a string entry into the config file
 */
bool U61_Config::write_str(FILE *file,char *name,char *value)
{
  char buffer[U61_CONST_STRING_SIZE];
  char buffer_value[U61_CONST_STRING_SIZE];
  int pos;
  bool ok;

  /*
   * Now we have to trap escape caracters such as \ if we
   * want the system to be bullet proof. It is required for
   * windows (\ in every path), but it can also be usefull on
   * UNIX, your never know...
   */

  pos=0;
  while (*value!=0)
    {
      switch (*value)
	{
	case '\\':
	  buffer_value[pos]='\\';
	  pos++;
	  /*
	   * no break; instruction, since we want the caracter to
	   * be added after the \
	   */
	default:
	  buffer_value[pos]=*value;
	  pos++;
	}
      value++;
    }
  buffer_value[pos]=0;

  strcpy(buffer,name);
  strcat(buffer," = \"");
  strcat(buffer,buffer_value);
  strcat(buffer,"\"\n");

  ok=(fwrite(buffer,strlen(buffer),1,file)>0);

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * saves a numeric entry into the config file
 */
bool U61_Config::write_int(FILE *file,char *name,int value)
{
  char buffer[U61_CONST_STRING_SIZE];
  char buffer2[20];
  bool ok;

  sprintf(buffer2,"%d",value);

  strcpy(buffer,name);
  strcat(buffer," = ");
  strcat(buffer,buffer2);
  strcat(buffer,"\n");

  ok=(fwrite(buffer,strlen(buffer),1,file)>0);

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * loads a string entry from the config file, for a specific player
 */
void U61_Config::read_player_str(char *buf,int pl,char *name,char *def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  read_str(buf,buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a number entry from the config file, for a specific player
 */
int U61_Config::read_player_int(int pl,char *name,int def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return read_int(buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a boolean entry from the config file, for a specific player
 */
bool U61_Config::read_player_bool(int pl,char *name,bool def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return read_bool(buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * saves a string entry into the config file, for a specific player
 */
bool U61_Config::write_player_str(int pl,FILE *file,char *name,char *value)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return write_str(file,buffer,value);
} 

/*--------------------------------------------------------------------------*/
/*
 * saves a numeric entry into the config file, for a specific player
 */
bool U61_Config::write_player_int(int pl,FILE *file,char *name,int value)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_PLAYER,pl);
  strcat(buffer,name);

  return write_int(file,buffer,value);
} 

/*--------------------------------------------------------------------------*/
/*
 * loads a string entry from the config file, for a specific key
 */
void U61_Config::read_key_str(char *buf,int k,char *name,char *def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_KEY,k);
  strcat(buffer,name);

  read_str(buf,buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a number entry from the config file, for a specific key
 */
int U61_Config::read_key_int(int k,char *name,int def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_KEY,k);
  strcat(buffer,name);

  return read_int(buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * loads a boolean entry from the config file, for a specific key
 */
bool U61_Config::read_key_bool(int k,char *name,bool def)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_KEY,k);
  strcat(buffer,name);

  return read_bool(buffer,def);
}

/*--------------------------------------------------------------------------*/
/*
 * saves a string entry into the config file, for a specific key
 */
bool U61_Config::write_key_str(int k,FILE *file,char *name,char *value)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_KEY,k);
  strcat(buffer,name);

  return write_str(file,buffer,value);
} 

/*--------------------------------------------------------------------------*/
/*
 * saves a numeric entry into the config file, for a specific key
 */
bool U61_Config::write_key_int(int k,FILE *file,char *name,int value)
{
  char buffer[U61_CONST_STRING_SIZE];

  sprintf(buffer,U61_CONFIG_ID_KEY,k);
  strcat(buffer,name);

  return write_int(file,buffer,value);
} 




