/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   keydef.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: A keydef is usefull to identify a key, that's to
 *              say make the link between a system code, an ascii code
 *              and a label. It also contains a "state" field which
 *              is used by the global input monitor to know wether it
 *              is pressed or not.
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <string.h>
#include <stdio.h>

#include <ClanLib/core.h>
#include <ClanLib/display.h>

#include "keydef.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_KEYDEF_CHECK_DELAY 1000

/*---------------------------------------------------------------------------
 methods
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default keydef
 */ 
U61_KeyDef::U61_KeyDef()
{
  define(U61_KEYDEF_ASCII_UNKNOWN,
	 U61_KEYDEF_LABEL_UNKNOWN,
	 CL_KEY_NONE_OF_THE_ABOVE);
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a keydef
 */ 
U61_KeyDef::~U61_KeyDef()
{
}

/*--------------------------------------------------------------------------*/
/* 
 * resets the state of a keydef
 */ 
void U61_KeyDef::reset()
{
  key_state=false;
  key_flag=false;
  last_time=0;
}

/*--------------------------------------------------------------------------*/
/* 
 * Sets the parameters of a keydef
 */ 
void U61_KeyDef::define(int ascii, char *label,int scancode)
{
  key_ascii=ascii;
  strncpy(key_label,label,U61_CONST_STRING_SIZE);
  key_label[U61_CONST_STRING_SIZE-1]='\0';
  key_scancode=scancode;

  update_label();

  reset();
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the scancode code of a keydef
 */ 
int U61_KeyDef::get_scancode()
{
  return key_scancode;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the ascii code of a keydef
 */ 
char U61_KeyDef::get_ascii()
{
  return key_ascii;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the label of a keydef
 */ 
char *U61_KeyDef::get_label()
{
  return key_label;
}

/*--------------------------------------------------------------------------*/
/* 
 * Returns the state (pressed or not) of a keydef
 * Note the use of the special "key_flag" value. This value is set to true
 * when the key is pressed, and polling the key with get_state sets it back
 * to false. This way if a key is pressed and then released, the first call
 * to get_state will still return true at least once. This prevents keys
 * from being "forgetted" if they are pressed during a period when the
 * keyboard is not polled at all (when there's heavy drawing for instance)
 */ 
bool U61_KeyDef::get_state()
{
  bool result;

  result=key_state||key_flag;
  key_flag=false;

  return result;
}

/*--------------------------------------------------------------------------*/
/* 
 * Updates the label accordingly to the ascii value
 */ 
void U61_KeyDef::update_label()
{
  if (key_ascii>32 && key_ascii<=127)
    {
      if (isalnum(key_ascii))
	{
	  if (key_scancode>=CL_KEY_0 && key_scancode<=CL_KEY_9)
	    {
	      /*
	       * Now for numbers we have a very special behavior:
	       * we always write the number whatever the ascii code
	       * might be. This is usefull for in the key settings
	       * menu, people remember much more easily of 1,2,3
	       * than of !, @ or #... And at the same time, we _must_
	       * allow the user to type those characters for for
	       * instance on azerty keyboards, you can't type "-" if
	       * you can't shift numbers 8-(
	       */
	      sprintf(key_label,"%d",key_scancode-CL_KEY_0);
	    }
	  else
	    {
	      sprintf(key_label,"%c",key_ascii);
	    }
	}
      else
	{
	  if (key_scancode>=CL_KEY_NONE_OF_THE_ABOVE)
	    {
	      sprintf(key_label,"%c (ascii %d)",key_ascii,key_ascii);
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/* 
 * Acknowledges a press on the key
 */ 
void U61_KeyDef::press(int scancode,int ascii)
{
  key_flag=true;
  key_state=true;

  last_time=CL_System::get_time();

  if (ascii>=0)
    {
      key_ascii=ascii;
    }
  key_scancode=scancode;

  update_label();
}

/*--------------------------------------------------------------------------*/
/* 
 * Acknowledges a release of the key
 */ 
void U61_KeyDef::release(int scancode,int ascii)
{
  key_state=false;

  last_time=0;

  if (ascii>=0)
    {
      key_ascii=ascii;
    }
  key_scancode=scancode;

  update_label();
}

/*--------------------------------------------------------------------------*/
/* 
 * Performs a check on the key: if it has not been pressed recently but
 * is still marked as being pressed, we force it to "unpressed"
 */ 
void U61_KeyDef::check()
{
  if (key_state && last_time<CL_System::get_time()-U61_KEYDEF_CHECK_DELAY)
    {
      release(key_scancode,key_ascii);
    }
}


