/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   bufferedsocket.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: subclass of socket which allow buffer sends/reads
 *              in a "buffered" context. Sends and receives never
 *              fail unless the socket is really closed. If send/receives
 *              are successfull, then 100% of the requested date is
 *              sent/received successfully.
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include "bufferedsocket.h"
#include "log.h"
#include "macro.h"
#include "time.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

/*
 * Number of bytes sent at once
 */
#define U61_BUFFEREDSOCKET_CHUNK_SIZE 100

/*
 * Max number of flushes per second
 */
#define U61_BUFFEREDSOCKET_FLUSH_INPUT_FREQ    5
#define U61_BUFFEREDSOCKET_FLUSH_OUTPUT_FREQ   5

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * Creates a socket
 */
U61_BufferedSocket::U61_BufferedSocket() : U61_Socket()
{
  last_flush_input_time=0;
  last_flush_output_time=0;
}

/*--------------------------------------------------------------------------*/
/*
 * Creates a buffered socket with the data of a socket object
 */
U61_BufferedSocket::U61_BufferedSocket(U61_Socket *s) : U61_Socket()
{
  last_flush_input_time=0;
  last_flush_output_time=0;

  alive = s->is_alive();
  mode = s->get_mode();
  sock = s->get_sock();
  U61_MACRO_STRCPY(sock_host, s->get_host());
  U61_MACRO_STRCPY(sock_ip, s->get_ip());
  sock_port = s->get_port();
}

/*--------------------------------------------------------------------------*/
/*
 * Destroys a socket 
 */
U61_BufferedSocket::~U61_BufferedSocket()
{
  close();
}

/*--------------------------------------------------------------------------*/
/*
 * Destroys a socket 
 */
U61_BufferedSocket *U61_BufferedSocket::accept()
{
  U61_Socket *temp=NULL;
  U61_BufferedSocket *result=NULL;
  
  temp=U61_Socket::accept();
  if (temp)
    {
      result=new U61_BufferedSocket(temp);

      /*
       * Very important to call this function. If it's not called, deleting
       * the temp socket will close the handle 8-/ Call this a hack...
       */
      temp->disable_close_on_delete();
      delete temp;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Pushes a buffer in the "send" stack
 */
bool U61_BufferedSocket::push_buf(char *buf, int size)
{
  bool result = false;

  if (is_alive())
    {
      result = true;

      if (result)
	{
	  result = output_buffer.append(buf,size);
	  
	  if (result)
	    {
	      U61_LOG_DEBUG("Pushed "<<size<<" bytes to connection output buffer.");
	    }
	}
      
      if (result && flush_output_needed())
	{
	  result = flush_output();
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Pops a buffer from the "receive" stack
 */
bool U61_BufferedSocket::pop_buf(char *buf, int size, bool peek)
{
  bool result = false;
  bool warning_printed = false;

  if (is_alive())
    {
      result = true;

      while (result && size>input_buffer.get_size())
	{
	  if (!warning_printed)
	    {
	      warning_printed=true;
	      U61_LOG_WARNING("Not enough data on input buffered socket ("<<input_buffer.get_size()<<"/"<<size<<").");
	    }
	  result = flush_input();
	}
      
      if (result)
	{
	  memcpy(buf,input_buffer.get_data(),size);

	  if (!peek)
	    {	  
	      U61_LOG_DEBUG("Popped "<<size<<" bytes from connection input buffer.");

	      input_buffer.cut(size);
	    }
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the size of input data available
 */
int U61_BufferedSocket::get_input_size()
{
  int result = 0;

  if (flush_input_needed())
    {
      /*
       * Here we ignore the error code, errors will
       * appear during call to other functions such as pop_buf
       * or is_alive.
       */
      flush_input();
    }

  result = input_buffer.get_size();

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Polls the socket (flushes buffers if needed)
 */
bool U61_BufferedSocket::flush_if_needed()
{
  bool result = true;
  bool result_input = true;
  bool result_output = true;

  if (flush_input_needed())
    {
      result_input = flush_input();
    }

  if (flush_output_needed())
    {
      result_output = flush_output();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Flushes the "send" stack
 */
bool U61_BufferedSocket::flush_output()
{
  bool result = true;
  int sent, to_send;
  bool full = false;

  while (result && !full && output_buffer.get_size()>0)
    {
      to_send = output_buffer.get_size();
      if (to_send>U61_BUFFEREDSOCKET_CHUNK_SIZE)
	{
	  to_send=U61_BUFFEREDSOCKET_CHUNK_SIZE;
	}
      sent = send_buf(output_buffer.get_data(),to_send);

      if (sent<0)
	{
	  result=false;
	}
      else
	{
	  if (sent<to_send)
	    {
	      full = true;
	    }

	  if (sent>0)
	    {
	      output_buffer.cut(sent);
	    }
	}
    }

  last_flush_output_time=U61_Time::for_effect();

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Flushes the "receive" stack
 */
bool U61_BufferedSocket::flush_input()
{
  bool result = true;
  int received;
  char buffer[U61_BUFFEREDSOCKET_CHUNK_SIZE];
  bool data_available=true;

  U61_LOG_DEBUG("flushing input");

  while (result && data_available)
    {
      received=recv_buf(buffer,U61_BUFFEREDSOCKET_CHUNK_SIZE);
      
      if (received<0)
	{
	  result=false;
	}
      else
	{
	  if (received==0)
	    {
	      data_available=false;
	    }
	  
	  if (received>0)
	    {
	      input_buffer.append(buffer,received);
	    }
	}
    }

  last_flush_input_time=U61_Time::for_effect();

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the input buffer needs to be flushed
 */
bool U61_BufferedSocket::flush_input_needed()
{
  bool needed=false;

  if (U61_Time::for_effect()>
      last_flush_input_time + 
      (U61_TIME_ONE_SECOND / U61_BUFFEREDSOCKET_FLUSH_INPUT_FREQ))
    {
      needed = true;
    }

  return needed;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the output buffer needs to be flushed
 */
bool U61_BufferedSocket::flush_output_needed()
{
  bool needed=false;

  if (U61_Time::for_effect()>
      last_flush_output_time + 
      (U61_TIME_ONE_SECOND / U61_BUFFEREDSOCKET_FLUSH_OUTPUT_FREQ))
    {
      needed = true;
    }

  return needed;
}
















