/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   chatdisplay.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: module responsible for getting chat messages from the
 *              network, interpret them and then display them on the
 *              screen.
 */

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

/*
 * Messages are displayed 30 seconds after last keypress.
 */
#define U61_CHATDISPLAY_DELAY 30

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "chatdisplay.h"
#include "log.h"
#include "time.h"
#include "global.h"

/*--------------------------------------------------------------------------*/
/*
 * Constructor of the chatdisplay class
 */
U61_ChatDisplay::U61_ChatDisplay()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * Destructor of the chatdisplay class
 */
U61_ChatDisplay::~U61_ChatDisplay()
{

}

/*--------------------------------------------------------------------------*/
/*
 * Resets all the items of a chatdisplay
 */
void U61_ChatDisplay::reset()
{
  lines.clear();
  set_player_id(0);
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the id of the locally talking player.
 */
void U61_ChatDisplay::set_player_id(int pid)
{
  player_id=pid;
}

/*--------------------------------------------------------------------------*/
/*
 * Puts a chat letter event in the queue
 */
void U61_ChatDisplay::put_chat_letter(U61_Event evt)
{
  int i,size,line;

  size=lines.size();
  line=-1;

  for (i=0;i<size;++i)
    {
      if (((unsigned int) lines[i].get_player_id()==evt.author) &&
	  (!lines[i].full()))
	{
	  line=i;
	}
    }

  if (line<0)
    {
      /*
       * If no line exists that fits our needs, we create one
       */
      line=size;
      lines.push_back(U61_ChatDisplayLine());
    }

  lines[line].put_chat_letter(evt);
}

/*--------------------------------------------------------------------------*/
/*
 * Draws the chat messages on the screen.
 */
void U61_ChatDisplay::draw(int x, int y, int w, int h)
{
  int i,size,dy;
  int current_line;
  bool cursor;

  remove_old_lines();
  remove_empty_lines();

  /*
   * First we remove lines which don't fit on the screen
   */
  size=lines.size();
  /*
   * Note that because of this "add line if missing feature" and
   * also because the trailing blinking "_" might make a line
   * longer and take 2 rows instead of 1, we need to intialize
   * dy to the height of the font... 
   */
  dy=U61_Global::data.font_info[U61_DATA_SIZE_SMALL]->get_height();
  for (i=size-1;i>=0;--i)
    {
      dy+=lines[i].get_height(w);
      if (dy>h)
	{
	  lines.erase(&(lines[i]));	  
	}
    }

  current_line=-1;
  /*
   * If we aren't in chat mode, we do not handle all
   * the "current line" stuff, we simply display
   * the existing lines, but won't add any blinking
   * cursor...
   */
  if (U61_Global::game.chat_input.is_active())
    {
      /*
       * We check if there's a line being edited currently
       */
      for (i=0;i<size;++i)
	{
	  if (lines[i].get_player_id()==player_id &&
	      !lines[i].full())
	    {
	      current_line=i;
	    }
	}
      /*
       * If there's no current line, we add one...
       */
      if (current_line<0)
	{
	  U61_ChatDisplayLine cdl;
	  cdl.set_player_id(player_id);
	  lines.push_back(cdl);
	  current_line=lines.size()-1;
	}
    }

  /*
   * then we draw
   */
  size=lines.size();
  dy=0;
  for (i=0;i<size;++i)
    {
      cursor=(i==current_line);
      lines[i].draw(x,y+dy,w,h,cursor);
      dy+=lines[i].get_height(w,cursor);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Removes lines which are too old.
 */
void U61_ChatDisplay::remove_old_lines()
{
  int i,size;

  size=lines.size();

  for (i=size-1;i>=0;--i)
    {
      if (((int) (U61_Time::for_event() - lines[i].get_last_update()))
	  >U61_CHATDISPLAY_DELAY*U61_TIME_ONE_SECOND)
	{
	  lines.erase(&(lines[i]));
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Removes useless empty lines, which might appear when one presses enter
 * and type nothing for instance.
 */
void U61_ChatDisplay::remove_empty_lines()
{
  int i,size;

  size=lines.size();

  for (i=size-1;i>=0;--i)
    {
      if (lines[i].empty())
	{
	  lines.erase(&(lines[i]));
	}
    }
}
