/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   chatdisplayline.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: holds information for a chat line, and provides methods
 *              to display it.
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/core.h>
#include <ClanLib/display.h>

#include "chatdisplay.h"
#include "log.h"
#include "macro.h"
#include "global.h"

/*--------------------------------------------------------------------------*/
/*
 * Constructor of the chatdisplayline class
 */
U61_ChatDisplayLine::U61_ChatDisplayLine()
{
  //U61_LOG_WARNING("create display line");
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * Destructor of the chatdisplayline class
 */
U61_ChatDisplayLine::~U61_ChatDisplayLine()
{
  //U61_LOG_WARNING("delete display line");
}

/*--------------------------------------------------------------------------*/
/*
 * Resets all the items of a chatdisplayline
 */
void U61_ChatDisplayLine::reset()
{
  player_id=-1;
  end_forced=false;
  U61_MACRO_MEMSET0(text);
  last_update=0;
}

/*--------------------------------------------------------------------------*/
/*
 * Puts a key strike in the queue
 */
void U61_ChatDisplayLine::put_chat_letter(U61_Event evt)
{
  unsigned int len;
  int ascii;

  len=strlen(text);

  /*
   * If this is the first message, we set up the player id
   */
  if (player_id<0)
    {
      player_id=evt.author;
    }

  if ((unsigned int) player_id != evt.author)
    {
      /*
       * This should never ever happen...
       */
      U61_LOG_WARNING("Inconsistency in chat messages, event with player id "<<evt.author<<" when it should be "<<player_id);
    }

  switch(evt.par)
    {
    case CL_KEY_ENTER:
      /*
       * Enter has been pressed, so we decide to consider that
       * the line is ended, even if we did not reach the size limit...
       */
      log_text();
      end_forced=true;
      break;
    case CL_KEY_LEFT:
    case CL_KEY_BACKSPACE:
      /*
       * Delete  char
       */
      if (len>0)
	{
	  text[len-1]='\0';
	}
      break;
    default:
      ascii=U61_Global::input_monitor.to_ascii(evt.par);
      if (ascii!=U61_KEYDEF_ASCII_UNKNOWN &&
	  ascii>=32 &&
	  ascii<=127)
	{
	  if (len<sizeof(text)-1)
	    {
	      text[len]=ascii;
	    }  
	  /*
	   * We display a log message if this was the last char.
	   */
	  if (len==sizeof(text)-2)
	    {
	      log_text();
	    }
	}
    }

  last_update=evt.time;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the chat line is full, and we need to start another one.
 */
bool U61_ChatDisplayLine::full()
{
  bool f=(end_forced || strlen(text)>=sizeof(text)-1);

  return f;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the chat line is empty, ie equal to "", " ", "  ", etc.
 */
bool U61_ChatDisplayLine::empty()
{
  bool e=true;
  int i;
  int pid;
  U61_Player *p;

  pid=get_player_id();
  if (pid>=0)
    {
      p=U61_Global::game.find_player_by_id(pid);	  
      if (p!=NULL)
	{
	  i=0;
	  while (text[i] && e)
	    {
	      if (text[i]!=' ')
		{
		  e=false;
		}
	    }
	}
    }

  return e;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the id of the player whom this line is associated to
 */
int U61_ChatDisplayLine::get_player_id()
{
  return player_id;
}

/*--------------------------------------------------------------------------*/
/*
 * Forces the player id, usefull in some (yet rare) cases, especially when
 * one wants to display the last editing line.
 */
void U61_ChatDisplayLine::set_player_id(int pid)
{
  player_id=pid;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the name of the player whom this line is associated to
 */
char *U61_ChatDisplayLine::get_player_name()
{
  U61_Player *p;
  int pid;
  char *pname=NULL;

  pid=get_player_id();
  if (pid>=0)
    {
      p=U61_Global::game.find_player_by_id(pid);	  
      if (p!=NULL)
	{
	  pname=p->get_name();
	}
    }

  if (pname==NULL)
    {
      pname="";
    }

  return pname;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the time of the last update (key event).
 */
unsigned int U61_ChatDisplayLine::get_last_update()
{
  return last_update;
}

/*--------------------------------------------------------------------------*/
/*
 * Logs the text on the console
 */
void U61_ChatDisplayLine::log_text()
{
  /*
   * We don't display empty lines.
   */
  if (!empty())
    {
      U61_LOG_MESSAGE("Player \""<<get_player_name()<<"\" says \""<<text<<"\".");
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Draws the text on the screen.
 */
void U61_ChatDisplayLine::draw(int x, 
			       int y,
			       int w, 
			       int h,
			       bool cursor)
{
  U61_TextDisplayer *td;

  td=prepare_draw(cursor);

  td->draw(x,y,U61_TEXTDISPLAYER_ALIGN_LEFT,w,h);

  delete td;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the height of the text when it will be drawn.
 */
int U61_ChatDisplayLine::get_height(int w, bool cursor)
{
  U61_TextDisplayer *td;
  int height;

  td=prepare_draw(cursor);

  height=td->get_height(w);

  delete td;

  return height;
}

/*--------------------------------------------------------------------------*/
/*
 * Internal function used by draw and get_height. The object returned
 * must be freed.
 */
U61_TextDisplayer *U61_ChatDisplayLine::prepare_draw(bool cursor)
{
  U61_TextDisplayer *td;
  char t[U61_CHATDISPLAYLINE_MAX_LENGTH+U61_DATA_PLAYER_NAME_SIZE+3];
  int size;

  if (cursor)
    {
      U61_MACRO_SPRINTF2(t,"%s: %s ",get_player_name(),text);
    }
  else
    {
      U61_MACRO_SPRINTF2(t,"%s: %s",get_player_name(),text);
    }

  td=new U61_TextDisplayer(U61_Global::data.font_info[U61_DATA_SIZE_SMALL],
			   t);

  if (cursor)
    {
      size=td->get_text_size();
      if (size-1>=0)
	{
	  td->enable_blink(size-1);
	}
    }

  return td;
}
