/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   clientprotocol.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the client sends informations about him and retrieves
 *              information from the server, including the rules and
 *              the state of all the other players
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include "clientprotocol.h"
#include "global.h"
#include "time.h"
#include "platform.h"
#include "log.h"
#include "macro.h"
#include "checksum.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_CLIENTPROTOCOL_CONNECTED_TO_SERVER      10
#define U61_CLIENTPROTOCOL_PASSWORD_SENT            30
#define U61_CLIENTPROTOCOL_PASSWORD_OK_RECEIVED     31
#define U61_CLIENTPROTOCOL_SCRIPT_NAME_RECEIVED     40
#define U61_CLIENTPROTOCOL_SCRIPT_RECEIVED          41
#define U61_CLIENTPROTOCOL_ID0_RECEIVED             50
#define U61_CLIENTPROTOCOL_ID1_RECEIVED             51
#define U61_CLIENTPROTOCOL_INITIAL_SPEED_RECEIVED   52
#define U61_CLIENTPROTOCOL_ACCELERATION_RECEIVED    53
#define U61_CLIENTPROTOCOL_CURSE_DELAY_RECEIVED     54
#define U61_CLIENTPROTOCOL_TIME_RECEIVED            55
#define U61_CLIENTPROTOCOL_PLAYING                  70   
#define U61_CLIENTPROTOCOL_CLOSED                   71

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a client protocol
 */
U61_ClientProtocol::U61_ClientProtocol(U61_ClientDispatcher *d, 
				       int p,
                                       char *srv)
  : U61_Protocol(p)
{
  char *pos;

  /*
   * we remove spaces at the end of the server name
   */
  pos=strchr(srv,' ');
  if (pos!=NULL)
    {
      (*pos)='\0';
    }

  dispatcher=d;
  server_id=srv;
  connection=NULL;

  connecting=false;
  connect_thread=NULL;
  connect_error=false;
  password_error=false;
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the protocol object
 */
U61_ClientProtocol::~U61_ClientProtocol()
{
  stop_connect_thread(true);
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if the server is still there...
 */
void U61_ClientProtocol::check_connection()
{
  int i;
  U61_Player *p;
  U61_Event evt;

  if (stage!=U61_CLIENTPROTOCOL_CLOSED 
      && connection)
    {
      if (!connection->is_alive())
	{
	  if (stage!=U61_CLIENTPROTOCOL_PLAYING)
	    {
	      if (password_error)
		{
		  U61_LOG_WARNING("Wrong password.");
		  U61_Menu::display_info("Wrong password");	      
		}
	      else
		{
		  U61_Menu::display_info("Connection closed");	      
		}
	    }
	  
	  stage=U61_CLIENTPROTOCOL_CLOSED;
	  U61_LOG_DEBUG("stage=CLOSED");
	  connection->close();
	  
	  /*
	   * We kill all the remote players and let the game keep going locally
	   */
	  for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
	    {
	      p=U61_Global::game.find_player_by_id(i);
	      if (p!=NULL && p->is_available() && !p->is_local())
		{
		  evt.time=0;
		  evt.author=evt.target=i;
		  
		  evt.code=U61_EVENT_READY;
		  dispatcher->put_out(evt);   
		  
		  evt.code=U61_EVENT_KILL;
		  dispatcher->put_out(evt);   
	      
		  p=U61_Global::game.find_player_by_id(i);
		  if (p!=NULL)
		    {
		      p->kill();
		    }
		}
	    }
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to send to the server
 */
int U61_ClientProtocol::send_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_CLIENTPROTOCOL_CONNECTED_TO_SERVER:
      U61_Menu::display_info("Sending program id...");	      
      if ((result=send_program(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_PROGRAM_SENT;
	  U61_LOG_DEBUG("stage=PROGRAM_SENT");
	}
      break;
    case U61_PROTOCOL_PROGRAM_RECEIVED:
      U61_Menu::display_info("Sending version id...");	      
      if ((result=send_version(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_SENT;
	  U61_LOG_DEBUG("stage=VERSION_SENT");
	}
      break;
    case U61_PROTOCOL_VERSION_RECEIVED:
      U61_Menu::display_info("Sending password...");	      
      if ((result=send_password())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_PASSWORD_SENT;
	  U61_LOG_DEBUG("stage=PASSWORD_SENT");
	}
      break;
    case U61_CLIENTPROTOCOL_TIME_RECEIVED:
      U61_Menu::display_info("Sending ready signal...");	      
      if ((result=send_ready(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_SENT;
	  U61_LOG_DEBUG("stage=READY_SENT");
	}
      break;
    case U61_PROTOCOL_READY_RECEIVED:
      U61_Menu::display_info("");	      
      start_game();
      dispatcher->open(connection);
      
      stage=U61_CLIENTPROTOCOL_PLAYING;
      U61_LOG_DEBUG("stage=PLAYING");
      
      break;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * receives the next packet. sometimes there's not really a packet send but
 * the point is to identify the sequence of messages to receive from the
 * server
 */
int U61_ClientProtocol::recv_next_packet()
{
  int result=U61_PROTOCOL_LATER;

  switch(stage)
    {
    case U61_PROTOCOL_BEGIN:
      U61_Menu::display_info("Connecting...");	      
      if ((result=connect_to_server())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_CONNECTED_TO_SERVER;
	  U61_LOG_DEBUG("stage=CONNECTED_TO_SERVER");
	}
      break;
    case U61_PROTOCOL_PROGRAM_SENT:
      U61_Menu::display_info("Receiving program id...");	      
      if ((result=recv_program(connection))==U61_PROTOCOL_DONE)
	{  
	  stage=U61_PROTOCOL_PROGRAM_RECEIVED;
	  U61_LOG_DEBUG("stage=PROGRAM_RECEIVED");
	}
      break;
    case U61_PROTOCOL_VERSION_SENT:
      U61_Menu::display_info("Receiving version id...");	      
      if ((result=recv_version(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_VERSION_RECEIVED;
	  U61_LOG_DEBUG("stage=VERSION_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_PASSWORD_SENT:
      U61_Menu::display_info("Checking password...");	      
      if ((result=recv_password_ok())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_PASSWORD_OK_RECEIVED;
	  U61_LOG_DEBUG("stage=PASSWORD_OK_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_PASSWORD_OK_RECEIVED:
      U61_Menu::display_info("Receiving script name...");	      
      if ((result=recv_script_name())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_SCRIPT_NAME_RECEIVED;
	  U61_LOG_DEBUG("stage=SCRIPT_NAME_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_SCRIPT_NAME_RECEIVED:
      U61_Menu::display_info("Receiving script code...");	      
      if ((result=recv_script())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_SCRIPT_RECEIVED;
	  U61_LOG_DEBUG("stage=SCRIPT_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_SCRIPT_RECEIVED:
      U61_Menu::display_info("Receiving id0...");	      
      if ((result=recv_id0())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_ID0_RECEIVED;
	  U61_LOG_DEBUG("stage=ID0_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_ID0_RECEIVED:
      U61_Menu::display_info("Receiving id1...");	      
      if ((result=recv_id1())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_ID1_RECEIVED;
	  U61_LOG_DEBUG("stage=ID1_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_ID1_RECEIVED:
      U61_Menu::display_info("Receiving init speed...");	      
      if ((result=recv_initial_speed())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_INITIAL_SPEED_RECEIVED;
	  U61_LOG_DEBUG("stage=INITIAL_SPEED_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_INITIAL_SPEED_RECEIVED:
      U61_Menu::display_info("Receiving init accel...");	      
      if ((result=recv_acceleration())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_ACCELERATION_RECEIVED;
	  U61_LOG_DEBUG("stage=ACCELERATION_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_ACCELERATION_RECEIVED:
      U61_Menu::display_info("Receiving curse delay...");	      
      if ((result=recv_curse_delay())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_CURSE_DELAY_RECEIVED;
	  U61_LOG_DEBUG("stage=CURSE_DELAY_RECEIVED");
	}
      break;
    case U61_CLIENTPROTOCOL_CURSE_DELAY_RECEIVED:
      U61_Menu::display_info("Receiving time...");	      
      if ((result=recv_time())==U61_PROTOCOL_DONE)
	{
	  stage=U61_CLIENTPROTOCOL_TIME_RECEIVED;
	  U61_LOG_DEBUG("stage=TIME_RECEIVED");
	}
      break;
    case U61_PROTOCOL_READY_SENT:
      U61_Menu::display_info("Waiting ready signal...");	      
      if ((result=recv_ready(connection))==U61_PROTOCOL_DONE)
	{
	  stage=U61_PROTOCOL_READY_RECEIVED;
	  U61_LOG_DEBUG("stage=READY_RECEIVED");
	}
      break;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * tries to find out if a game has been found on the server
 * and obtain a connection from this server
 */
int U61_ClientProtocol::connect_to_server()
{
  int result=U61_PROTOCOL_LATER;

  connect_mutex.enter();

  if (!connection)
    {
      if (connect_error)
	{
	  U61_MACRO_STRCPY(diagnostic,"Unable to connect on server");
	  result=U61_PROTOCOL_FAILED;
	}
      else
	{
	  /*
	   * connection is NULL, we're not connected yet...
	   */
	  if (!connecting)
	    {
	      stop_connect_thread(true);

	      try 
		{
		  connecting = true;
		  connect_thread=CL_Thread::create(&connect_to_server_callback, (void *) this);
		  if (connect_thread)
		    {
		      connect_thread->start();
		      U61_LOG_DEBUG("U61_ClientProtocol connect thread started");
		    }
		  else
		    {
		      U61_LOG_DEBUG("Could not create CL_Thread object");
		      U61_MACRO_STRCPY(diagnostic,"Could not create connect thread");
		      stop_connect_thread(true);
		      result=U61_PROTOCOL_FAILED;
		    }
		}
	      catch (CL_Error e)
		{
		  U61_LOG_DEBUG("Could not launch U61_ClientProtocol connect thread");
		  U61_MACRO_STRCPY(diagnostic,"Could not start connect thread");
		  stop_connect_thread(true);
		  result=U61_PROTOCOL_FAILED;
		}
	    }
	}
    }
  else
    {
      /*
       * connection is not NULL, we consider we're connected
       */
      stop_connect_thread(false);
      result=U61_PROTOCOL_DONE;
    }    

  connect_mutex.leave();

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Callnack called by connect_to_server in a separate thread
 */
int U61_ClientProtocol::connect_to_server_callback(void *data)
{
  int result=0;
  U61_ClientProtocol *client_protocol;
  U61_BufferedSocket *sock;
  U61_Connection *connection=NULL;

  U61_LOG_DEBUG("Client connection thread started");

  client_protocol=(U61_ClientProtocol *) data;

  U61_LOG_DEBUG("Connecting on \""<<client_protocol->server_id<<":"<<client_protocol->port<<"\".");

  sock=new U61_BufferedSocket();
  if (sock) 
    {
      if (sock->connect(client_protocol->server_id,client_protocol->port))
	{
	  result = 1;
	  /*
	   * The connection object will handle the destruction of
	   * the socket objet
	   */
	  connection=new U61_Connection(sock);
	}
      else
	{
	  delete sock;
	}
    }

  if (connection != NULL)
    {
      connection->authorize_all();
    }

  client_protocol->connect_mutex.enter();

  client_protocol->connect_error = (connection==NULL);
  client_protocol->connection = connection;
  client_protocol->connecting = false;

  client_protocol->connect_mutex.leave();

  U61_LOG_DEBUG("Client connection thread finished");

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends the password
 */
int U61_ClientProtocol::send_password()
{
  int result=U61_PROTOCOL_LATER;
  U61_Checksum checksum;
  char *ascii_sig;

  if (connection!=NULL)
    {
      if (U61_Global::config.use_password)
	{
	  checksum.calc(U61_Global::config.password,
			strlen(U61_Global::config.password));
	  ascii_sig=checksum.get_ascii_sig();
	}
      else
	{
	  ascii_sig="";
	}

      if (connection->send(ascii_sig, strlen(ascii_sig)))
	{
	  /*
	   * We set password error to true, so from now
	   * any error will be interpreted as a bad
	   * password. This is necessary for the server does
	   * not answer "password is wrong" but simply closes
	   * the connection...
	   */
	  password_error=true;

	  result=U61_PROTOCOL_DONE;
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }
 
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the answer "password is correct".
 */
int U61_ClientProtocol::recv_password_ok()
{
  int result=U61_PROTOCOL_LATER;
  char password_ok[U61_CONST_STRING_SIZE];

  if (connection!=NULL) 
    {
      if (connection->peek_str())
	{
	  if (connection->recv(password_ok,sizeof(password_ok)-1))
	    {
	      if (strcmp(password_ok,U61_PROTOCOL_OK_MESSAGE)==0)
		{
		  password_error=false;
		  result=U61_PROTOCOL_DONE;		  
		  U61_LOG_DEBUG("Password is OK");
		}
	      else
		{
		  result=U61_PROTOCOL_FAILED;		  
		}
	    }
	}      
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the script name, this value will be used to create
 * a filename for the server remote script on the client
 */
int U61_ClientProtocol::recv_script_name()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_str())
	{
	  if (connection->recv(U61_Global::remote_script.name_buffer,
			       sizeof(U61_Global::remote_script.name_buffer)-1))
	    {
	      U61_Platform::strip_remote_script_path
		(U61_Global::remote_script.name_buffer);
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("script_name=\""<<U61_Global::remote_script.name_buffer<<"\"");
	    }
	}      
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the script itself, this can be a long packet,
 * I have no idea wether ClanLib handles this correctly yet (I mean that
 * I fear this transmission could "block" the game for a while)
 */
int U61_ClientProtocol::recv_script()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_str())
	{
	  if (connection->recv(U61_Global::remote_script.code_buffer,
			       sizeof(U61_Global::remote_script.code_buffer)-1))
	    {
	      result=U61_PROTOCOL_DONE;
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the ID that will be given to the first local player
 */
int U61_ClientProtocol::recv_id0()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_int())
	{
	  if (connection->recv(&(id[0])))
	    {
	      connection->authorize_player_id(id[0],false);
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("id0="<<id[0]);
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the ID that will be given to the second local player
 */
int U61_ClientProtocol::recv_id1()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_int())
	{
	  if (connection->recv(&(id[1])))
	    {
	      connection->authorize_player_id(id[1],false);
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("id1="<<id[1]);
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the initial speed that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_initial_speed()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_int())
	{
	  if (connection->recv(&initial_speed))
	    {
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("initial_speed="<<initial_speed);
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the acceleration that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_acceleration()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_int())
	{
	  if (connection->recv(&acceleration))
	    {
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("acceleration="<<acceleration);
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the curse delay that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_curse_delay()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_int())
	{
	  if (connection->recv(&curse_delay))
	    {
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("curse_delay="<<curse_delay);
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Receives the time that will be used to initialize the local maps
 */
int U61_ClientProtocol::recv_time()
{
  int result=U61_PROTOCOL_LATER;

  if (connection!=NULL) 
    {
      if (connection->peek_int())
	{
	  if (connection->recv(&time))
	    {
	      result=U61_PROTOCOL_DONE;
	      U61_LOG_DEBUG("time="<<time);
	    }
	}
    }
  else
    {
      result=U61_PROTOCOL_FAILED;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Starts the game (at last!)
 */
void U61_ClientProtocol::start_game()
{
  stop_connect_thread(false);

  U61_Global::game.do_network_join(U61_Global::remote_script.code_buffer,
				   id[0],id[1],
				   initial_speed,acceleration,curse_delay,
				   time);
}

/*--------------------------------------------------------------------------*/
/*
 * Stops the connection thread
 */
void U61_ClientProtocol::stop_connect_thread(bool close_connection)
{
  connect_mutex.enter();

  if (connect_thread!=NULL)
    {
      U61_LOG_DEBUG("Stopping client connection thread");

      //#ifdef U61_DEF_WIN32
      /*
       * Threads suck on Win98, so we disable the "force terminate"
       * feature on this platform and wait for the thread to
       * terminate alone instead
       */
      //      connect_thread->wait();
      //#else
      connect_thread->terminate();
      //#endif

      U61_LOG_DEBUG("Client connection thread stopped");

      delete connect_thread;
      connect_thread = NULL;
    }

  if (close_connection)
    {
      if (connection!=NULL)
	{
	  delete connection;
	  connection=NULL;
	}
    }
      
  connecting = false;
  connect_error = false;

  connect_mutex.leave();
}


