/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   connection.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the connection object is a low-level wrapper over
 *              ClanLib's network API, it's the equivalent of a socket
 */


/*---------------------------------------------------------------------------
  includes
  ---------------------------------------------------------------------------*/

#include "connection.h"
#include "log.h"
#include "global.h"
#include "serial.h"

/*---------------------------------------------------------------------------
  constants
  ---------------------------------------------------------------------------*/

#define U61_CONNECTION_MAX_STRING_SIZE U61_CONST_SCRIPT_SIZE

/*---------------------------------------------------------------------------
  functions
  ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a connection on a specific channel
 */
U61_Connection::U61_Connection(U61_BufferedSocket *s)
{
  opened=true;
  sock=s;
  authorize_none();
}

/*--------------------------------------------------------------------------*/
/*
 * deletes a connection
 */
U61_Connection::~U61_Connection()
{
  close();

  if (sock)
    {
      delete sock;
      sock=NULL;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sends a packet on the connection.
 * returns false if the sending operation failed
 */
bool U61_Connection::send(U61_Packet *p)
{
  int size;
  bool result=false;

  if (opened)
    {
      size=p->size;
      if (size>U61_PACKET_MAX_DATA)
	{
	  U61_LOG_WARNING("Sent packet size is "<<size<<" but max is "<<U61_PACKET_MAX_DATA<<".");
	  size=U61_PACKET_MAX_DATA;
	}

      /*
       * We send the size of the packet
       */
      if (send(p->size))
	{
	  /*
	   * And now we send the data
	   */
	  if (sock->push_buf(p->data,size))
	    {
	      U61_LOG_DEBUG("Sent packet size="<<p->size);
	      result = true;
	    }
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * sends a string on the connection.
 * returns false if the sending operation failed
 * The string will be truncated at the "max" length if it's too long
 */
bool U61_Connection::send(char *str,int max)
{
  int size;
  bool result=false;

  if (opened)
    {
      if (max>U61_CONNECTION_MAX_STRING_SIZE)
	{
	  max=U61_CONNECTION_MAX_STRING_SIZE;
	}

      size=strlen(str);
      if (size>max)
	{
	  U61_LOG_WARNING("Sent string length is "<<size<<" but max is "<<max<<".");
	  size=max;
	}

      /*
       * We send the size of the string
       */
      if (send(size))
	{
	  /*
	   * And now we send the string
	   */
	  if (sock->push_buf(str,size))
	    {
	      U61_LOG_DEBUG("Sent string length="<<size);
	      result = true;
	    }
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * sends an int on the connection.
 * returns false if the sending operation failed
 */
bool U61_Connection::send(int i)
{
  bool result = false;
  unsigned char buffer[U61_SERIAL_INT_SERIALIZED_SIZE];
  unsigned char *buf=buffer;

  if (opened)
    {
      U61_Serial::serialize_int(i,&buf,buf+U61_SERIAL_INT_SERIALIZED_SIZE);

      /*
       * We send the data
       */
      if (sock->push_buf((char *) buffer,U61_SERIAL_INT_SERIALIZED_SIZE))
	{
	  U61_LOG_DEBUG("Sent integer value="<<i);
	  result = true;
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * sends an event on the connection.
 * returns false if the sending operation failed
 */
bool U61_Connection::send(U61_Event evt)
{
  bool result = false;
  unsigned char buffer[U61_EVENT_SERIALIZED_SIZE]; 
  unsigned char *buf=buffer;

  if (opened)
    {
      U61_LOG_DEBUG("Sending event "<<evt<<" on the network");

      evt.serialize(&buf,buf+U61_EVENT_SERIALIZED_SIZE);

      /*
       * We send the data
       */
      if (sock->push_buf((char *) buffer,U61_EVENT_SERIALIZED_SIZE))
	{
	  U61_LOG_DEBUG("Sent event "<<evt);
	  result = true;
	}
    }

  return result;
}
   
/*--------------------------------------------------------------------------*/
/*
 * returns the next packet available on this connection
 */
bool U61_Connection::recv(U61_Packet *p)
{
  int size;
  bool result=false;

  if (opened)
    {
      U61_LOG_DEBUG("receiving packet");

      if (recv(&size))
	{
	  if (size>U61_PACKET_MAX_DATA)
	    {
	      U61_LOG_WARNING("Received packet size is "<<size<<" but max is "<<U61_PACKET_MAX_DATA<<".");
	      size=U61_PACKET_MAX_DATA;
	    }

	  if (sock->pop_buf(p->data,size,false))
	    {
	      p->size = size;
	      U61_LOG_DEBUG("Received packet size="<<p->size);
	      result = true;
	    }
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next string available on this connection
 * The string is written in a buffer, which should be large enough
 * to receive all the data... Beware, the available size must be
 * of "max+1" since a trailing 0 must be added
 * However, the data will be truncated at the size given be the "max" arg.
 */
bool U61_Connection::recv(char *str,int max)
{
  int size;
  bool result=false;

  if (opened)
    {
      U61_LOG_DEBUG("receiving string");

      if (max>U61_CONNECTION_MAX_STRING_SIZE)
	{
	  max=U61_CONNECTION_MAX_STRING_SIZE;
	}

      if (recv(&size))
	{
	  if (size>max)
	    {
	      U61_LOG_DEBUG("Received string length is "<<size<<" but max is "<<max<<".");
	      size=max;
	    }

	  if (sock->pop_buf(str,size,false))
	    {
	      str[size] = '\0';
	      U61_LOG_DEBUG("Received string length="<<size);
	      result = true;
	    }
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next int available on this connection
 */
bool U61_Connection::recv(int *i)
{
  bool result = false;
  unsigned char buffer[U61_SERIAL_INT_SERIALIZED_SIZE];
  unsigned char *buf=buffer;

  if (opened)
    {
      U61_LOG_DEBUG("receiving int");

      if (sock->pop_buf((char *) buffer,U61_SERIAL_INT_SERIALIZED_SIZE,false))
	{
	  U61_Serial::unserialize_int(i,&buf,buf+U61_SERIAL_INT_SERIALIZED_SIZE);

	  U61_LOG_DEBUG("Received integer value="<<*i);

	  result = true;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next event available on this connection
 */
bool U61_Connection::recv(U61_Event *evt)
{
  bool result = false;
  unsigned char buffer[U61_EVENT_SERIALIZED_SIZE];
  unsigned char *buf=buffer;

  if (opened)
    {
      U61_LOG_DEBUG("receiving event");

      if (sock->pop_buf((char *) buffer,U61_EVENT_SERIALIZED_SIZE,false))
	{
	  evt->unserialize(&buf,buf+U61_EVENT_SERIALIZED_SIZE);

	  U61_LOG_DEBUG("Received event "<<evt);

	  result = true;
	}
    }


  /*
   * We check if this player is authorized to send events
   */
  if (result)
    {
      if ((authorized_player_id[evt->author] &&
	   (evt->author==evt->target || 
	    evt->code==U61_EVENT_REQUEST_CURSE)) ||
	  evt->code==U61_EVENT_READY_ALL)	 
	{
	  U61_LOG_DEBUG("event is authorized");
	}
      else
	{
	  U61_LOG_WARNING("Received an event with an incorrect player id.");
	  U61_LOG_DEBUG(*evt);
	  result = false;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if a connection is ready to receive a packet.
 */
bool U61_Connection::peek_packet()
{
  bool result = false;
  unsigned char buffer[U61_SERIAL_INT_SERIALIZED_SIZE];
  unsigned char *buf=buffer;
  int size;
  int input_size;

  if (opened)
    {
      U61_LOG_DEBUG("peeking packet");

      if (peek_int())
	{
	  if (sock->pop_buf((char *) buffer,U61_SERIAL_INT_SERIALIZED_SIZE,true))
	    {
	      U61_Serial::unserialize_int(&size,&buf,buf+U61_SERIAL_INT_SERIALIZED_SIZE);
	      
	      input_size=sock->get_input_size();

	      if (size>U61_CONNECTION_MAX_STRING_SIZE)
		{
		  U61_LOG_DEBUG("Packet is too big (size="<<size<<", input_size="<<input_size<<").");
		}

	      if (input_size >= size+U61_SERIAL_INT_SERIALIZED_SIZE)
		{
		  U61_LOG_DEBUG("packet OK");
		  result = true;
		}
	    }
	}
    }
  
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if a connection is ready to receive a string.
 */
bool U61_Connection::peek_str()
{
  bool result=false;

  if (opened)
    {
      U61_LOG_DEBUG("peeking str");

      result=peek_packet();
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if a connection is ready to receive an integer.
 */
bool U61_Connection::peek_int()
{
  bool result=false;

  if (opened)
    {
      U61_LOG_DEBUG("peeking int");

      if (sock->get_input_size()>=U61_SERIAL_INT_SERIALIZED_SIZE)
	{
	  result = true;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if a connection is ready to receive an event.
 */
bool U61_Connection::peek_event()
{
  bool result=false;

  if (opened)
    {
      U61_LOG_DEBUG("peeking event");

      if (sock->get_input_size()>=U61_EVENT_SERIALIZED_SIZE)
	{
	  result = true;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Authorizes all player ids to send events on this connection
 */
void U61_Connection::authorize_all()
{
  int i;

  for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
    {
      authorized_player_id[i] = true;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Forbids all player ids to send events on this connection
 */
void U61_Connection::authorize_none()
{
  int i;

  for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
    {
      authorized_player_id[i] = false;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Authorizes incoming events for a given player_id.
 */
void U61_Connection::authorize_player_id(int player_id, bool state)
{
  if (player_id>=0 && player_id<U61_DISPATCHER_MAX_PLAYER_ID)
    {
      authorized_player_id[player_id] = state;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns authorization concerning a given player.
 */
bool U61_Connection::is_authorized(int player_id)
{
  return authorized_player_id[player_id];
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the connection is alive
 */
bool U61_Connection::is_alive()
{
  bool alive=false;

  if (sock)
    {
      sock->flush_if_needed();
      alive=sock->is_alive();
    }
  
  return alive;
}

/*--------------------------------------------------------------------------*/
/*
 * closes the connection
 */
void U61_Connection::close()
{
  if (sock)
    {
      sock->close();
    }
  opened=false;
}





































