/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   data.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the data object holds all the data of the game
 *              this includes the graphics, the sounds and anything
 *              which is included in the datafile
 *              it provides an easy and standard way for all other
 *              classes to access the game resources
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <ClanLib/mikmod.h>
#include <string.h>

#include "data.h"
#include "const.h"
#include "log.h"
#include "platform.h"
#include "version.h"
#include "utils.h"
#include "macro.h"

/*---------------------------------------------------------------------------
 constants
 ---------------------------------------------------------------------------*/

#define U61_DATA_ID_PROGRAM "id/program"
#define U61_DATA_ID_VERSION "id/version"

#define U61_DATA_ID_PROPERTY_SCREEN_W "property/screen_w"
#define U61_DATA_ID_PROPERTY_SCREEN_H "property/screen_h"

#define U61_DATA_ID_PROPERTY_PLAYER_W_BIG "property/player_w_big"
#define U61_DATA_ID_PROPERTY_PLAYER_H_BIG "property/player_h_big"

#define U61_DATA_ID_PROPERTY_OFFSET_MAP_X_BIG "property/offset_map_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_MAP_Y_BIG "property/offset_map_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_INFO_X_BIG "property/offset_info_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_INFO_Y_BIG "property/offset_info_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_SQUARES_X_BIG "property/offset_squares_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_SQUARES_Y_BIG "property/offset_squares_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_NAME_X_BIG "property/offset_name_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_NAME_Y_BIG "property/offset_name_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_SCORE_X_BIG "property/offset_score_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_SCORE_Y_BIG "property/offset_score_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_TARGET_X_BIG "property/offset_target_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_TARGET_Y_BIG "property/offset_target_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_X_BIG "property/offset_preview_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_Y_BIG "property/offset_preview_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_CURSES_X_BIG "property/offset_curses_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_CURSES_Y_BIG "property/offset_curses_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_X_BIG "property/offset_antidotes_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_Y_BIG "property/offset_antidotes_y_big"
#define U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_X_BIG "property/offset_next_curse_x_big"
#define U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_Y_BIG "property/offset_next_curse_y_big"

#define U61_DATA_ID_PROPERTY_PLAYER_W_SMALL "property/player_w_small"
#define U61_DATA_ID_PROPERTY_PLAYER_H_SMALL "property/player_h_small"

#define U61_DATA_ID_PROPERTY_OFFSET_MAP_X_SMALL "property/offset_map_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_MAP_Y_SMALL "property/offset_map_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_INFO_X_SMALL "property/offset_info_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_INFO_Y_SMALL "property/offset_info_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_SQUARES_X_SMALL "property/offset_squares_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_SQUARES_Y_SMALL "property/offset_squares_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_NAME_X_SMALL "property/offset_name_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_NAME_Y_SMALL "property/offset_name_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_SCORE_X_SMALL "property/offset_score_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_SCORE_Y_SMALL "property/offset_score_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_TARGET_X_SMALL "property/offset_target_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_TARGET_Y_SMALL "property/offset_target_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_X_SMALL "property/offset_preview_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_Y_SMALL "property/offset_preview_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_CURSES_X_SMALL "property/offset_curses_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_CURSES_Y_SMALL "property/offset_curses_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_X_SMALL "property/offset_antidotes_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_Y_SMALL "property/offset_antidotes_y_small"
#define U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_X_SMALL "property/offset_next_curse_x_small"
#define U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_Y_SMALL "property/offset_next_curse_y_small"

#define U61_DATA_ID_FONT_MENU "font/menu"
#define U61_DATA_ID_FONT_INFO_BIG "font/info_big"
#define U61_DATA_ID_FONT_INFO_SMALL "font/info_small"
#define U61_DATA_ID_FONT_SYMBOL_BIG "font/symbol_big"
#define U61_DATA_ID_FONT_SYMBOL_SMALL "font/symbol_small"

#define U61_DATA_ID_SOUND_GAME_START "sound/game_start"
#define U61_DATA_ID_SOUND_MENU_MOVE "sound/menu_move"
#define U61_DATA_ID_SOUND_MENU_VALIDATE "sound/menu_validate"
#define U61_DATA_ID_SOUND_BLOCK_TOUCH "sound/block_touch"
#define U61_DATA_ID_SOUND_BLOCK_PATTERN "sound/block_pattern"
#define U61_DATA_ID_SOUND_PLAYER_START "sound/player_start"
#define U61_DATA_ID_SOUND_CURSE_RECEIVE "sound/curse_receive"
#define U61_DATA_ID_SOUND_PLAYER_LOOSE "sound/player_loose"
 
#define U61_DATA_ID_MUSIC_MOD "music/mod_"
#define U61_DATA_ID_MUSIC_NUMBER "music/number"

#define U61_DATA_ID_SQUARE_BIG "square/big_n"
#define U61_DATA_ID_SQUARE_SMALL "square/small_n"
#define U61_DATA_ID_SQUARE_CURSE "curse"
#define U61_DATA_ID_SQUARE_ANTICIPATION "anticipation"
#define U61_DATA_ID_SQUARE_EXPLOSION "explosion"
#define U61_DATA_ID_SQUARE_FILLER "filler"

#define U61_DATA_ID_MAP_BIG "map/big_"
#define U61_DATA_ID_MAP_SMALL "map/small_"
#define U61_DATA_ID_MAP_NUMBER "map/number"

#define U61_DATA_ID_BACK_MAIN "back/main"
#define U61_DATA_ID_SHADE "back/shade"

#define U61_DATA_ID_INFO_BIG "info/big"
#define U61_DATA_ID_INFO_SMALL "info/small"

#define U61_DATA_ID_MOUSE_CURSOR "mouse/cursor"


#define U61_DATA_MOUSE_CURSOR_NB_FRAMES         16
#define U61_DATA_MOUSE_CURSOR_FRAME_DELAY       30

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates an uninitialized data object
 */
U61_Data::U61_Data()
{
  int i,j;

  /*
   * Resetting public stuff
   */
  screen_w = 0;
  screen_h = 0;

  nb_map = 0;
  nb_music = 0;

  for (i=0;i<U61_DATA_NB_SIZE;++i)
    {
      square_w[i] = 0;
      square_h[i] = 0;
      map_w[i] = 0;
      map_h[i] = 0;
      info_w[i] = 0;
      info_h[i] = 0;
      player_w[i] = 0;
      player_h[i] = 0;
      
      offset_map_x[i] = 0;
      offset_map_y[i] = 0;
      offset_info_x[i] = 0;
      offset_info_y[i] = 0;
      offset_squares_x[i] = 0;
      offset_squares_y[i] = 0;
      offset_name_x[i] = 0;
      offset_name_y[i] = 0;
      offset_score_x[i] = 0;
      offset_score_y[i] = 0;
      offset_target_x[i] = 0;
      offset_target_y[i] = 0;
      offset_preview_x[i] = 0;
      offset_preview_y[i] = 0;
      offset_curses_x[i] = 0;
      offset_curses_y[i] = 0;
      offset_antidotes_x[i] = 0;
      offset_antidotes_y[i] = 0;
      offset_next_curse_x[i] = 0;
      offset_next_curse_y[i] = 0;

      font_info[i] = NULL;
      font_symbol[i] = NULL;

      for (j=0;j<U61_DATA_NB_SQUARE_TYPE;++j)
	{
	  square[j][i] = NULL;
	}

      curse[i] = NULL;
      anticipation[i] = NULL;
      explosion[i] = NULL;
      filler[i] = NULL;

      for (j=0;j<U61_DATA_NB_MAP_MAX;++j)
	{
	  map[j][i] = NULL;
	}

      info[i] = NULL;
    }

  font_menu = NULL;

  sound_game_start = NULL;
  sound_menu_move = NULL;
  sound_menu_validate = NULL;
  sound_block_touch = NULL;
  sound_block_pattern = NULL;
  sound_player_start = NULL;
  sound_curse_receive = NULL;
  sound_player_loose = NULL;

  back = NULL;
  shade = NULL;

  mouse_cursor=NULL;

  /*
   * Protected stuff
   */

  manager=NULL;
  packed_data=true;
  
  /*
   * We keep the initial working directory in memory, to
   * chdir back to it later, if needed...
   */
  U61_MACRO_STRCPY(initial_dir,U61_Platform::getcwd());
  U61_MACRO_STRCPY(data_dir,U61_Platform::getcwd());
  U61_MACRO_STRCPY(data_file,"");
}

/*--------------------------------------------------------------------------*/
/*
 * destroys a data object
 */
U61_Data::~U61_Data()
{
}

/*--------------------------------------------------------------------------*/
/*
 * unloads the content of the last loaded datafile
 */
void U61_Data::unload()
{
  /*
   * now we unload everything the manager loaded
   */
  U61_LOG_DEBUG("unload");
  if (manager!=NULL)
    {
      /*
       * I do not really understand things very well, but if I uncomment
       * the following code, I get a bunch of junk messages, complaining
       * about not unloading resources. Asking for the load count before
       * unload_all_resources returns 2, and if you call unload on the
       * resource it returns 1 then -1. And never 0. I just don't get it.
       * However the game seems to cleanup things quite correctly when
       * it exits (at least under Linux) so I don't care much.
       */

      //manager->unload_all_resources();

      //delete manager;
    }

  chdir_to_initial_dir();

  manager=NULL;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the data from a disk .dat file
 * returns true if load is successful, false otherwise
 * this is the first part of loading, here we only load things that
 * do not require to have clanlib graphical mode set for instance
 */
bool U61_Data::load1(char *name)
{
  bool success=true;
  int len;
  
  packed_data=true;

  /*
   * we first cehck that the file exists
   */
  if (!U61_Utils::file_exists(name))
    {
      success=false;
      
      U61_LOG_ERROR("Unable to load data from \""<<name<<"\" (file does not exist).");
    }
  else
    {
      /*
       * creation of the resource manager, which will be
       * used to load all the sections
       */
      U61_LOG_MESSAGE("Loading data from \""<<name<<"\".");
      /*
       * We use a global try to catch errors from ClanLib's
       * datafile reader. This try should only catch global
       * errors such as being unable to find the datafile
       */
      try
	{
	  /*
	   * We check if the datafile is packed (.dat) or directory based (.scr)
	   */
	  len=strlen(name);
	  if (len>4)
	    {
	      if (strcmp(name+len-4,".scr")==0 || strcmp(name+len-4,".SCR")==0)
		{
		  packed_data=false;
		}
	    }

	  /*
	   * Now we store in dir the directory where the datafile is.
	   * Later we will chdir to this directory, otherwise the
	   * ClanLib datafile reader whill pathetically fail since
	   * it opens "." as the main parent directory.
	   */
	  U61_MACRO_STRCPY(data_dir,name);
	  U61_Platform::keep_dir_only(data_dir);
	  /*
	   * We store the name of the datafile, without the path
	   */
	  U61_MACRO_STRCPY(data_file,name);
	  U61_Platform::keep_file_only(data_file);

	  chdir_to_data_dir();
	  manager=new CL_ResourceManager(data_file,packed_data);

	  if (manager!=NULL)
	    {
	      /*
	       * We use a another try to catch errors from ClanLib's
	       * datafile reader.
	       */
	      try
		{
		  if (success)
		    {
		      U61_LOG_DEBUG("Checking id");
		      check_id();

		      U61_LOG_DEBUG("Loading properties");
		      success=success && load_properties();
		    }
		  if (!success)
		    {
		      U61_LOG_ERROR("Unknown error reading datafile, step 1");
		    }
		}
	      catch(CL_Error e)
		{
		  success=false;
		  U61_LOG_ERROR("Error reading datafile, step 1: "<<e.message.c_str());
		}
	    }
	  else
	    {
	      U61_LOG_ERROR("Unknown error creating resource manager, step 1"); 
	      success=false;
	    }
	}
      catch(CL_Error e)
	{
	  success=false;
	  U61_LOG_ERROR("Error creating resource manager, step 1: "<<e.message.c_str());
	}
    }

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * this function performs the second part of loading of data
 * here, most of the time, the graphic mode has to be set
 */
bool U61_Data::load2()
{
  bool success=true;
    
  if (manager!=NULL)
    {
      /*
       * We use a global try to catch errors from ClanLib's
       * datafile reader.
       */
      try
	{
	  /*
	   * the following code loads all the different sections
	   * if one section fails, the load process is aborted
	   */
	  if (success)
	    {
	      U61_LOG_DEBUG("Loading fonts");
	      success=success && load_fonts();
	    }
        
	  if (success)
	    {
	      U61_LOG_DEBUG("Loading sounds");
	      success=success && load_sounds();
	    }
        
	  if (success)
	    {
	      U61_LOG_DEBUG("Loading musics");
	      success=success && load_musics();
	    }

	  if (success)
	    {
	      U61_LOG_DEBUG("Loading square graphics");
	      success=success && load_squares();
	    }

	  if (success)
	    {
	      U61_LOG_DEBUG("Loading map graphics");
	      success=success && load_maps();
	    }

	  if (success)
	    {
	      U61_LOG_DEBUG("Loading background graphics");
	      success=success && load_back();
	    }

	  if (success)
	    {
	      U61_LOG_DEBUG("Loading info graphics");
	      success=success && load_info();
	    }

	  if (success)
	    {
	      U61_LOG_DEBUG("Loading mouse graphics");
	      success=success && load_mouse();
	    }

	  if (!success)
	    {
	      U61_LOG_ERROR("Unknown error reading datafile, step 2");
	    }
	  
	  manager->load_all();
	}
      catch(CL_Error e)
	{
	  success=false;
	  U61_LOG_ERROR("Error reading datafile, step 2: "<<e.message.c_str());
	}
    }
  else
    {
      U61_LOG_ERROR("Unable to create resource manager, step 2"); 
      success=false;
    }

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * chdir to the data directory 
 */
void U61_Data::chdir_to_data_dir()
{
  U61_Platform::chdir(data_dir);
}

/*--------------------------------------------------------------------------*/
/*
 * cancels the effects of chdir_to_data_dir
 */
void U61_Data::chdir_to_initial_dir()
{
  U61_Platform::chdir(initial_dir);
}


/*--------------------------------------------------------------------------*/
/*
 * checks that the datafile is a correct U61 datafile
 */
bool U61_Data::check_id()
{
  bool success=true;
  char *program_read;
  char *version_read;
  char version[U61_CONST_STRING_SIZE];

  U61_Version::get_version_text(version);

  program_read=CL_String(U61_DATA_ID_PROGRAM,manager).get_string();
  version_read=CL_String(U61_DATA_ID_VERSION,manager).get_string();

  if (program_read==NULL || strcmp(program_read,U61_CONST_PROGRAM))
    {
      success=false;
      U61_LOG_WARNING("The theme file does not seem to be a valid U61 theme file, you should check that U61 is correctly installed.");
    }
  else
    {
      if (version_read==NULL || strcmp(version_read,version))
	{
	  success=false;
	  if (version_read==NULL)
	    {
	      version_read="NULL";
	    }
	  
	  U61_LOG_WARNING("The data file version is \""<<version_read<<"\" but the game version is \""<<version<<"\", so you get errors when loading it. Maybe you should just get rid of it and/or reinstall U61.");
	}
    }

  return success;    
}

/*--------------------------------------------------------------------------*/
/*
 * loads global properties 
 */
bool U61_Data::load_properties()
{
  bool success=true;

  screen_w=CL_Integer(U61_DATA_ID_PROPERTY_SCREEN_W,manager);
  screen_h=CL_Integer(U61_DATA_ID_PROPERTY_SCREEN_H,manager);

  player_w[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_PLAYER_W_BIG,manager);
  player_h[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_PLAYER_H_BIG,manager);

  offset_map_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_MAP_X_BIG,manager);
  offset_map_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_MAP_Y_BIG,manager);
  offset_info_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_INFO_X_BIG,manager);
  offset_info_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_INFO_Y_BIG,manager);
  offset_squares_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SQUARES_X_BIG,manager);
  offset_squares_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SQUARES_Y_BIG,manager);
  offset_name_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NAME_X_BIG,manager);
  offset_name_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NAME_Y_BIG,manager);
  offset_score_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SCORE_X_BIG,manager);
  offset_score_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SCORE_Y_BIG,manager);
  offset_target_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_TARGET_X_BIG,manager);
  offset_target_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_TARGET_Y_BIG,manager);
  offset_preview_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_X_BIG,manager);
  offset_preview_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_Y_BIG,manager);
  offset_curses_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_CURSES_X_BIG,manager);
  offset_curses_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_CURSES_Y_BIG,manager);
  offset_antidotes_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_X_BIG,manager);
  offset_antidotes_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_Y_BIG,manager);
  offset_next_curse_x[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_X_BIG,manager);
  offset_next_curse_y[U61_DATA_SIZE_BIG]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_Y_BIG,manager);

  player_w[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_PLAYER_W_SMALL,manager);
  player_h[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_PLAYER_H_SMALL,manager);

  offset_map_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_MAP_X_SMALL,manager);
  offset_map_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_MAP_Y_SMALL,manager);
  offset_info_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_INFO_X_SMALL,manager);
  offset_info_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_INFO_Y_SMALL,manager);
  offset_squares_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SQUARES_X_SMALL,manager);
  offset_squares_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SQUARES_Y_SMALL,manager);
  offset_name_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NAME_X_SMALL,manager);
  offset_name_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NAME_Y_SMALL,manager);
  offset_score_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SCORE_X_SMALL,manager);
  offset_score_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_SCORE_Y_SMALL,manager);
  offset_target_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_TARGET_X_SMALL,manager);
  offset_target_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_TARGET_Y_SMALL,manager);
  offset_preview_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_X_SMALL,manager);
  offset_preview_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_PREVIEW_Y_SMALL,manager);
  offset_curses_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_CURSES_X_SMALL,manager);
  offset_curses_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_CURSES_Y_SMALL,manager);
  offset_antidotes_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_X_SMALL,manager);
  offset_antidotes_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_ANTIDOTES_Y_SMALL,manager);
  offset_next_curse_x[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_X_SMALL,manager);
  offset_next_curse_y[U61_DATA_SIZE_SMALL]=
    CL_Integer(U61_DATA_ID_PROPERTY_OFFSET_NEXT_CURSE_Y_SMALL,manager);

  return success;    
}

/*--------------------------------------------------------------------------*/
/*
 * loads the fonts
 */
bool U61_Data::load_fonts()
{
  bool success=true;

  font_menu=CL_Font::load(U61_DATA_ID_FONT_MENU,manager);
  cl_assert(font_menu!=NULL);
   
  font_info[U61_DATA_SIZE_BIG]=CL_Font::load(U61_DATA_ID_FONT_INFO_BIG,manager);
  cl_assert(font_info[U61_DATA_SIZE_BIG]!=NULL);
   
  font_info[U61_DATA_SIZE_SMALL]=CL_Font::load(U61_DATA_ID_FONT_INFO_SMALL,manager);
  cl_assert(font_info[U61_DATA_SIZE_SMALL]!=NULL);
   
  font_symbol[U61_DATA_SIZE_BIG]=CL_Font::load(U61_DATA_ID_FONT_SYMBOL_BIG,manager);
  cl_assert(font_symbol[U61_DATA_SIZE_BIG]!=NULL);
   
  font_symbol[U61_DATA_SIZE_SMALL]=CL_Font::load(U61_DATA_ID_FONT_SYMBOL_SMALL,manager);
  cl_assert(font_symbol[U61_DATA_SIZE_SMALL]!=NULL);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the sounds
 */
bool U61_Data::load_sounds()
{
  bool success=true;

  sound_game_start=CL_SoundBuffer::load(U61_DATA_ID_SOUND_GAME_START,manager);
  cl_assert(sound_game_start!=NULL);

  sound_menu_move=CL_SoundBuffer::load(U61_DATA_ID_SOUND_MENU_MOVE,manager);
  cl_assert(sound_menu_move!=NULL);

  sound_menu_validate=CL_SoundBuffer::load(U61_DATA_ID_SOUND_MENU_VALIDATE,manager);
  cl_assert(sound_menu_validate!=NULL);

  sound_block_touch=CL_SoundBuffer::load(U61_DATA_ID_SOUND_BLOCK_TOUCH,manager);
  cl_assert(sound_block_touch!=NULL);

  sound_block_pattern=CL_SoundBuffer::load(U61_DATA_ID_SOUND_BLOCK_PATTERN,manager);
  cl_assert(sound_block_pattern!=NULL);

  sound_player_start=CL_SoundBuffer::load(U61_DATA_ID_SOUND_PLAYER_START,manager);
  cl_assert(sound_player_start!=NULL);

  sound_curse_receive=CL_SoundBuffer::load(U61_DATA_ID_SOUND_CURSE_RECEIVE,manager);
  cl_assert(sound_curse_receive!=NULL);

  sound_player_loose=CL_SoundBuffer::load(U61_DATA_ID_SOUND_PLAYER_LOOSE,manager);
  cl_assert(sound_player_loose!=NULL);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the musics
 */
bool U61_Data::load_musics()
{
  bool success=true;
  char buffer[U61_CONST_STRING_SIZE];
  int i;

  nb_music=CL_Integer(U61_DATA_ID_MUSIC_NUMBER,manager);
  if (nb_music>=U61_DATA_NB_MUSIC_MAX)
    {
      nb_music=U61_DATA_NB_MUSIC_MAX-1;
    }

  for (i=0;i<nb_music && success;++i)
    { 
      U61_MACRO_SPRINTF2(buffer,"%s%d",U61_DATA_ID_MUSIC_MOD,i);
      music[i]=CL_Streamed_MikModSample::load(buffer,manager); 
      cl_assert(music[i]!=NULL);
    } 
   
  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the squares
 */
bool U61_Data::load_squares()
{
  bool success=true;
  int type;
  char id[U61_CONST_STRING_SIZE];
  //    CL_Surface *temp;

  for (type=0; type<U61_DATA_NB_SQUARE_TYPE; ++type)
    {
      U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_SMALL);
      id[strlen(id)-1]='0'+type;
      square[type][U61_DATA_SIZE_SMALL]=CL_Surface::load(id,manager);
      cl_assert(square[type][U61_DATA_SIZE_SMALL]!=NULL);

      U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_BIG);
      id[strlen(id)-1]='0'+type;
      square[type][U61_DATA_SIZE_BIG]=CL_Surface::load(id,manager);
      cl_assert(square[type][U61_DATA_SIZE_BIG]!=NULL);
    }

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_SMALL);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_CURSE);
  curse[U61_DATA_SIZE_SMALL]=CL_Surface::load(id,manager);
  cl_assert(curse[U61_DATA_SIZE_SMALL]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_BIG);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_CURSE);
  curse[U61_DATA_SIZE_BIG]=CL_Surface::load(id,manager);
  cl_assert(curse[U61_DATA_SIZE_BIG]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_SMALL);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_ANTICIPATION);
  anticipation[U61_DATA_SIZE_SMALL]=CL_Surface::load(id,manager);
  cl_assert(anticipation[U61_DATA_SIZE_SMALL]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_BIG);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_ANTICIPATION);
  anticipation[U61_DATA_SIZE_BIG]=CL_Surface::load(id,manager);
  cl_assert(anticipation[U61_DATA_SIZE_BIG]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_SMALL);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_EXPLOSION);
  explosion[U61_DATA_SIZE_SMALL]=CL_Surface::load(id,manager);
  cl_assert(explosion[U61_DATA_SIZE_SMALL]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_BIG);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_EXPLOSION);
  explosion[U61_DATA_SIZE_BIG]=CL_Surface::load(id,manager);
  cl_assert(explosion[U61_DATA_SIZE_BIG]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_SMALL);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_FILLER);
  filler[U61_DATA_SIZE_SMALL]=CL_Surface::load(id,manager);
  cl_assert(filler[U61_DATA_SIZE_SMALL]!=NULL);

  U61_MACRO_STRCPY(id,U61_DATA_ID_SQUARE_BIG);
  id[strlen(id)-1]=0;
  U61_MACRO_STRCAT(id,U61_DATA_ID_SQUARE_FILLER);
  filler[U61_DATA_SIZE_BIG]=CL_Surface::load(id,manager);
  cl_assert(filler[U61_DATA_SIZE_BIG]!=NULL);

  if (success)
    {
      square_w[U61_DATA_SIZE_SMALL]=square[0][U61_DATA_SIZE_SMALL]->get_width();
      square_h[U61_DATA_SIZE_SMALL]=square[0][U61_DATA_SIZE_SMALL]->get_height();
      square_w[U61_DATA_SIZE_BIG]=square[0][U61_DATA_SIZE_BIG]->get_width();
      square_h[U61_DATA_SIZE_BIG]=square[0][U61_DATA_SIZE_BIG]->get_height();
    }

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the map images
 */
bool U61_Data::load_maps()
{
  bool success=true;
  int i;
  char id[U61_CONST_STRING_SIZE];

  nb_map=CL_Integer(U61_DATA_ID_MAP_NUMBER,manager);
  if (nb_map>=U61_DATA_NB_MAP_MAX)
    {
      nb_map=U61_DATA_NB_MAP_MAX-1;
    }
  
  for (i=0; i<nb_map; ++i)
    {
      U61_MACRO_SPRINTF2(id,"%s%d",U61_DATA_ID_MAP_SMALL,i);
      map[i][U61_DATA_SIZE_SMALL]=CL_Surface::load(id,manager);
      cl_assert(map[i][U61_DATA_SIZE_SMALL]!=NULL);
      map[i][U61_DATA_SIZE_SMALL]->convert_system();

      U61_MACRO_SPRINTF2(id,"%s%d",U61_DATA_ID_MAP_BIG,i);
      map[i][U61_DATA_SIZE_BIG]=CL_Surface::load(id,manager);
      cl_assert(map[i][U61_DATA_SIZE_BIG]!=NULL);
      map[i][U61_DATA_SIZE_BIG]->convert_system();
    }


  if (success)
    {
      map_w[U61_DATA_SIZE_SMALL]=map[0][U61_DATA_SIZE_SMALL]->get_width();
      map_h[U61_DATA_SIZE_SMALL]=map[0][U61_DATA_SIZE_SMALL]->get_height();
      map_w[U61_DATA_SIZE_BIG]=map[0][U61_DATA_SIZE_BIG]->get_width();
      map_h[U61_DATA_SIZE_BIG]=map[0][U61_DATA_SIZE_BIG]->get_height();
    }

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the background image 
 */
bool U61_Data::load_back()
{
  bool success=true;

  back=CL_Surface::load(U61_DATA_ID_BACK_MAIN,manager);
  cl_assert(back!=NULL);
  back->convert_system();

  shade=CL_Surface::load(U61_DATA_ID_SHADE,manager);
  cl_assert(shade!=NULL);

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the info textures (where scores are displayed)
 */
bool U61_Data::load_info()
{
  bool success=true;

  info[U61_DATA_SIZE_BIG]=CL_Surface::load(U61_DATA_ID_INFO_BIG,manager);
  cl_assert(info[U61_DATA_SIZE_BIG]!=NULL);
  info[U61_DATA_SIZE_BIG]->convert_system();

  info[U61_DATA_SIZE_SMALL]=CL_Surface::load(U61_DATA_ID_INFO_SMALL,manager);
  cl_assert(info[U61_DATA_SIZE_SMALL]!=NULL);
  info[U61_DATA_SIZE_SMALL]->convert_system();

  if (success)
    {
      info_w[U61_DATA_SIZE_SMALL]=info[U61_DATA_SIZE_SMALL]->get_width();
      info_h[U61_DATA_SIZE_SMALL]=info[U61_DATA_SIZE_SMALL]->get_height();
      info_w[U61_DATA_SIZE_BIG]=info[U61_DATA_SIZE_BIG]->get_width();
      info_h[U61_DATA_SIZE_BIG]=info[U61_DATA_SIZE_BIG]->get_height();
    }

  return success;
}

/*--------------------------------------------------------------------------*/
/*
 * loads the mouse sprites
 */
bool U61_Data::load_mouse()
{
  bool success=true;
  CL_SurfaceProvider *surface_provider;
  static std::vector<int> frame_delays;
  int i;

  frame_delays.clear();
  for (i=0;i<U61_DATA_MOUSE_CURSOR_NB_FRAMES;++i)
    {
      frame_delays.push_back(U61_DATA_MOUSE_CURSOR_FRAME_DELAY);
    }

  surface_provider=CL_SurfaceProvider::load(U61_DATA_ID_MOUSE_CURSOR,manager);
  cl_assert(surface_provider!=NULL);
  
  mouse_cursor=CL_MouseCursorProvider::create(surface_provider,
					      true,
					      true,
					      &frame_delays);
  cl_assert(mouse_cursor!=NULL);

  return success;
}

