/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   dispatcher.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: the dispatcher is responsible for sending & receiving
 *              messages from the network, and send them to the right person
 *              it's the main "complex" interface with the low level
 *              network API
 */


/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "dispatcher.h"
#include "time.h"
#include "log.h"

/*---------------------------------------------------------------------------
 variants
 ---------------------------------------------------------------------------*/

#define U61_DISPATCHER_MAX_AT_ONCE    5000
#define U61_DISPATCHER_READY_DELAY_SEC   5

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*
 * creates a dispatcher
 */
U61_Dispatcher::U61_Dispatcher()
{
  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * deletes a dispatcher
 */
U61_Dispatcher::~U61_Dispatcher()
{
}

/*--------------------------------------------------------------------------*/
/*
 * Empties all the queues
 */
void U61_Dispatcher::reset()
{
  int i;

  outgoing_queue.clear();
  for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
    {
      items[i].reset();
    }
  ready_all_flag=false;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there's been a message for this player when
 * the dispatcher was processed last time
 */
bool U61_Dispatcher::is_player_busy(int player_id)
{
  return items[player_id].get_busy();
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there are no more incoming events for a given player
 */
bool U61_Dispatcher::empty_in(int player_id)
{
  return items[player_id].empty_in();
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there are no more outgoing messages
 */
bool U61_Dispatcher::empty_out()
{
  return outgoing_queue.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * Puts an event in the outgoing queue
 */
void U61_Dispatcher::put_out(U61_Event event)
{
  U61_LOG_DEBUG("Event "<<event<<" in dispatcher");
  outgoing_queue.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Gets an event for a given player
 */
U61_Event U61_Dispatcher::get_in(int player_id)
{
  return items[player_id].get_in();
}

/*--------------------------------------------------------------------------*/
/*
 * Gets an event from the outgoing queue
 */
U61_Event U61_Dispatcher::get_out()
{
  U61_Event event;

  event=outgoing_queue.front();
  outgoing_queue.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * Puts an event in the incoming queue
 */
void U61_Dispatcher::put_in(U61_Event event, int player_id)
{
  items[player_id].put_in(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Distributes all the events
 */
bool U61_Dispatcher::process()
{
  U61_Event event;
  bool ok=true;
  int count=0;

  while (peek() && count<U61_DISPATCHER_MAX_AT_ONCE)
    {
      if (!recv(&event))
	{
	  ok=false;
	}
      else
	{
	  look(event);

	  if (event.time>U61_Time::for_event())
	    {
	      U61_LOG_DEBUG("Speeding up");
	      U61_Time::speed_up();
	    }
          put_in(event,event.target);
	}
      count++;
    }
 
  if (count>U61_DISPATCHER_MAX_AT_ONCE/2)
    {
      U61_LOG_WARNING("Dispatcher is filled up with events");
    }

  while (!empty_out())
    {
      event=get_out();

      look(event);

      switch (event.code)
	{
	case U61_EVENT_REQUEST_CURSE:
	case U61_EVENT_READY_ALL:
	  /*
	   * REQUEST_CURSE events are copied into the local queue
	   * since the target can be on a the same machine
	   * than the sender. READY_ALL events too are copied to
	   * the local queue, for they concern everybody.
	   */
	  put_in(event,event.target);
	  break;
	}

      if (!send(event))
	{
	  ok=false;
	}
    }

  return ok;
}


/*--------------------------------------------------------------------------*/
/*
 * sends an event to no one 
 */
bool U61_Dispatcher::send(U61_Event evt)
{
  bool ok=true;

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * returns always false by default 
 */
bool U61_Dispatcher::recv(U61_Event *evt)
{
  bool ok=false;

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * returns always false by default
 */
bool U61_Dispatcher::peek()
{
  bool ok=false;

  return ok;
} 

/*--------------------------------------------------------------------------*/
/*
 * Looks a message and does the necessary.
 */
void U61_Dispatcher::look(U61_Event event)
{
  switch (event.code)
    {
    case U61_EVENT_KILL:
      items[event.target].set_busy(false);
      items[event.target].set_ready(false);
      break;
    case U61_EVENT_START_STOP:
    case U61_EVENT_NAME_LETTER:
    case U61_EVENT_CHAT_LETTER:
      items[event.target].set_busy(true);
      break;
    case U61_EVENT_READY:
      check_ready_all(event);
      break;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Checks if all the players are ready, and sends a "READY_ALL" mesage
 * if needed.
 */
void U61_Dispatcher::check_ready_all(U61_Event evt)
{
  int i;

  items[evt.target].set_ready(true);

  if (!ready_all_flag)
    {
      ready_all_flag=true;
      for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
	{
	  U61_LOG_DEBUG("Checking if everybody's ready "<<i<<" / busy="<<items[i].get_busy()<<" / ready="<<items[i].get_ready());
	  if (items[i].get_busy())
	    {
	      if (!items[i].get_ready())
		{
		  ready_all_flag=false;
		}
	    }  
	}
    }

  if (ready_all_flag)
    {
      evt.code=U61_EVENT_READY_ALL;
      for (i=0;i<U61_DISPATCHER_MAX_PLAYER_ID;++i)
	{
	  evt.target=i;
	  put_out(evt);
	}
    }
}

