/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   edit.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: a menu item which allows the user to type text
 *              this is usefull to edit the player's name for instance
 */



/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include <string.h>

#include "edit.h"
#include "const.h"
#include "menuinput.h"
#include "global.h"
#include "inputmonitor.h"
#include "macro.h"
#include "log.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * Variable which is true if we're in insert mode (ie new chars push other
 * chars to the end of the string) or in standard mode.
 */ 
bool U61_Edit::insert_mode = false;

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a 'edit' item
 *
 * Note that the buffer must be able to contain at least size+1 chars.
 */ 
U61_Edit::U61_Edit(char *label, char *val, int size) : U61_MenuItem()
{
  name=label;

  if (val!=NULL)
    {
      value=val;
    }
  else
    {
      U61_MACRO_STRCPY(buffer,"");
      value=buffer;
    }

  size_max=size;
  if (size_max>U61_EDIT_MAX_SIZE)
    {
      U61_LOG_ERROR("size too long ("<<size_max<<") for U61_Edit object");
      size_max=U61_EDIT_MAX_SIZE;
    }

  value[size_max]='\0';

  pos=0;
  update();
}

/*--------------------------------------------------------------------------*/
/*
 * updates the text displayed so that it reflects the value of the item
 */
void U61_Edit::update()
{
  char buffer[U61_CONST_STRING_SIZE];

  U61_MACRO_SPRINTF3(buffer,
		      "%s%s%s ",
		      name,
		      U61_MENUITEM_SEPARATOR,
		      value);

  set_text(buffer);
}

/*--------------------------------------------------------------------------*/
/*
 * callback for key presses
 */
bool U61_Edit::action(int key)
{
  bool changed=false;
  int len;
  int i;
  int ascii;
  int old_pos;

  len=strlen(value); 
  old_pos=pos;

  switch (key)
    {
    case CL_KEY_RIGHT:
      pos++;
      break;
    case CL_KEY_LEFT:
      pos--;
      break;
    case CL_KEY_DELETE:
      if (pos<len && len>0)
	{
	  suppress();
	  changed=true;
	}
      break;
    case CL_KEY_BACKSPACE:
      if (pos>0)
	{
	  pos--;
	  suppress();
	  changed=true;
	}
      break;
    case CL_KEY_HOME:
      if (pos>0)
	{
	  pos=0;
	  changed=true;
	}
      break;
    case CL_KEY_END:
      if (pos<len)
	{
	  pos=len;
	  changed=true;
	}
      break;
    case CL_KEY_INSERT:
      insert_mode = !insert_mode;
      break;
    case CL_KEY_ENTER:
      /*
       * Value check, this is usefull for childs which handle
       * numbers for instance.
       */
      if (!check_value())
	{
	  changed=true;
	}
      break;
    case CL_KEY_UP:
    case CL_KEY_DOWN:
      /*
       * we trap UP and DOWN for these should not generate any key
       */
    case CL_KEY_LSHIFT:
    case CL_KEY_RSHIFT:
    case CL_KEY_ALTGR:
    case CL_KEY_CAPSLOCK:
    case CL_KEY_NUMLOCK:
      /*
       * we trap the modifiers keys, for we do not want them to interfere
       */
      break;

      /*
       * Here, a "real" key has been typed
       */
    default:
      ascii=translate_to_ascii(key);
      /*
       * We exclude non-ascii characters
       */
      if (ascii>=32 && ascii<=127) 
	{
	  /*
	   * Handle the insert / non-insert modes
	   */
	  if (insert_mode)
	    {
	      for (i=len;i>=pos;--i)
		{
		  value[i+1]=value[i];
		}
	    }
	  
	  value[pos]=ascii;
	  value[size_max]=0;
	  pos++;
	  changed=true;
	}
      break;
    }

  /*
   * little hack: we remove all the spaces at the beginning of
   * the edit zone. It should avoid some (minor) problems,
   * when resolving the server name for instance
   */
  while (value[0]==' ')
    {
      pos=0;
      suppress();
      changed=true;
    }

  check_pos();

  /*
   * If the old position and the new one are different, the we force
   * an unselect/select, so that there's no problem with blinking
   */
  if (pos!=old_pos)
    {
      int sel;
      
      sel=strlen(name)+strlen(U61_MENUITEM_SEPARATOR)+pos; 

      /*
       * We disable blinking and dancing manually
       * without calling unselect for we do not
       * want the check_value() function to be called
       */
      unselect_ex();

      select(sel);
      changed=true;
    }

  return changed;
}

/*--------------------------------------------------------------------------*/
/*
 * deletes the current letter
 */
void U61_Edit::suppress()
{
  int len,i;

  len=strlen(value);
  for (i=pos;i<len;++i)
    {
      value[i]=value[i+1];
    }
}

/*--------------------------------------------------------------------------*/
/*
 * checks & corrects the pos of the cursor
 */
void U61_Edit::check_pos()
{
  int len_value;
  int len_static;

  len_static = strlen(name)+strlen(U61_MENUITEM_SEPARATOR);
  len_value = strlen(value);

  if (pos>len_value)
    {
      pos=len_value;
    }
  if (pos>=size_max)
    {
      pos=size_max-1;
    }
  if (pos<0)
    {
      pos=0;
    }

  selection=len_static+pos;
}

/*--------------------------------------------------------------------------*/
/*
 * selects the current character
 */
void U61_Edit::select(int sel)
{
  int len_static;
  int len_value;
  int old_pos;

  old_pos=pos;

  len_static = strlen(name)+strlen(U61_MENUITEM_SEPARATOR);
  len_value = strlen(value);

  if (sel<len_static)
    {
      sel=len_static;
    }

  pos=sel-len_static;
  check_pos();

  /*
   * Now this might seem stupid since the same check is performed
   * in the action function. However it's needed since select(int )
   * might also be called on a mouse move event.
   */
  if (pos!=old_pos)
    {
      unselect_ex();
    }

  /*
   * We use a "<=len..." because we know there's a space at the
   * end of the menu item.
   */
  if (sel<=len_static+len_value)
    {
      select_char(sel);
      blink_char(sel);
    }
}


/*--------------------------------------------------------------------------*/
/*
 * translate a key code to an ascii code. This function is virtual,
 * therefore sub-classes might decide to block some characters and/or
 * change the general behavior of the keyboard.
 */
int U61_Edit::translate_to_ascii(int key)
{
  int ascii;

  ascii=U61_Global::input_monitor.to_ascii(key);
  if (ascii==U61_KEYDEF_ASCII_UNKNOWN)
    {
      ascii=' ';
    }  

  return ascii;
}

/*--------------------------------------------------------------------------*/
/*
 * Value check performed when the item is de-selected or deleted.
 * Think of it as a "lost focus" event.
 */
bool U61_Edit::check_value()
{
  value[size_max]='\0';

  return true;
}











