/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000-2003 Christian Mauduit (ufoot@ufoot.org)              *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on Savannah (http://savannah.gnu.org)     *
 ****************************************************************************/

/*
 * file name:   event.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: one of the most fundamental classes of U61. An event is
 *              basically what u61 uses to control a map and send info
 *              to other computers on the network. There's a difference
 *              between U61_Event and CL_Event. CL_Event is GUI oriented
 *              for it's associated to a mouse click, a key press etc...
 *              but U61_Event is very U61 specific since it can be a
 *              drop action or a curse request.
 *              almost all events are dated.
 */

/*---------------------------------------------------------------------------
 includes
 ---------------------------------------------------------------------------*/

#include "event.h"
#include "log.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a meaningless event
 */ 
U61_Event::U61_Event()
{
  code=0;
  author=0;
  target=0;
  par=0;
  time=0; 
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of an event
 */ 
U61_Event::~U61_Event()
{

}

/*--------------------------------------------------------------------------*/
/*
 * Puts the event into a portable char buffer (for network transmission)
 */
bool U61_Event::serialize(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=false;

  if (buffer_limit-(*buffer)>=U61_EVENT_SERIALIZED_SIZE)
    {
      (*buffer)[0]=code;
      (*buffer)[1]=(author<<4) | target;
      (*buffer)[2]=0xFF & par;
      (*buffer)[3]=par>>8;
      (*buffer)[4]=0xFF & time;
      (*buffer)[5]=0xFF & time>>8;
      (*buffer)[6]=0xFF & time>>16;
      (*buffer)[7]=time>>24;

      (*buffer)+=U61_EVENT_SERIALIZED_SIZE;
      result=true;
    }
  else
    {
      U61_LOG_DEBUG("Buffer too short to serialize event!");
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Gets the event from a portable char buffer (for network transmission)
 */
bool U61_Event::unserialize(unsigned char **buffer, unsigned char *buffer_limit)
{
  bool result=false;

  if (buffer_limit-(*buffer)>=U61_EVENT_SERIALIZED_SIZE)
    {
      code=(*buffer)[0];
      target=(*buffer)[1] & 0x0F;
      author=(*buffer)[1]>>4;
      par=(((unsigned int) (*buffer)[3])<<8)
	| ((unsigned int) (*buffer)[2]);
      time=(((unsigned int) (*buffer)[7])<<24) 
	| (((unsigned int) (*buffer)[6])<<16) 
	| (((unsigned int) (*buffer)[5])<<8) 
	| ((unsigned int) (*buffer)[4]);

      (*buffer)+=U61_EVENT_SERIALIZED_SIZE;
      result=true;
    }
  else
    {
      U61_LOG_DEBUG("Buffer too short to unserialize event!");
    }

  return result;
}

/*------------------------------------------------------------------------- */
/*
 * Usefull to print an event using iostreams
 */
ostream &operator <<(ostream &o, U61_Event event)
{
  char *event_label="";

  switch (event.code)
    {
    case U61_EVENT_NOTHING :
      event_label="NOTHING";
      break;
    case U61_EVENT_READY :
      event_label="READY";
      break;
    case U61_EVENT_READY_ALL :
      event_label="READY_ALL";
      break;
    case U61_EVENT_START_STOP :
      event_label="START_STOP";
      break;
    case U61_EVENT_KILL :
      event_label="KILL";
      break;
    case U61_EVENT_NAME_LETTER :
      event_label="NAME_LETTER";
      break;
    case U61_EVENT_CHAT_LETTER :
      event_label="CHAT_LETTER";
      break;
    case U61_EVENT_CHANGE_BACKGROUND:
      event_label="CHANGE_BACKGROUND";
      break; 
    case U61_EVENT_MOVE_RIGHT :
      event_label="MOVE_RIGHT";
      break;
    case U61_EVENT_ROTATE_LEFT :
      event_label="ROTATE_LEFT";
      break;
    case U61_EVENT_ROTATE_RIGHT :
      event_label="ROTATE_RIGHT";
      break;
    case U61_EVENT_MOVE_DOWN :
      event_label="MOVE_DOWN";
      break;
    case U61_EVENT_DROP :
      event_label="DROP";
      break;
    case U61_EVENT_SET_VICTIM :
      event_label="SET_VICTIM";
      break;
    case U61_EVENT_NEXT_VICTIM :
      event_label="NEXT_VICTIM";
      break;
    case U61_EVENT_PREV_VICTIM :
      event_label="PREV_VICTIM";
      break;
    case U61_EVENT_USE_ANTIDOTE :
      event_label="USE_ANTIDOTE";
      break;
    case U61_EVENT_NEW_BLOCK :
      event_label="NEW_BLOCK";
      break;
    case U61_EVENT_LOOSE :
      event_label="LOOSE";
      break;
    case U61_EVENT_NEW_CURSE :
      event_label="NEW_CURSE";
      break;
    case U61_EVENT_REQUEST_CURSE :
      event_label="REQUEST_CURSE";
      break;
    case U61_EVENT_DO_CURSE :
      event_label="DO_CURSE";
      break;
    case U61_EVENT_HISTORY_BEGIN :
      event_label="HISTORY_BEGIN";
      break;
    case U61_EVENT_HISTORY_END :
      event_label="HISTORY_END";
      break;
    case U61_EVENT_SERIAL_BEGIN :
      event_label="SERIAL_BEGIN";
      break;
    case U61_EVENT_SERIAL_DATA :
      event_label="SERIAL_DATA";
      break;
    case U61_EVENT_SERIAL_END :
      event_label="SERIAL_END";
      break;
    case U61_EVENT_REQUEST_CHECKSUM :
      event_label="REQUEST_CHECKSUM";
      break;
    case U61_EVENT_VERIFY_CHECKSUM :
      event_label="VERIFY_CHECKSUM";
      break;
    default :
      event_label="UNKNOWN";
      break;
    }

  return o<<"code="<<event.code
	  <<" ("<<event_label<<")"
	  <<" / author="<<event.author
	  <<" / target="<<event.target
	  <<" / par="<<event.par
	  <<" / time="<<event.time;
}

